package Utils::CatalogiaUtils;
use strict;

use utf8;

use base qw(Exporter);
use Time::HiRes qw(usleep gettimeofday tv_interval);
use POSIX qw(strftime);
use Encode qw(encode_utf8);

use Utils::LightCommon;

our @EXPORT = (
    'create_subchunks',
    'glue_subchunks'
);

our @EXPORT_OK = (
    'save_query',
    'set_log_file',
    'close_log_file',
    'print_err',
    'check_alive',
    'handle_errors',
);

# разрезаем запрос на чанки меньшего размера (для распараллеливания)
sub create_subchunks {
    my ($path, $fn, $max_chunk_size, $data) = @_;
    my $num_subchunks = 0;
    my $curr_fn = "";
    my $result = [];
    my $n = 0;

    for my $line (@$data) {
        if(!$curr_fn) {
            $num_subchunks++;
            $curr_fn = $fn."_$num_subchunks";
            open F, "> $path/$curr_fn"  or die "Cannot open $path/$curr_fn for writing ($!)";
        }

        print F $line;
        $n++;

        my $is_last_line = $n == @$data ? 1 : 0;
        if(!($n % $max_chunk_size) or $is_last_line) {
            close F;
            rename("$path/$curr_fn", "$path/in/$curr_fn") or die qq[Cannot rename("$path/$curr_fn", "$path/in/$curr_fn") ($!)];
            chmod(0777, "$path/in/$curr_fn") or die qq[Cannot chmod(0777, "$path/in/$curr_fn") ($!)];
            $curr_fn = "";
        }
    }

    return $num_subchunks;
}

# склеиваем ответ на запрос
sub glue_subchunks {
    my ($path, $fn, $num_subchunks, $timeout, $on_timeout, %prm) = @_;
    my $start = time; # TODO   не округлять до целого числа секунд?
    my $result = [];
    my @input_files = map { "$path/in/$fn"."_$_" } (1 .. $num_subchunks);
    my @output_files = map { "$path/out/$fn"."_$_" } (1 .. $num_subchunks);
    for my $curr_fn (@output_files) {
        until(-e $curr_fn) {
            my $now = time;
            if($now - $start >= $timeout) {
                if($on_timeout && $on_timeout eq "http_error") {
                    print "Status: 504 Gateway Timeout\n\nTIMEOUT $now $start $timeout\n";
                } else {
                    print "content-type: text/plain\n\nTIMEOUT $now $start $timeout\n";
                }

                # Удаляем @input_files, т.к. результат их обработки больше не нужен
                unlink $_ for grep { -e $_ } @input_files;

                if ($prm{dont_exit_on_error}) {
                    # TODO сделать это поведение дефолтным
                    return;
                } else {
                    exit(0);
                }
            }
            usleep(10_000);
        }
        open F, $curr_fn  or die "Cannot open $curr_fn ($!)";
        push @$result, $_ for <F>;
        close F;
        unlink $curr_fn;
    }

    return $result;
}

my $fh_log;
sub set_log_file {
    my ($log_path) = @_;
    open $fh_log, ">> $log_path"
        or return;
    STDERR->fdopen($fh_log, 'w');
    $fh_log->autoflush(1); # Нужно для корректного парсинга лога для отправки данных в Графит  TODO - ???
    return 1;
}
sub close_log_file {
    close $fh_log;
    $fh_log = undef;
    return 1;
}

# Напечатать строку в лог-файл (заданный в set_log_file)
sub print_log_to_file {
    my ($line) = @_;

    return unless $fh_log;

    my $time = Time::HiRes::time();
    my $microseconds = ($time - int($time)) * 1e6;
    my $now = sprintf("%s.%06.0f", strftime("%Y-%m-%d %H:%M:%S", localtime($time)), $microseconds);
    print $fh_log join("\t", $now, "[$$]", $line)."\n";
}

# Сохранить запрос в файл (например, для использования позже при тестировании)
sub save_query {
    my ($data, %prm) = @_;
    my $dir = $prm{dir} || die "Void dir!";
    my $filename = $prm{filename} || die "Void filename!";

    my $date_now = strftime("%Y-%m-%d", localtime);
    my $dir_for_saved_queries = "$dir/$date_now";
    system("mkdir -p $dir_for_saved_queries") and do {
        print_log_to_file("WARN: Could not `mkdir -p $dir_for_saved_queries` $? ($!)");
        return;
    };
    open my $fh_save, ">", "$dir_for_saved_queries/$filename"  or do {
        print_log_to_file("WARN: Could not open $dir_for_saved_queries/$filename ($!)");
        return;
    };

    print $fh_save "$_" for @$data;
    close $fh_save  or do {
        print_log_to_file("WARN: Could not close $dir_for_saved_queries/$filename ($!)");
        return;
    };
    return 1;
}

sub process_query {
    my @args = @_;

    my $text_result = eval { _process_query(@args) };
    if ($@) {
        print_err("ERROR: _process_query failed ($@)");
        print "Status: 503 Internal error\n\n";
        print_log_to_file("ERROR: _process_query failed ($@)");
        return;
    }

    print "content-type: text/plain; charset=\"UTF-8\"\n\n";
    print $text_result;
    return 1;
}

sub _process_query {
    my ($max_chunk_size, $path, $log_path, %prm) = @_;
    my $input_data = $prm{input_data};  # Если input_data не задано, будем читать из STDIN
    my $save_queries = $prm{save_queries};
    my $start_time = $prm{start_time};
    my $prefix = $prm{prefix} // '';
    my $timeout = $prm{timeout} // 13;   # TODO - default timeout - ?
    my $on_timeout = $prm{on_timeout} // 'http_error';
    my $glue_subchunks_prm = $prm{glue_subchunks_prm} // {};
    my $queue = $prm{queue};

    set_log_file($log_path);

    my $ts = Time::HiRes::time();

    unless ($input_data) {
        $input_data = [];
        while (<STDIN>) {
            push @$input_data, $prefix . $_;
        }
    }
    my $length = 0;
    $length += length($_) for @$input_data;
    my $n_items = @$input_data;
    my $fn = join("_", "q", ($queue || ()), $ts, $$);

    my $num_subchunks = create_subchunks($path, $fn, $max_chunk_size, $input_data);

    if ($save_queries) {
        eval { save_query($input_data, dir => "$path/saved_queries/", filename => $fn ) };
        if ($@) {
            print_log_to_file("ERROR: save_query failed ($@)");
        }
    }

    my $prefix_out = $prefix || 'empty';
    $prefix_out =~ s/\s+$//g; # Т.к. $prefix может оканчиваться на \t
    my $env_HTTP_X_FORWARDED_FOR = $ENV{HTTP_X_FORWARDED_FOR} // '';
    my $log_str = "ADDR: $ENV{REMOTE_ADDR} HTTP_X_FORWARDED_FOR: $env_HTTP_X_FORWARDED_FOR PREFIX:$prefix_out LENGTH:$length NITEMS:$n_items NCHUNKS:$num_subchunks TIMEOUT:$timeout";
    print_log_to_file("START $log_str");

    my $result = glue_subchunks($path, $fn, $num_subchunks, $timeout, $on_timeout, %$glue_subchunks_prm);
    my $duration = tv_interval($start_time);
    my $state = $result ? 0 : 1; # state=0 - OK.  glue_subchunks возвращает undef в случае timeout

    my $time_per_item = $n_items  ?  $duration / $n_items  :  undef;
    print_log_to_file("DONE  $log_str  STATE:$state TIME:$duration " . (defined $time_per_item ? " TIME_PER_ITEM:$time_per_item" : ""));

    close_log_file();

    unless ($result) {
        die "Void result!";
    };

    my $text_result = join("", @$result);
    return $text_result;
}

sub check_alive {
    my ($prm) = @_;

    # TODO сделать use, если это не замедлит get_*categs.pl
    require LWP::UserAgent;

    my $is_alive;
    # проверка функционала
    my $host = $prm->{host} // 'localhost';
    my $uri = $prm->{uri} // '';
    my $ua = LWP::UserAgent->new(timeout => $prm->{timeout});
    my $resp = $ua->post("http://$host$uri", Content => Encode::encode_utf8($prm->{content}));
    if($resp->is_success) {
        my $result = $resp->decoded_content;
        #print_err("result: $result");
        if ($prm->{check_re}) {
            if ($result =~ $prm->{check_re}) {
                $is_alive = 1;
            } else {
                print_err("Bad result: '$result'");
            }
        } else {
            $is_alive = 1;
        }
    }
    $is_alive //= 0;
    print_err("check_alive: " . join(" ", map { "$_: " . ($prm->{$_} // 'UNDEF') } qw[ host uri timeout content check_re ]) . " is_alive: $is_alive");
    return $is_alive;
}

sub check_cached_alive {
    my ($server_options) = @_;

    my $monitor_file = $server_options->{monitor_file} || die "Void monitor_file";
    #print_err("check_cached_alive $monitor_file");
    my $max_allowed_monitor_file_delay = $server_options->{max_allowed_monitor_file_delay}; # seconds  Максимально допустимый возраст файла $monitor_file  # TODO decrease

    unless (-f $monitor_file) {
        die "$monitor_file does not exist";
    }

    my $mtime = mtime($monitor_file);
    if (time - $mtime > $max_allowed_monitor_file_delay) {
        die "Too old $monitor_file ($mtime)";
    }

    open my $fh, "< $monitor_file" or die "Could not open $monitor_file ($!)";
    my @lines = map { chomp; $_ } <$fh>;
    close $fh or die "Could not close $monitor_file ($!)";

    my ($is_alive) = map { (split /: /)[1] } grep { m/^is_alive: / } @lines;

    return $is_alive;
}


sub mtime {
    my ($file) = @_;  # filename or filehandle
    my @stat = stat $file;
    return undef if not @stat;
    return $stat[9];
}

sub log_time_fmt {
    return POSIX::strftime("%Y-%m-%d %H:%M:%S", localtime);
}

sub log_msg_fmt {
    my $time = log_time_fmt();
    return join("\t", $time, "[$$]", shift);
}

sub print_err {
    print STDERR log_msg_fmt(shift), "\n";
    return 1;
}

sub handle_errors {
    my (%prm) = @_;

    $SIG{__DIE__} = sub {
        my $msg = log_msg_fmt('ERROR: died: '.shift);
        die($msg);
    };

    $SIG{__WARN__} = sub {
        my $msg = log_msg_fmt('WARN: '.shift);
        warn($msg);
    };

    $SIG{TERM} //= sub {
        print_err("ERROR: SIG{TERM} was received! Exiting...");
        exit(1);
    };

    $SIG{INT} //= sub {
        print_err("ERROR: SIG{INT} was received! Exiting...");
        exit(1);
    }
}

1;
