package Utils::Common;

use strict;
use utf8;

use Sys::Hostname;

use Utils::LightCommon;
use Utils::Hosts;
use Utils::Funcs qw(patch_hash get_python_options);

use BM::BannersMaker::Options;

BEGIN {
    eval {
        require JSON;
        JSON->import();
    };
    eval {
        require JSON::XS;
        JSON::XS->import();
    } if $@;
    eval {
        require JSON::PP;
        JSON::PP->import();
    } if $@;
    die $@ if $@;
    if (0) {
        *decode_json = sub{die};  # Fix static check
    }
}

# Конфиг (хэш %$options) формируется следующим образом:
# 1) собираем конфиг из нескольких источников:
#   (при конфликте разных конфигов делается die)
#   - light - базовые опции, для веб-серверов, без импортов
#   - Common - общие для всех опции, задаваемые здесь и в CommonCluster
#   - конфиги разных проектов:
#       * BannerLand
#       * ...
#   - новые конфиги:
#       * или конфиг из внешнего json файла
#       * или конфиги из python-модулей
# 2) применяем патчи:
#   (могут переопределять уже имеющиеся опции)
#   - патчи для конкретных хостов/ролей, задаваемые в этом модуле
#   - произвольные патчи из patchfile и $ENV{BM*PATCH*}

our ($options);

my $dirs = $Utils::LightCommon::dirs;
my $light_options = $Utils::LightCommon::options;

my @add_options;
push @add_options, $light_options;
push @add_options, BM::BannersMaker::Options::get_options();

my $bm_options_json_file = $ENV{BM_OPTIONS_JSON_FILE};
if ($bm_options_json_file) {
    require File::Slurp;
    # новый конфиг: из json-файла
    # задаётся, например, в BMYT
    my $opts_jsondoc = File::Slurp::read_file($bm_options_json_file);
    my $opts = decode_json($opts_jsondoc);
    push @add_options, $opts;
} else {
    # Конфиги из Питонячих модулей
    push @add_options, get_python_options($dirs->{scripts} . '/bannerland_options/bannerland_options', 0);
    push @add_options, get_python_options($dirs->{scripts} . '/irt_common_options/irt_common_options', 0);
}

my $BM_MDB_HOSTS = ['c-mdbdl4e1lh1lqge3t0h3.rw.db.yandex.net', 'sas-6ant1ybleuvfhsdy.db.yandex.net', 'vla-gj2vrppbvhaosczx.db.yandex.net'];
my $BM_MDB_PASSPATH = $dirs->{secrets}. "/passwords/mysql-prod-mdb_password";

$options = {
    # TODO: (modest) оторвать костыль
    'bannerland_perf_kostyl_order2feedurl' => {
        # orderid => feed_url
        160150180 => 'https://market.feed/?site=kupimebel.ru&business_id=1048031&shop_id=1048030',
        160244845 => 'https://market.feed/?site=magic-svet.ru&business_id=1048031&shop_id=1048045',
        160245650 => 'https://market.feed/?site=svetlux.ru&business_id=1025853&shop_id=1025915',
        160251819 => 'https://market.feed/?site=frontmaster.su&business_id=1048031&shop_id=1051890',
    },

    ansible_mark_file => '/var/spool/ansible/marks',

    cmdlog => $dirs->{log}.'/cmdlog',

    dicts => $dirs->{dicts},

    scripts => $dirs->{scripts},

    bin => $dirs->{bin},

    work => $dirs->{work},
    temp => $dirs->{temp},

    host_info_files => {
        role => $dirs->{work}."/host_role",
    },

    lru_cache => {
        MaxBytes => 100000000,
    },

    # дамп некоторых таблиц
    db_dump => {
        bannerland => {
            dbh_name => 'bannerland_dbh',
            tables => [qw(
                PerfPremiumGoodsDomains
                PerfUtmProblemDomains
                PerfAllowCustomPhrases
                PerfCustomTemplates
                PerfParamMapping
            )],
            producer => 'bannerland:master',
            project => 'DYNAMICBANNERS',
        },
    },

    Monitor => {
        cdict_monitor           => "$dirs->{work}/cdict-monitor",

        update_test_db_mark_file => "$dirs->{work}/update_test_db.success",
    },

    # вливание фраз
    infuse_web_result_file => $dirs->{temp} . "/infuse_web_result",

    # перезапуск сервисов
    RestartServices_params => {
        log_file    => $dirs->{temp} . "/restart_services_log",
    },


    switchman_client_params => {
        admins_config => '/etc/switchman.conf',
        params => {
            prefix   => '/broadmatch',
            logfile  => $dirs->{log}.'/switchman.log',
            loglevel => 'debug',
        },
    },
    yt_client_params => {
        raw_data_dir => $dirs->{raw_data}.'/YTClient',
        temp_dir => $dirs->{temp}."/YTClient",
        work_dir => $dirs->{work}."/YTClient",
        bin => "/usr/bin/yt",
        params => {
            # cluster, proxy, token_path => см. ниже
            tries               => 1,
            sleep_between_tries => 0,
        },
        bm_events_script => $dirs->{scripts}."/stat/yt_tasks/bm_chevent_hourstat.py --use-banner-phrase-stat",
        bm_banners_script => $dirs->{scripts}."/stat/yt_tasks/bm_banners.py",
    },

    site_tree_fix => $dirs->{dicts} . '/site_tree_fix',
    site_filters  => $dirs->{dicts} . '/site_filters',
    site_urifix   => $dirs->{dicts} . '/site_urifix',
    site_name_prefilter => $dirs->{dicts} . '/site_name_prefilter',
    site_subsections => $dirs->{dicts} . '/site_subsections',
    site_subsections_re => $dirs->{dicts} . '/site_subsections_re',
    site_homonymy_names => $dirs->{dicts} . '/site_homonymy_names',
    site_region_domains => $dirs->{dicts} . '/site_region_domains',

    # Словарь префильтров
    dict_prefilters => $dirs->{dicts} . '/dict_prefilters',
    #словарь флагов модерации для доменов
    dict_domain_flags => $dirs->{dicts} . '/dict_domain_flags',

    #Данные о пользователях для CatMedia
    auth_users => {
        dbhname => 'catalogia_media_dbh',
        table   => 'Users',
        group_table => 'UsersGroups',
        log_table => 'UsersVisitsLog',
    },

    # db_uploader
    db_uploader_params => {
        temp_dir                     => $dirs->{temp} . '/DbUploader',
        work_dir                     => $dirs->{work} . '/DbUploader',
    },

    # broad_kyoto cache
    broad_kyoto => {
#	    servers  => [{ address => 'bmcache01e.yandex.ru:1115'}], # переезд h в Ивантеевку, временно включам e
#	    servers  => [{ address => 'bmcache01h.yandex.ru:1115'}],
        servers  => [{ address => 'bmcache01i.ppc.yandex.ru:1115'}],
            max_size => 32 * 1024 * 1024,        # it means 16M not 30M cause of base-64 encoding
            keys_pack => 50,			 # how many keys to be processed once
    },

    # kyoto cache
    kyoto => {
            #servers =>[{ address => 'bmcache01f.yandex.ru:1115'}],
            servers =>[{ address => 'bmcache01e.yandex.ru:1115'}],
            max_size => 500 * 1024 * 1024, # 500M max size of record
            salt     => "salt_2015_11_19 18:56", # salt
    },

    PrefProjSrv_params => {
        pid_file            => $dirs->{lock} . "/prefprojsrv.pid",
        port                => 20202,
        host                => '127.0.0.1',
        num_forks           => 25,
        min_free_forks      => 3,
        client_timeout      => 40,
    },

    BaseHttpServer_params => {
        pid_file            => $dirs->{lock} . "/basehttp.pid",
        port                => 20303,
        num_forks           => 50,
    },

    # HTTP поведнческого
    BmHttp_params => {
        pid_file            => $dirs->{lock} . "/BmHttp.pid",
        port                => 20303,
        host                => '127.0.0.1',
        num_forks           => 20,
    },

    # UserAgent
    UserAgent_params => {
        # здесь пусто, но некоторые флаги сюда ставятся из CommonPatches
    },

    # zora-connections
    zora_params => {
        command           => '/usr/local/bin/zoracl fetch --input=- --output=- --max-redirects=9 --priority=0 --ignore-low-priority=yes --redirect-mode=last --freshness=14400 --source=drf --send-quota=1000',
        multiget_command  => '/usr/local/bin/zoracl fetch --format=document    --max-redirects=9 --priority=0 -E --ignore-low-priority=yes --redirect-mode=last --freshness=14400 --source=drf --send-quota=1000',
        default_timeout   => 14000,
        temp_dir => $dirs->{temp} . '/Zora',
        work_dir => $dirs->{work} . '/Zora',
        tvm => {
            clientid => '2011076',
            secret_locations => [
                $dirs->{root}.'/secrets/tvm/app_zoracl_secret',
                '/home/broadmatching/.zoracl/tvm_secret', #todo: оторвать, когда bmapi будет переведен на доставку в первое место
            ],
        },
    },

    zora_fast_params => {
        command           => '/usr/local/bin/zoracl fetch --input=- --output=- --max-redirects=9 --priority=0 --ignore-low-priority=yes --redirect-mode=last --freshness=14400 --source=dynbnr --send-quota=1000',
        multiget_command  => '/usr/local/bin/zoracl fetch --format=document    --max-redirects=9 --priority=0 -E --ignore-low-priority=yes --redirect-mode=last --freshness=14400 --source=dynbnr --send-quota=1000',
        default_timeout   => 14000,
        temp_dir => $dirs->{temp} . '/ZoraFast',
        work_dir => $dirs->{work} . '/ZoraFast',
        tvm => {
            clientid => '2011076',
            secret_locations => [
                $dirs->{root}.'/secrets/tvm/app_zoracl_secret',
                '/home/broadmatching/.zoracl/tvm_secret', #todo: оторвать, когда bmapi будет переведен на доставку в первое место
            ],
        },
    },

    catalogia_media_db_inf => {
        hosts => $BM_MDB_HOSTS,
        database => 'bm',
        port => 3306,
        password_path=>$BM_MDB_PASSPATH,
        user => 'bmclient',
        utf8 => 1,
    },

    dbAttr   => {
        RaiseError => 1,
        PrintError => 1,
        AutoCommit => 1,
        mysql_enable_utf8 => 1,
        mysql_init_command => q{SET NAMES 'utf8'},
        #mysql_multi_statements  => 1,
        #mysql_init_command => q{SET NAMES 'utf8'; SET CHARACTER SET 'utf8'},
    },

    monitoring_db_inf => {
        hosts => $BM_MDB_HOSTS,
        database => 'monitoring',
        port => 3306,
        password_path=>$BM_MDB_PASSPATH,
        user => 'bmclient',
        utf8 => 1,
    },

    dirs => $dirs,

    RUserClicks_params => {
        raw_data_dir        =>  $dirs->{raw_data}.'/RUserClicks',
        temp_dir            =>  $dirs->{temp}.'/RUserClicks',
        work_dir            =>  $dirs->{work}.'/RUserClicks',
        weighted_graph      =>  $dirs->{work}.'/RUserClicks/weighted_graph',

        result_nonsyn_gz1   =>  $dirs->{simpgraphs} . '/RUserClicks1.nonsyn_simpgraph.gz',
        result_nonsyn_gz2   =>  $dirs->{simpgraphs} . '/RUserClicks2.nonsyn_simpgraph.gz',
        result_nonsyn_gz3   =>  $dirs->{simpgraphs} . '/RUserClicks3.nonsyn_simpgraph.gz',
        result_nonsyn_gz_sergio   =>  $dirs->{simpgraphs} . '/RUserClicks_sergio.nonsyn_simpgraph.gz',
        result_nonsyn_gz90  =>  $dirs->{simpgraphs} . '/RUserClicks90.nonsyn_simpgraph.gz',
        result_syn_gz       =>  $dirs->{simpgraphs} . '/RUserClicks.simpgraph.gz',
        result_syn_gz1      =>  $dirs->{simpgraphs} . '/RUserClicks1.simpgraph.gz',
        result_syn_gz_sergio      =>  $dirs->{simpgraphs} . '/RUserClicks_sergio.simpgraph.gz',
        result_syn_gz2      =>  $dirs->{simpgraphs} . '/RUserClicks2.simpgraph.gz',
        result_syn_gzf      =>  $dirs->{simpgraphs} . '/RUserClicksf.simpgraph.gz', # last day filtered simpgraph
        result_syn_gz3      =>  $dirs->{simpgraphs} . '/RUserClicks3.simpgraph.gz',
        result_syn_gz90     =>  $dirs->{simpgraphs} . '/RUserClicks90.simpgraph.gz',

        # RUserClicks + RBindings = UCB
        result_nonsyn_ucb_gz    =>  $dirs->{simpgraphs} . '/UCB.nonsyn_simpgraph.gz',
        result_nonsyn_ucb_gzf   =>  $dirs->{simpgraphs} . '/UCBf.nonsyn_simpgraph.gz',
        result_syn_ucb_gz   =>  $dirs->{simpgraphs} . '/UCB.simpgraph.gz',

        min_vertex_weight   =>  5,
        max_friends         =>  2000,	   # top по частоте
        max_pairs           =>  1000,	   # сколько пар выгружаем максимально из профиля
        max_associations    =>  80,        # сколько максимально ассоциаций в симпграфе
        max_advq_count      =>  1_000_000, # максимальное по частоте из advq - фразы чаще не используются DEPRECATED!!!
        max_files_to_leave  =>  120,
        max_dates_to_process_once => 10,
        first_day_ago       =>  30,
    },

    # настройки для BroadmatchDirectPhrases
    BroadmatchDirectPhrases_params => {
        temp_dir                     => $dirs->{temp} . '/BroadmatchDirectPhrases',
        work_dir                     => $dirs->{work} . '/BroadmatchDirectPhrases',

        phrases_http_source          => 'http://yacotools/cgi-bin/direct_phrases_new.py',                           # путь к фразам лиректа

        harmonized_file              => $dirs->{work} . "/QueryLogStat/norm_orig_ru.gz",
        phrases_filename             => $dirs->{work} . '/BroadmatchDirectPhrases/direct_phrases.txt.gz',              # исходный файл с фразами директа
        phrases_broadmatched_filename=> $dirs->{work} . '/BroadmatchDirectPhrases/direct_phrases_broadmatched.txt.gz', # файл с broadmatched-фразами
        united_simpgraph             => $dirs->{work} . '/BroadmatchDirectPhrases/united_simpgraph',                   # объединенный симпграф
        united_simpgraph_extended    => $dirs->{work} . '/BroadmatchDirectPhrases/united_simpgraph_extended',          # объединенный симпграф с весами без фимльтрации по фразам директа
        united_simpgraph_context_syn => $dirs->{work} . '/BroadmatchDirectPhrases/united_simpgraph_context_syn',       # объединенный симпграф с контекстными синонимами
        united_simpgraph_user_clicks => $dirs->{work} . '/BroadmatchDirectPhrases/united_simpgraph_user_clicks',       # объединенный симпграф с пользовательскими кликами
        united_simpgraph_extended_exp1    => $dirs->{work} . '/BroadmatchDirectPhrases/united_simpgraph_extended_exp1',          # объединенный симпграф с весами без фимльтрации по фразам директа


        united_simpgraph_winter      => $dirs->{work} . '/BroadmatchDirectPhrases/united_simpgraph_user_clicks.good',
    },

    RunItConfig => {
        services_configs_dir    => $dirs->{configs},
        services_with_configs   => [ qw(apache2 nginx) ], # запускаются только на тех ролях, для которых есть конфиги
        services_without_config => [ qw(atop haproxy) ],  # запускаются на всех хостах
        host_specific_services  => {
            deploy => [ qw(graphite rsyslog) ],
            qloud => [ qw(cron graphite) ],
            iron => [ ],
        }
    },

    # BM::CateforiesTree и BM::PhraseCategs
    CategoriesTree_params => {
        # здесь пусто, но некоторые флаги сюда ставятся из CommonPatches
    },

    # categs_mapper
    # когда-то это было частью bmcollect'а
    CategsMapper_params => {
        queries_reg_serp_month_stat_compressed  => $dirs->{work} . '/CategsMapper/queries_reg_serp_month_stat_compressed',
    },

    yt_direct_banners_mapping => [
        { yt_field => 'bid', mysql_field => 'id' },
        { yt_field => 'cid', mysql_field => 'campaign_id' },
        { yt_field => 'uid', mysql_field => 'uid' },
        { yt_field => 'title', mysql_field => 'title' },
        { yt_field => 'body', mysql_field => 'body' },
        { yt_field => 'yacontextCategories', mysql_field => 'categories' },  # dummy empty field
        { yt_field => 'geo', mysql_field => 'targetting' },
        { yt_field => 'phrases', mysql_field => 'phrases' },
        { yt_field => 'href', mysql_field => 'url' },
        { yt_field => 'is_active', mysql_field => 'active_flag' },
        { yt_field => 'BannerID', mysql_field => 'bs_banner_id' },
        { yt_field => 'OrderID', mysql_field => 'bs_order_id' },
        { yt_field => 'lang', mysql_field => 'lang_old' },
        { yt_field => 'domain', mysql_field => 'site_domain' },
        { yt_field => 'pid', mysql_field => 'group_id' },
        { yt_field => 'is_moderated', mysql_field => 'is_moderated' },
        { yt_field => 'image_flag', mysql_field => 'image_flag' },
        { yt_field => 'is_show', mysql_field => 'status_show' },
        { yt_field => 'is_archived', mysql_field => 'status_archived' },
        { yt_field => 'banner_type', mysql_field => 'banner_type' },
    ],

    # менеджер словарей
    DictManager_params => {
    },
    # Настройки для DictOptions.pm
    DictOptions_params => {
        biwords_source => "cat " . join(" ", map { $dirs->{dicts}."/$_" } qw[ dict_biwords_auto ] ) . " |",
        biwords_strict_source => "cat " . join(" ", map { $dirs->{dicts}."/$_" } qw[ dict_biwords dict_biwords_geo ] ) . " |",
    },

    # перегенерация словарей для нормализации слов (norm, snorm)
    DictNorm => {
        work_dir                     => $dirs->{work}.'/Norm',
        temp_dir                     => $dirs->{temp}.'/Norm',

        # данные для построения norm_dict
        add_wordforms_source         => 'zcat '.join(' ', map { "$dirs->{bin_dicts}/normalize/gen/$_" } "freq_hyphen_wordforms.gz", "tr_wiki_wordforms.gz", "tr_tatiana_wordforms.gz").'|',
        lemmer_bin                   => $dirs->{bin_dicts}.'/normalize/lemmer-test-debian-64-2014.01',
        frequency_dict               => $dirs->{dicts}.'/normalize/frequency_dict.csv',
        pairs_AS_dict                => $dirs->{dicts}.'/normalize/pairs_AS',
        pairs_AS_fix                 => $dirs->{dicts}.'/normalize/pairs_AS.fix',
        tr_suffixes                  => $dirs->{dicts}.'/tr_suffixes',
        lemmer_fix_source            => 'cat '.join(' ', map { $dirs->{dicts}.'/normalize/'.$_ } qw(lemmer_fix bm_lemmer_fix tr_lemmer_fix lemmer_fix_web lemmer_fix_web_tr)).'|',
        lemmer_fix_ru                => $dirs->{dicts}.'/normalize/lemmer_fix',
        lemmer_fix_tr                => $dirs->{dicts}.'/normalize/tr_lemmer_fix',

        badfirstparts       => $dirs->{dicts}.'/normalize/dict_compound_badfirstparts',

        # данные для построения syn_cells
        synonyms => [
            (map { $dirs->{bin_dicts}.'/syns/'.$_ } qw(
                translit-good-201011.gz
            )),
            (map { $dirs->{dicts}.'/syns/'.$_ } qw(
                dict_translit_add
                dict_syns_changelayout
                dict_eng_singular_and_plural
                dict_translit_moria
	    )),
            # Транслит из Moria - см. ниже
            map { $dirs->{dicts}.'/'.$_ } qw(
                dict_deriv
                dict_cars
                dict_synonyms
                dict_synonyms_geo2adj
                dict_wikisyn
                dict_orfovars_201011

                dict_translate_add
                dict_deriv_add
            )
        ],
        misprints_source_dir         => $dirs->{root}.'/../bmimport',
        reask_file                   => $dirs->{work}.'/Norm/REASK.sug.gz',
        misprints_add                => $dirs->{dicts}.'/dict_misprints_add',
        misprints_bad                => $dirs->{dicts}.'/dict_misprints_bad',
        synonyms_badwords            => $dirs->{dicts}.'/dict_synonyms_badwords',
        synonyms_badpairs            => $dirs->{dicts}.'/dict_synonyms_badpairs',
        synonyms_badpairs_tr         => $dirs->{dicts}.'/dict_synonyms_badpairs_tr',

        synonyms_tr => [
            $dirs->{dicts}.'/dict_synonyms_tr',
        ],

        # словари для синонимов, добавленных через веб-интерфейс
        synonyms_web_ru => $dirs->{dicts}.'/dict_synonyms',
        synonyms_web_tr => $dirs->{dicts}.'/dict_synonyms_tr',

        # словари для слов из lemmer-fix, добавленных через веб-интерфейс
        lemmer_fix_web_ru => $dirs->{dicts}.'/normalize/lemmer_fix_web',
        lemmer_fix_web_tr => $dirs->{dicts}.'/normalize/lemmer_fix_web_tr',

        # файлы с доп. информацией о нормализаторе
        info => {
            syn_ru => $dirs->{work}.'/Norm/syn_cells_ru.index',
            syn_tr => $dirs->{work}.'/Norm/syn_cells_tr.index',
        },
    },
    DictSynonyms => {
        moria_file => $dirs->{bin_dicts}.'/syns/moria-ru-2015.txt.gz',
        moria_url => 'https://arcadia.yandex.ru/arc/trunk/arcadia_tests_data/wizard_source_data/thesaurus/txt/moria/ru.txt.gz',

        work_dir  => $dirs->{work}.'/Synonyms',

        translit_raw_file => $dirs->{work}.'/Synonyms/translit_raw.tsv',
        translit_file => $dirs->{work}.'/Synonyms/translit.tsv',
        translit_file_frozen => $dirs->{dicts}.'/syns/translit.tsv', # Закоммиченный файл work/Synonyms/translit.tsv (последнее обновление файла - Aug 25 2016)
    },
    DictCSyns => {
        work_dir  => $dirs->{work}.'/CSyns',
        temp_dir  => $dirs->{temp}.'/CSyns',
        smap_translit_file => $dirs->{gen_dicts}.'/csyns_translit.smap',
        smap_translit_ru2en_file => $dirs->{gen_dicts}.'/csyns_translit_ru2en.smap',
        smap_translit_en2ru_file => $dirs->{gen_dicts}.'/csyns_translit_en2ru.smap',
        smap_translit_en2ru_strict_file => $dirs->{gen_dicts}.'/csyns_translit_en2ru.strict.smap',
    },
    # опции для генератора словаря словоформ
    DictGrammar => {
        file => $dirs->{gen_dicts}.'/dict_wordforms',
        raw_data_dir    => $dirs->{raw_data}.'/Market',
        temp_dir        => $dirs->{temp}.'/Grammar',
    },

    # обновление словаря эквивалентных доменов
    DictEquivDomains => {
        work_dir                     => $dirs->{work}.'/EquivDomains',
        temp_dir                     => $dirs->{temp}.'/EquivDomains',
        file                         => $dirs->{gen_dicts}.'/dict_equiv_domains',
        mirrors_src                  => 'ya.yandex.ru:/berkanavt/bin/data/mirrors.res',  # assume rsync
    },
    # обновление словаря трагичных слов
    DictTragicword => {
        url => 'http://bssoap.yandex.ru/export/export_tragicword.cgi',
        file => $dirs->{gen_dicts}.'/dict_tragicword',
        temp_dir => $dirs->{temp},
    },
    # обновление словаря особо отмеченных фраз
    DictDirectMod => {
        soap_params => {
            uri    => 'SOAPDaemon',
            proxy  => 'http://intapi.direct-mod.yandex.ru:7080/soap',
            method => 'getHighlightedPhrases',
        },
        file_full => $dirs->{gen_dicts}.'/dict_directmod_full',  # с комментариями
        file      => $dirs->{gen_dicts}.'/dict_directmod',
        temp_dir  => $dirs->{temp},
    },
    # многословные синонимы
    DictMulWordSyns => {
        misspell_file     => $dirs->{gen_dicts}.'/mulword_syns_misspell',
    },

    # омонимы
    Homonyms_params => {
        file          => "$dirs->{dicts}/dict_context_homonymy",
    },

    Capitalization_params => {
        file_ru         => "$dirs->{dicts}/casevcb_filtered_ru",
        file_tr         => "$dirs->{dicts}/casevcb_filtered_tr",
    },

    Collapsed_words_params => {
        file_ru         => "$dirs->{dicts}/dict_collapsed_words_ru",
    },

    # контекстные синонимы - словари
    ContextSyns => {
        Hyperonyms => {
            file => "$dirs->{dicts}/dict_hyperonyms",
        },
        MulWordSyns => {
            # см. ниже
        },

        # BM::ContextSyns::ContextLinks
        precise => {
            dicts => [qw[
                dict_csyns_precise dict_csyns_precise_web
                dict_csyns_precise_mulwords
                dict_csyns_precise_abbrev
                dict_csyns_precise_brandassoc
                dict_csyns_wiki_precise

                dict_csyns_compoundwords_brands
                dict_csyns_compoundwords_banners

            ]],
            store_dict_names => 1,
        },
        precise_tr => {
            dicts => [qw[ dict_csyns_precise_web_tr ]],
            lang => 'tr',
            store_dict_names => 1,
        },
        careful => {
            dicts => [ 'dict_csyns_misc-careful' ],
            store_dict_names => 1,
        },
        medium => {
            dicts => [ qw(
                dict_csyns_medium
            ) ],
            store_dict_names => 1,
        },
        medium_exp => {
            dicts => [qw[
                dict_csyns_medium_exp
                dict_csyns_medium_moria
                dict_csyns_compoundwords
                dict_csyns_mulwords
                dict_csyns_compoundwords_misc
                dict_csyns_compoundwords_adjective
            ]],
            store_dict_names => 1,
        },
        medium_tr => {
            dicts => [qw[ dict_csyns_medium_tr ]],
            lang => 'tr',
            store_dict_names => 1,
        },
        brands_assoc => {
            file => $dirs->{dicts}.'/csyns/dict_csyns_brands_assoc',
        },
        models => {
            # см. ниже
        },
        diacr => {
            file => $dirs->{dicts}.'/csyns/dict_csyns_diacr',
        },
        orfovars_tr => {
            file => $dirs->{dicts}.'/dict_orfovars_tr',
            lang => 'tr',
        },
        translit_ru2en => { parse_params => { expand_main => 1 } },  # файлы см. ниже
        translit_en2ru => { parse_params => { expand_main => 1 } },  # файлы см. ниже
        translit_en2ru_strict => { parse_params => { expand_main => 1 } },  # файлы см. ниже

        merchant_sim => {
            dicts => [qw[
                dict_csyns_compoundwords_banners
                dict_csyns_translit_moria
                dict_csyns_compoundwords_merchants
                dict_csyns_translit_merchants
                dict_csyns_orfovars_merchants
            ]],
            store_dict_names => 1,
        },

        (map {
            my $type = $_;
            ( "smpg_$type" => {
                    store_dict_names => 1,
                    dicts => [ map { "dict_csyns_smpg_${_}_$type" } qw[ Tags SerpClicks HitLog Bindings QueryLogQQ Slotstat Banners QueryLog ]],
                }
            )
        } qw[ spell medium ]),

        translit_smpg => {
            dicts => [qw[ dict_csyns_translit_smpg ]],
            store_dict_names => 1,
        },

        CatmediaContextSyns => {
            dbh_name => "catalogia_media_dbh",
            table => "ContextSynFresh",
            field => "InitialPhrase",
            condition => "not Status = 'Deleted'",
            reload_always => 1,
        },
    },

    # контекстные синонимы - дополнительные параметры
    ContextSyns_params => {
        dir_dicts => $dirs->{dicts}."/csyns",
        dicts_badpairs => [map { $dirs->{dicts}."/csyns/$_" } qw[ dict_csyns_badpairs ]],

        # словари для контекстных синонимов, добавленных через веб-интерфейс
        context_synonyms_web_ru => $dirs->{dicts}.'/csyns/dict_csyns_precise_web',
        context_synonyms_web_tr => $dirs->{dicts}.'/csyns/dict_csyns_precise_web_tr',
    },

    #спам-фразы
    SpamPhrases_params => {
        spam_phrases_wide_ru => $dirs->{dicts} . '/dict_wide_spam_phrases_ru',
        spam_phrases_wide_tr => $dirs->{dicts} . '/dict_wide_spam_phrases_tr',
    },

    # медиагруппы
    MediaGroups_params => {
        name            => "mediagroups",
        file            => $dirs->{dicts} . "/mediagroups.json",
        db_categ2group  => "categs_mediagroups",
        db_categ2group_fld_group    => "mediagroup",
        db_fix          => "mediagroups_fix",
        db_brandgroups  => "brandgroups",
        db_secondlevel  => "mediagroups_second_level",
    },

    # видеоподложки
    VideodirectGroups_params => {
        name            => "videodirectgroups",
        file            => $dirs->{dicts} . "/videodirectgroups.json",
        db_categ2group  => "categs_videodirectgroups",
        db_categ2group_fld_group    => "videodirectgroup",
        db_fix          => "videodirectgroups_fix",
        #db_brandgroups  => "brandgroups",
        db_secondlevel  => "videodirectgroups_second_level",
        direct_mapping_file => $dirs->{dicts} . "/dict_videodirectgroups2directid",
    },

    # маппинг для интересов (Рекламный Дзен, CATALOGIA-1230)
    InterestsMapping_params => {
        name            => "interests_mapping",
        file            => $dirs->{dicts} . "/interests_mapping.json",
        db_categ2group  => "interests_mapping",
        db_categ2group_fld_group => "mapped_name",
    },

    # Настройки, специфичные для разных языков
    Language_params => {
        ru  => {
            atoms_source    => "cat $dirs->{dicts}/categs_atoms |",
            notwidephrs     => "cat $dirs->{dicts}/dict_not_widephrs |",
            widephrs        => "cat $dirs->{dicts}/dict_widephrs |",
            widephr_statinf => "$dirs->{dicts}/dict_widephr_statinf",
            hyphen          => "$dirs->{dicts}/Hyphen_Ru_Ru",
            dict_complex_words   => "$dirs->{dicts}/dict_complex_words",
            wide_words      => [
                $dirs->{gen_dicts}.'/models/wide_models',  # широкие модели (если соответствуют > 1 типу товара)
                map { $dirs->{dicts}.'/'.$_ } qw(
                    dict_wide
                    dict_widebrands
                    dict_namesm dict_namesw
                    dict_patronyms
                    dict_navig
                    dict_widecolours
                    dict_cities
                    dict_metrical
                    dict_ambiguous
                    dict_wide_medicine
                )
            ],
        },

        tr  => {
            atoms_source    => "cat $dirs->{dicts}/categs_atoms_tr |",
            notwidephrs     => "cat $dirs->{dicts}/dict_not_widephrs_tr |",
            widephrs        => "cat $dirs->{dicts}/dict_widephrs_tr |",
            widephrs_statinf => "cat $dirs->{dicts}/dict_widephr_statinf |",
            wide_words      => [ $dirs->{dicts} . "/dict_wide_tr" ],
            hyphen          => "$dirs->{dicts}/Hyphen_Tr",
        },

        en  => {
            hyphen          => "$dirs->{dicts}/Hyphen_En",
            widephrs_statinf => "cat $dirs->{dicts}/dict_widephr_statinf |",
        },
    },

    #Настройки работы со словами - статические
    Words_params => {
        # словари с указанием языка
        stopword_dicts  => [
            ( map { [ "$dirs->{dicts}/dict_stwords_$_", $_ ] } qw(ru en tr uk) ),
        ],

        bsstop_words    => $dirs->{dicts}.'/dict_bsstops',
        bad_words       => [
            $dirs->{dicts}.'/dict_bad',
            $dirs->{dicts}.'/dict_polit',
            $dirs->{gen_dicts}.'/dict_tragicword',
        ],

        # словари
        norm_dict           => $dirs->{gen_dicts}.'/norm_dict',
        syn_dicts => [
            # словари синонимов для разных языков
            map { [ "$dirs->{gen_dicts}/syn_cells_$_", $_ ] } qw(ru tr)
        ],
        wordcount_dict      => $dirs->{gen_dicts}.'/wordcount_dict',
        word2snorm_dict     => $dirs->{gen_dicts}.'/word2snorm_dict',
    },

    #Настройки работы с фразами
    grammar_dict        => $dirs->{dicts}.'/gramm_dict',
    notwidephrs       => "cat $dirs->{dicts}/dict_not_widephrs |",
    widephrs          => "cat $dirs->{dicts}/dict_widephrs |",
    porn_words        => "< $dirs->{dicts}/dict_porn",
    quotes            => "< $dirs->{dicts}/dict_quotes",

    #Настройки для работы с баннерами
    minuswords        => "< $dirs->{dicts}/dict_minuswords",
    minus_media       => "< $dirs->{dicts}/dict_minus_media",
    diff_words        => "< $dirs->{dicts}/dict_diff",
    service_words     => "< $dirs->{dicts}/dict_service",
    goods_words       => "< $dirs->{dicts}/dict_goods",
    info_words        => "< $dirs->{dicts}/dict_info",
    cities_words      => "cat $dirs->{dicts}/dict_cities $dirs->{dicts}/tr_dict_cities |",
    dict_towns        => "< $dirs->{dicts}/dict_towns",
    dict_towns_world  => "< $dirs->{dicts}/dict_towns_world",
    dict_category_premium_brands  => "< $dirs->{dicts}/dict_category_premium_brands",

    domain_categs => "$dirs->{dicts}/domain_categs",
    dict_stopwords => "$dirs->{dicts}/dict_stopwords",
    dict_alibaba_badwords => "$dirs->{dicts}/dict_alibaba_badwords",
    dict_categs_nephews => "$dirs->{dicts}/categsspec_nephews",
#    dict_brands       => "cat $dirs->{dicts}/dict_brands $dirs->{dicts}/tr_dict_brands |",
    dict_brands       => "$dirs->{dicts}/dict_brands",
    dict_auto_brands  => "cat $dirs->{dicts}/dict_auto_brands |",
    dict_brands_market => "$dirs->{dicts}/dict_brands_market",
    dict_context_brands  => "$dirs->{dicts}/dict_context_brands",
    dict_noises       => "cat $dirs->{dicts}/dict_noises |",
    dict_acc_brands   => "cat $dirs->{dicts}/dict_acc_brands |",
    dict_metrical     => "cat $dirs->{dicts}/dict_metrical |",
    dict_goods        => "cat $dirs->{dicts}/dict_goods |",
    dict_surnames_ru  => "cat $dirs->{dicts}/dict_surnames_ru |",
    dict_surnames_en  => "cat $dirs->{dicts}/dict_surnames_en |",
    dict_surnames_wide_ru  => "cat $dirs->{dicts}/dict_surnames_wide_ru |",
    dict_names_ru     => "cat $dirs->{dicts}/dict_names_ru |",
    dict_banks     => "cat $dirs->{dicts}/dict_banks |",
    dict_notmodels     => "cat $dirs->{dicts}/dict_notmodels |",
    dict_categskeys_with_minuswords     => "cat $dirs->{dicts}/dict_categskeys_with_minuswords |",
    dict_market_freq    => "$dirs->{dicts}/dict_market_freq",
    dict_geo    => "cat $dirs->{dicts}/dict_towns_new $dirs->{dicts}/dict_towns_world |",
    dict_wear_properties => "cat $dirs->{dicts}/parser/dict_wear_properties |",
    ends_of_banners   => "$dirs->{dicts}/dict_ends_of_banners",
    catalogia_markers => "< $dirs->{dicts}/catalogia_markers",
    catalogia_flags_plus_minus => "< $dirs->{dicts}/catalogia_flags_plus_minus",
    catalogia_med_workaround => "< $dirs->{dicts}/med_workaround",
    categs_direct_ids => "$dirs->{dicts}/categsspec_direct_ids",
    categs_virtual    => "< $dirs->{dicts}/categsspec_virtual",
    categs_virtual_gen=> "$dirs->{dicts}/categsspec_virtual_gen",
    categs_diff       => "< $dirs->{dicts}/categsspec_diff",
    categs_diff_file  => "$dirs->{dicts}/categsspec_diff",
    categs_siblings   => "< $dirs->{dicts}/categsspec_siblings",
    categs_siblings_add => "< $dirs->{dicts}/categsspec_siblings_add",
    categs_never_sibl => "< $dirs->{dicts}/categsspec_never_sibl",
    categs_nephews    => "< $dirs->{dicts}/categsspec_nephews",
    categs_minus_old  => "< $dirs->{dicts}/categsspec_minus_old",
    categs_minus      => "< $dirs->{dicts}/categsspec_minus",
    categs_plus       => "< $dirs->{dicts}/categsspec_plus",
    categs_flags      => "< $dirs->{dicts}/categsspec_flags",
    categs_add_flags  => "$dirs->{dicts}/categsspec_add_flags",
    categs_deleted_flags  => "$dirs->{dicts}/categsspec_deleted_flags",
    categs_uncertain  => "cat $dirs->{dicts}/categsspec_uncertain |",
    categs_addphr     => "cat $dirs->{dicts}/caddphr_infusion_20140205 $dirs->{dicts}/caddphr_infusion_20111212 $dirs->{dicts}/caddphr_market $dirs->{dicts}/caddphr_domains |",
    categs_deleted    => "$dirs->{dicts}/categsspec_deleted",
    categs_parents    => "$dirs->{dicts}/categsspec_parents",

    mobile_categs_mapping => "$dirs->{dicts}/apps/apps_hierarchy",
    mobile_categs_siblings => "$dirs->{dicts}/apps/mobAppsCategs2OrdinaryCategs",

    categs_antiwords  => {
        ru  =>  "$dirs->{dicts}/categsspec_antiwords",
        en  =>  "$dirs->{dicts}/categsspec_antiwords_en",
        tr  =>  "$dirs->{dicts}/categsspec_antiwords_tr",
    },

    deprecated_categs_words => {
        ru  => "$dirs->{dicts}/dict_deprecated_categs_words_ru",
        en  => "$dirs->{dicts}/dict_deprecated_categs_words_en",
        tr  => "$dirs->{dicts}/dict_deprecated_categs_words_tr",
    },

    # слои категорий и их свойства
    # возможные свойства:
    # use_norm  использовать нормализацию без выкидывания стоп-слов синонимов
    categs_layers     => {
        categs      => {},
        dim2        => {},
        atoms       => {},
        atoms_named => {},
        uncertain   => {},
        catalysts   => {},
        catalysts2  => {},
        csnorms     => {},
        multiwords  => {
            use_norm_with_stops => 1,
            max_words_dist      => 1,
            dont_sort           => 1
        },
    },

    # фразы, добавленные через веб-интерфейс
    categs_web  => {
        map{$_ => $dirs->{dicts}."/caddphr_web_$_"} qw(ru en tr)
    },

    # фразы, удалённые через веб-интерфейс
    categs_web_deleted  => {
        map{$_ => $dirs->{dicts}."/cdelphr_web_$_"} qw(ru en tr)
    },

    categs_languages  => {
        en  => "$dirs->{dicts}/categs_en",
        tr  => "$dirs->{dicts}/categs_tr"
    },
    categs_virtual_languages  => {
        ru  => "$dirs->{dicts}/categsspec_virtual",
        en  => "$dirs->{dicts}/categsspec_virtual_en",
        tr  => "$dirs->{dicts}/categsspec_virtual_tr"
    },
    dict_not_wide     => "< $dirs->{dicts}/dict_not_wide",

    # сервер bender
    bender_params => {
        port                => 11322,
        host                => "127.0.0.1",
        temp_dir            => $dirs->{temp} . "/bender",
        work_dir            => $dirs->{work} . "/bender",
        server_dir          => $dirs->{scripts} . "/cpp-source/bender/bin",
        data_file           => "",
        encoded_file        => "",
        index_file          => ""
    },

    CdictClient_params => {
        host                => "bmchronicle.yandex.ru",
    },

    ChronicleCdict_params => {
        dict_file           => $dirs->{work} . "/cdict/chronicle_dict",
        index_file          => $dirs->{work} . "/cdict/chronicle_index",
        single_file         => $dirs->{work} . "/cdict/cdict_chronicle_binary",
    },

    DatotekaCdict_params => {
        host                => "bmdatoteka.yandex.ru",
        dict_file           => $dirs->{work} . "/cdict/datoteka_dict",
        index_file          => $dirs->{work} . "/cdict/datoteka_index",
        single_file         => $dirs->{work} . "/cdict/cdict_datoteka_binary",
        raw_files_dir       => $dirs->{work} . "/cdict/raw_files",
    },

    # сервер cdict
    cdict_params => {
        port                => 11330,
        work_dir            => $dirs->{work} . "/cdict",
        temp_dir            => $dirs->{temp} . "/cdict",
        server_dir          => $dirs->{scripts} . "/cpp-source/cdict",
        norm_config_file    => $dirs->{dicts} . "/cpp_norm_config",
    },

    # параметры для static_map
    static_map_params => {
        work_dir    => $dirs->{'work'} . '/static_map',
        temp_dir    => $dirs->{'temp'} . '/static_map',
    },

    # параметры для static_zmap
    static_zmap_params => {
        work_dir    => $dirs->{work} . '/static_zmap',
        temp_dir    => $dirs->{temp} . '/static_zmap',
        proj_dir    => $dirs->{scripts} . '/cpp-source/static_zmap/encode_data/',
        server_dir  => $dirs->{scripts} . '/cpp-source/static_zmap/encode_data/',
    },


    # bender по баннерам
    BannersBender_params => {
        port                => 11323,
        host                => "bmbender-front01h.yandex.ru",
        binary_file         => $dirs->{work} . "/bender/banners_bin",
        data_file           => $dirs->{work} . "/bender/banners_data",
        index_file          => $dirs->{work} . "/bender/banners_index",
        encoded_file        => $dirs->{work} . "/bender/banners_encoded",
    },
    DistrBannersBender_params => {
        port                => 11323,
        hosts               => [
            "bmbender-gen02i.bm.yandex.net",
            "bmcdict-gen01f.yandex.ru",
            "bmbender-front02i.yandex.ru",
            "bmbender-front02f.bm.yandex.net",
            "bmbender-front03i.bm.yandex.net"
        ],
        binary_file         => $dirs->{work} . "/bender/banners_bin",
        data_file           => $dirs->{work} . "/bender/banners_data",
        index_file          => $dirs->{work} . "/bender/banners_index",
        encoded_file        => $dirs->{work} . "/bender/banners_encoded",
    },

    # статистика по запросам из логов запросов
    QueryLogStat_params => {
        temp_dir            => $dirs->{temp} . "/QueryLogStat",
        work_dir            => $dirs->{work} . "/QueryLogStat",

        yt_table_bnr_counts          => $dirs->{yt_cdict_generation} . "/bnr_counts",
        yt_table_counts_full         => $dirs->{yt_cdict_generation} . "/counts_full",

        # логи из advq
        advq_mr_cmd         => "YT_TOKEN_PATH=/opt/broadmatching/secrets/tokens/yt_plato /usr/bin/mapreduce-yt -server hahn",
        languages => {
            ru => {
                mr_dir              => "//home/advq/broadmatch/rus/phits30d",
                mr_dir_mob          => "//home/advq/broadmatch/rus/phits30d",

                # yt generation
                yt_advq_phits_dir        => '//home/advq/broadmatch/rus/phits30d',
                yt_table_norms           => $dirs->{yt_cdict_generation}.'/ru/norms',
                yt_table_snorms          => $dirs->{yt_cdict_generation}.'/ru/snorms',
                yt_table_syns            => $dirs->{yt_cdict_generation}.'/ru/syns',
                yt_table_syns_categs     => $dirs->{yt_cdict_generation}.'/ru/syns_categs',
                yt_table_tails           => $dirs->{yt_cdict_generation}.'/ru/tails',
                yt_table_categs_regions  => $dirs->{yt_cdict_generation}.'/ru/categs_regions',
                yt_table_categs          => $dirs->{yt_cdict_generation}.'/ru/categs',
                yt_table_regions         => $dirs->{yt_cdict_generation}.'/ru/regions',

                yt_table_norms_orig      => $dirs->{yt_cdict_generation}.'/ru/norms_orig',
                yt_table_norms_full      => $dirs->{yt_cdict_generation}.'/ru/norms_full',
                yt_table_counts_query    => $dirs->{yt_cdict_generation}.'/ru/counts_query',
                yt_table_counts_full     => $dirs->{yt_cdict_generation}.'/ru/counts_full',
                yt_table_counts          => $dirs->{yt_cdict_generation}.'/ru/counts',
                yt_table_counts_geo      => $dirs->{yt_cdict_generation}.'/ru/counts_geo',
                yt_table_counts_geo_aggr => $dirs->{yt_cdict_generation}.'/ru/counts_geo_aggr',
                yt_table_harm            => $dirs->{yt_cdict_generation}.'/ru/harm',

                yt_table_norms_mob          => $dirs->{yt_cdict_generation}.'/ru/norms_mob',
                yt_table_norms_full_mob     => $dirs->{yt_cdict_generation}.'/ru/norms_full_mob',
                yt_table_counts_mob         => $dirs->{yt_cdict_generation}.'/ru/counts_mob',
                yt_table_counts_full_mob    => $dirs->{yt_cdict_generation}.'/ru/counts_full_mob',

                yt_table_bnr_counts         => $dirs->{yt_cdict_generation}.'/ru/bnr_counts',
                yt_table_flags              => $dirs->{yt_cdict_generation}.'/ru/flags',
            },
            tr => {
                mr_dir              => "//home/advq/broadmatch/tur/phits30d",
                mr_dir_mob          => "//home/advq/broadmatch/tur/phits30d",

                # yt generation
                yt_advq_phits_dir        => '//home/advq/broadmatch/tur/phits30d',
                yt_table_norms           => $dirs->{yt_cdict_generation}.'/tr/norms',
                yt_table_snorms          => $dirs->{yt_cdict_generation}.'/tr/snorms',
                yt_table_syns            => $dirs->{yt_cdict_generation}.'/tr/syns',
                yt_table_syns_categs     => $dirs->{yt_cdict_generation}.'/tr/syns_categs',
                yt_table_tails           => $dirs->{yt_cdict_generation}.'/tr/tails',
                yt_table_categs_regions  => $dirs->{yt_cdict_generation}.'/tr/categs_regions',
                yt_table_categs          => $dirs->{yt_cdict_generation}.'/tr/categs',
                yt_table_regions         => $dirs->{yt_cdict_generation}.'/tr/regions',

                yt_table_norms_orig      => $dirs->{yt_cdict_generation}.'/tr/norms_orig',
                yt_table_norms_full      => $dirs->{yt_cdict_generation}.'/tr/norms_full',
                yt_table_counts_query    => $dirs->{yt_cdict_generation}.'/tr/counts_query',
                yt_table_counts_full     => $dirs->{yt_cdict_generation}.'/tr/counts_full',
                yt_table_counts          => $dirs->{yt_cdict_generation}.'/tr/counts',
                yt_table_counts_geo      => $dirs->{yt_cdict_generation}.'/tr/counts_geo',
                yt_table_counts_geo_aggr => $dirs->{yt_cdict_generation}.'/tr/counts_geo_aggr',
                yt_table_harm            => $dirs->{yt_cdict_generation}.'/tr/harm',


                yt_table_norms_mob          => $dirs->{yt_cdict_generation}.'/tr/norms_mob',
                yt_table_norms_full_mob     => $dirs->{yt_cdict_generation}.'/tr/norms_full_mob',
                yt_table_counts_mob         => $dirs->{yt_cdict_generation}.'/tr/counts_mob',
                yt_table_counts_full_mob    => $dirs->{yt_cdict_generation}.'/tr/counts_full_mob',

                yt_table_bnr_counts         => $dirs->{yt_cdict_generation}.'/tr/bnr_counts',
                yt_table_flags              => $dirs->{yt_cdict_generation}.'/tr/flags',
            },
        },
    },

    subphrasers_hosts => [qw/catalogia-media-front01e.yandex.ru
                             catalogia-media-front01i.yandex.ru
                          /],

    # сервер подфраз категорий
    subphraser_params => {
        port                    => 11222,
        host                    => '127.0.0.1',
        dir_server              => $dirs->{'cpp-source'} . '/subphraser',
        init_data_file          => $dirs->{'temp'} . '/subphraser_init_data',
        comptrie_file           => $dirs->{'bin-dicts'} . '/categorization/subphraser.comptrie',
        work_dir                => $dirs->{'work'} . '/subphraser',
        temp_dir                => $dirs->{'temp'} . '/subphraser',
        max_connect_attempts    => 150,
        sandbox_resource        => 'BROADMATCH_CATEGORIZATION_COMPTRIE',
    },

    # эталонный сервер
    perfect_subphraser_params => {
        port                    => 11225,
        host                    => "127.0.0.1",
        dir_server              => $dirs->{"cpp-source"}."/subphraser",
        init_data_file          => $dirs->{temp}."/subphraser_init_data",
        max_connect_attempts    => 150,
    },

    categs_from_web   => $dirs->{dicts}."/categs_web",
    minicategs        => "cat $dirs->{dicts}/categs_information $dirs->{dicts}/categs_services $dirs->{dicts}/categs_equipment $dirs->{dicts}/categs_goods $dirs->{dicts}/categs_web $dirs->{dicts}/categs_dim2 |",
    celebrities       => "cat $dirs->{dicts}/dict_celebrities $dirs->{dicts}/dict_characters |",

    # иерархия категорий (для внешних пользователей)
    categs_tree_report => $dirs->{work}."/categs_tree",
    categs_tree_from_infuse => $dirs->{dicts}."/categs_tree/categs_tree_from_infuse",
    categs_tree_from_infuse_json => $dirs->{dicts}."/categs_tree/categs_tree_from_infuse.json",
    mediagroups_tree_report => $dirs->{work}."/mediagroups_tree",

    # Логи для мониторинга
    cdict_monitor           => "$dirs->{work}/cdict-monitor",

    #Настройки для работы с моделями
    Models_params => {
        not_vendors_file    => $dirs->{dicts}.'/dict_not_vendors',
        vendors_file    => $dirs->{dicts}.'/dict_static_vendors',

        dict_types      => $dirs->{dicts}.'/models/dict_types',
        work_dir        => $dirs->{work}.'/models',
        wide_models     => $dirs->{gen_dicts}.'/models/wide_models',
        temp_dir        => $dirs->{temp}.'/models',
        dict_dir        => $dirs->{gen_dicts}.'/models',

        # генерируемые словари вендоров и типов
        gen_vendors_file    => $dirs->{gen_dicts}."/dict_vendors",
        gen_modeltypes_file  => $dirs->{gen_dicts}."/dict_modeltypes",
        gen_modelsyns_file  => $dirs->{gen_dicts}."/dict_modelsyns",
    },

    # Настройки для работы с категориями моделей
    ModelsCategories_params => {
        categs_file         => $dirs->{dicts}."/caddphr/categs_models", # автоматически коммитим при инфьюзе
        cluster_max_count   => 500,
        temp_file           => $dirs->{temp}."/ContextFiltering/categs_models",
    },

    # Настройки для работы с категориями фильмов
    MoviesCategories_params => {
        categs_file_gen     => $dirs->{gen_dicts}."/caddphr_movies", # регулярно обновляемый файл
        categs_file         => $dirs->{dicts}."/caddphr/caddphr_movies", # автоматически коммитим при инфьюзе
    },

    #Настройки для работы с симпграфами
    simpgraph_files   => {
        map { $_ => $dirs->{simpgraphs}.'/'.$_.'.simpgraph.gz' } qw(
            Tags
            QueryLog
            QueryLogQQ
            HitLog
            SerpClicks
            Bindings
            Banners
            Slotstat
            RBindings
            ConvStat
            Thes
            RUserClicks2
            RUserClicks3
            RRBindings
            RRBindingsExperimental
            RUserClicks1
            UserSessionsQQ
        ),
    },

    # категоризатор страниц
    PagesCategories_params => {
        domains_file    => $dirs->{dicts}."/pcategs_domains",   # категории для указанных доменов
    },

    # категории баннеров
    BannersCategories_params => {
        categs_file_direct  => $dirs->{work}.'/BannersExtended/banners-categories-direct',
        camp_categs_file    => $dirs->{work}.'/BannersExtended/campaigns-categories',
        domains_categs_file => $dirs->{work}.'/BannersExtended/domains-categories',
        sold_phrases_file   => $dirs->{work}.'/BannersExtended/sold-phrases-categories',

        categ2mediagroup_file   => $dirs->{work}."/categ2mediagroup",
        id2mediagroup_file      => $dirs->{work}."/id2mediagroup",
        mediagroup2videodirectgroup_file => $dirs->{work}."/mediagroup2videodirectgroup",
        videodirectgroups_second_level_file => $dirs->{work}."/videodirectgroups_second_level",
        brandgroups_file        => $dirs->{work}."/brandgroups",
        interests_mapping_file  => $dirs->{work}."/interests_mapping_file",
        yt_table_path           => "//home/catalogia",
    },

    #файл с частотами пар букв русского языка для определения кодировки
    symbfile => $dirs->{dicts} . '/Russian_chst',

    #настройки работы со страницами
    pageinf => {
        cache_dir => $dirs->{temp}.'/page_cache',
        banner_maker_dir => $dirs->{temp}.'/banner_maker',
        wide_url_names => "cat $dirs->{dicts}/dict_wide_url_names $dirs->{dicts}/tr_dict_wide_url_names $dirs->{dicts}/en_dict_wide_url_names |",
        dict_brands => "cat $dirs->{dicts}/dict_brands $dirs->{dicts}/tr_dict_brands |",
        dict_context_brands => "cat $dirs->{dicts}/dict_context_brands |",
        dict_noises => "cat $dirs->{dicts}/dict_noises |",
        dict_surnames_ru => "cat $dirs->{dicts}/dict_surnames_ru |",
        dict_surnames_en => "cat $dirs->{dicts}/dict_surnames_en |",
        dict_surnames_wide_ru => "cat $dirs->{dicts}/dict_surnames_wide_ru |",
        dict_names_ru => "cat $dirs->{dicts}/dict_names_ru |",
        dict_acc_brands => "cat $dirs->{dicts}/dict_acc_brands |",
        dict_metrical => "cat $dirs->{dicts}/dict_mertical |",
        dict_goods => "cat $dirs->{dicts}/dict_goods |",
        dict_banks => "cat $dirs->{dicts}/dict_banks |",
        dict_notmodels => "cat $dirs->{dicts}/dict_notmodels |",
        dict_categskeys_with_minuswords => "cat $dirs->{dicts}/dict_categskeys_with_minuswords |",
        dict_geo    => "cat $dirs->{dicts}/dict_towns_new $dirs->{dicts}/dict_towns_world |",
        dict_wear_properties => "cat $dirs->{dicts}/parser/dict_wear_properties |",
    },

    # адрес, куда посылать письма об ошибках
    admin_mail => 'bm-dev@yandex-team.ru',

    #Данные по группам категорий для медиапланеров (mediagroups)
    mediagroups => $dirs->{dicts}.'/mediagroups',

    transl_intrf => $dirs->{dicts}.'/transl_intrf',
    cmd_transl_intrf => 'cat '.$dirs->{dicts}.'/transl_intrf '.$dirs->{dicts}.'/transl_intrf_mine |',

    QTests => {
        results_file        => "$dirs->{QTests}/qtests_results-current.log",
    },

    domain_mirrors_url => 'https://intapi.direct.yandex.ru/jsonrpc/Mirrors', #Зеркальщик Директа
    domain_ids_url => 'https://yabs-id-generator.yabs.yandex.ru/domain',

    #/ Динамические и перфоманс баннеры

    RedButton_params => {
        dbh_name => 'catalogia_media_dbh',
        log_table => 'RedButtonLog',
        svnup_prm => {
            bm_path => $dirs->{root},
            paths => '', # Обновлять всё
            make_targets => [
                'rt-research/broadmatching', # дефолтная цель "ya make" нашего кода
                'rt-research/monitoring', # код мониторинга
                'solomon/agent/bin', # бинарь Solomon-агента (IRT-1931)
            ],
        },
        RedButtonLog_ID_label => 'RedButtonLog_ID',
    },

    mr_catalogia => {
        work_dir => $dirs->{temp} . "/build_mr_catalogia",
    },

    sandbox_params => {
        user       => 'robot-bm-admin',
        owner      => 'BROADMATCH_ROBOT',
        api_url    => 'https://sandbox.yandex-team.ru/api/v1.0',
        proxy_url  => 'https://proxy.sandbox.yandex-team.ru',
        token_path => $dirs->{'root'} . '/secrets/tokens/sandbox',
        temp_dir   => $dirs->{'temp'} . '/sandbox',
        work_dir   => $dirs->{'work'} . '/sandbox',
        resources  => {
            BROADMATCH_CATEGORIZATION_COMPTRIE      => {
                file    => $dirs->{'work'} . '/subphraser/subphraser.comptrie',
            },
            BROADMATCH_CATEGORIES_SUPPRESSION_DICT  => {
                file    => $dirs->{'work'} . '/static_zmap/categories_suppression.szmap',
            },
        },
    },

    categories_suppression_params => {
        dict                => $dirs->{'dicts'} . '/categsspec_diff',
        szmap_dict          => $dirs->{'work'} . '/static_zmap/categories_suppression.szmap',
        sandbox_szmap_dict  => $dirs->{'bin-dicts'} . '/categorization/categories_suppression.szmap',
        sandbox_resource    => 'BROADMATCH_CATEGORIES_SUPPRESSION_DICT',
    },

    mail_lists => {

        # catalogia

        'automoderate' => {
            'to' => [ 'kuzinaksu@yandex-team.ru' ],
            'cc' => [ 'breqwas@yandex-team.ru' ],
        },

        'moderation_superiors' => {
            'cc' => [
                'shelkovnikova@yandex-team.ru',
                'stitov@yandex-team.ru',
            ],
        },

        'moderation_notifications' => {
            'cc' => [
                'directmod-cat@yandex-team.ru',
                'kuzinaksu@yandex-team.ru',
                'mkadyrova@yandex-team.ru',
                'bratchikova@yandex-team.ru',
                'catmedia-moderation@yandex-team.ru',
            ],
        },

        'banners_benchmark' => {
            'to' => [
                'sergio@yandex-team.ru',
                'emurav@yandex-team.ru',
                'kuzinaksu@yandex-team.ru',
            ],
        },

        'categs_diff_manager' => {
            'bcc' => [ 'emurav@yandex-team.ru' ],
        },


        'banners_recategorization' => {
            'to' => [
                'apovetkin@yandex-team.ru',
                'bendyna@yandex-team.ru',
                'april-dev@yandex-team.ru',
                'vip-planners@yandex-team.ru',
            ],
        },


        # bannerland

        'bannerland_perf_errors' => {
            'to' => [
                'sergio@yandex-team.ru',
                'malykhin@yandex-team.ru',
            ],
        },

        'bannerland_yabs_errors' => {
            'to' => [
                'pgriffin@yandex-team.ru',
                'yarulin@yandex-team.ru',
                'dkozhenov@yandex-team.ru',
                'sivkov@yandex-team.ru',
                'temafedotov@yandex-team.ru',
            ],
        },

        'bannerland_state' => {
            'to' => [
                'sergio@yandex-team.ru',
                'breqwas@yandex-team.ru',
                'johnyh@yandex-team.ru',
                'apovetkin@yandex-team.ru',
                'modest@yandex-team.ru',
                'rysmiatova@yandex-team.ru',
                'fawkes@yandex-team.ru'
            ],
        },

        'bannerland_moderation' => {
            'to' => [ 'johnyh@yandex-team.ru' ],
        },


        # other

        'broadmatch_phrases' => {
            'bcc' => [ 'emurav@yandex-team.ru' ],
        },

    },

    fcgi_http_server => {
        n_processes => 30,
    },

    dt_proxy_params => {
        proxy_port => 10203,
        proxy_binary => $dirs->{scripts} . "/cpp-source/dyntables_proxy/bm-dyntables-proxy",
        proxy_config => $dirs->{work} . "/bm-dyntables-proxy.conf",
        proxy_log => $dirs->{log} . "/bm-dyntables-proxy.log",
        proxy_lockname => "run_dyntables_proxy",

        max_client_timeout => 120,  # seconds
        log_requests_longer_than => 10,  # seconds

        resources => [
            {
                src_table => "//home/catalogia/banners_extended",
                dst_table => "//home/catalogia/dyntables/resources/banners_extended",
                primary_key => "bid",
                secondary_keys => ["cid", "uid", "pid", "BannerID", "OrderID", "domain"],
                extra_key_table_fields => ["is_active"],
            },
        ],

        proxy_acl => [
            "//home/catalogia/dyntables/resources/",
            "//home/bannerland/perf/dyntables/",
            "//home/bannerland/dyn/dyntables/",
        ],

        local_proxy_hosts => [
            'catalogia-media-dev02e.yandex.ru',
            Utils::Hosts::get_hosts(role => 'catalogia-media-front'),
        ],

        public_proxy_hosts => [
            Utils::Hosts::get_hosts(role => 'catalogia-media-front'),
        ],
    },

    GraphiteClient_params => {
        host => 'localhost',
        port => 42000,
        debug_mode => 0,  # для переопределения в debug_env
    },

    SolomonClient_params => {
        push_host   => "http://localhost",
        agent_port => 10050,
    },
};

while (my ($group, $opt) = each %{$options->{db_dump}}) {
    $opt->{dir} = "$dirs->{work}/db_dump/$group";
    $opt->{table_file} = { map { $_ => "$opt->{dir}/$_.json" } @{$opt->{tables}} };
};

# экв. домены баннеров
$options->{equiv_domains} = '< ' . $options->{DictEquivDomains}{file};

# плохие фразы
$options->{directmodphrs} = '< ' . $options->{DictDirectMod}{file};

push @{$options->{Words_params}{wide_words}}, map { $_->[0] } @{$options->{Words_params}{stopword_dicts}};

# доопределяем настройки генератора norm_dict
$options->{DictNorm}{reask_source} = 'zcat '.$options->{DictNorm}{reask_file}.'|';
$options->{DictNorm}{homonyms} = $options->{Homonyms_params}{file};
$options->{DictNorm}{stopword_dicts} = $options->{Words_params}{stopword_dicts};
push @{$options->{DictNorm}{synonyms}}, $options->{DictSynonyms}{translit_file_frozen};

$options->{DictNorm}{export}{$_} = $options->{Words_params}{$_} for qw(
    norm_dict
    syn_dicts
    wordcount_dict
    word2snorm_dict
);

$options->{DictMulWordSyns}{reask_source} = $options->{DictNorm}{reask_source};

$options->{ContextSyns}{MulWordSyns}{file} = $options->{DictMulWordSyns}{misspell_file};

$options->{ContextSyns}{translit_ru2en}{staticmap_files} = [ $options->{DictCSyns}{smap_translit_ru2en_file} ];
$options->{ContextSyns}{translit_en2ru}{staticmap_files} = [ $options->{DictCSyns}{smap_translit_en2ru_file} ];
$options->{ContextSyns}{translit_en2ru_strict}{staticmap_files} = [ $options->{DictCSyns}{smap_translit_en2ru_strict_file} ];



#
# Объединение разных конфигов
#

for my $add (@add_options) {
    while (my ($k, $v) = each %$add) {
        if (defined $options->{$k}) {
            die "ERROR: Options conflict for key [$k]!";
        }
        $options->{$k} = $v;
    }
}

#
# параметры из других конфигов прокидываем в объединенный конфиг
#

$options->{secrets} = {
    server_dir => "/opt/broadmatching/secrets/",
    id => $options->{bm_tokens_secret_id},

    items => {
        tokens => {
            list  => [ qw(yql_robot_bm_admin yt_plato yt_banach yt_freud juggler sandbox direct_api hw_watcher.conf
                          solomon_oauth_token juggler_oauth_token qloud_oauth_token startrek_oauth_token tvm_demon_secrets.json
                          artmon_oauth_token ) ],
            chgrp => {
                'hw_watcher.conf' => 'hw-watcher',
                'tvm_demon_secrets.json' => 'www-data',
            }
        },

        passwords => {
            list => [ qw (mysql-preprod-mdb_password mysql-test-mdb_password mysql-prod-mdb_password ) ],
        },

        tvm => {
            list => [ qw(tvm_bmfront_secret tvm_catmedia_secret tvm_catmedia-dev_secret app_zoracl_secret) ],
        },

        certificates => {
            list  => [ qw( bmfront.bm.yandex-team.ru.pem ) ],
        },
    }
};

$options->{yt_client_params}{params}{cluster} = $options->{yt_default_cluster};
$options->{yt_client_params}{params}{proxy} = $options->{yt_default_proxy};
$options->{yt_client_params}{params}{token_path} = $options->{yt_token_path};

$options->{MarketData_params}{id_category_vendor_model} = $dirs->{root} . "/$options->{market_data_rel_path}";
$options->{subphraser_market}{comptrie_file} = $dirs->{root} . "/$options->{market_comptrie_rel_path}";
$options->{subphraser_market}{staticmap_file} = $dirs->{root} . "/$options->{market_staticmap_rel_path}";

#
# Патчи
#

# "патчи" для конкретных ролей/хостов
my $role = eval { Utils::Hosts::get_host_role() } // ''; # get_host_role может не работать, например в джобах на YT
if ($role eq 'bmfront') {
    $options->{subphraser_params}{host} = 'catalogia-media-front01e.yandex.ru';
    $options->{PrefProjSrv_params}{client_timeout} = 300;
} elsif($role eq "catmedia-dev") {

    #$options->{subphraser_params}{host} = 'catalogia-media01e.yandex.ru';
    #$options->{perfect_subphraser_params}{host} = 'catalogia-media01e.yandex.ru';
    $options->{subphraser_params}{host} = 'catalogia-media-front01e.yandex.ru';
    $options->{perfect_subphraser_params}{host} = 'catalogia-media-front01e.yandex.ru';

    $options->{PrefProjSrv_params}{num_forks} = 10;
    $options->{PrefProjSrv_params}{min_free_forks} = 2;
}

# применение патчей
my @patches;

my $patchfile = $dirs->{root} . '/patchfile';
if ( -f $patchfile ) {
    open(my $patches_fh, $patchfile) or die "ERROR: $!";
    push @patches, grep { $_ } map { chomp; s/#.*//; s/^\s+//; s/\s+$//; $_ } <$patches_fh>;
}

push @patches, split(m/,/, $ENV{BM_USE_PATCH}) if $ENV{BM_USE_PATCH};
push @patches, 'env-patch' if $ENV{BM_PATCH_STR};

if ($ENV{BM_NO_PATCH}) {
    my %exclude_patches = map { $_ => 1 } split(m/,/, $ENV{BM_NO_PATCH});
    @patches = grep { !$exclude_patches{$_} } @patches;
}

if (scalar @patches) {
    require Utils::CommonPatches;

    for my $patch (@patches) {
        my $patch_data = $Utils::CommonPatches::patch_options->{$patch};
        if (ref($patch_data) eq 'HASH') {
            patch_hash($options, $patch_data, clone_source => 0);
        }
    }
}

1;
