package Utils::DB;
use strict;

use Dates;

use utf8;
use open ':utf8';

use Utils::Sys qw(get_uniq_name);
use BM::DB::Tables qw(get_table_def);

use base qw(Exporter);

our @EXPORT_OK = (
    'get_tbl_update_time',
    'get_tbl_records_number',
    'get_tables',
    'copy_table',
    'array2text',
    'rewrite_table',
    'flush_buffer',
    'get_db_lock', 'release_db_lock',
);


my $dates_obj = Dates->new;

# get_tbl_update_time($dbh, $tbl_name);
#   database => название базы
sub get_tbl_update_time {
    my $dbh = shift;
    my $tbl = shift;
    my %prm = @_;
    my $database = $prm{database};
    my $database_str = $database ? " in $database " : "";
    my $tbl_info = $dbh->selectrow_hashref("show table status $database_str like '$tbl'", { Slice => {} });
    return if !$tbl_info;
    my $str = $tbl_info->{Update_time}
        or return;
    return $dates_obj->trdate('db_time', 'sec', $str);
}

# get_tbl_records_number($dbh, $tbl_name);
#   database => название базы
sub get_tbl_records_number {
    my $dbh = shift;
    my $tbl = shift;
    my %prm = @_;
    my $database = $prm{database};
    my $database_str = $database ? "$database." : "";
    my $info = $dbh->selectrow_hashref("select count(*) cnt from $database_str$tbl", { Slice => {} });
    return if !$info;
    return $info->{cnt};
}

# get_tables($dbh, $template)  -  получить список таблиц
#   database => название базы
sub get_tables {
    my $dbh = shift;
    my $tmpl = shift;
    my %prm = @_;
    my $database = $prm{database};
    my $database_str = $database ? " in $database " : "";
    my $tbl_data = $dbh->List_SQL("show tables $database_str like '$tmpl'");
    return map { values %$_ } @$tbl_data;
}

# статусы всех таблиц
sub get_tables_status {
    my $dbh = shift;
    my $tmpl = shift // '%';
    return $dbh->List_SQL("show table status like ".$dbh->quote($tmpl));
}

# copy_table($tbl, $src_dbh, $dst_dbh)
# params:
#   pack_size => insert pack size; default: 50k
#   def => use custom table def for dst
sub copy_table {
    my $tbl = shift;
    my $src_dbh = shift;
    my $dst_dbh = shift;
    my %par = (
        pack_size => 50_000,
        @_,
    );
    my $create_sql;
    if ($par{def}) {
        $create_sql = "create table $tbl $par{def}";
    } else {
        $create_sql = $src_dbh->List_SQL("show create table $tbl")->[0]->{'Create Table'};
    }
    $dst_dbh->do($create_sql);
    my $rows = $src_dbh->List_SQL("select * from $tbl");
    return if !@$rows;
    my @fld = sort keys %{$rows->[0]};
    while (@$rows) {
        my @pack = splice(@$rows, 0, $par{pack_size});
        $dst_dbh->do("
            insert into $tbl (".join(',', @fld).") values ".
            array2text($dst_dbh, \@pack, fields => \@fld)
        );
    }
}

# array2text($dbh, $data, %par)
# переводит список значений в строку для запроса
# доп. параметры:
#   fields =>  если заданы поля, предполагается, что $data - список хэшей
sub array2text {
    my $dbh = shift;
    my $data = shift;
    my %par = @_;

    my $arr;
    if ($par{fields}) {
        my @fld = @{$par{fields}};
        $arr = [ map { [ @{$_}{@fld} ] } @$data ];
    } else {
        $arr = $data;
    }

    my @val_str = map { '('.$_.')' } map { join(',', map { $dbh->quote($_) } @$_) } @$arr;
    return join(',', @val_str);
}

# перезаписать таблицу данными в памяти
# вообще-то, это частный случай reload_table, но там запись только из файла
sub rewrite_table {
    my $dbh = shift;
    my $tbl = shift;
    my $rows = shift;
    my %par = @_;

    my %add_par;
    $add_par{fields} = $par{fields} if $par{fields};
    my $fldstr = $par{fields} ? '('.join(',', @{$par{fields}}).')' : '';

    my $new = get_uniq_name($tbl);
    $dbh->do("create table $new ".(get_table_def($tbl) // "like $tbl"));
    while (@$rows) {
        my @pack = splice(@$rows, 0, 50_000);
        $dbh->do("
            insert into $new $fldstr
            values ".array2text($dbh, \@pack, %add_par)."
        ");
    }
    $dbh->do("drop table if exists Done$tbl");
    $dbh->do("rename tables $tbl to Done$tbl, $new to $tbl");
}

# flush_buffer($dbh, $tbl)
# записать содержимое ${tbl}Buffer в ${tbl}YYYYMMDD
# параметры: $dbh, $tbl,
#   def => table def (default: like old Buffer)
#   replace => replace вместо insert
#   pack_size => макс. кол-во строк при вставке
#   verbose => ...
sub flush_buffer {
    my $dbh = shift;
    my $tbl = shift;
    my %par = @_;

    my $buf = $tbl.'Buffer';
    my $new = $buf.'New';
    my $done = $buf.'Done';
    my $month = substr($dates_obj->cur_date('bs'), 0, 6);
    my $log = $tbl.$month;

    my $def = $par{def} // "like $buf";

    unless (get_tables($dbh, $done)) {
        $dbh->do("drop table if exists $new");
        $dbh->do("create table $new $def");
        $dbh->do("rename table $buf to $done, $new to $buf");
    }

    my $cmd = $par{replace} ? 'replace' : 'insert';
    $dbh->do("create table if not exists $log $def");
    if (!$par{pack_size}) {
        $dbh->do("$cmd into $log select * from $done");
    } else {
        my $tfld = 'UpdateTime';
        my $cnt = 0;
        while (1) {
            my $time = $dbh->List_SQL("
                select max($tfld) as MaxTime
                from (select $tfld from $done order by $tfld limit $par{pack_size}) Q
            ")->[0]{MaxTime};
            last if !$time;
            if ($par{verbose}) {
                $dbh->log("iter ".(++$cnt).": process $tfld <= '$time'");
            }
            $dbh->do("$cmd into $log select * from $done where $tfld <= '$time'");
            $dbh->do("delete from $done where $tfld <= '$time'");
        }
    }
    $dbh->do("drop table $done");
}


sub get_db_lock {
    my $dbh = shift;
    my $name = shift;
    my $unlimited = shift // 0; # бесконечный lock
    my $qname = $dbh->quote($name);
    my $command = "select get_lock($qname," . ($unlimited ? "-1" : "0") . ") as res";
    my $res = $dbh->List_SQL( $command );
    return if !$res or !@$res;
    return $res->[0]{res};
}

sub release_db_lock {
    my $dbh = shift;
    my $name = shift;
    my $qname = $dbh->quote($name);
    $dbh->do("select release_lock($qname)");
}


1;
