package Utils::Hosts;
use strict;

use utf8;

use Storable qw(dclone);
use Socket;

use Utils::Funcs qw(expand_names get_python_options);
use Utils::LightCommon;

use base qw(Exporter);
our @EXPORT_OK = qw(
    get_hosts
    get_curr_host
    get_readable_hostname
    get_host_info
    get_host_role
    get_host_group
    get_host_project
    get_host_datacenter
    get_hosts_domains
    get_short_hostname
    get_crontab_files
    get_alive_hosts
    check_addrinfo
    get_from_conductor_api
    ip2hostname
    is_cur_host_in_cloud
    is_cur_host_in_deploy
    get_deploy_project_id
    get_deploy_stage_id
    get_deploy_unit_id
    get_deploy_workload_id
);

# Информация о хосте (ключ -- полное имя хоста, FQDN)
#   project =>      проект (ДРФ, Каталогия и т.п.)
#   role    =>      роль хоста внутри проекта
#   group   =>      группа хостов (кластер)
#   resources =>    "кластер" ресурсов, при выборе производителя ресурса отдаем предпочтение хостам с тем же значением
my %default_host_info = (
    project => '',
    role    => '',
    group   => '',
);

my $hosts_py_path = $Utils::LightCommon::dirs->{scripts} . '/hosts/hosts';
my %host_info = %{get_python_options($hosts_py_path, 0)};

# получить список хостов
# доп. параметры (в хэше) позволяют фильтровать список, например:
#   role => $role  -  хосты с этой ролью
#   role => [ $r1, $r2 ]  -  хосты с ролью $r1 или $r2
# exclude => хэш настроек для get_hosts - хосты, которые нужно исключить из результата
# ext_prm => хэш дополнительных настроек:
#   conductor_group => группа в кондукторе
sub get_hosts {
    my %par = @_;

    my @hosts = sort keys %host_info;

    while (my ($par, $val) = each %par) {
        next if $par eq 'ext_prm' or $par eq 'exclude';
        my @val = (ref $val) ? @$val : $val;
        my %val = map { $_ => 1 } @val;
        @hosts = grep { defined $host_info{$_}{$par} and $val{ $host_info{$_}{$par} } } @hosts;
    }
    return @hosts unless @hosts;

    if (my $ext_prm = $par{ext_prm}) {
        if (my $conductor_group = $ext_prm->{conductor_group}) {
            my @conductor_hosts = get_from_conductor_api(groups2hosts => $conductor_group);
            my %conductor_hosts = map { $_ => 1 } @conductor_hosts;
            @hosts = grep { $conductor_hosts{$_} } @hosts;
        }
        if (my $datacenter = $ext_prm->{datacenter}) {
            my %datacenter = map { $_ => 1 } (ref($datacenter) eq 'ARRAY' ? @$datacenter : ($datacenter));
            @hosts = grep { $datacenter{ get_host_datacenter($_) // '' } } @hosts;
        }
    }

    if (my $exclude = $par{exclude}) {
        my %exclude_hosts = map { $_ => 1 } get_hosts(%$exclude);
        @hosts = grep { ! $exclude_hosts{$_} } @hosts;
    }

    return @hosts;
}

sub is_cur_host_in_cloud {
    return 1 if is_cur_host_in_deploy();
    return 0;
}

sub get_deploy_project_id {
    return $ENV{DEPLOY_PROJECT_ID};
}

sub get_deploy_stage_id {
    return $ENV{DEPLOY_STAGE_ID};
}

sub get_deploy_unit_id {
    return $ENV{DEPLOY_UNIT_ID};
}

sub get_deploy_workload_id {
    return $ENV{DEPLOY_WORKLOAD_ID};
}

sub is_cur_host_in_deploy {
    return 1 if $ENV{'DEPLOY_PROJECT_ID'};
    return 0;
}

# локальный хост; полное доменное имя (FQDN)
my $curr_hostname = undef;
sub get_curr_host {
    my %opts = @_;

    return $ENV{BM_CURR_HOST} if $ENV{BM_CURR_HOST};
    return $curr_hostname if $curr_hostname && !$opts{force_reload};

    $curr_hostname = determine_curr_host();
    return $curr_hostname;
}

# локальный хост; полное читабельное имя
sub get_readable_hostname {
    my %opts = @_;

    if ($ENV{DEPLOY_POD_PERSISTENT_FQDN}) {
        return $ENV{DEPLOY_POD_PERSISTENT_FQDN};
    }

    return get_curr_host(%opts);
}

# имя локального хоста без кэширования и переопределений
# не используйте без явной на то причины, get_curr_host почти всегда подойдёт лучше
sub determine_curr_host {
    return $ENV{QLOUD_DISCOVERY_INSTANCE} if $ENV{QLOUD_DISCOVERY_INSTANCE};
    return $ENV{DEPLOY_POD_PERSISTENT_FQDN} if $ENV{DEPLOY_POD_PERSISTENT_FQDN};
    my $host = `hostname --fqdn`;
    if ($? or (not ($host // ''))) {
        die "ERROR in get_curr_host (" . join(", ", map {$_//""} ($host, $?, $!)) .")";  # Здесь нужно ERROR, т.к. get_host_info вызывается на этапе 'use Utils::CommonCluster3' (до handle_errors())
                                                                                         # однако Utils::CommonClusters3 больше нет, так что если мешает можно убрать
    }
    chomp $host;
    return $host;
}

# информацию о хосте
# на входе:
#   $host -  полное доменное имя хоста (по умолчанию: локальный хост)
sub get_host_info {
    my $host = shift || get_curr_host();
    return $host_info{$host} if exists $host_info{$host};
    return \%default_host_info;
}

sub get_host_role {
    my $info = get_host_info(@_) || {};
    return $info->{role};
}

sub get_host_group {
    my $info = get_host_info(@_) || {};
    return $info->{group};
}

sub get_host_project {
    my $info = get_host_info(@_) || {};
    return $info->{project};
}

my %letter2datacenter = (
    'e' => 'iva',  # Ивантеевка
    'f' => 'myt',  # Мытищи
    'i' => 'sas',  # Сасово
    'k' => 'man',  # Мянтсяля
);

# Возвращает датацентр (определяем по полю datacenter в host_info или по имени хоста)
sub get_host_datacenter {
    my $host = shift;
    return lc($ENV{QLOUD_DATACENTER}) if $ENV{QLOUD_DATACENTER};
    return lc($ENV{DEPLOY_NODE_DC}) if $ENV{DEPLOY_NODE_DC};

    my $datacenter = get_host_info($host)->{datacenter};
    return $datacenter if defined $datacenter;

    if ($host =~ m/^([^\.]+?)([a-z])\..*yandex\.(ru|net)$/i) {
        my $host_letter = $2;
        if (exists $letter2datacenter{$host_letter}) {
            return $letter2datacenter{$host_letter};
        }
    }
}

# Домены первого и второго уровней наших машинок
my $hosts_domains = [qw[ yandex.ru yandex.net ]];

# Возвращает список доменов вида "SLD.TLD" наших машинок
sub get_hosts_domains {
    return $hosts_domains;
}

# Возвращает короткое имя хоста
sub get_short_hostname {
    my $host = shift // '';
    for my $host_domain (@$hosts_domains) {
        my $re_host_domain = $host_domain =~ s/\./\\\./r;
        my $short = ( $host =~ m/^(.+)\.$re_host_domain$/ )[0] || '';
        return $short if $short;
    }
    return $host;
}

# На входе:
#   $api_par - название параметра в кондукторе (см. https://c.yandex-team.ru/api/ )
#   $api_val - значение
# На выходе:
#   массив строк из кондуктора
# Пример:
#   my @hosts = get_from_conductor_api(groups2hosts => 'bm_ng_databases_bmdata');
sub get_from_conductor_api {
    my ($api_par, $api_val) = @_;
    my $url = "https://c.yandex-team.ru/api/$api_par/$api_val";
    my @values = grep {$_} map {chomp; $_} `curl -s '$url'`;
    if ($?) {
        die "Error in downloading ($url): $? ($!)";
    }
    return @values;
}

# Возвращает ссылку на массив - список файлов для set-crontab.pl
# Берем следующие файлы:
#   common.cron.d  (если роль не указана в списке исключений @roles_without_common_crontab)
#   qloud.common.cron.d  (для qloud хостов, если роль не указана в списке исключений @roles_without_common_crontab)
#   iron.common.cron.d  (для не qloud хостов, если роль не указана в списке исключений @roles_without_common_crontab)
#   Если для хоста задан crontab_name, то $crontab_name.cron.d; иначе - $host_role.cron.d
#       Если задан crontab_name => '', то $host_role.cron.d не используется
#   Если для хоста задан флаг master => 1, то также берем соответствующий файл *-master.cron.d
#   Если для хоста задан arrayref extra_crontabs, то также берем соответствующие файлы *.cron.d
sub get_crontab_files {
    my $host = get_curr_host();
    my $is_cloud = is_cur_host_in_cloud();

    my $crontab_dir = $Utils::Common::options->{dirs}{scripts} . "/crontabs";
    my $info = get_host_info($host);
    my @crontabs;

    my $role = $info->{role}
        || return;

    my @roles_without_common_crontab = qw[ idle ];
    my %roles_without_common_crontab = map { $_ => 1 } @roles_without_common_crontab;
    unless ($roles_without_common_crontab{$role}) {
        push @crontabs, "$crontab_dir/common.cron.d";
        my $cn = $is_cloud ? 'qloud.common.cron.d': 'iron.common.cron.d';
        push @crontabs, "$crontab_dir/$cn";
    }

    my $crontab_name = $info->{crontab_name} // $role;
    if ($crontab_name) {
        my $crontab_file = "$crontab_dir/$crontab_name.cron.d";

        if (! -f $crontab_file) {
            die "Can't find the crontab file (role: $role, crontab_name: " . ($info->{crontab_name} // "") . ", crontab_file: $crontab_file)\n";
        }
        push @crontabs, $crontab_file;
    }

    if ($info->{master}) {
        my $file = "$crontab_dir/$crontab_name-master.cron.d";
        push @crontabs, $file if -f $file;
    }

    for (@{$info->{extra_crontabs}}) {
        my $file = "$crontab_dir/$_.cron.d";
        push @crontabs, $file if -f $file;
    }

    return \@crontabs;
}

sub check_addrinfo {
    my ($host) = @_;
    my $res;
    eval {
        my ( $err, @addrs ) = Socket::getaddrinfo( $host, 0, { 'protocol' => Socket::IPPROTO_TCP, } );
        $res = (!$err and @addrs)  ?  1 : 0;
    };
    return $res;
}

# получить список хостов, для которых check_addrinfo возвращает ненулевой результат
# доп. параметры (в хэше) - аналогично get_hosts
sub get_alive_hosts {
    my %par = @_;
    my @hosts = get_hosts(%par);
    @hosts = grep { check_addrinfo($_) } @hosts;
    return @hosts;
}

# Получает ip, возвращает его hostname
# На вxодe:
#   ip
# Доп. параметры (в хеше)
#   use_cache => 0 - кеширование не испольуем
#   use_cache => 1 - используем кеширование
# Если параметров нет. Кеширование не используем
# На выходе:
#   hostname - если все прошло успешно
#   undef - если произошла ошибка
my %ip2hostname;
sub ip2hostname {
    my $ip = shift;
    my %flags = @_;


    if ($flags{use_cache} and exists $ip2hostname{$ip}) {
        return $ip2hostname{$ip};
    }
    my $type = AF_INET;
    my $host;

    if ($ip =~ /:/) {
        $type = AF_INET6;
    }
    my $adress = Socket::inet_pton($type, $ip);

    if (! defined($adress)) {
        return;
    }

    if ($host = gethostbyaddr($adress, $type)) {
        if ($flags{use_cache}) {
            $ip2hostname{$ip} = $host;
        }
        return $host;
    }
    return;
}

1;
