package Utils::LWP::Override;
use strict;
use warnings;
use utf8;

use LWP::Protocol;

use Utils::Urls qw(external_dns_resolver);
use Utils::Sys qw(transform_v4_ip is_ipv4_host);

use base qw(Exporter);

use constant LWP_OVERRIDE_NONE => 1;
use constant LWP_OVERRIDE_V4 => 2;
use constant LWP_OVERRIDE_V6 => 3;
use constant LWP_OVERRIDE_V4TOV6 => 4;

our @modes = qw/LWP_OVERRIDE_NONE LWP_OVERRIDE_V4 LWP_OVERRIDE_V6 LWP_OVERRIDE_V4TOV6/;

our @EXPORT = (
    qw/
        lwp_override_mode
        get_peer_addr
    /,
    @modes,
);

our %EXPORT_TAGS = (
    'mode' => \@modes,
);


our @protocols = qw(http https);
our %implementors;
our $current_mode = LWP_OVERRIDE_NONE;

our %mode_methods = (
    &LWP_OVERRIDE_NONE => [],
    &LWP_OVERRIDE_V4 => [qw(
        market_resolver_fix
    )],
    &LWP_OVERRIDE_V6 => [qw(
        v4_host_fix
        market_resolver_fix
    )],
    &LWP_OVERRIDE_V4TOV6 => [qw(
        v4_host_fix
        market_resolver_fix
        broken_ipv6_fix
    )],
);

#Настройки внешнего ресолвера - список хешрефов с наборами опций. Если есть несколько хешрефов, они пробуются по очереди, до первого совпадения.
#type      - тип записи
#transform - нужно ли конвертировать из ipv4 в ipv6
our %external_dns_opts = (
    &LWP_OVERRIDE_V4     => [ { type => 'A',    transform => 0 } ],
    &LWP_OVERRIDE_V6     => [ { type => 'AAAA', transform => 0 }, { type => 'A',    transform => 1 } ],
    &LWP_OVERRIDE_V4TOV6 => [ { type => 'A',    transform => 1 } ],
);

our %methods = (
    v4_host_fix  => \&_v4_host_fix,
    market_resolver_fix => \&_market_resolver_fix,
    broken_ipv6_fix   => \&_external_dns_addr,
);

sub lwp_override_mode { 
    my $mode = shift;
    $mode //= LWP_OVERRIDE_NONE;
    die "Unknown lwp override mode: $mode" unless exists $mode_methods{$mode};
    if ($current_mode eq LWP_OVERRIDE_NONE && $mode ne LWP_OVERRIDE_NONE) {
        _enable_override();
    }
    if ($current_mode ne LWP_OVERRIDE_NONE && $mode eq LWP_OVERRIDE_NONE) {
        _disable_override();
    }
    $current_mode = $mode;
}

sub get_peer_addr {
    my $host = shift;
    my $addr = undef;;
    foreach my $method ( @{$mode_methods{$current_mode}} ) {
        $addr = $methods{$method}->($host);
        last if defined $addr;
    }
    return $addr;
}

#################
# address methods
#################

sub _v4_host_fix {
    my $host = shift;
    my $addr = undef;
    if (is_ipv4_host($host)) {
        $addr = transform_v4_ip($host);
    }

    return $addr;
}

sub _market_resolver_fix {
    my $host = shift;
    my $addr = undef;
    if ($host =~ /\.(market|company)$/) {
        $host .= '.';
        $addr = _external_dns_addr($host);
    }

    return $addr;
}

our %external_dns_cache;
sub _external_dns_addr {
    my $host = shift;
    if ( !exists $external_dns_cache{$current_mode}{$host} ) {
        my $addr = undef;
    	my $reply = external_dns_resolver->search($host);
    	if ($reply) {
            foreach my $opts ( @{$external_dns_opts{$current_mode}} ) {
                foreach my $rr ($reply->answer) {
                    next unless $rr->type eq $opts->{type};
                    $addr = $rr->address;
                    $addr = transform_v4_ip($addr) if defined $addr && $opts->{transform};
                    last if defined $addr;
                }
                last if defined $addr;
            }
    	}
        $external_dns_cache{$current_mode}{$host} = $addr;
    }
    return $external_dns_cache{$current_mode}{$host};
}

##################
# override methods
##################
sub _enable_override {
    for my $proto (@protocols) {
        if (my $orig = LWP::Protocol::implementor($proto)) {
            my $impl = _implementor($proto);
            if (eval "require $impl; 1") {
                LWP::Protocol::implementor($proto => $impl);
                $implementors{$proto} = $orig;
            }
        }
    }
}

sub _disable_override {
    for my $proto (@protocols) {
        if (my $impl = $implementors{$proto}) {
            LWP::Protocol::implementor($proto, $impl);
        }
    }
}

sub _implementor {
    my $proto = shift;
    return sprintf 'Utils::LWP::Protocol::%s' => $proto;
}

1;
