package Utils::LibTrie;
use strict;
use vars qw(@EXPORT_OK @ISA @EXPORT);

require Exporter;
@ISA       = qw(Exporter);
@EXPORT    = qw(); # no default export
@EXPORT_OK = qw(construct_trie find_in_trie append_trie union_tries);

use  Storable qw(dclone);

=header         DESCRIPTION

    Package to operate with Trie data structure.
    Trie presented as hash of hashrefs and terminals.
=cut


# Construct trie
# Params:
#   1) src_chains : arrayref                    // acceptable source chains for trie
#   2) terminal   : string                      // string to label end-of-chain (terminal)
#   3) separator  : string    [default = '\s+'] // string which contains regexp which used to split chains
# Returns:
#   hashref                                     // Trie

sub construct_trie {
    my ($src_chains, $terminal, $separator) = @_;
    
    # default separator
    $separator //= '\s+';
    my $trie = {};
#   print $separator."\n"; 
    for my $part(@$src_chains) {
        my @words = split(/$separator/, $part);
        my $cur = $trie;
        for my $word(@words) {
            $cur->{$word} = {} unless $cur->{$word};
            $cur = $cur->{$word};
        }
        $cur->{$terminal} = 1;
    }

    return $trie;
}

# TODO comment

sub find_in_trie {
    my ($trie, $terminals, $start_index, $words) = @_;
    
    my $res = { map{ $_ => '' } @$terminals };
    my $sub_trie = $trie->{$words->[$start_index]} || '';

    return $res unless $sub_trie;

    my $res_index = { map {$_ => $sub_trie->{$_} ? $start_index : -1} @$terminals };
    my $cur_node = $sub_trie;
    for(my $i = $start_index+1; $cur_node && $i < scalar(@$words); $i++) {
        $cur_node = $cur_node->{$words->[$i]} || ();
        $res_index->{$_} = $i for grep { $cur_node->{$_} } @$terminals;
    }
    $res->{$_} = join(' ', @$words[$start_index..$res_index->{$_}]) for ( grep { $res_index->{$_} != -1 } @$terminals );
    
    return $res;
}

sub append_trie {
    my ($dest_trie, $src_trie) = @_;

    for my $key(keys %$src_trie) {
        if ( ref($src_trie->{$key}) eq 'HASH' ) {
            $dest_trie->{$key} = {} unless $dest_trie->{$key};
            append_trie($dest_trie->{$key}, $src_trie->{$key});
        } else {
            $dest_trie->{$key} = $src_trie->{$key};
        }
    }

    return $dest_trie;
}

sub union_tries {
    my @tries = @_;

    my $res = {};

    $res = dclone($tries[0]) if $tries[0];
    for(my $i=1; $i < scalar(@tries); $i++) {
        append_trie($res, $tries[$i]);
    }

    return $res;
}

