package Utils::Server;
use strict;

use utf8;
use open ':utf8';

use POSIX;
use IO::File;
use IO::Handle;
use IO::Socket;

use Utils::Common;
use Utils::Sys qw(
    print_log print_err
    get_file_lock
);

use base qw(Exporter);

our @EXPORT = qw(
    start_server
    stop_server
    shutdown_server
    simple_loop
    multiprocess_loop
);

# параметры:
#   pid_file - путь к pid-файлу (по умолчанию: lock/servername.pid)
#   log_file - файл для STDOUT
#   err_file - файл для STDERR
sub start_server {
    my $pid_file = shift;
    my $log_file = shift;
    my $err_file = shift || $log_file;

    my $old_pid = get_pid($pid_file);
    if (defined $old_pid) {
        if ($old_pid && kill(0, $old_pid)) {
            print_log "Server already running with pid $old_pid!";
            exit(0);
        }
        print_log "pid_file '$pid_file' already exists, deleting it";
        unlink $pid_file
            or die "Can't unlink pid_file '$pid_file': $!";
    }

    # открываем так, чтобы только один процесс мог получить filehandle
    my $pid_fh = IO::File->new($pid_file, O_WRONLY|O_CREAT|O_EXCL, 0644)  
        or die "Can't create pid_file '$pid_file': $!";

    print_log("starting server, see logs [$log_file], [$err_file]");
    my $pid = daemonize();
    print $pid_fh $pid, "\n";
    close $pid_fh;

    open STDOUT, '>>', $log_file
        or die "Can't redirect STDOUT to file '$log_file': $!";
    open STDERR, '>>', $err_file
        or die "Can't redirect STDERR to file '$err_file': $!";
    STDOUT->autoflush(1);
    STDERR->autoflush(1);

    get_file_lock($pid_file, filename => 1)
        or die "Can't get lock!";
}

# параметры:
#   $pid_file  -  файл, в котором записан pid процесса
# доп. параметры:
#   dont_wait  =>  не дожидаться остановки процесса
#   signal  =>  номер посылаемого сигнала, прокидывается в kill_process*
sub stop_server {
    my $pid_file = shift;
    my %par = @_;
    if (get_file_lock($pid_file, filename => 1)) {
        print_log("Lock acquired, server is not running!");
        return undef;
    }
    my $pid = get_pid($pid_file);

    my %kill_par;
    $kill_par{signal} = $par{signal} if defined $par{signal};
    if ($par{dont_wait}) {
        kill_process_once($pid, %kill_par);
    } else {
        kill_process($pid, %kill_par);
    }
}

sub shutdown_server {
    my $pid_file = shift;
    unlink $pid_file;
}


# создать сокет и войти в цикл с ожиданием tcp соединения;
# считать одну строку, обработать, послать результат и закрыть соединение
# параметры:
#   port     =>  $port
#   timeout  =>  $timeout
#   func     =>  функция обработки сообщений (строка => строка в кодировке UTF-8)
sub simple_loop {
    my %par = @_;

    my $quit = 0;
    $SIG{INT} = $SIG{TERM} = sub { 
        print_log("Signal to stop called, stopping server");
        $quit = 1;
    };
    $SIG{HUP} = $SIG{PIPE} = 'IGNORE';

    my $sock = IO::Socket::INET->new(
        LocalPort => $par{port},
        Listen	  => 20,
        Proto	  => 'tcp',
        Reuse	  => 1,
        Timeout	  => $par{timeout},
    ) or die "Cannot create listening socket: $!\n";

    print_log("Server ready, awaiting connection on port " . $par{port});
    while (!$quit) {
        print_log("Waiting connection ...");
        next unless my $session = $sock->accept();
        $session->autoflush;
        my $host = gethostbyaddr( $session->peeraddr, AF_INET) || $session->peerhost;
        my $port = $session->peerport;
        print_log("Connection from [$host, $port] accepted");

        my $cmd_bytes = <$session>;
        my $cmd = Encode::decode('UTF-8', $cmd_bytes);
        chomp $cmd;
        print_log("Received command '$cmd'");
        my $res = $par{func}->($cmd);
        my $res_bytes = Encode::encode('UTF-8', $res);
        print $session $res_bytes, "\n";
        close $session;

        print_log("Connection closed");
    }
    close $sock;
}

# создать сокет и войти в цикл с ожиданием tcp соединения;
# при приходе клиента сделать fork и обработать задание в дочернем процессе
# считать одну строку, обработать, послать результат и закрыть соединение
# параметры:
#   port     =>  $port
#   timeout  =>  $timeout   таймаут на коннект (по умолчанию: час)
#   max_proc =>  $proc      максимальное количество дочерних процессов (по умолчанию: 10)
#   func     =>  \&func     функция обработки сообщений (строка => строка в кодировке UTF-8)
sub multiprocess_loop {
    my %par = (
        timeout => 3600,
        max_proc => 10,
        @_,
    );

    my %process;  # таблица процессов

    my $quit = 0;
    $SIG{INT} = $SIG{TERM} = sub { 
        print_log("Signal to stop called, stopping server");
        $quit = 1;
    };
    $SIG{HUP} = $SIG{PIPE} = 'IGNORE';
    $SIG{CHLD} = sub {
        while () {
            my $kid = waitpid(-1, POSIX::WNOHANG);
            last unless $kid > 0;
            next if !defined $process{$kid};
            my $exit_status = ($? >> 8);
            delete $process{$kid};
            print_log("process [$kid] finished; exit status: $exit_status; running: ".keys(%process));
        }
    };

    my $sock = IO::Socket::INET->new(
        LocalPort => $par{port},
        Listen	  => $par{max_proc} + 2,  # с запасом
        Proto	  => 'tcp',
        Reuse	  => 1,
        Timeout	  => $par{timeout},
    ) or die "Cannot create listening socket: $!\n";

    print_log("Server ready, awaiting connection on port " . $par{port});
    while (!$quit) {
        # обеспечим ограничение на количество процессов
        if (keys %process > $par{max_proc}) {
            sleep 1;
            next;
        }

        print_log("Waiting connection ...");
        next unless my $session = $sock->accept();
        $session->autoflush;
        my $host = gethostbyaddr( $session->peeraddr, AF_INET) || $session->peerhost;
        my $port = $session->peerport;
        print_log("Connection from [$host, $port] accepted");

        # пытаемся сделать форк
        my $child = fork();
        die "Can't fork" if !defined $child;

        if ($child == 0) {
            # дочерний процесс
            close $sock;

            my $cmd_bytes = <$session>;
            my $cmd = Encode::decode('UTF-8', $cmd_bytes);
            chomp $cmd;
            print_log("Received command '$cmd'");
            my $res = $par{func}->($cmd);
            my $res_bytes = Encode::encode('UTF-8', $res);
            print $session $res_bytes, "\n";
            close $session;
            exit(0);
        }

        $process{$child} = { start => time };
        print_log("process $child started, total running: ".keys(%process));
        close $session;
    }
    close $sock;
}

#
# вспомогательные функции
#

sub daemonize {
    my $dpid = fork();
    die "Can't fork to daemonize: $!" if !defined $dpid;
    exit(0) if $dpid;

    POSIX::setsid()
        or die "Can't start a new session: $!";

    open STDIN, '< /dev/null';
    open STDOUT, '> /dev/null';
    open STDERR, '> /dev/null';

    chdir '/';
    return $$;
}

sub kill_process_once {
    my $pid = shift;
    my %par = @_;
    my $signal = $par{signal} // 15;
    print_log("kill_process_once: sending signal $signal to process $pid");
    kill $signal, $pid;
}

sub kill_process {
    my $pid = shift;
    my %par = @_;
    my $signal = $par{signal} // 15;
    print_log("kill_process: sending signal $signal to process $pid");
    kill $signal, $pid;
    for (my $sec = 0; $sec < 3600; ++$sec) {
        if (!kill(0, $pid)) {
            print_log("process $pid terminated");
            return 1;
        }
        sleep 1;
    }
    die "Can't kill process $pid: $!";
}

sub get_pid {
    my $pid_file = shift;
    if (-f $pid_file ) {
        open my $fh, '<', $pid_file
            or die "Can't open pid_file '$pid_file': $!";
        my $pid = <$fh>;
        close $fh;
        return int($pid);
    }
    return undef;
}

1;
