package Utils::Sys;
use strict;

use utf8;
use open ':utf8';

use Utils::Common;

use Cwd qw();
use Encode qw(_utf8_off _utf8_on);
use Sys::Hostname;
use Data::Dumper;
use Fcntl ':flock';
use POSIX qw(:sys_wait_h strftime);
use IO::Handle;
use Text::Iconv;
use Encode;
use URI::Escape;
use LWP::UserAgent;
use Digest::MD5 qw();
use File::Temp qw(tempfile tempdir);
use File::Basename qw(dirname);
use File::Path qw();
use File::Spec qw();
use File::Copy;
use Time::Local qw(timelocal timegm);
use IO::Socket::INET;
use IO::Socket::SSL qw(inet4);
use List::Util qw(max);
use JSON;
use MIME::Base64 qw();
use IPC::Open2;
use Time::HiRes qw(usleep gettimeofday tv_interval);
use Scalar::Util qw(weaken);
use Storable qw(dclone);

use Utils::Array qw(in_array);
use Utils::Funcs qw(encode_tsv decode_tsv);
use Utils::Hosts qw(get_hosts);

use IO::Uncompress::AnyUncompress qw(anyuncompress $AnyUncompressError);

no warnings 'utf8'; # из-за варнингов ловим segfault в bmapi, глушим варнинги до решения DYNSMART-863

use base qw(Exporter);
#no warnings 'recursion';

our @EXPORT = (
    'h2sa',                   #Превращаем хэш в массив с обратной сортировкой по значению
    'hh2h',                   #Хэш хэшей превращает в хэш со сзначениями количества ключей в хэшах второго уровня
    'h2top',                  #Хэш превращает в ТОП по 2 порогам
    'u2w', 'w2u',             #Преобразование кодировок
    'uniq',                   # удалить из списка повторы
    'load_map',               #Загружает хэш из файла
    'get_file_lock',          # залочиться. Если lock занят, то вернуть 0.
    'wait_for_file_lock',     # залочиться. Если lock занят, то залочиться после того, как он освободится.
    'get_lock_pid',           # процесс, залочивший файл
    'release_file_lock',      # разлочиться
    'mem_usage',              # сколько памяти (в байтах) занимает процесс (pid)
    'get_children_pids',      # PID'ы дочерних процессов
    'dir_files',              # содержимое директории (обвязка вокруг opendir/closedir)
    'lines_count',            # кол-во строк в файле
    'mtime',                  # modification time файла
    'modtime',                # alias для mtime
    'check_modtime',          # проверить, требуется ли обновление
    'fork_tasks',             # распараллеливание задач с ограничением на память
    'do_safely',              # вызов заданной функции с ограничением по времени, с возможностью перехватить die в вызываемой функции
    'mem_bytes',              # перевод строкового представления памяти (1M,1G,etc) в байты
    'file_bytes',             # сколько места на диске (в байтах) занимает файл
    'rotate_files',           # ротация данных
    'get_encoder',    # закэшированные енкодеры (Encode) для более быстрых операций
    'make_good_utf_string',   # выбрасываем не-utf символы из строки (perl internal string)
    'get_timestamp',          # timestamp в формате yyyymmdd_hhmmss
    'log_time_fmt',           # timestamp в формате yyyy-mm-dd hh:mm:ss
    'log_time_hires_fmt',     # timestamp в формате yyyy-mm-dd hh:mm:ss.ms
    'log_msg_fmt',            # формат сообщений для логов
    'print_log',              # логирование в STDOUT
    'print_err',              # логирование в STDERR
    'handle_errors',          # переопределить $SIG{__DIE__}, $SIG{__WARN__} для получения timestamp-ов в err-логах
    'md5int',                 # md5, сжатый до 64-bit integer по методу БК
    'md5int_base64',          # то же, что md5int, но в base64-записи, символы [a-zA-z0-9+/]
    'time_unix2db',           # 1296682768 -> '2011-02-03 00:39:15'
    'time_db2unix',           # '2011-02-03 00:39:15' -> 1296682768
    'time_log2unix',          # '2011-02-03 00:39:15' или '2011-02-03 00:39:15.746921' -> 1296682768
    'time_yt2unix',           # "2016-10-19T14:35:56.173961Z" -> 1476876956
    'is_ipv4_host',           # проверить похож ли адрес хоста на ipv4
    'switch_to_ipv6',         # copy IO::Socket::INET6 -> IO::Socket::INET
    'switch_to_ipv4',         # copy IO::Socket::INET4 -> IO::Socket::INET
    'get_runlevel',           # текущий runlevel
    'read_ts_file',           # загрузить данные в tabtools-формате из файла
    'read_ts_header',         # прочитать заголовок tabtools-файла
    'load_json',              # загрузить данные в json-формате (аналог Storable::retrieve)
    'save_json',              # сохранить данные в json-формате (аналог Storable::store)
    'load_or_eval',           # eval с кэшированием результата в файле
    'number_of_fields',       # кол-во полей в файле
    'round',                  # округление
    'format_number',          # приведение чисел в красивый формат
    'md5_shuffle',
    'rsync',                  # Вызов rsync с заданными параметрами
    'unzipdata',              # Распаковка данных
    'uncompressdata',
    'uncompressfile',
    'staticmap',              # Возвращает объект StaticMap по заданному пути к файлу.
    'staticmap_value',        # Возвращает value объекта StaticMap по заданному пути к файлу для заданного ключа.
    'get_bash_cmd',           # Получить команду для вызова исходной команды через bash
    'wait_children',            # Дождаться завершения процессов, порожденых fork
    'file_text_sub',          #($text, $sub) Нужно для огромных текстов, выполняет функцию для каждой строки через файлы и возвращает результирующий текст
    'file_result_sub',
    'get_uniq_name',
    '_get_lockname',
    'split_csv_line',         # Сплит строки csv по разделителю с учетом отзеркаленных кавычек
    'are_strings_equal',      # проверяет равенство строк с учетом undef
    'are_refs_same',          # проверяет, что ссылки указывают на один объект с учетом undef
    'merge_hashes',           # merge хэшей (а-ля update в Python), подробнее в определении метода
    'url_encode',
    'url_decode',
    'url_decode_safe',
    'transform_v4_ip',
    'assert_curr_host',       # проверяет, что скрипт запущен на хосте из списка, поведение переопределяется через debug_env
    'configure_runit_services', # запускает/останавливает сервисы с помощью runit + подкладывает конфиги для web сервисов
    'uncompress_web_file',
);

our @EXPORT_OK = (
    'get_tempfile',
    'cleanup_tempfiles',    # Удалить временные файлы, созданные sub get_tempfile
    'do_sys_cmd',           # Вызов system для заданной команды с проверкой exit code и логированием
    'read_sys_cmd',         # Возвращает текст, напечатанный командой в STDOUT, с проверкой exit code и логированием.
    'ref_to_weaken_ref',    # Возвращает ссылку на weak версию переданной ссылки
    'make_comptrie_subphraser',
    'pack_files_from_arcadia_root',
    'is_ucs2_compatible',   # Проверяет, все ли символы в строке находятся в подмножестве ucs-2
    'include_cpan',
    'get_files_md5',
);

my $lock_handlers = {}; # for get_file_lock(), release_file_lock()

#####
# merge хешей (рекурсивно), результат в хеш 1
# по совпадающим SCALAR и ARRAY ключам пишутся значения хеша 2
# по несовпадающим типам ключа записываются значения хеша 2
sub merge_hashes {
    my ($h_1, $h_2) = @_;
    return unless ($h_2);

    die('wrong parameters (not hashes)') unless (ref($h_1) eq 'HASH' && ref($h_2) eq 'HASH');

    for my $key(keys %$h_2) {
        my $type = ref( $h_2->{$key} );
#        print $h_2->{$key}, ",type: '$type'","\n";
        if ($type eq 'HASH') {
            if ($h_1->{$key} && ref($h_1->{$key}) eq 'HASH') {
                merge_hashes($h_1->{$key}, $h_2->{$key});
            } else {
                $h_1->{$key} = dclone($h_2->{$key});
            }
        }
        elsif ($type eq 'ARRAY') {
            $h_1->{$key} = dclone($h_2->{$key});
        }
        else {
            $h_1->{$key} = $h_2->{$key};
        }
    }

    return $h_1;
}

my $include_cpan_done = 0;
sub include_cpan {
    return if $include_cpan_done++;
    # push @INC делать плохо, т.к. сначала будет поиск в системных директориях - лишний disk IO
    my $options = $Utils::Common::options;
    unshift @INC, $options->{dirs}{scripts} . '/cpan';
}

sub hh2h {
    my ($h)  = @_;
    return { map { $_ =>  0 + keys %{$h->{$_}} } keys %$h };
}


sub h2sa {
    my ($h)  = @_;
    return sort {$b->[1] <=> $a->[1]} map { [$_, 0+$h->{$_} ] } keys %$h;
}

sub h2top {
    my ($h, $nextdiff, $topdiff, $abslim)  = @_;
    $abslim = 0 unless $abslim;
    my $first = 0;
    my $prev  = 0;
    my @arr = ();
    for my $ctg ( h2sa( $h ) ){
        last if $ctg->[1] < $abslim;
        last if $ctg->[1] < $nextdiff * $prev;
        last if $ctg->[1] < $topdiff * $first;
        push(@arr, $ctg);
        $prev = $ctg->[1];
        $first = $ctg->[1] unless $first;
    }
    return @arr;
}

sub u2w {
    my ($text) = @_;
    my $t = Encode::encode("cp1251", $text);
    return $t;
}

sub w2u {
    my ($text) = @_;
    my $t = Encode::decode('cp1251',$text);
    $t = Encode::encode("utf-8", $t);
    return $t;
}

sub uniq {
    my %seen;
    my @uniq;
    my $seen_undef = 0;
    for my $elem (@_) {
        if (defined $elem) {
            next if $seen{$elem};
            $seen{$elem} = 1;
        } else {
            # Если undef, то $seen{$elem} работает некорректно
            next if $seen_undef;
            $seen_undef = 1;
        }
        push @uniq, $elem;
    }
    return @uniq;
}

# загрузить tab-separated файл с двумя полями в виде хэша { $fld1 => $fld2 }
# my $hash = load_map($file);
# параметры:
#   $filename  -  путь к файлу
# доп. параметры:
#   reverse     => поменять местами поля при загрузке
sub load_map {
    my $filename = shift;
    my %par = @_;
    my $rev = $par{'reverse'};
    my %map;
    open MAP, "<", $filename  or die("can't open map $filename: $!");
    while (<MAP>) {
	chomp;
	my ($p, $q) = split /\t/, $_, -1;
        ($p, $q) = ($q, $p) if $rev;
	$map{$p} = $q;
    }
    close MAP;
    return \%map;
}

# вспомогательная функция для file_lock
sub _get_lockname($) {
    my $script_name = shift;

    if (not defined $script_name) {
        my ($str) = $0 =~ /.*?([^\/]+)\.pl$/;
        unless(defined $str) {
            print_err("Can't parse filename: $0");
            return 0;
        }
        $script_name = $str;
    }

    return $Utils::Common::options->{'dirs'}{'lock'}.'/'.$script_name.'.lock';
}

# returns 1 on success, 0 if lock if busy, undef if error occured
sub get_file_lock {
    my $script_name = shift;
    my %par = @_;

    my $lockname = $par{filename} ? $script_name : _get_lockname($script_name);

    unless(-e $lockname) {
        my $dir = dirname($lockname);
        if ($dir && ! -d $dir) {
            print_err("Create non-existing $dir ...");
            system("mkdir -p $dir") == 0 or die "Could not create $dir ($!)";
        }
        open(TMP, ">$lockname")
            or print_err("Can't create $lockname: $!")
            and return;
        close(TMP);
    }

    my $handle;
    open($handle, "<$lockname")
        or print_err("Can't open $lockname: $!")
        and return;

    flock($handle,LOCK_EX|LOCK_NB)  or return 0;

    chmod(0664,$lockname)
        or print_err("Can't chmod $lockname: $!");

    $lock_handlers->{$lockname} = $handle;
    return 1;
}

# Сделать get_file_lock. Если lock занят, то сделать get_file_lock после того, как он освободится.
# returns 1 on success, undef if error occured
sub wait_for_file_lock {
    my $res = 0;
    while ( ! ($res = get_file_lock(@_))) {
        unless (defined $res) {
            print_err("Error in get_file_lock: $!");
            return;
        }
        sleep 1;
    }
    return $res;
}

sub get_lock_pid {
    my $script_name = shift;
    my $lockname = _get_lockname($script_name);

    open F, "lsof -Fpl $lockname |";
    while(my $pid = <F>) {
        my $lock = <F>;
        if($lock && $lock =~ /W/) {
            ($pid) = $pid =~ /(\d+)/;
            return $pid;
        }
    }
    close F;

    return 0;
}

sub release_file_lock {
    my $script_name = shift;
    my %par = @_;
    my $lockname = $par{filename} ? $script_name : _get_lockname($script_name);

    unless(exists $lock_handlers->{$lockname}) {
        print_err("I'm trying to unlock unknown handler: $lockname");
        return 0;
    }
    my $handle = $lock_handlers->{$lockname};

#    truncate($handle,0);
    flock($handle,LOCK_UN);
    close($handle);

    unlink $lockname if $par{remove_file};
    delete($lock_handlers->{$lockname});

    return 1;
}

# используемая процессом память (в байтах) Linux only!
sub mem_usage {
    my $pid = shift || $$;
    open STAT, "< /proc/$pid/stat"
        or return undef;
    my $line = <STAT>;
    close STAT;
    $line =~ s/.*\)//;
    my @fields = split ' ', $line;
    return $fields[20];
}

# PID'ы дочерних процессов. Linux only!
sub get_children_pids {
    my $pid = shift || $$;
    my $file = "/proc/$pid/task/$pid/children";
    open F, "< $file"
        or die "Cannot open $file ($!)";
    my $line = <F>;
    close F;
    $line //= '';
    my @pids = $line =~ m/(\d+)/g;
    #print_err("get_children_pids: $pid ($line) (@pids)");
    return @pids;
}

sub dir_files {
    my ($dir) = @_;
    opendir DIR, $dir
        or print_err("can't open dir $dir: $!")
        and return undef;
    my @files = grep { not /^\./ } readdir(DIR);
    closedir DIR;
    return @files;
}

sub lines_count {
    my $filename = shift;
    open FILE, "<", $filename
        or return undef;
    while (<FILE>) {
        1;
    }
    my $lines = $.;
    close FILE;
    return $lines;
}

sub mtime {
    my ($file) = @_;  # filename or filehandle
    my @stat = stat $file;
    return undef if not @stat;
    return $stat[9];
}

sub modtime { mtime(@_) }

# требуется ли строить данные по source_file
# в info_file лежит последняя дата обновления
sub check_modtime {
    my ($source_file, $info_file) = @_;
    my $curr_time = &modtime($source_file);
    return 1 if not defined $curr_time;
    my $prev_time = 0;
    if (-f $info_file) {
        open INFO, "<", $info_file;
        $prev_time = int(<INFO>);
        close INFO;
    }
    return $prev_time != $curr_time;
}


# распараллелить задачи с ограничением на память
# основано на Pool::forkTasts из БК
#
# на входе -- список задач вида
# task = {
#   name => 'build graph',
#   func => sub { $graph->build(); },
#   max_mem  => "2G", # допустимые множители: K, M, G
#   max_time => 3600, # секунды
# }
# и хэш параметров (max_proc, max_mem)
sub fork_tasks {
    my $tasks = shift;
    my $opt   = shift || {};

    # опции по умолчанию
    my %options = (
        max_proc => 6,          # кол-во одновременных задач
        max_mem  => "10G",
    );
    $options{$_} = $opt->{$_} for keys %$opt;
    $options{max_bytes} = mem_bytes($options{max_mem});
    my $start_time = time;
    my $mem_peak = 0;
    my $used_bytes = mem_usage($$);
    my @tasks;
    my %done_task;
    my %process;
    STDOUT->autoflush(1);
    STDERR->autoflush(1);

    print_log("fork_tasks ...");

    # сразу отбрасываем то, что не влезет в память
    for my $task (@$tasks) {
        my $task_bytes = mem_bytes($task->{max_mem});
        if ($used_bytes + $task_bytes < $options{max_bytes}) {
            $task->{max_bytes} = $task_bytes;
            push @tasks, $task;
        } else {
            print_log("ERROR: task '".$task->{name}."' requires too much memory (used: $used_bytes)");
        }
    }

    # основной цикл раз в 3 секунды
    while (1) {
        my $curr_mem = mem_usage($$);
        $mem_peak = $curr_mem if $mem_peak < $curr_mem;
        # проверяем, что никто не вылез за рамки
        while (my ($pid, $data) = each %process) {
            next if $data->{killed};
            my $max_time_to_kill15 = 10;   # Максимальное время, которое дается процессу для выхода по kill(15)  (после которого делаем kill(9))
            next if  ($data->{killed15_time} && (time - $data->{killed15_time}) <= $max_time_to_kill15);  # Мы уже попытались убить этот процесс через kill(15), даем ему время для выхода

            my ($mem, $maxmem, $time, $maxtime) = (
                mem_usage($pid) || 0,  # на случай зомби
                $data->{max_bytes},
                time - $data->{start},
                $data->{max_time},
            );
            $data->{mem_peak} = $mem if $data->{mem_peak} < $mem;
            if ($mem > $maxmem or $time > $maxtime) {
                my ($sig, $sig_name) = $data->{killed15_time} ? (9, 'KILL') : (15, 'TERM');    # Сначала посылаем kill(15). Если kill(15) уже посылали и прошло время, большее $max_time_to_kill15, но процесс всё еще работает, то посылаем kill(9)
                my $msg = sprintf("ERROR: killing($sig:$sig_name) task '%s' [%d] (mem: %d of %d, time: %d of %d)",
                    $data->{name}, $pid,
                    $mem, $maxmem, $time, $maxtime,
                );
                print_log($msg);
                kill($sig, $pid);
                $data->{killed15_time} = time   if $sig == 15;
                $data->{killed} = 1   if $sig == 9;
            }
        }

        # следующая задача для выполнения
        my $task;

        # находим задачи, которые пролезают по памяти и у которых завершили работу зависимости
        if (keys %process < $options{max_proc}) {
            for my $t (@tasks) {
                my @d = @{$t->{depends} // []};
                next if grep { !$done_task{$_} } @d;
                next if $used_bytes + $t->{max_bytes} > $options{max_bytes};
                $task = $t;
                last;
            }
        }

        if ($task) {
            # нашли задачу, запускаем!
            @tasks = grep { $_->{name} ne $task->{name} } @tasks;

            my $pid = fork();
            die("can't fork") if not defined $pid;
            if ($pid > 0) {
                # parent
                my $msg = sprintf("task '%s' [%d] started",
                    $task->{name},
                    $pid,
                );
                print_log($msg);
                $process{$pid} = {
                    start      => time,
                    name       => $task->{name},
                    max_time   => $task->{max_time},
                    max_bytes  => $task->{max_bytes},
                    mem_peak   => 0,
                    killed     => 0,
                };
                $used_bytes += $process{$pid}{max_bytes};
            } else {
                # child
                $task->{func}->();
                exit(0);
            }
        }

        # ловим отработавшие или убитые процессы
        my $kid = waitpid(-1, WNOHANG);
        if (defined $process{$kid}) {
            my $exit_status = ($? >> 8);
            my $msg = sprintf(
                "task '%s' [%d] finished; exit status: %d, elapsed time: %d seconds, peak memory usage: %d bytes", # см. парсинг этой строки в scripts/lib/BM/Monitor/Indicators.pm
                $process{$kid}{name},
                $kid,
                $exit_status,
                time - $process{$kid}{start},
                $process{$kid}{mem_peak},
            );
            print_log($msg);
            if ($exit_status == 0) {
                print_log('TASK_'.$process{$kid}{name}.'_OK'); # Чтобы проверять в мониторинге last_success_hours_old
            } else {
                print_log(sprintf("ERROR: task %s failed!", $process{$kid}{name}));
            }
            $used_bytes -= $process{$kid}{max_bytes};
            $done_task{ $process{$kid}{name} } = 1;
            delete $process{$kid};
        }

        last if not @tasks and not (keys %process);
        sleep 5;
    }

    my $final_msg = sprintf(
        "fork_tasks finished; elapsed time: %d seconds, peak memory usage: %d bytes",
        time - $start_time,
        $mem_peak,
    );
    print_log($final_msg);
}

# Вызвать заданную функцию с ограничением по времени, с возможностью перехватить die в вызываемой функции
# На входе - вызываемая функция
# Дополнительные параметры:
#   timeout         => максимально допустимое время выполнения функции (в секундах). Если tries > 1, то это таймаут для каждой попытки вызова функции
#   timeout_handler => функция, вызываемая при превышении timeout
#   no_die          => не вызывать die, при die в вызываемой функции возвращать undef
#   die_handler     => функция, вызываемая в случае die в вызываемой функции
#   no_error        => не выводить ERROR в случае die в вызываемой функции (имеет смысл, если выставлен флаг no_die)
#   die_if_timeout  => вызвать die, если превышен timeout
#   verbose         => писать лог выполнения
#   tries           => количество попыток вызова функции (если функция сделала die, то повторяем вызов функции)
#   sleep_between_tries => время (в секундах) для sleep то между попытками (если задан параметр 'tries')
#   logger          => Если задано, то сообщения выводятся методом $prm{logger}->log()
# Возвращает значение, которое вернула вызываемая функция;
# если функция не отработала корректно или превышен таймаут, возвращает undef или вызывает die (в зависимости от флагов no_die и die_if_timeout)
# Пример:
#   my $result = do_safely(sub { sleep 3; return 42; },  timeout => 2, no_die => 1);
sub do_safely {
    my ($func, %prm) = @_;
    my $timeout = $prm{timeout} // 0;
    my $no_die = $prm{no_die} // 0;
    my $die_if_timeout = $prm{die_if_timeout} // 0;
    my $wantarray = wantarray;
    my $verbose = $prm{verbose} // 0;

    # Если задан $prm{logger}, то выводим все сообщения в него, иначе - в print_err
    my $print_err_ref = $prm{logger}  ?  sub {
        my ($msg) = @_;
        $prm{logger}->log($msg);
    }  :  \&print_err;

    &$print_err_ref("do_safely... timeout=$timeout no_die=$no_die die_if_timeout=$die_if_timeout wantarray=".($wantarray // 'UNDEF') . " tries=".($prm{tries} // 'UNDEF')) if $verbose;

    my $error_msg;  # Если undef, то всё ОК.
    my $should_die = $no_die ? 0 : 1;   # Делать die или возвращать undef в случае ошибки (непустого $error_msg)
    my $timeout_exceeded;   # Был ли превышен таймаут

    if (ref($func) ne 'CODE') {
        if (my $die_handler = $prm{die_handler}) {
            if (ref($die_handler) eq 'CODE') {
                &$print_err_ref("do_safely die_handler ...");
                $die_handler->();
                &$print_err_ref("do_safely die_handler done");
            } else {
                &$print_err_ref("ERROR: do_safely: bad die_handler!");
                $error_msg .= " Bad die_handler!";
            }
        }
        $error_msg = "bad func (" . ref($func) . ")";
        if ($should_die) {
            die $error_msg;
        } else {
            &$print_err_ref("ERROR: $error_msg");
            return;
        }
    }

    my $msg_timeout = "do_safely timeout exceeded!";    # Сообщение для die, если превышен таймаут

    sub _error_re {
        my ($msg) = @_;
        # Для проверки сообщения об ошибке (формат сообщений от die мог быть изменен в handle_errors)
        # TODO обрабатывать спецсимволы в $msg
        #&$print_err_ref("_error_re($msg)");
        return qr/^(\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d\t\[\d+\]\tERROR: died: )*$msg( at [^\s]+ line \d+.)?$/;
    };

    my ($res, @res);    # Результат, который вернула вызываемая функция (в зависимости от wantarray)
    #local $SIG{__DIE__} = undef; # Чтобы убрать $SIG{__DIE__}, установленный в handle_errors       # TODO
    if (($prm{tries} // 1) != 1) {
        # Несколько попыток вызова функции
        my $success = 0;
        for my $try ( 1 .. $prm{tries} ) {
            &$print_err_ref("Try($try) ...")   if $verbose;
            my %prm_call = (%prm, tries => undef, no_die => 0, die_if_timeout => 1, timeout_handler => undef);
            eval {
                if (not defined $wantarray) {
                    do_safely($func, %prm_call);
                } elsif ($wantarray) {
                    @res = do_safely($func, %prm_call);
                } else {
                    $res = do_safely($func, %prm_call);
                }
                $success = 1;
            };
            #if ($@) {
            #    &$print_err_ref("Try($try) died: $@")   if $verbose;
            #}
            &$print_err_ref("Try($try): success: $success" . ($@ ? " ($@)" : ""))  if $verbose;
            last if $success;
            if ($prm{sleep_between_tries} and $try != $prm{tries}) {
                &$print_err_ref("sleep_between_tries: " . $prm{sleep_between_tries})  if $verbose;
                sleep($prm{sleep_between_tries});
            }
        }
        unless ($success) {
            my $msg = $@;
            $timeout_exceeded = 1    if ($msg =~ _error_re($msg_timeout));
            chomp $msg;
            $error_msg = "Unsuccess ($prm{tries} tries) ($msg)";
            &$print_err_ref("Tries msg: $msg")  if $verbose;
        }
    } else {
        my $msg_alarm = "do_safely alarm\n"; # NB: \n required        http://perldoc.perl.org/functions/alarm.html
        my $timeout_prev;
        my $tm_begin = time;
        local $SIG{ALRM} = sub { die $msg_alarm };
        if ($timeout) {
            $timeout_prev = alarm $timeout;   # Сохраняем предыдущий alarm (для вложенных вызовов do_safely)
            &$print_err_ref("do_safely timeout_prev: $timeout_prev")  if $timeout_prev and $verbose;
            alarm $timeout_prev   if $timeout_prev  and  $timeout_prev < $timeout;  # Если предыдущий alarm должен был произойти раньше
        }
        &$print_err_ref("do_safely: start eval") if $verbose;
        eval {
            if (not defined $wantarray) {
                $func->();
            } elsif ($wantarray) {
                @res = $func->();
            } else {
                $res = $func->();
            }
        };
        &$print_err_ref("do_safely: end eval") if $verbose;
        if ($timeout) {
            if ($timeout_prev) {
                # Если до вызова do_safely был установлен alarm, переустанавливаем его
                my $tm_end = time;
                $timeout_prev -= ($tm_end - $tm_begin);
                $timeout_prev = 1  if $timeout_prev <= 0;
                alarm $timeout_prev;
            } else {
                alarm 0;
            }
        };
        if ($@) {
            #die unless $@ eq "alarm\n"; # propagate unexpected errors
            $error_msg = $@;
            &$print_err_ref("do_safely: error_msg: $error_msg") if $verbose;
            if ($timeout  and  $error_msg =~ /do_safely alarm/) {
                # timed out
                &$print_err_ref("do_safely: timeout_exceeded") if $verbose;
                $timeout_exceeded = 1;
            }
        }
        #$SIG{ALRM} = $sig_alrm; #      if $sig_alrm;
    }

    if ($timeout_exceeded) {
        &$print_err_ref("WARN: do_safely timeout_exceeded: $timeout_exceeded")   if $verbose;
        $error_msg = $msg_timeout;
        $should_die = $die_if_timeout ? 1 : 0;

        if (my $timeout_handler = $prm{timeout_handler}) {
            if (ref($timeout_handler) eq 'CODE') {
                &$print_err_ref("do_safely timeout_handler ...");  # if $verbose;
                $timeout_handler->();
                &$print_err_ref("do_safely timeout_handler done");  # if $verbose;
            } else {
                &$print_err_ref("ERROR: do_safely: bad timeout_handler!");
                $error_msg .= " Bad timeout_handler!";
            }
        }
    }

    if (defined $error_msg) {
        # Error
        unless ($timeout_exceeded) {
            if (my $die_handler = $prm{die_handler}) {
                if (ref($die_handler) eq 'CODE') {
                    &$print_err_ref("do_safely die_handler ...");
                    $die_handler->();
                    &$print_err_ref("do_safely die_handler done");
                } else {
                    &$print_err_ref("ERROR: do_safely: bad die_handler!");
                    $error_msg .= " Bad die_handler!";
                }
            }
        }
        chomp $error_msg;
        if ($should_die) {
            &$print_err_ref("do_safely: failed: $error_msg") if $verbose;
            die $error_msg;
        } else {
            &$print_err_ref(($prm{no_error} ? "WARN": "ERROR") . ": do_safely: $error_msg");
            return;
        }
    } else {
        # OK
        &$print_err_ref("do_safely: success") if $verbose;
        return ($wantarray ? @res : $res);
    }
}

sub mem_bytes {
    my $mem_str = shift;
    return undef unless $mem_str =~ /^\s*(\d+)\s*([gGmMkK])\s*$/;
    my ($num, $mult) = ($1, $2);
    if ($mult eq 'g' or $mult eq 'G') {
        return $num * 2**30;
    } elsif ($mult eq 'm' or $mult eq 'M') {
        return $num * 2**20;
    } else {
        return $num * 2**10;
    }
}

sub file_bytes {
    my ($file) = @_;
    my @stat = stat $file;
    return undef if not @stat;
    return $stat[7];
}

# ротация файлов; дописываем timestamp
sub rotate_files {
    my $file = shift;
    my $n_files = shift || 5;

    my $timestamp = &get_timestamp(&modtime($file));

    my ($dir, $filename) = ($file =~ m/^(.*)\/([^\/]*)$/);
    opendir DIR, $dir;
    my @old_files = sort { $a cmp $b } grep { /^$filename\.\d{8}_\d{6}$/ } readdir(DIR);
    close DIR;

    return if @old_files and $old_files[$#old_files] eq $filename.'.'.$timestamp;  # nothing to do!

    link $file, $file.'.'.$timestamp;

    unlink map { $dir.'/'.$_ } @old_files[0..($#old_files - $n_files + 1)];
}


my $encoders = {};
sub get_encoder {
    my $encoding = shift;
    unless ( exists $encoders->{$encoding} ) {
        $encoders->{$encoding} = Encode::find_encoding($encoding) or die "Unknown encoding: $encoding";
    }
    return $encoders->{$encoding};
}

sub make_good_utf_string {
    my $str = shift;
    return get_encoder('UTF-8')->decode(get_encoder('UTF-8')->encode($str));
}

sub get_timestamp {
    my $unixtime = shift || time;
    return POSIX::strftime("%Y%m%d_%H%M%S", localtime($unixtime));
}

sub log_time_fmt {
    return POSIX::strftime("%Y-%m-%d %H:%M:%S", localtime);
}

sub log_time_hires_fmt {
    my ($sec, $msec) = Time::HiRes::gettimeofday();
    my $str = POSIX::strftime("%Y-%m-%d %H:%M:%S", localtime($sec));
    $str .= sprintf ".%06d", $msec;
    return $str;
}

sub log_msg_fmt {
    my $time = log_time_fmt();
    return join("\t", $time, "[$$]", shift);
}

sub print_log {
    print STDOUT log_msg_fmt(shift), "\n";
    return 1;
}

sub print_err {
    print STDERR log_msg_fmt(shift), "\n";
    return 1;
}

# Input:
#   DIE => { хэш настроек обработки DIE }
#       stack_trace => 0/1  выводить также stack_trace
#       format => 'compact'  -  выводить stack_trace в компактном формате (для stack_trace => 1)
#   WARN => { хэш настроек обработки WARN } - аналогично настройкам DIE => { }
#   log_warnings_to_file => Файл, в который будет перенаправлен вывод warn
# Examples:
#   handle_errors();
#   handle_errors(log_warnings_to_file => $log_file);
#   handle_errors(WARN => {stack_trace => 1}, DIE => {stack_trace => 1}, );
#   handle_errors(WARN => {stack_trace => 0}, DIE => {stack_trace => 1, format => 'compact'}, );
my $fh_warnings;
sub handle_errors {
    my (%prm) = @_;

    $SIG{__DIE__} = sub {
        my $msg = log_msg_fmt('ERROR: died: '.shift);
        if ($prm{DIE}{stack_trace}) {
            my $format = $prm{DIE}{format} // '';
            chomp $msg   if $format eq 'compact';
            my %st_prm = $format eq 'compact' ? (format => 'one-line') : (prefix => '__DIE__  ');
            $msg .= " " . stack_trace(%st_prm);
        }
        die($msg);
    };

    if (defined $prm{log_warnings_to_file}) {
        if ($prm{log_warnings_to_file}) {
            open $fh_warnings, ">>", $prm{log_warnings_to_file} or die "Could not open $prm{log_warnings_to_file} ($!)";
            $fh_warnings->autoflush(1);
        } else {
            $fh_warnings = undef;
        }
    }

    $SIG{__WARN__} = sub {
        my $msg = log_msg_fmt('WARN: '.shift);
        if ($prm{WARN}{stack_trace}) {
            my $format = $prm{WARN}{format} // '';
            chomp $msg   if $format eq 'compact';
            my %st_prm = $format eq 'compact' ? (format => 'one-line') : (prefix => '__WARN__ ');
            $msg .= " " . stack_trace(%st_prm);
        }
        if ($fh_warnings) {
            print $fh_warnings $msg;
        } else {
            warn($msg);
        }
    };

    $SIG{TERM} //= sub {
        print_err("ERROR: SIG{TERM} was received! Exiting...");
        exit(1);
    };

    $SIG{INT} //= sub {
        print_err("ERROR: SIG{INT} was received! Exiting...");
        exit(1);
    }

    # Не меняем SIG{PIPE} в продакшне, но можно использовать в отладочных целях.
    # См. https://st.yandex-team.ru/BSDEV-61728
    # TODO - сделать мониторинг корректной работы с SIG{PIPE}
    #$SIG{PIPE} //= sub {
    #    print_err("SIG{PIPE} was received!");
    #};
}

sub url_encode {
    my $rv = shift;
    $rv =~ s/([^a-z\d\Q.-_~ \E])/sprintf("%%%2.2X", ord($1))/geix;
    $rv =~ tr/ /+/;
    return $rv;
}

sub url_decode {
    my $rv = shift;
    $rv =~ tr/+/ /;
    $rv =~ s/\%([a-f\d]{2})/ pack 'C', hex $1 /geix;
    return $rv;
}

sub url_decode_safe { 
    my $url = shift;
    if ($url =~ m/\%[a-f\d]{2}/i) {
        eval { $url = Encode::encode('utf8', $url); $url = url_decode($url); $url = Encode::decode('utf8', $url); };
    }
    return $url;
}


# same as bs: https://a.yandex-team.ru/arc/trunk/arcadia/library/digest/md5/md5.cpp?rev=5077534#L191
# same as pylib: bm.utils.md5int
# md5int("qwertyuiopqwertyuiopasdfghjklasdfghjkl") == 11753545595885642730
# предполагается, что на входе - UTF-8 строка
sub md5int {
    my ($str) = @_;
    my $str_bytes = get_encoder('UTF-8')->encode($str);
    my @a = unpack("N4", Digest::MD5::md5($str_bytes));
    return ($a[1] ^ $a[3]) << 32 | ($a[0] ^ $a[2]);
}

sub md5int_base64 {
    my ($str) = @_;
    my $str_bytes = get_encoder('UTF-8')->encode($str);
    my @a = unpack("N4", Digest::MD5::md5($str_bytes));
    my $hash_bytes = pack("N2", $a[1] ^ $a[3], $a[0] ^ $a[2]);  # big-endian
    my $base64 = MIME::Base64::encode_base64($hash_bytes, '');
    $base64 =~ s/=$//;  # убираем padding
    return $base64;
}

sub time_unix2db {
    my $time = shift;
    return strftime("%Y-%m-%d %H:%M:%S", localtime $time);
}

sub time_db2unix {
    my $time_str = shift;
    return unless defined $time_str;
    my @fld = ($time_str =~ /^(\d{4})-(\d\d)-(\d\d) (\d\d):(\d\d):(\d\d)$/);
    return undef unless @fld;
    my ($year, $mon, $mday, $hour, $min, $sec) = @fld;
    $year -= 1900;
    $mon  -= 1;
    return timelocal($sec, $min, $hour, $mday, $mon, $year);
}

# Время в логах может быть записано как '2018-11-06 17:50:30' или '2018-11-06 17:50:30.922125'
sub time_log2unix {
    my $time_str = shift;
    my ($time_db_str) = ($time_str =~ /^(\d{4}-\d\d-\d\d \d\d:\d\d:\d\d)(\.\d{6})?$/);
    return time_db2unix($time_db_str);
}

# подходит также и для SB
# примеры:
# YT: "2018-12-05T13:23:45.383174Z"
# SB: "2018-12-05T13:23:45Z"
sub time_yt2unix {
    my $time_str = shift;
    my @fld = ($time_str =~ /^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})(?:\.\d{6})?Z$/);
    return undef unless @fld;
    my ($year, $mon, $mday, $hour, $min, $sec) = @fld;
    $year -= 1900;
    $mon  -= 1;
    return timegm($sec, $min, $hour, $mday, $mon, $year);
}

# обертка вокруг File::Temp::tempfile
# не экспортируем, а то конфликтует с методом Projpart
# get_tempfile($name, DIR => $dir, ...) - см. File::Temp
sub get_tempfile {
    my $name = shift || 'Utils_Sys_tmp';
    my %par = (
        DIR => $Utils::Common::options->{dirs}{temp},
        @_,
    );
    unless (-d $par{DIR}) {
        system('mkdir -p '.$par{DIR});
    }
    my ($file_handle, $file_name) = File::Temp::tempfile("$name.tmp.$$.XXXX", %par);
    close $file_handle;
    chmod 0644, $file_name;
    return $file_name;
}

# Удалить временные файлы, созданные sub get_tempfile
# На входе - префикс имени файла
# Дополнительные параметры:
#   DIR         - директория временных файлов (по умолчанию - $Utils::Common::options->{dirs}{temp})
#   hours_old   - удалять файлы только старше этого возраста
sub cleanup_tempfiles {
    my $name = shift || 'Utils_Sys_tmp';
    my %par = (
        DIR => $Utils::Common::options->{dirs}{temp},
        hours_old => undef,
        @_,
    );
    my $dir = $par{DIR}
        or die "Void DIR in cleanup_tempfiles!";
    my $name_re = $name;
    $name_re =~ s/\./\\./g;
    my @file_names = grep { /^$name_re\.....$/ } dir_files($dir);
    if (defined $par{hours_old}) {
        @file_names = grep { (time - mtime("$dir/$_")) > $par{hours_old} * 3600 } @file_names;
    }
    print_err("cleanup_tempfiles $dir $name ( @file_names )");
    for my $file (map {"$dir/$_"} @file_names) {
        print_err("unlink $file");
        unlink $file;
    }
}

#Получаем текст и функцию
#Обрабатываем через внешний файл каждую строку функцией
#Возвращает обработанную строку
sub file_text_sub {
    my $text = shift;
    my $sbr  = shift;
    my $dir  = shift @_ || '';
    my $filename_src;
    my $filename_dst;
    if ($dir) {
        $filename_src = get_tempfile('file_sub_src', UNLINK => 1, DIR => $dir);
        $filename_dst = get_tempfile('file_sub_dst', UNLINK => 1, DIR => $dir);
    } else {
        $filename_src = get_tempfile('file_sub_src', UNLINK => 1);
        $filename_dst = get_tempfile('file_sub_dst', UNLINK => 1);
    }
    open(F, "> $filename_src") || die("Cant open file '".$filename_src."' - $@");
    binmode(F);
    print F $text;
    close(F);
    open(F2, "< $filename_src") || die("Cant open file '".$filename_src."' - $@");
    open(F3, "> $filename_dst") || die("Cant open file '".$filename_dst."' - $@");
    while(<F2>){
        my $el = $_;
        print F3 &$sbr;
    }
    close(F2);
    close(F3);

    my $prev = $/;
    $/ = undef;
    open(F4, "< $filename_dst") || die("Cant open file '".$filename_dst."' - $@");
    my $res = <F4>;
    close(F4);
    $/ = $prev;
    unlink($filename_src);
    unlink($filename_dst);
    return $res;
}

sub file_result_sub {
    my $filename_src = shift;
    my $sbr  = shift;
    my $dont_need_result = shift;
    my $dir  = shift @_ || '';
    my $filename_dst;
    if ($dir) {
        $filename_dst = get_tempfile('file_sub_dst', UNLINK => 1, DIR => $dir);
    } else {
        $filename_dst = get_tempfile('file_sub_dst', UNLINK => 1);
    }

    open(F2, "< $filename_src") || die("Cant open file '".$filename_src."' - $@");
    open(F3, "> $filename_dst") || die("Cant open file '".$filename_dst."' - $@");
    while(<F2>){
        my $el = $_;
        print F3 &$sbr;
    }
    close(F2);
    close(F3);

    my $str = '';
    if ($dont_need_result) {
        unlink($filename_dst);
        return \$str;
    }

    my $prev = $/;
    $/ = undef;
    open(F4, "< $filename_dst") || die("Cant open file '".$filename_dst."' - $@");
    my $res = <F4>;
    close(F4);
    $/ = $prev;
    unlink($filename_src);
    unlink($filename_dst);
    return \$res;
}

# cat file1 file2 .. | md5sum
sub get_files_md5 {
    my @files = @_;
    my $md5_obj = Digest::MD5->new;
    for my $file (@files) {
        open my $fh, '<', $file
            or die "Can't open file `$file': $!";
        binmode $fh;
        $md5_obj->addfile($fh);
        close $fh;
    }
    return $md5_obj->hexdigest;
}

sub switch_to_ipv6 {
    eval {
        require IO::Socket::INET6;
    };
    if ($@) {
        return 0;
    } else {
        IO::Socket::INET6->import;
    }
    # взято из Net::INET6Glue::INET_is_INET6
    ############################################################################
    ## copy IO::Socket::INET to IO::Socket::INET4
    ## copy IO::Socket::INET6 to IO::Socket::INET
    #############################################################################

    return 1 if $INC{'IO/Socket/INET.pm'} eq $INC{'IO/Socket/INET6.pm'};

    $INC{'IO/Socket/INET4.pm'} = $INC{'IO/Socket/INET.pm'};
    $INC{'IO/Socket/INET.pm'}  = $INC{'IO/Socket/INET6.pm'};

    {
        # copy subs
        no strict 'refs';
        
        # redefining constant subroutine ignores no warnings pragma
        local $SIG{'__WARN__'} = sub { CORE::warn(@_) if $_[0] !~ m/[sS]ubroutine.*redefined/ };
        for ( keys %{IO::Socket::INET::} ) {
            ref(my $v = $IO::Socket::INET::{$_}) and next;
            *{'IO::Socket::INET4::'.$_} = \&{ "IO::Socket::INET::$_" } if *{$v}{CODE};
        }

        for ( keys %{IO::Socket::INET6::} ) {
            ref(my $v = $IO::Socket::INET6::{$_}) and next;
            *{'IO::Socket::INET::'.$_} = \&{ "IO::Socket::INET6::$_" } if *{$v}{CODE};
        }
    }
    return 1;
}

sub switch_to_ipv4 {
    eval {
        require IO::Socket::INET6;
    };
    if ($@) {
        return 0;
    } else {
        IO::Socket::INET6->import;
    }
    # взято из Net::INET6Glue::INET_is_INET6
    ############################################################################
    ## copy IO::Socket::INET4 to IO::Socket::INET
    #############################################################################

    return 1 if $INC{'IO/Socket/INET.pm'} ne $INC{'IO/Socket/INET6.pm'};
    $INC{'IO/Socket/INET.pm'}  = $INC{'IO/Socket/INET4.pm'};

    {
        # copy subs
        no strict 'refs';
        # redefining constant subroutine ignores no warnings pragma
        local $SIG{'__WARN__'} = sub { CORE::warn(@_) if $_[0] !~ m/[sS]ubroutine.*redefined/ };

        for ( keys %{IO::Socket::INET4::} ) {
            ref(my $v = $IO::Socket::INET4::{$_}) and next;
            *{'IO::Socket::INET::'.$_} = \&{ "IO::Socket::INET4::$_" } if *{$v}{CODE};
        }
    }
    return 1;
}

sub is_ipv4_host {
    my $host = shift;
    my $section_re = '(25[0-5]|2[0-4]\d|[0-1]?\d{1,2})';
    my $ipv4_re = '^'."$section_re(\.$section_re){3}".'$';
    return $host =~ /$ipv4_re/;
}

sub get_runlevel {
    my $rl_result = `/sbin/runlevel`;
    my ($rl) = $rl_result =~ /(\d+)$/;

    return $rl;
}

# прочитать данные из tsv-файла с заголовком (как в tabtools)
# доп. параметры:
#   max_rows => N     прочитать не более N записей
#   fields => [ .. ]  прочитать только поля из заданного списка
#   file_fields => [..]  список полей, вместо хедера
#   skip_lines  => N  пропустить первые N строк
#   skip_decode_tsv => 1  вернуть значения "как есть", без расквочивания
sub read_ts_file {
    my $file = shift;
    my %par  = @_;

    my @rows;
    open my $fh, '<', $file
        or return;
    my @fld;
    if ($par{file_fields}) {
        @fld = @{$par{file_fields}};
    } else {
        my $header = <$fh>;
        chomp $header;
        return if $header !~ /^#/;
        $header =~ s/^\#\s*//;
        @fld = split /\s*\t\s*/, $header;
    }

    my $use_ind;
    if ($par{fields}) {
        my %use = map { $_ => 1 } @{$par{fields}};
        $use_ind = [ grep { $use{$fld[$_]} } (0 .. $#fld) ];
        @fld = @fld[@$use_ind];
    }

    my $line_cnt = 0;
    while (<$fh>) {
        chomp;
        $line_cnt++;
        next if defined $par{skip_lines} and $line_cnt <= $par{skip_lines};
        my %r;
        my @d = split /\t/, $_;
        @d = @d[@$use_ind] if $use_ind;
        @d = map { decode_tsv($_) } @d unless $par{skip_decode_tsv};
        @r{@fld} = @d;
        push @rows, \%r;
        last if defined($par{max_rows}) and @rows >= $par{max_rows};
    }
    close $fh;

    return \@rows;
}

sub read_ts_header {
    my $file = shift;
    my @rows;
    open my $fh, '<', $file
        or return;
    my $header = <$fh>;
    chomp $header;
    return if $header !~ /^#/;
    $header =~ s/^\#\s*//;
    my @fld = split /\s*\t\s*/, $header;
    close $fh;

    return \@fld;
}


# return undef if can't open or invalid json
sub load_json {
    my $file = shift;
    my $json_flags = shift || {};
    local $/;
    open my $fh, '<', $file
        or return;
    my $json_str = <$fh>;
    close $fh;
    my $data = eval { from_json($json_str, { pretty => 1, utf8 => 0, %$json_flags}) };
    return $@ ? undef : $data;
}

sub save_json {
    my $data = shift;
    my $file = shift;
    my $json_str = to_json($data, { pretty => 1, utf8 => 0});
    my $dir = dirname($file);
    my $temp_file = get_tempfile('json', DIR => $dir);
    open my $fh, '>', $temp_file
        or return;
    print $fh $json_str;
    close $fh
        or return;
    rename $temp_file, $file
        or do { unlink $temp_file; return };  # fail
    return 1;  # OK
}

# eval с кэшированием результата в файле (json-encoded; функция должна вовзращать ref)
# load_or_eval($file, sub { .... })
# доп. параметры:
#   force =>  не читать данные из файла, вызывать функцию
sub load_or_eval {
    my $file = shift;
    my $func = shift;
    my %par  = @_;
    my $data;
    if (-f $file and !$par{force}) {
        $data = load_json($file);
    } else {
        $data = $func->();
        save_json($data, $file);
    }
    return $data;
}

sub number_of_fields {
    my $file = shift;
    open my $fh, '<', $file
        or return undef;
    my $line = <$fh>;
    close $fh;
    return undef if !defined $line;
    my @fld = split /\t/, $line, -1;
    return scalar @fld;
}


sub round {
    my ($number, $prec, %opts) = @_;

    $number ||= 0;
    $prec ||= 2;

    my $half = 0.50000000000008; # число взято из модуля Math::Round

    my $multiply = 10**$prec;
    if ($number >= 0) {
        $number = int($number*$multiply+$half)/$multiply;
    } else {
        $number = int($number*$multiply-$half)/$multiply;
    }

    return sprintf("%.${prec}f", $number);
};

sub format_number {
    my ($num, $precision) = @_;
    $num ||= 0;
    $precision ||= 0;

    $num = round($num, $precision);

    my ($n, $d) = split /\./, sprintf( "%.".$precision."f", $num );
    if (abs($n) > 9999) { # do not separate thousands in 4 digit numbers (1234 and 12 345)
        my $separator = ' ';
        my @a = ();
        while ($n =~ /\d\d\d\d/) {
            $n =~ s/(\d\d\d)$//;
            unshift @a, $1;
        }
        unshift @a,$n;
        $n = join $separator, @a;
    }
    $n = "$n\,$d" if $d && $d =~ /\d/;
    return $n;
}

# псевдослучайная перестановка на основе md5
# на входе:
#   $arr  -  ссылка на массив строк
#   $seed -  seed для "случайности", по умолчанию seed=time
sub md5_shuffle {
    my $arr = shift;
    my $seed = shift // time;
    return map { $_->[0] } sort { $a->[1] <=> $b->[1] } map { [ $_, md5int($seed.' '.$_) ] } @$arr;
}


# Вызывает rsync с заданными параметрами
# На входе:
#   Путь, передаваемый в rsync
#   Дополнительные параметры:
#       contimeout, timeout   -  соответствующие параметры rsync (по умолчанию: 120)
#       system_timeout => 1/0 Использовать команду timeout (полезно, если rsync "зависает")
#       bwlimit, min-size  -   соотв. параметры rsync
#       ipv4, ipv6 => 1/0   Использовать ipv4 или ipv6. Если не задан ни один из этих параметров, запускается с параметром ipv6, а в случае неудачи - с параметром ipv4
#       logger              Если задано, то сообщения выводятся методом $par{logger}->log()
#       add_opt             Строка опций, которые будут дописаны при вызове rsync
#       out                 Путь для записи скачанного файла
#       max_attempt_count   Максимальное число попыток вызова rsync (по умолчанию: 5)
#       WARN_ONLY => 1/0    В случае неудачи, в лог вместо ERROR пишется WARN
#       READ => 1/0         Вернуть выдачу rsync
#       ignore_stderr => 1/0    Не писать в лог выдачу STDERR от rsync (STDERR будет перенаправлен в rsync_ignore.err, который может быть исключен из мониторинга)
# Пример вызова:
#   my $result = rsync("rsync://$host/bmexport/$link", out => "$tmpd/$host.json", add_opt => "--copy-links", logger => $proj, WARN_ONLY => 1,)
#   my $output = rsync("rsync://$host/bmexport/", add_opt => "--copy-links", logger => $proj, WARN_ONLY => 1, READ => 1, );
sub rsync {
    my $path = shift;
    my %par  = (
        contimeout => 120,
        timeout => 120,
        max_attempt_count => 5,
        @_,
    );

    return if not $path;

    my $logger = $par{logger};
    my $out = $par{out} // '';
    my $add_opt = $par{add_opt} // '';

    my %rsync_opt = (
        (map { $_ => $par{$_} }  grep { defined $par{$_} }  qw( contimeout timeout bwlimit min-size )),
        (map { $_ => '' }  grep { defined $par{$_} } qw( ipv4 ipv6 )),
    );
    my $rsync_opt = join(' ',
        (map { "--$_" . ($rsync_opt{$_} ne ''  ?  "=".$rsync_opt{$_} : '') } keys %rsync_opt),
        $add_opt,
    );

    #my $cmd_rsync = "rsync --contimeout=120 --timeout=120 --copy-links rsync://$host/bmexport/$link $tmpd/$host.indc.log";

    my $max_attempt_count = $par{max_attempt_count};
    my $attempt_count = 0;
    my $res = 0;
    my $output = '';
    my @add_ipv_opts = (defined $rsync_opt{ipv4} or defined $rsync_opt{ipv6}) ? ('') : ('--ipv6', '--ipv4');

    while (!$res  and  $attempt_count < $max_attempt_count) {
        for (@add_ipv_opts) {
            my $cmd_rsync = "rsync $rsync_opt $path $out $_ ";
            if ($par{system_timeout}) {
                my $system_timeout = int($par{system_timeout});
                $cmd_rsync = "timeout --kill-after 600s $system_timeout $cmd_rsync";
            }
            $cmd_rsync .= " 2>>" . $Utils::Common::options->{dirs}{log} . "/rsync_ignore.err"   if $par{ignore_stderr};
            $logger->log("rsync ($cmd_rsync)")   if $logger;
            $output = `$cmd_rsync`;
            if ($? == 0) {
                $res = 1;
                last;
            };
            $logger->log("WARN: rsync failed ($cmd_rsync) (".($?>>8)."): $! (attempt $attempt_count)")   if $logger;
        }
        ++$attempt_count;

        sleep(5 + int rand 5)  if !$res  and  $attempt_count < $max_attempt_count;
    };

    if ($logger) {
        $logger->log($res  ?
                  "rsync ($path) done"
                : ($par{WARN_ONLY} ? 'WARN' : 'ERROR') . ": rsync ($path) failed, $attempt_count attempts done");
    }

    return $output   if $par{READ};
    return $res;
}

sub unzipdata {
    my ($data) = @_;
    use IO::Uncompress::Unzip qw(unzip $UnzipError) ;

    my $res = "";
    IO::Uncompress::Unzip::unzip( \$data => \$res )
        or die "unzip failed: $UnzipError\n";

    _utf8_off($res);

    return $res;
}

sub uncompressdata {
    my ($data) = @_;

    my $res = "";

    anyuncompress( \$data => \$res )
        or die "anyuncompress failed: $AnyUncompressError\n";
    _utf8_off($res);

    return $res;
}

#Распаковка файлов
sub uncompressfile {
    my ($filein, $fileout) = @_;

    my $res = "";
    my $status = 0;
    $status = anyuncompress( $filein => $fileout, MultiStream => 1) or do_sys_cmd("unzip -p $filein > $fileout", timeout => 600, no_die => 1) or die "anyuncompress failed: $AnyUncompressError\n";
#    do_sys_cmd("cp $fileout ~/TMP");
    return $status;
}

our $staticmaps;    # Хэш: ( путь к файлу  =>  объект StaticMap )
# Возвращает объект StaticMap по заданному пути к файлу. Если StaticMap для этого файла уже создан, он не пересоздается.
# На входе - путь к файлу.
# die, если файл не существует
# Параметры:
#   reload      =>  1/0   Пересоздать StaticMap для заданного файла
sub staticmap {
    my ($file, %par) = @_;
    $staticmaps //= {};
    if (not exists $staticmaps->{$file}  or  $par{reload}) {
        if ( ! -f $file ) {
            die "StaticMap file ($file) does not exist!";
        }
        $staticmaps->{$file} = StaticMap->new($file);
    }
    return $staticmaps->{$file};
}

# Возвращает value объекта StaticMap по заданному пути к файлу для заданного ключа. Если StaticMap для этого файла уже создан, он не пересоздается (в соответствии с параметрами для sub staticmap)
# На входе:
#   путь к файлу для StaticMap
#   ключ
# Параметры:
#   staticmap_prm   => хэш настроек для создания StaticMap - передается в sub staticmap
sub staticmap_value {
    my ($file, $key, %prm) = @_;
    return undef   unless defined $key; # Чтобы избежать "Segmentation fault"
    my $str = staticmap($file, %{ $prm{staticmap_prm} || {} })->Value($key);
    _utf8_on($str);
    return $str;
}

# Получить команду для вызова исходной команды через bash
sub get_bash_cmd {
    my ($cmd) = @_;
    $cmd =~ s/'/'"'"'/g;
    $cmd = "/bin/bash -c 'set -o pipefail; $cmd'";
    return $cmd;
}

# Вызов system для заданной команды с проверкой результата выполнения и логированием
# На входе:
#   вызываемая команда
# Параметры:
#   read        => 0/1  Вернуть текст, напечатанный командой в STDOUT (аналог read_sys_cmd). Если у команды ненулевой exit code, возвращает undef
#                       TODO  С опцией no_die возвращать текст даже при ненулевом exit code? Сделать для этого отдельный параметр?
#   silent      => 0/1  Не выводить сообщений в лог
#   logger      => $obj  Объект для логирования
#   log_to_STDOUT => 0|1 Выводить сообщения в STDOUT вместо STDERR (если не задан logger)
#   no_die      => 0|1  Не вызывать die в случае exit_status!=0
#   no_error    => 0|1  Не выводить ERROR в случае exit_status!=0
#   timeout     => timeout в секундах. Через это время выполняемой команде будет отправлен SIGTERM. Если вызываемая команда состоит из нескольких команд, то timeout будет действовать на все команды при указании shell => 'bash'
#   kill_after  => если задан параметр timeout - время в секундах: also send a KILL signal if COMMAND is still running this long after the initial signal was sent.
#   shell       => 'bash', ''  Выполнять команду в дефолтном интерпретаторе или в bash
sub do_sys_cmd {
    my ($cmd, %prm) = @_;
    my $print = sub {
        my $msg = shift;
        return if $prm{silent};
        my $logger = $prm{logger};
        if ($logger) {
            $logger->log($msg);
        } elsif ($prm{log_to_STDOUT}) {
            print_log($msg);
        } else {
            print_err($msg);
        }
    };

    if (($prm{shell} // '') eq 'bash') {
        $cmd = get_bash_cmd($cmd);
    }

    if ($prm{timeout}) {
        $cmd = join(" ",
            "timeout",
            $prm{kill_after} ? "--kill-after ".$prm{kill_after}."s" : (),
            $prm{timeout}."s",
            $cmd,
        );
        # TODO
        #$exitstatus2msg{124} = "Timeout";
        #$exitstatus2msg{128+9} = "TimeoutKill";
    }
    my $title_msg = ($prm{read} ? "read " : "") . "system cmd { $cmd }";
    $print->("$title_msg ...");

    my $exit_status = '';
    my $result;
    my @errors;
    my $start_time = [gettimeofday];
    if ($prm{read}) {
        open my $fh, $cmd . '|' or do {
            push @errors, "open failed ($!)";
        };
        if ($fh) {
            $result = '';
            while (my $line = <$fh>) {
                $result .= $line;
            }
            close $fh or do {
                push @errors, "close failed ($!)";
            };
            $exit_status = $?;
        }
    } else {
        $exit_status = system($cmd);
        if ($exit_status != 0) { # Error
            push @errors, "($!)";
        } else { # OK
            $result = 1;
        }
    }
    my $duration = tv_interval($start_time);

    if (@errors) {
        my $err_msg = join(" ",
            "failed($exit_status): " . join("; ", @errors),
            ". Duration: $duration",
        );
        $print->(($prm{no_error} ? "WARN" : "ERROR") . ": $title_msg $err_msg");
        if ($prm{no_die}) {
            return undef;
        } else {
            die "$title_msg $err_msg";
        }
    } else {
        $print->("$title_msg done. Duration: $duration");
    }
    return $result;
}

sub read_sys_cmd {
    my ($command, %prm) = @_;
    return do_sys_cmd($command, read => 1, %prm);
}

# получить уникальный (для хоста) идентификатор (для временного файла, таблицы и т.п.)
sub get_uniq_name {
    my $tmpl = shift // 'uniq_name';
    usleep(10);
    my $pid = $$;
    my ($sec, $msec) = gettimeofday();
    return join('_', 'Tmp', $tmpl, $pid, $sec, $msec);
}

sub _stack_trace {
    my $i = 1;
    my @st = ();
    while(++$i) {
        my ($package, $filename, $line, $subroutine, $hasargs,
            $wantarray, $evaltext, $is_require, $hints, $bitmask) = caller($i);

        last if !defined($package);

        push(@st, {
            package     => $package,
            filename    => $filename,
            line        => $line,
            subroutine  => $subroutine,
        });
    };
    return \@st;
};

sub stack_trace {
    my (%prm) = @_;
    my $arr = _stack_trace;
    my $text;
    my @lines = map { $_->{subroutine}." <= ".$_->{package}." ".$_->{line} } reverse @$arr;
    if ( ($prm{format} // '') eq 'one-line' ) {
        $text = join " // ", @lines;
    } else {
        my $prefix = $prm{prefix} // '    ';
        $text = join "", map { $prefix . $_ . "\n" } @lines;
    }
    return $text;
}

# Дождаться завершения процессов, порожденых fork
# На входе:
#   pids    => pid'ы, которых нужно дождаться. Если не задано, то все  waitpid(-1, WNOHANG)
# На выходе:
#   1, если процессы завершились успешно
#   0, иначе
sub wait_children {
    my %prm = @_;
    print_log("wait_children ... " . to_json(\%prm));
    my $pid2isfinished = $prm{pids} ? { map { $_ => 0 } @{$prm{pids}} } : undef;
    my @pids_failed;
    my @pids_ok;
    my $kid = 1;
    while ( $kid != -1 ) {
        $kid = waitpid(-1, WNOHANG);
        if ($kid > 0) {
            #print_log("kid: $kid");
            if ( not $pid2isfinished or $pid2isfinished and defined $pid2isfinished->{$kid} ) {
                my ($rc, $sig, $core) = ($? >> 8, $? & 127, $? & 128);
                my $ok = ($rc or $sig or $core) ? 0 : 1;
                print_log("wait_children kid:$kid res:$ok (rc:$rc, sig:$sig, core:$core) '$!'");
                if ($ok) {
                    push @pids_ok, $kid;
                } else {
                    push @pids_failed, $kid;
                }
                $pid2isfinished->{$kid} = 1   if $pid2isfinished;
            }
        } else {
            sleep(1);
        }
        if ($pid2isfinished) {
            last unless grep { not $_ } values %$pid2isfinished; # Выходим, если все интересующие нас процессы завершили работу
        }
    }
    if ($pid2isfinished) {
        my @bad = grep { not $pid2isfinished->{$_} } sort keys %$pid2isfinished;
        if (@bad) {
            # Smth strange
            print_log("wait_children: warn: bad: @bad");
            push @pids_failed, @bad;
        }
    }

    if (@pids_failed) {
        print_log("wait_children done. Failed children: @pids_failed. Done children: @pids_ok");
        return 0;
    } else {
        print_log("wait_children done. Done children: @pids_ok");
        return 1;
    }
}

#####
# Разбивает строку csv по разделителю delim
# Default delim = ','
#
sub split_csv_line {
    my ( $text, $delim ) = @_;
    $delim = ',' unless $delim;

    # заменяем отзеркаленные кавычки на спецстроку
    my $has_specquote = $text =~ s/\\"/___SPECQUOTE___/g;

    my @res = ();
    my @buffer = ();
    my $qq = 0;
    my $re = quotemeta($delim);
    foreach my $part ( split /$re/, $text, -1 ) {
        $qq += ($part =~ tr/"//); 
        push @buffer, $part;
        unless ($qq % 2) {
            push @res, join($delim,@buffer);
            @buffer = (); 
            $qq = 0;
        }
    }
    push @res, join(',',@buffer) if @buffer;

    @res = map { $_ =~ s/^\"|\"$//g; $_ } @res;
    if ( $has_specquote ) {
        #если все встреченные ранее specquote закончились до конца массива, не проверяем оставшуюся часть
        @res = map { my $line = $_; if ($has_specquote){ $has_specquote -= $line =~ s/___SPECQUOTE___/\"/g }; $line } @res;
    }
#    print STDERR Dumper ( [ $re, $text, \@res ] );
    return @res;
}

# 1 если строки равны или обе undef
sub are_strings_equal {
    my ($a, $b) = @_;
    if (defined $a && defined $b) {
        return $a eq $b;
    } else {
        return ! defined $a && ! defined $b;
    }
}

# 1 обе ссылки undef или ссылки указывают на тот же объект
# если какой-то объект не ссылка, то вернет пустую строку
sub are_refs_same {
    my ($a, $b) = @_;
    if (defined $a && defined $b) {
        return ref($a) && ref($b) && $a == $b;
    } else {
        return ! defined $a && ! defined $b;
    }
}

sub ref_to_weaken_ref {
    my $weak_ref = shift;
    weaken($weak_ref);
    return \$weak_ref;
}

# скомпилировать comptrie_subphraser
# на входе:
#   input =>  init_data_file для обычного сабфрейзера
#   output =>  comptrie file
#   temp_dir =>  subj (необязательный параметр)
# нужен ya make --checkout ads/quality/bm_subphraser/bin/converter
sub make_comptrie_subphraser {
    my %par = @_;
    my $options = $Utils::Common::options;
    my %temp_par = (UNLINK => 1);
    $temp_par{DIR} = $par{temp_dir} if defined $par{temp_dir};
    my $temp = get_tempfile('subphraser_data.comptrie', %temp_par);
    my $binary = $options->{dirs}{arcadia} . 'ads/quality/bm_subphraser/bin/converter/subphraser_init_data_converter';
    do_sys_cmd("$binary --init-data=$par{input} --trie=$temp");
    rename($temp, $par{output}) or die($!);
}

sub transform_v4_ip {
    my $ipv4 = shift // die "provide ipv4";
    my $nat64_prefix = shift // "64:ff9b::";  # https://tools.ietf.org/html/rfc6052#section-2.1

    return sprintf("[$nat64_prefix%x%x:%x%x]", split /\./, $ipv4);
}

# проверить, что скрипт запущен на хосте из списка, поведение переопределяется через debug_env
# на входе:
#   allow => список параметров для Utils::Hosts::get_hosts(), или строка "nothing"
#   host => хост, для которого проверить (необязательный, по умолчанию - текущий)
sub assert_curr_host {
    my %params = @_;
    
    my $allowed = $params{allow} // die "provide allow";
    my $host = $params{host};

    return 1 if $Utils::Common::options->{override_assert_curr_host};
    return 0 if !ref($allowed) && $allowed eq 'nothing';
    
    $host //= Utils::Hosts::determine_curr_host();

    # почему не get_curr_host(), а именно determine_curr_host():
    # нужно уметь отлавливать запуски в духе `BM_CURR_HOST=production01i.yandex.ru ./production_script.pl`
    # потому что имелось в виду скорее всего `BM_CURR_HOST=production01i.yandex.ru BM_USE_PATCH=debug_env ./production_script.pl`
    
    my @allowed_hosts = get_hosts(%$allowed);
    return scalar grep { $host eq $_ } @allowed_hosts;
}

# пакует файлы в .tgz из корня аркадии
# параметры:
#   $input_files:  список файлов
#   $output_file:  куда положить результат
sub pack_files_from_arcadia_root {
    my $input_files = shift;
    my $output_file = shift;

    my $dirs = $Utils::Common::options->{dirs};

    # у нас обычно симлинк /opt/arcadia/rt-research/broadmatching -> /opt/broadmatching
    # поэтому смотрим путь к файлу относительно broadmatching
    my $arcadia_root = Cwd::abs_path($dirs->{arcadia});  # also removes trailing slash
    my $bm_root = Cwd::abs_path($dirs->{root});
    my @arc_paths;
    for my $file (@$input_files) {
        my $rel_path = File::Spec->abs2rel($file, $bm_root);
        if (grep { $_ eq '..' } split '/', $rel_path) {  # можем выйти из аркадии, ну нафиг
            die "Bad abs2rel path for file $file: $rel_path!";
        }
        my $arc_path = "rt-research/broadmatching/$rel_path";
        if (!-f "$arcadia_root/$arc_path") {
            die "Bad filename, not reachable from arcadia root: $file";
        }
        push @arc_paths, $arc_path;
    }

    # линкуем файлы в отдельную директорию, на случай если они поменяются в момент запаковки
    my $temp_dir = File::Temp::tempdir("arc_root.$$.XXXX", DIR => $dirs->{temp}, CLEANUP => 1);
    my $temp_arc_root = "$temp_dir/arcadia";
    for my $path (@arc_paths) {
        my $target = Cwd::abs_path("$arcadia_root/$path");
        my $link = "$temp_arc_root/$path";
        File::Path::make_path(File::Basename::dirname($link));
        # делаем hard link, чтобы файл не менялся в процессе запаковки
        print_err("arcadia path $path: link $target <-- $link");
        link($target, $link) or die "Can't make link $target <-- $link: $!";
    }

    my $tar_file = "$temp_dir/tar_file";
    do_sys_cmd("tar cf $tar_file --dereference -C $temp_arc_root @arc_paths");

    my $tmp_out_file = "$temp_dir/out_file";

    # чтобы не менялся md5, важно задать опцию -n для gzip
    do_sys_cmd("gzip -n -c $tar_file > $tmp_out_file");

    rename($tmp_out_file, $output_file)  or die "Can't rename!";
}

sub is_ucs2_compatible {
    my ( $text ) = @_;
    for my $char ( split //, $text ) {
        my $ord = ord($char);
        if ( $ord > 65535 or $ord == 65533 or $char =~ /\p{gc=Cc}/ ) {
            return 0;
        }
    }
    return 1;
}

sub set_sv {
    # Включает/выключает сервис использую runit
    my $service = shift;
    my $is_need_enable = shift;
    if ($is_need_enable) {
        do_sys_cmd("ln -sf /etc/sv/$service /etc/service") if (! -l "/etc/service/$service");
    } else {
        do_sys_cmd("rm /etc/service/$service") if (-l "/etc/service/$service");
    }
}

sub set_web_config {
    # Данный метод управляет запуском и конфигом веб сервиса (apache2/nginx)
    # Если $config_full_filename_to_enable не указан: удаляет симлинки на все действующие конфиги (из sites-enabled)
    # Если указан то тестит его на синтаксис, выставляет его единственным симлинком в sites-enabled (остальные удаляет)
    # Если набор симлинков изменился и сервис был запущен до этого, то перезагружает конфиги для сервиса(для apache2 на данный момент полный рестарт)
    my $web_service = shift;
    my $config_full_filename_to_enable = shift // "";
    my $is_changed = 0;

    print_err "start set_web_config for $web_service";
    opendir(DIR, "/etc/$web_service/sites-enabled") or die "can't opendir /etc/$web_service/sites-enabled: $!";
    while (my $config_filename = readdir(DIR)) {
        next if $config_filename eq '.' or $config_filename eq '..';
        my $config_full_filename = "/etc/$web_service/sites-available/$config_filename";

        if (!$config_full_filename_to_enable or $config_full_filename ne "$config_full_filename_to_enable") {
            do_sys_cmd("rm /etc/$web_service/sites-enabled/$config_filename");
            $is_changed = 1;
        } elsif ($config_full_filename_to_enable) {
            $config_full_filename_to_enable = "";
        }
    }
    closedir(DIR);
    if ($config_full_filename_to_enable or $is_changed) {
        do_sys_cmd("ln -sf $config_full_filename_to_enable /etc/$web_service/sites-enabled");

        if (-l "/etc/service/$web_service") { # перезагружаем конфиги, если был запущен до этого
            print_err "need restart service $web_service";

            if ($web_service eq "nginx") { # для nginx подгружаем конфиг без перезагрузки nginx
                do_sys_cmd("nginx -s reload") or die("nginx reload config fail: $!");
            } else { # в apache такого не умеем, поэтому просто рестартуем сервис целиком
                do_sys_cmd("sv restart $web_service") or die("$web_service restart fail: $!");
            }
        }
    }
    print_err "end set_web_config for $web_service";
}

sub configure_runit_services {
    my $host_role = shift;
    # $host_role нужен для сервисов с веб конфигами: если будет найден конфиг данной роли, то данный сервис будет добавлен
    # в автозапауск runit, плюс данный веб конфиг будет добавлен в site-enable данного сервиса

    my $host_specific = shift;
    # нужен, чтобы на qloud дополнительно прописать в runit крон (см. host_specific_services в RunItConfig)

    my $run_it_config = $Utils::Common::options->{RunItConfig};

    if (!in_array($host_specific, [keys $run_it_config->{host_specific_services}])) {
        die "host_specific $host_specific not found; avaliable values: " .join(';', keys $run_it_config->{host_specific_services});
    };
    my @services_with_webconfigs = @{$run_it_config->{services_with_configs}};
    my @services_without_webconfigs = @{$run_it_config->{services_without_config}};
    push @services_without_webconfigs, @{$run_it_config->{host_specific_services}->{$host_specific}};

    my @avaliable_services = ();
    opendir(DIR, '/etc/sv') or die "ERROR: Cannot open directory: $!";;
    while (my $service_name = readdir(DIR)) {
        next if $service_name eq '.' or $service_name eq '..';
        push @avaliable_services, $service_name;
    }
    closedir(DIR);

    my @not_found_services = (grep {!in_array($_, \@avaliable_services)} @services_without_webconfigs);
    if (@not_found_services) {
        die "services '@not_found_services' not found in /etc/sv";
    };

    my $enabled_services = {};
    opendir(DIR, '/etc/service') or die "ERROR: Cannot open directory: $!";;
    while (my $service_name = readdir(DIR)) {
        next if $service_name eq '.' or $service_name eq '..';
        $enabled_services->{$service_name} = 1;
    }
    closedir(DIR);

    for my $web_service (@services_with_webconfigs) {
        my $web_config = '';
        # ищем конфиг для данной роли
        for my $config_filename ("${host_role}_$host_specific.conf", "$host_role.conf") {
            my $config_full_name = "/etc/$web_service/sites-available/$config_filename";
            if (-e $config_full_name) {
                $web_config = $config_full_name;
                last;
            }
        }

        print_err "$web_service: found config: $web_config";

        if ($web_config) { # нужно включить данный сервис с нужным конфигом
            # проверяем, что данный сервис доступен
            die "service '$web_service' not found in /etc/sv" if (!in_array($web_service, \@avaliable_services));
            set_web_config($web_service, $web_config);
            set_sv($web_service, 1);
        } elsif ($web_config eq '' and $enabled_services->{$web_service}) { # конфиг не нашли => если сервис включен выключаем
            set_web_config($web_service);
            set_sv($web_service, 0)
        }

        # убираем данный сервис из хеша, т.к. его уже обработали
        delete $enabled_services->{$web_service};
    }

    for my $service (@services_without_webconfigs) {
        if (!$enabled_services->{$service}) {
            set_sv($service, 1)
        }
        # убираем данный сервис из хеша, т.к. его уже обработали
        delete $enabled_services->{$service};
    }

    # в $enabled_services остались только выключеные, т.к. остальные обошли, выключаем их все
    for my $service (keys $enabled_services) {
        set_sv($service, 0)
    }
}

sub decode_base64_file {
    my ($in_file, $out_file) = @_;
    require MIME::Base64;

    open my $in_fh, '<', $in_file or die $!;
    binmode $in_fh;
    open my $out_fh, '>', $out_file or die $!;
    binmode $out_fh;

    while (1) {
        my $buffer;
        my $bytes_read = read $in_fh, $buffer, 100000; #безопасно, так как каждый символ в base64 дает ровно 6 бит, и любой буфер, кратный 4, даст целое число байт при декодировании
        die $! if !defined $bytes_read;
        last if !$bytes_read;
        print $out_fh MIME::Base64::decode_base64($buffer);
    }
    close $in_fh;
    close $out_fh;
}

sub decode_qp_file {
    my ($in_file, $out_file) = @_;
    require MIME::QuotedPrint;

    open my $in_fh, '<', $in_file or die $!;
    binmode $in_fh;
    open my $out_fh, '>', $out_file or die $!;
    binmode $out_fh;

    while (my $buffer = <$in_fh>) { #quoted-printable по стандарту бьются на короткие строки
        chomp $buffer;
        print $out_fh MIME::QuotedPrint::decode_qp($buffer);
    }
    close $in_fh;
    close $out_fh;
}

sub uncompress_web_file {
    my ($filein, $fileout, $content_encoding) = @_;
    $content_encoding //= '';
    my $from_file = get_tempfile('uncompress_web_file', UNLINK => 1);
    File::Copy::copy($filein, $from_file);

    for my $encoding (reverse split(/\s*,\s*/, lc($content_encoding))) {
        next if !$encoding || $encoding eq 'identity' || $encoding eq 'none';
        my $to_file = get_tempfile('uncompress_web_file', UNLINK => 1);
        if ($encoding eq 'base64') {
            decode_base64_file($from_file, $to_file);
        }
        elsif ($encoding eq 'quoted-printable') {
            decode_qp_file($from_file, $to_file);
        }
        else {
            anyuncompress($from_file, $to_file);
        }
        unlink $from_file;
        $from_file = $to_file;
    }
    File::Copy::move($from_file, $fileout);
}

1;
