package Utils::Urls;
use strict;

#Работа с урлами

use utf8;
use open ':utf8';
use URI;
use URI::_idna;
use Net::DNS;

use base qw(Exporter);
our @EXPORT = qw/
    validate_url
    get_sec_level_domain
    get_region_domains
    get_free_hostings
    canonical_url
    url_to_domain
    effective_domain
    banner_domain
    get_url_parts
    normalize_url
    remove_url_get_params
    url_to_punycode
    url_to_unicode
    safe_punycode_decode safe_punycode_encode
    external_dns_resolver
    get_internal_yandex_domains
    is_internal_yandex_url
    check_lwp_whitelist
    url_sort_get_params
    fix_url_scheme
/;

# Описание основных функций
#
# url_to_domain                    По урлу получить домен (почти без нормализации)
# effective_domain                 По домену получить домен 2-го уровня с учетом рег. доменов (3-й уровень) и бесплатных хостингов (выкидываем)
# banner_domain                    Склейка для баннеров: зеркальщик + экв. объявления, потом эфф. домен
# (везде, где что-то неправильно, отдается пустая строка)

my %eqdm;
my $region_str = join('|', map { $_ =~ s/\./\\\./g; $_; } get_region_domains());
my $region_regexp = qr/([\w_-]+\.(?:$region_str))$/o;

my $hostings_str = join('|', map { $_ =~ s/\./\\\./g; $_ } get_free_hostings());
my $hostings_regexp = qr/(?:^|\.)(?:$hostings_str)$/o;

my $class_init_has_done = 0;
sub class_init {
    my ($class, $opt) = @_;
    my $log = $opt->{logger};

    return if $class_init_has_done; #Исключаем повторный запуск
    $class_init_has_done = 1;

    #Заполняем массив доменов для склейки
    $log->log("equiv_domains");
    open(F, $opt->{equiv_domains});
    while(<F>){
        chomp;
        my ($main, $all) = split /\t/;
        $eqdm{$_} = $main for split /,/, $all;
    }
    close(F);
    $log->log("/equiv_domains (".(keys %eqdm).")");
}

sub banner_domain {
    my $domain = shift;
    my $eqdm = $eqdm{$domain} || $domain;
    return effective_domain($eqdm);
}

sub validate_url {
    my $url = shift;
    # After changing - use auto-test in tests/domains-check.pleck.pl
    $url=~s/(^\s+|\s+$)//g;
    return 0 unless $url;
    my $l = '[-A-Za-z0-9]';
    my $w = $l.'+';
    my $s = '\.';
    my $u = '^(https?\:\/\/)?'."($w$s)+".$l.'{1,4}($|\/|:\d+)';

    return $url =~ /$u/ ? 1 : 0;
}

sub url_to_domain {
    my $url = shift;
    $url = lc($url);
    if ($url =~ /^\s*(?:https?:\/\/)?(?:www\.)?((?:[\w_-]+\.)*[\w_-]+)(\s*$|\/|:\d+|\?)/o) {
        return $1;
    } else {
        return '';
    };
};

# Убирает http, https, www
sub canonical_url {
    my $url = shift;
    $url = lc($url);
    $url =~ s/^https?\:\/\/?//;
    $url =~ s/^www\.//;
    return $url
}

sub url_sort_get_params {
    my $url = shift;
    # sort get-params
    if (index($url, '&') != -1) {
        my $params_begin = rindex(substr($url, 0, rindex($url, '&')), '?');
        if ($params_begin != -1) {
            my @params = sort { $a cmp $b } split('&', substr($url, $params_begin+1));
            $url = substr($url, 0, $params_begin+1) . join('&', @params);
        }
    }
    return $url;
}

###### 
# Нормализация урла (для динамических и перфомансов)
# 1) доклеиваем http://
# 2) удаляем utm - метки
# 3) сортируем get-параметры
sub normalize_url {
    my ($url, %par) = @_;

    $url =~ s/^\s+|\s+$//g;
    if (length $url < 4 || substr($url, 0, 4) ne 'http') {
        $url = "http://" . $url;
    }

    $url =~ s/(?<=[&?])[^&?=]*?utm\_[^&?]*?(\&|$)//g;
    $url =~ s/(?<=[&?])pm\_[^&?]*?(\&|$)//g;
    $url =~ s/(?<=[&?])yd\_[^&?]*?(\&|$)//g;
    $url =~ s/(?<=[&?])r1=[^&?]*?(\&|$)//g;
    $url =~ s/(?<=[&?])r2=[^&?]*?(\&|$)//g;
    $url =~ s/(?<=[&?])roistat=[^&?]*?(\&|$)//g;

    $url =~ s/^[\?\&]+|[\?\&]+$//g;

    $url = url_sort_get_params($url);

    $url = lc $url unless $par{nolc};

    return $url;
}

#####
# Удаление GET-параметров в урле
#
sub remove_url_get_params {
    my $url = shift;
    my $ind = index($url, '?');
    if ( $ind != -1 ) {
        $url = substr($url, 0, $ind);
    }
    return $url;
}


sub get_sec_level_domain {
    my $domain = shift;
    $domain = url_to_domain($domain);
    $domain =~ /([\w_-]+\.[\w-]+)$/;  # дефис в домене первого уровня может быть в punycode доменах
    return $1;
}

sub get_free_hostings {
    return qw{
bbmy.ru ltalk.ru nsknet.ru polubomu.ru xost.ru beon.ru mindmix.ru ya.ru
0pk.ru 2bb.ru 2x4.ru 3bb.ru 4bb.ru 5bb.ru 6bb.ru 70mb.ru 7bb.ru
9bb.ru alfamoon.com alltrades.ru am9.ru appee.ru at.ua ath.cx august4u.ru
awardspace.com axer.ru babyhost.ru blog.ru ifolder.ru boxmail.biz bridelove.ru
build2.ru co.cc communityhost.ru croe.net cwx.ru dbpart.ru depositfiles.com
do.am ds8.ru e-gloryon.com ex6.ru fidep.ru flybb.ru flyfolder.ru forever.kz
forum24.ru freehostia.com g3g.ru galaxyhope.com googlepages.com h17.ru h18.ru
h2m.ru ho.ua homeip.net hoter.ru idknet.com infostore.org inmarket.biz int.ru
intway.com intway.info intwayblog.com intwayshop.com intwaystore.com ipb.su
ipsys.net isrv.ru jino.ru jlt.ru jp-net.ru kmx.ru krovatka.su lact.ru
lafekaka.ru letnick.com liveinternet.ru loveplanet.ru mcdir.ru md8.ru
metroland.ru ucoz.com ucoz.de ucoz.kz ucoz.lv ucoz.net ucoz.org ucoz.ru ucoz.ua
ukoz.ru vkontakte.ru vo.uz vsem.ru vsemp.ru w-ru.com w777w.ru web-box.ru
webasyst.net webest.net webklass.com webpromote.ru webstolica.ru webtalk.ru
wen.ru wml.su wmsite.ru wordpress.com wowex.ru x53.ru yndex.ru z16.ru mirbb.ru
mybb.ru mybb2.ru mylivepage.ru zx6.ru odnoklassniki.ru testbox.ru testsbox.ru
mpchat.com mpchat.ru nc.ru my1.ru no-ip.info no-ip.org novoya.com nx0.ru nxt.ru
or.kz pbnet.ru rifo.net ru.gg rxfly.net sbn.bz okis.ru pp.ru sdrom.ru se-ua.net
servegame.com site88.net siteedit.ru siting.ru sk6.ru spybb.ru sytes.net
telenet.ru tu1.ru tu2.ru umoz.ru tut.su uaprom.net v-teme.com vdelo.ru vipik.ru
index.html vsem.ru vsemp.ru yadviga.ru yadviga.com saidme.ru ilovethis.ru
vhoste.ru clann.ru aaa4.ru specblog.ru a4a4.ru page.tl hmsite.net hoter.ru
ru.gg okis.ru tu2.ru cwx.ru tut.su hostland.ru litehosting.ru g3g.ru ex6.ru
ha.by x53.ru webstolica.ru xan.su sk6.ru xam.su ds8.ru 70mb.ru zx6.ru naxx.ru
kzet.ru so.kz zs9.ru jlt.ru testbox.ru testsbox.ru ru.gg 12mb.com 20MegsFree.com
50megs.com i8.com 5u.com 100free.com 100megsfree.com addr.ru anzwers.net
elance.ru freeservers.com freewebsitehosting.com infhost.com joinme.ne
megspace.com mvm.su onlinehoster.net prohosting.com sbn.bz topcities.com
web1000.com xaper.com se-ua.net ho.ua tu1.ru w-ru.com h18.ru okis.ru
mulivepage.ru ucoz.kz pp.ru 2x4.ru houa.org intway.info mylivepage.com
piczo.com tut.su web-box.ru siteedit.ru kiss.su ruserv.com velion.org
com-info.ru allmy.ru 1me.ru my-page.ru my-page.name itmy.name once.name
ifree.name i7i.ru ifolder.ru ucoz.ua ukoz.lv ukoz.kz jit.ru dn.ru vo.uz
pp.net.ua fxcity.ru 110mb.com mylivepage.ru 3dn.ru 4u.ru afaik.ru agava.ru
aha.ru aiq.ru al.ru amillo.net az.ru bip.ru boom.ru borda.ru bos.ru by.ru
chat.ru clan.su come.ru dax.ru dem.ru down.ru dtn.ru ero.ru fanat.ru far.ru
fastbb.ru fatal.ru flip.ru formoza.ru fromru.com front.ru fud.ru geocities.com
go.ru h1.ru h10.ru h11.ru h12.ru h14.ru h15.ru h16.ru ho.com.ua hobi.ru
hoha.ru holm.ru hop.ru hostweb.ru hotbox.ru hotmail.ru hut.ru hut1.ru hut2.ru
i-am.ru i-nets.ru id.ru iwebs.ru jino-net.ru krovatka.net land.ru lgg.ru
livejournal.com mail15.com mail333.com mailru.com mchost.ru meoko.com moy.su
my1.ru myfreehost.biz myrunet.com newmail.ru nightmail.ru nm.ru non.ru ok.ru
p0.ru pips.ru pisem.net plex.ru pochta.org pochta.ru pochtamt.ru pop3.ru
pud.ru quake.ru r2.ru rbcmail.ru ru.ru ruwh.com sitecity.ru smtp.ru sp.ru
subs.ru supercharts.ru tora.ru tut.by ucoz.ru vio.ru vip.su vipcentr.ru
vippochta.ru vipshop.ru viptop.ru vov.ru w6.ru wallst.ru webhost.ru
webservis.ru yard.ru zerkalo.ru zmail.ru
narod.ru
    };
}

sub get_region_domains {
    return qw{
abkhazia.su ac.ru adygeya.ru altai.ru amur.ru amursk.ru arkhangelsk.ru
astrakhan.ru baikal.ru bashkiria.ru belgorod.ru belgorod.su bir.ru biz.ua
bryansk.ru bryansk.su bukhara.su buryatia.ru cbg.ru chel.ru chelyabinsk.ru
cherkassy.ua chernigov.ua chernovtsy.ua chimkent.su chita.ru chukotka.ru
chuvashia.ru ck.ua cmw.ru cn.ua co.il co.ua co.uk com.ru com.ua crimea.ua
cv.ua dagestan.ru dagestan.su dn.ua dnepropetrovsk.ua donetsk.ua dp.ua
dudinka.ru e-burg.ru east-kazakhstan.su edu.ru exnet.su fareast.ru georgia.su
grozny.ru grozny.su if.ua in.ua int.ru irkutsk.ru ivano-frankivsk.ua
ivanovo.ru ivanovo.su izhevsk.ru jamal.ru jar.ru joshkar-ola.ru k-uralsk.ru
kalmykia.ru kalmykia.su kaluga.ru kaluga.su kamchatka.ru karacol.su
karaganda.su karelia.ru karelia.su kazan.ru kchr.ru kemerovo.ru kh.ua
khabarovsk.ru khakassia.ru khakassia.su kharkov.ua kherson.ua khmelnitskiy.ua
khv.ru kiev.ua kirov.ru kirovograd.ua km.ua kms.ru koenig.ru komi.ru komi.su
kostroma.ru kr.ua krasnodar.su krasnoyarsk.ru ks.ua kuban.ru kurgan.ru
kurgan.su kursk.ru kustanai.ru kustanai.su kuzbass.ru kv.ua lg.ua lipetsk.ru
lugansk.ua lutsk.ua lviv.ua magadan.ru magnitka.ru mangyshlak.su mari-el.ru
mari.ru marine.ru mk.ua mordovia.ru mordovia.su mosreg.ru msk.ru msk.su
murmansk.ru murmansk.su mytis.ru nakhodka.ru nalchik.ru nalchik.su navoi.su
net.ru net.ua nikolaev.ua nkz.ru nn.ru nnov.ru norilsk.ru north-kazakhstan.su
nov.ru nov.su novosibirsk.ru nsk.ru obninsk.su od.ua odessa.ua omsk.ru
orenburg.ru org.ru org.ua oryol.ru oskol.ru palana.ru penza.ru penza.su perm.ru
pl.ua pokrovsk.su poltava.ua pp.ru pskov.ru ptz.ru pyatigorsk.ru rnd.ru
rovno.ua ru.com ru.net rubtsovsk.ru rv.ua ryazan.ru sakhalin.ru samara.ru
samara.su saratov.ru sebastopol.ua semsk.su simbirsk.ru smolensk.ru snz.ru
sochi.su spb.ru spb.su stavropol.ru stv.ru sumy.ua surgut.ru syzran.ru tagil.ru
tambov.ru tashkent.su tatarstan.ru te.ua termez.su ternopil.ua togliatti.su
tom.ru tomsk.ru troitsk.su tsaritsyn.ru tsk.ru tula.ru tula.su tuva.ru tuva.su
tver.ru tyumen.ru udm.ru udmurtia.ru ulan-ude.ru ur.ru uz.ua uzhgorod.ua
vdonsk.ru vinnica.ua vladikavkaz.ru vladikavkaz.su vladimir.ru vladimir.su
vladivostok.ru vn.ua volgograd.ru vologda.ru vologda.su voronezh.ru vrn.ru
vyatka.ru yakutia.ru yakutia.su yamal.ru yar.ru yaroslavl.ru yekaterinburg.ru
ykt.ru yuzhno-sakhalinsk.ru zaporizhzhe.ua zgrad.ru zhitomir.ua zp.ua zt.ua
        };
}

sub get_internal_yandex_domains {
    return qw{
        yandex.net
        yandex-team.ru
        yandex.ru
    };
}

sub check_lwp_whitelist {
    my $url = shift;
    my $domain = url_to_domain($url);
    my $is_whitelisted = (
        $domain eq 'storage-int.mds.yandex.net' ||
        $domain eq 'storage-int.mdst.yandex.net' ||
        $domain eq 'storage.yandex-team.ru' ||
        $domain =~ /(^|.*\.)s3\.yandex\.net$/ ||
        $domain =~ /(^|.*\.)s3\.mdst?\.yandex\.net$/ ||
        $domain eq 'feeds-yandex.booking.com' ||
        $domain eq 'proxy.sandbox.yandex-team.ru' ||
        $domain eq 'paste.yandex-team.ru'
    );
    return $is_whitelisted;
}

sub is_internal_yandex_url {
    my $url = shift;
    my $domain = get_sec_level_domain($url);
    return grep { $domain eq $_ } get_internal_yandex_domains();
}

sub is_datacamp_feed_url {
    my $url = shift;
    return url_to_domain($url) eq 'market.feed';
}

sub get_cgi_url_param {
    my $url = shift;
    my $param = shift;
    my $q = URI->new($url);
    my %query_params = $q->query_form;
    return Encode::decode('utf8', $query_params{$param});
}

sub effective_domain {
    my $domain = shift;
    $domain =~ s/^www\.//;
    if ($domain =~ /$hostings_regexp/) {
        return '';
    } elsif ($domain =~ /$region_regexp/) {
        return $1;
    } elsif ($domain =~ /([\w_-]+\.[\w]+)$/o) {
        return $1;
    } else {
        return '';
    }
}

sub get_url_parts {
    my ($url) = @_;
    my @parts = split '/', canonical_url($url);
    return \@parts;
}

sub safe_punycode_decode {
    my $text = shift;
    return eval { URI::_idna::decode($text) } // $text;
}

sub safe_punycode_encode {
    my $text = shift;
    return eval { URI::_idna::encode($text) } // $text;
}

# приведение к пуникоду
#   keep_case   =>  0|1 - не приводить домен к нижнему регистру
sub url_to_punycode {
    my $url = shift;
    my %par = @_;
    if ($url =~ m!(^https?)://([^/\?\:]+\.[^/\?\:\.]+)(.*)$!) {
        # main case
        my ($scheme, $domain, $path) = ($1, $2, $3);
        $domain = URI::_idna::encode($domain);
        $domain = lc($domain) unless $par{keep_case};
        $url = join('', $scheme, '://', $domain, $path);
    } elsif ($url =~ m!^([^/\?\:]+\.[^/\?\:\.]+)(.*)$!) {
        my ($domain, $path) = ($1, $2);
        $domain = URI::_idna::encode($domain);
        $domain = lc($domain) unless $par{keep_case};
        $url = $domain . $path;
    }
    return $url;
}

sub url_to_unicode {
    my ($url) = @_;
    if ($url =~ /^https?:\/\//) {
        $url =~ s/(^https?:\/\/)([^\/\?\:]+\.[^\/\?\:\.]+)/$1.safe_punycode_decode($2)/e;
    } else {
        $url =~ s/^([^\/\?\:]+\.[^\/\?\:\.]+)/safe_punycode_decode($1)/e;
    }
    return $url;
}

sub fix_url_scheme {
    my $url = shift;
    return $url unless $url;
    return $url if ($url =~ /^(https?|ftp)/i);
    $url =~ s/^:?\/\/?([^\/])/http:\/\/$1/i; # фикс, если ссылка начинается на //, /, :/, ://
    if ( $url !~ /^(https?|ftp)/ ) {
        $url = 'http://'.$url;  # фиксим, если она начинается со всего остального
    }
    return $url;
}

our $external_dns_resolver;
sub external_dns_resolver {
    if (!defined $external_dns_resolver) {
        $external_dns_resolver = Net::DNS::Resolver->new( nameservers => ['2a02:6b8::feed:ff', '77.88.8.8'] ); # dns.yandex.ru
    }
    return $external_dns_resolver;
}

1;
