package Utils::Words;
use strict;
use utf8;

# токенизация, нормализация слов, синонимы, плохие слова и т.п.
# новая версия Utils::Norm, Utils::Word, Utils::Dicts
# генерация данных в BM::Dicts::Words

use Data::Dumper;
use open ':utf8';
use Encode qw{_utf8_on};

use Utils::Common;
use Utils::Sys qw[
    print_err
    uniq
];
use StaticMap;

use base qw(Exporter);
our @EXPORT = (
    # склейка слов
    'text2words',                     # разбить текст на слова
    'text2words_nolc',                # разбить текст на слова без приведения к нижнему регистру
    'text2origwords',                 # разбить текст на слова без приведения к нижнему регистру
    'word2norm',                      # нормальная грамматическая форма (бывш. Norm::normword)
    'text2normwords',                 # нормальная грамматическая форма (бывш. Norm::normword)
    'text2snormwords_fast',
    'norm2good',                      # исправить ошибку в слове
    'norm2snorm',                     # представитель класса слов-синонимов (Norm::snormword)
    'word2snorm',                     # композиция word2norm и norm2snorm

    # расклейка
    'snorm2allsyns',                  # по snorm-слову получить список всех синонимов
    'snorm2goodsyns',                 # по snorm-слову получить список "правильных" синонимов (не опечаток)
    'norm2misprints',                 # по norm-слову получить список опечаток

    # доп. словари
    'stop4norm',                      # является ли стоп-словом (Word::stopword)
    'bsstop4word',                    # является ли стоп-словом в БК
    'bad4snorm',                      # является ли плохим словом (Word::bad4snorm)

    'load_dict',                      # загрузка словаря с учетом norm/snorm/..
    'phrases2norm',
    'phrases2snorm',
    'phrases2uniqnorm',

    'lmr',                            # бинарный леммер
    'clear_edges_fast',               # отрезание \W+ с концов текста
);

our (%goodnormlang, %mainsynlang);
our (%goodsynslang, %misprintslang);
our (%stop, %stoplang, %bad, %bsstop);

our $default_lang = 'ru';

our $token_regexp;
do {
    my $dot = '(?<=\d)\.(?=\d)';
    my $word = '\w(?:[\w-]|'.$dot.')*\w|\w';
    $word = '(?:'.$word.')';

    # токен это либо слово, либо -слово, либо !слово, либо -!слово
    # перед минусом или ! должен быть пробел или начало слова
    my $pretoken = qr/(?:^|(?<=\s))(?:!|-|-!|\+)/;
    $token_regexp = qr/($pretoken?$word)/o;
};

our $lm; # = StaticMap->new('/home/broadmatching-skreling/scripts/users/skreling/test_mmap_norm_dict');
our $lm_lite; # Леммер без нормализации - для использования в sub text2words
our $snlm; # Леммер для снормализации

sub lmr {
    unless( $lm ){
        my $file = $_[0] || $Utils::Common::options->{Words_params}{norm_dict};
        #print STDERR "lmr file: '$file'\n";
        unless ( -f $file ) {
            my $msg = "Could not create lmr: file '$file' does not exist";
            print STDERR "ERROR: $msg\n";
            die $msg;
        }
        $lm = StaticMap->new($file);
    }
    return $lm;
}

# Леммер без нормализации - для использования в sub text2words
sub lmr_lite {
    return $lm if $lm;
    unless( $lm_lite ){
        $lm_lite = StaticMap->new("");
    }
    return $lm_lite;
}

sub snlmr {
    unless( $snlm ){
        my $file = $_[0] || $Utils::Common::options->{Words_params}{word2snorm_dict};
        #print STDERR "snlmr file: '$file'\n";
        unless ( -f $file ) {
            my $msg = "Could not create snlmr: file '$file' does not exist";
            print STDERR "ERROR: $msg\n";
            die $msg;
        }
        $snlm = StaticMap->new($file);
    }
    return $snlm;
}

sub text2origwords {
    my $text = $_[0];
    my @w = ($text =~ m/$token_regexp/g);
    return @w;
}

sub text2words {
    return () if not defined $_[0];
    return @{lmr_lite->words($_[0])};
}

sub text2words_nolc {
    my $text = $_[0];
    return () if not defined $_[0];
    my $lang = $_[1] || $default_lang;

    $text =~ tr/Iİ/ıi/ if $lang eq 'tr';
    my @w = ($text =~ m/$token_regexp/g);
    return @w;
}

sub word2norm {
    return '' if not defined $_[0];
    return lmr->norm_word($_[0], $_[1] || $default_lang);
}

sub text2normwords {
    my $text = shift // return ();
    my $lang = shift || $default_lang;
    my $stp = $stoplang{$lang};
    my $res = lmr->norm_words($text, $lang);
    return grep { ! $stp->{$_} } @$res;
}

sub text2snormwords_fast {
    my $text = shift // return ();
    my $lang = shift || $default_lang;
    my $stp = $stoplang{$lang};
    my $res = snlmr->norm_words($text, $lang);
    return grep { ! $stp->{$_} } @$res;
}

sub phrases2norm {
    my $arr = shift;
    my $lang = shift || $default_lang;
    my @narr = ();
    for my $t ( @$arr ){
        my $t2 = join ' ', sort( text2normwords($t, $lang) );
        push(@narr, $t2);
    }
    return @narr;
}

sub phrases2uniqnorm {
    my $arr = shift;
    my $lang = shift || $default_lang;
    my @narr = ();
    for my $t ( @$arr ){
        my $t2 = join ' ', sort keys %{{ map { $_ => 1 } text2normwords($t, $lang) }};
        push(@narr, $t2);
    }
    return @narr;
}


sub phrases2snorm {
    my $arr = shift;
    my $lang = shift || $default_lang;
    my @narr = phrases2norm($arr, $lang);
    my $gnrm = $goodnormlang{$lang};
    my $mns = $mainsynlang{$lang};
    @narr = map { join ' ', sort map { $mns->{$_} // $_ } map { $gnrm->{$_} // $_}  @{[split '\ ',  $_]}} @narr;
    return @narr;
}

sub norm2good {
    my $norm = $_[0];
    my $lang = $_[1] || $default_lang;
    my $good = $goodnormlang{$lang}{$norm};
    return defined $good ? $good : $norm;
}

# deprecated, use word2snorm instead
sub norm2snorm {
    return word2snorm(@_);
}

sub word2snorm {
    my ($word, $lang) = @_;
    return snlmr()->norm_word($word, $lang // $default_lang);
}

#пакетная обработка
sub words2norm {
    my $lang = shift;
    if( $lang eq 'tr' ){
        tr/Iİ/ıi/ for @_;
    }
    my $stp = $stoplang{$lang};
    return grep { ! $stp->{$_} } map { word2norm($_) } @_;
}

#пакетная обработка
sub words2snorm {
    my $lang = shift;
    if( $lang eq 'tr' ){
        tr/Iİ/ıi/ for @_;
    }
    my $gnrm = $goodnormlang{$lang};
    my $mns = $mainsynlang{$lang};
    return map { $mns->{$_} // $_ } map { $gnrm->{$_} // $_} words2norm( @_ );
}

sub snorm2allsyns {
    my $snorm = $_[0];
    my $lang  = $_[1] || $default_lang;
    map { $_, norm2misprints($_, $lang) } snorm2goodsyns($snorm, $lang);
}


sub snorm2goodsyns {
    my $snorm = $_[0];
    my $lang  = $_[1] || $default_lang;
    my $arr = $goodsynslang{$lang}{$snorm};
    defined $arr ? @$arr : $snorm;
}

sub norm2misprints {
    my $norm = $_[0];
    my $lang = $_[1] || $default_lang;
    my $msp = $misprintslang{$lang}{$norm};
    defined $msp ? @$msp : ();
}

sub stop4norm {
    my $norm = $_[0];
    my $lang = $_[1] || $default_lang;
    return $stop{$norm.':'.$lang};
}

sub bsstop4word {
    $bsstop{$_[0]};
}

sub bad4snorm {
    $bad{$_[0]};
}

my $init_has_done = 0;

# параметры:
# norm_dict           словарь для norm
# syn_cells           словарь синонимов в специальном виде
# bad_words  -- соотв. словари
# stopword_dicts      словари стоп-слов для разных языков
# logger              логгер - объект с методом log
# force               форсировать повторную инициализацию
sub class_init {
    my ($class, $par) = @_;
    return if !$par->{force} and $init_has_done++;

    my $logger = $par->{logger};
    $logger->log('Utils::Words::class_init ...') if $logger;

    $logger->log("loading norm_dict from `$par->{norm_dict}' ...") if $logger;
    $lm = undef;  # force lemmer reload
    lmr($par->{norm_dict});

    $logger->log("loading word2snorm_dict from `$par->{word2snorm_dict}' ...") if $logger;
    $snlm = undef;
    snlmr($par->{word2snorm_dict});

    # загружаем стоп-слова без нормализации, но с учетом языка
    $logger->log('loading stopword dicts ...') if $logger;
    %stop = ();
    %stoplang = ();
    for my $data (@{$par->{stopword_dicts}}) {
        my ($file, $lang) = @$data;
        open my $fh, '<', $file
            or die "Can't open stopword dict '$file': $!";
        while (<$fh>) {
            s/#.*//g;   # Строки комментариев в файле
            $stop{$_.':'.$lang} = 1 for text2words($_);
            $stoplang{$lang}{$_} = 1 for text2words($_);
        }
        close $fh;
    }

    %mainsynlang = ();
    %goodsynslang = ();
    %misprintslang = ();
    %goodnormlang = ();
    for my $data (@{$par->{syn_dicts}}) {
        my ($file, $lang) = @$data;
        $logger->log("loading synonyms for language '$lang' ...") if $logger;
        _load_syn_dict($file, $lang);
    }


    $logger->log('loading miscellaneous dicts ...') if $logger;
    *bad  = load_dict($par->{bad_words},  snorm => 1);
    *bsstop = load_dict($par->{bsstop_words});

    #Добавляем данные по стоп-словам в леммер
    for my $lng (keys %stoplang){ #Добавляем данные стоп-слов
        $lm->AddStop($_, $lng) for keys %{$stoplang{$lng}};
        $snlm->AddStop($_, $lng) for keys %{$stoplang{$lng}};
    }

    $logger->log('Utils::Words::class_init done!') if $logger;
}

# параметры:
# dict   - файл или список файлов
# хэш опций:
# norm   - применять word2norm при загрузке
# snorm  - применять word2snorm
sub load_dict {
    my ($dict, %par) = @_;
    my %dict;
    my @files = ref $dict ? @$dict : $dict;
    for my $file (@files) {
        open my $dict_fh, '<', $file
            or die "Can't open dict '$file': $!";
        while (<$dict_fh>) {
            my @words = text2words($_);
            if ($par{norm}) {
                @words = map { word2norm($_) } @words;
            } elsif ($par{snorm}) {
                @words = map { word2snorm($_) } @words;
            }
            $dict{$_} = 1 for @words;
        }
        close $dict_fh;
    }
    return \%dict;
}

# Загружаем %mainsynlang, %goodsynslang, %misprintslang, %goodnormlang
sub _load_syn_dict {
    my ($file, $lang) = @_;
    my @clus;
    open my $fh, '<', $file
        or die "Can't open file '$file': $!";
    while (<$fh>) {
        chomp;
        if ($_ eq '') {
            $mainsynlang{$lang}{$_} = $clus[0] for @clus[1 .. $#clus];
            $goodsynslang{$lang}{$clus[0]} = [ @clus ];
            @clus = ();
            next;
        }
        my ($norm, $msp_str, $bdl_str) = split /;/;
        my @msp = split /,/, $msp_str;
        my @bdl = split /,/, $bdl_str;
        $misprintslang{$lang}{$norm} = [ @msp, @bdl ] if @msp or @bdl;
        $goodnormlang{$lang}{$_} = $norm for @msp, @bdl;
        push @clus, $norm;
    }
    close $fh;
}

# Для подсчета диффа категоризации (в интерфейсе - cmd=banners_categs_diff)
# Только для русского языка!
# На входе:
#   norm2snorm      => { norm => snorm, ... }
#   snorm2norms     => { snorm => [ @norms ], ... }
#   change_snorm    => { snorm_old => snorm_new, ... }
#   always_add_words => [ @words ] - Дописать эти слова в word2snorm_dict, даже если их там нет
#   correct_misspell    => 0/1
#   dont_clean      => 0/1
# На выходе:
#   Путь к новому словарю word2snorm_dict
sub create_patched_word2snorm_dict {
    my (%prm) = @_;
    print_err("create_patched_word2snorm_dict ... correct_misspell:" . ($prm{correct_misspell} // 0));

    my $lang = 'ru'; # TODO other languages

    my $change_snorm = $prm{change_snorm} || {};
    print_err("change_snorm: " . join(", ", map { $_." => ".$change_snorm->{$_} } sort keys %$change_snorm));
    my $snorm2norms = $prm{snorm2norms} || {};
    my $norm2snorm = \%{ $prm{norm2snorm} || {} };
    print_err("snorm2norms: " . join(", ", map { $_." => [ ".join(" ", @{$snorm2norms->{$_} // []}) . " ]" } sort keys %$snorm2norms));
    print_err("norm2snorm input: " . join(", ", map { $_." => ".$norm2snorm->{$_} } sort keys %$norm2snorm));
    for my $snorm (keys %$snorm2norms) {
        my @norms = @{ $snorm2norms->{$snorm} // [] };
        $norm2snorm->{$_} = $snorm   for @norms;
        $norm2snorm->{$snorm} = $snorm;
    }
    my (%mainsynlang_prev, %goodsynslang_prev, %misprintslang_prev, %goodnormlang_prev);
    if ($prm{correct_misspell}) {
        print_err("loading dicts for correct_misspell");
        %mainsynlang_prev = %mainsynlang;
        %goodsynslang_prev = %goodsynslang;
        %misprintslang_prev = %misprintslang;
        %goodnormlang_prev = %goodnormlang;
        (%mainsynlang, %goodsynslang, %misprintslang, %goodnormlang) = ();
        my $syn_dicts = $Utils::Common::options->{Words_params}{syn_dicts}; # TODO - ?
        for my $data (@$syn_dicts) { # TODO сделать без копипаста
            my ($file, $lang) = @$data;
            print_err("loading synonyms for language '$lang' ...");
            _load_syn_dict($file, $lang);
        }
    }

    for my $norm (keys %$norm2snorm) { # Нормализация могла измениться - меняем norm в %$norm2snorm
        my $snorm = $norm2snorm->{$norm};
        my $snorm_new = Utils::Words::word2norm($snorm, $lang);
        #if ($prm{correct_misspell}) {
        # TODO - ?     norm2good($snorm_new, $lang);
        #}
        $norm2snorm->{$norm} = $snorm_new;
    }
    for my $norm_old (keys %$norm2snorm) { # Нормализация могла измениться - меняем snorm в %$norm2snorm
        my $norm_new = Utils::Words::word2norm($norm_old, $lang);
        if ($prm{correct_misspell}) {
            $norm_new = norm2good($norm_new, $lang);
        }
        $norm2snorm->{$norm_new} = $norm2snorm->{$norm_old}   if $norm_old ne $norm_new;
    }

    print_err("norm2snorm final: " . join(", ", map { $_." => ".$norm2snorm->{$_} } sort keys %$norm2snorm));

    my $word2snorm_dict_old = $prm{input} || $Utils::Common::options->{Words_params}{word2snorm_dict};
    my $word2snorm_dict_new = $prm{output} || Utils::Sys::get_tempfile("word2snorm_dict_patched.tmp", UNLINK => ($prm{dont_clean} ? 0 : 1), );

    my $word2newsnorm_func = sub {
        my ($word, $snorm) = @_;
        my $snorm_old = $snorm;
        #print "word2newsnorm_func ($word, $snorm)\n";

        my $norm = Utils::Words::word2norm($word, $lang);
        if (my $new = $norm2snorm->{$norm} // (
                $prm{correct_misspell} ? $norm2snorm->{ norm2good($norm, $lang) } : undef
            )
        ) {
            $snorm = $new;
        }
        if (my $new = $change_snorm->{$snorm}) {
            $snorm = $new;
        }

        $snorm = Utils::Words::word2norm($snorm, $lang); # Т.к. нормализация могла измениться
        print_err("word2newsnorm_func: ($word, $snorm_old) -> $snorm")  if $snorm ne $snorm_old;
        return $snorm;
    };

    create_patched_dict(
        $word2snorm_dict_old,
        $word2snorm_dict_new,
        $word2newsnorm_func,
        [ uniq( keys %$norm2snorm, keys %$change_snorm, @{ $prm{always_add_words} // [] } )],
        (map { $_ => $prm{$_} } qw[ dont_clean ]),
    );

    if ($prm{correct_misspell}) { # Возвращаем исходные значения
        %mainsynlang = %mainsynlang_prev;
        %goodsynslang = %goodsynslang_prev;
        %misprintslang = %misprintslang_prev;
        %goodnormlang = %goodnormlang_prev;
    }

    print_err("create_patched_word2snorm_dict done. $word2snorm_dict_new");
    return $word2snorm_dict_new;
}

# Для подсчета диффа категоризации (в интерфейсе - cmd=banners_categs_diff)
# На входе:
#   word2norm      => { word => norm, ... }
#   norm2words     => { norm => [ @words ], ... }
#   change_norm    => { norm_old => norm_new, ... }
#   dont_clean      => 0/1
# На выходе:
#   Путь к новому словарю norm_dict
sub create_patched_norm_dict {
    my (%prm) = @_;
    print_err("create_patched_norm_dict ...");

    my $change_value = $prm{change_norm} || {};
    my $value2keys = $prm{norm2words} || {};
    my $key2value = \%{ $prm{word2norm} || {} };
    for my $value (keys %$value2keys) {
        my @keys = @{ $value2keys->{$value} // [] };
        $key2value->{$_} = $value   for @keys;
        $key2value->{$value} = $value;
    }

    my $dict_old = $prm{input} || $Utils::Common::options->{Words_params}{norm_dict};
    my $dict_new = $prm{output} || Utils::Sys::get_tempfile("norm_dict_patched.tmp", UNLINK => ($prm{dont_clean} ? 0 : 1), );

    my $transform_func = sub {
        my ($word, $norm) = @_;

        if (my $new = $key2value->{$word}) {
            $norm = $new;
        }
        if (my $new = $change_value->{$norm}) {
            $norm = $new;
        }
        return $norm;
    };

    create_patched_dict(
        $dict_old,
        $dict_new,
        $transform_func,
        [ keys %$key2value, keys %$change_value ],
        (map { $_ => $prm{$_} } qw[ dont_clean ]),
    );

    print_err("create_patched_norm_dict done. $dict_new");
    return $dict_new;
}

# "Применяем патч" к словарю. Словарь в формате key-value, разделитель - \t
# На входе:
#   $dict_old
#   $dict_new
#   $transform_func
#   $add_keys   (Для словаря, ключи которого отсортированы!) Ключи, которые нужно дописать в словарь, даже если их там нет.
sub create_patched_dict {
    my ($dict_old, $dict_new, $transform_func, $add_keys, %prm) = @_;

    print_err("dict_old: $dict_old");
    print_err("dict_new: $dict_new");
    my $dict_temp = Utils::Sys::get_tempfile("dict_patched.tmp.tmp", UNLINK => ($prm{dont_clean} ? 0 : 1), );
    print_err("dict_temp: $dict_temp");

    open my $fh_in, "<", $dict_old or die "Cannot open < $dict_old ($!)";
    open my $fh_out, ">", $dict_temp or die "Cannot open > $dict_temp ($!)";
    my $prev_key = '';
    while (my $line = <$fh_in>) {
        chomp $line;
        my ($key, $value) = split /\t/, $line, 2;

        if ($add_keys) {
            # В словаре может не быть слов, для которых key = value. Дописываем их
            for my $add_key (grep { $_ gt $prev_key and $_ lt $key } sort(uniq(@$add_keys)) ) {
                my $add_value = $transform_func->($add_key, $add_key);
                if (defined $add_value) {
                    print $fh_out join("\t", $add_key, $add_value), "\n";
                }
            }
        }

        $value = $transform_func->($key, $value) // $value;
        print $fh_out join("\t", $key, $value), "\n";
        $prev_key = $key;
    }
    close $fh_in or die "Cannot close input $dict_old ($!)";
    close $fh_out or die "Cannot close output $dict_temp ($!)";
    Utils::Sys::do_sys_cmd("mv $dict_temp $dict_new");
    return 1;  # TODO  return $dict_new ?
}

# На входе - см. параметры sub create_patched_word2snorm_dict
# На выходе:
#   1, если всё отработало
sub patch_word2snorm_dict {
    my (%prm) = @_;
    $Utils::Common::options->{Words_params}{word2snorm_dict} = create_patched_word2snorm_dict(%prm);
    return 1;
}

# На входе - см. параметры sub create_patched_norm_dict и sub create_patched_word2snorm_dict
# На выходе:
#   1, если всё отработало
sub patch_normalization {
    my (%prm) = @_;

    $Utils::Common::options->{Words_params}{norm_dict} = create_patched_norm_dict(%prm);

    my $always_add_words = [ uniq (
            keys %{ $prm{change_norm} // {} },
            values %{ $prm{change_norm} // {} },
            keys %{ $prm{word2norm} // {} },
            values %{ $prm{word2norm} // {} },
            keys %{ $prm{norm2words} // {} },
            ( map { @{$_} } values %{ $prm{norm2words} // {} }),
            @{ $prm{always_add_words} // [] },
    ) ];
    $Utils::Common::options->{Words_params}{word2snorm_dict} = create_patched_word2snorm_dict(
        #(map { $_ => $prm{$_} } qw[ dont_clean correct_misspell ]),
        %prm,
        always_add_words => $always_add_words,
    );

    return 1;
}

sub clear_edges_fast {
    my ($text, $side) = @_;
    $text =~ s/^[\W_]+//i;
    $text =~ s/[\W_]+$//i;
    return $text;
}

1;
