package Utils::XLS;
use strict;

# some useful functions for testing

use utf8;
use open ':utf8';
no warnings "utf8";

binmode (STDIN, ":utf8");
binmode (STDOUT, ":utf8");
binmode (STDERR, ":utf8");

use base qw(Exporter);
use Utils::Sys qw(get_tempfile);
use File::Temp;
use Encode qw(from_to);
use File::Spec;
use IO::Uncompress::Unzip qw(unzip $UnzipError);

our @EXPORT = qw(
    csv2array
    xls2array
    xlsx2array
    array2xls
    array2xlsx
    array2xls_multi
    xlsx2array_allsheets
    csv2array_all
    xls2array_allsheets
    xls_edit_line
    xls_lines_list
    spreadsheet2arr
    xls2array_allsheets_fixed
);

#       xlsx2array                 первый столбец первого листа возвращает в виде массива
#       xlsx2array_allsheets       парсит всю книгу по листам, возвращает 1) книгу в виде массива массивов табулированных строк 2) массив названий листов книги
#       csv2array_all              парсит сsv в массив табулированных строк 
#       array2xls                  сохраняет переданный массив на первом листе книги (с указанным sheetname), возвращает бинарник
#       xls2array_allsheets        парсит всю книгу по листам, возвращает 1) книгу в виде массива массивов табулированных строк 2) массив названий листов книги

#       xls_edit_line              автоматическое редактирование строк
#       xls_lines_list             возвращает массив массивов данных таблицы

#####
# Метод из библиотеки с измененным поведением при исключениях
# Отличается от оригинала тем, что при некорректных числовых знаечениях (#0, #VALUE!, #NaN) возвращает '' 
# без warning'а
#
sub XLSX_FmtString_Custom {
    my($oThis, $oCell, $oBook) =@_;

    my $sFmtStr;
    unless(defined($sFmtStr)) {
        if ($oCell->{Type} eq 'Numeric') {
            if ($oCell->{Format}) {
                $sFmtStr=$oCell->{Format};
            } elsif ($oCell->{Val} =~ /^[\d\.\,\-\+]+$/) {
                if (int($oCell->{Val}) != $oCell->{Val}) {
                    $sFmtStr = '0.00';
                } else {
                    $sFmtStr = '0';
                }
            } else {
                $sFmtStr = '@';
            }
        }
        elsif($oCell->{Type} eq 'Date') {
            if($oCell->{Format}){
                $sFmtStr=$oCell->{Format};
            } elsif (int($oCell->{Val}) <= 0) {
                $sFmtStr = 'h:mm:ss';
            } else {
                $sFmtStr = 'm-d-yy';
            }
        }
        else {
            $sFmtStr = '@';
        }
    }
    return $sFmtStr;
}

my $_xlsx_fmt2007_inited = 0;
sub _init_xlsx_fmt2007 {
    return if $_xlsx_fmt2007_inited;
    # подменяем библиоечную функцию исправленной
    no warnings "all";
    require Spreadsheet::XLSX::Fmt2007;
    *Spreadsheet::XLSX::Fmt2007::FmtString = \&XLSX_FmtString_Custom;
    use warnings "all";
    $_xlsx_fmt2007_inited = 1;
}



# Параметры:
#   get_all_cells_array     1/0  Вернуть массив ячеек из всех столбцов
sub xls2array { #Не протестировано
    my ($xls_data, %prm) = @_;
    require Spreadsheet::ParseExcel;
    my ($file_handle, $file) = File::Temp::tempfile("xls2array.XXXX", DIR => $Utils::Common::options->{dirs}{temp}, UNLINK => 1);

    binmode($file_handle);
    print $file_handle $xls_data;
    close($file_handle);

    my @res = ();
    eval {
        my $parser   = Spreadsheet::ParseExcel->new();
        my $workbook = $parser->parse($file);
    
        if ( !defined $workbook ) {
            unlink ($file);
            die $parser->error(), ".\n";
        }
    
        my @ws = $workbook->worksheets();
        my $worksheet = shift @ws;
        my ( $row_min, $row_max ) = $worksheet->row_range;
        my ( $col_min, $col_max ) = $worksheet->col_range;
        for my $row ( $row_min .. $row_max ) {
            if ($prm{get_all_cells_array}) {
                my @res_raw;
                for my $col ( $col_min .. $col_max ) {
                    my $cl = $worksheet->get_cell( $row, $col );
                    push(@res_raw, $cl ? $cl->value : undef );
                }
                push(@res, \@res_raw);
            } else {
                my $cl = $worksheet->get_cell( $row, $col_min );
                push(@res, $cl->value ) if $cl;
            }
        }
    };
    if ($@) {
        unlink($file);
        die $@;
    }
    unlink($file);

    return @res;
}

#####
# Обертка для чтения xls/xlsx в массив
#
sub spreadsheet2arr {
    my ($filepath, $format) = @_;

    my ($sheets, $shls);
    eval {
        if ($format eq 'xlsx' || $filepath =~ /\.xlsx$/) {
            ($sheets, $shls) = _xlsxfile2array_allsheets($filepath);
        } elsif ($format eq 'xls') {
            ($sheets, $shls) = _xlsfile2array_allsheets_fixed($filepath);
        }
    };
    $sheets = undef if ($@);

    return $sheets->[0] || [];
}

#####
# Fixed-версия xls2array_allsheets
# тут пустые ячейки не игнорируются, а принимаются равными ''
#
sub xls2array_allsheets_fixed {
    my ($xls_data) = @_;

    my ($file_handle, $file) = File::Temp::tempfile("xls2array.XXXX", DIR => $Utils::Common::options->{dirs}{temp}, UNLINK => 1);

    binmode($file_handle);
    print $file_handle $xls_data;
    close($file_handle);

    my $res = [];
    my $sheetnames = [];

    eval {
        ($res, $sheetnames) = _xlsfile2array_allsheets_fixed($file);
    };
    if ($@) {
        unlink($file);
        die $@;
    }
    unlink($file);
    return $res, $sheetnames;
}

sub _xlsfile2array_allsheets_fixed {
    my $file = shift;
    my @res = ();
    my @sheetnames = ();

    #бесплатный по памяти тест на то, что это ole-файл
    #это не обязательно xls, это может быть любой MSOffice-документ
    #но если это не MSOffice-документ, делать дальше точно нечего
    #Spreadsheet::ParseExcel, однако, все равно полностью прокачивает в память любой файл вне зависимости от его типа, даже если там точно не xls
    require OLE::Storage_Lite;
    my $ole_storage = OLE::Storage_Lite->new($file);
    my $ole_tree = $ole_storage->getPpsTree(0);
    die "$file is not in xls format" unless defined $ole_tree;

    require Spreadsheet::ParseExcel;

    my $parser   = Spreadsheet::ParseExcel->new();
    my $workbook = $parser->parse($file);

    if ( !defined $workbook ) {
        die $parser->error(), ".\n";
    }

    for my $worksheet ( $workbook->worksheets ){
        my $sheetname = $worksheet->get_name;
        Encode::_utf8_on( $sheetname );
        push @sheetnames, $sheetname;
        my @rows = ();
        my ( $row_min, $row_max ) = $worksheet->row_range;
        for my $row ( $row_min .. $row_max ){
            my @cols = ();
            my ( $col_min, $col_max ) = $worksheet->col_range;
            for my $col ( $col_min .. $col_max ){
                my $cell = $worksheet->get_cell( $row, $col );
                my $value = $cell ? $cell->value : '';
                Encode::_utf8_on( $value );
                push @cols, $value;
            }
            push @rows, join( "\t", @cols );
        }
        push @res, \@rows;
    }
    return \@res, \@sheetnames;
}

# есть баг, но оставил, т.к. используется везде и никто не жалуется
sub xls2array_allsheets {
    my ($xls_data) = @_;
    require Spreadsheet::ParseExcel;
    my ($file_handle, $file) = File::Temp::tempfile("xls2array.XXXX", DIR => $Utils::Common::options->{dirs}{temp}, UNLINK => 1);

    binmode($file_handle);
    print $file_handle $xls_data;
    close($file_handle);

    my @res = ();
    my @sheetnames = ();

    eval {
        my $parser   = Spreadsheet::ParseExcel->new();
        my $workbook = $parser->parse($file);
    
        if ( !defined $workbook ) {
            unlink ($file);
            die $parser->error(), ".\n";
        }
    
        for my $worksheet ( $workbook->worksheets ){
            my $sheetname = $worksheet->get_name;
            Encode::_utf8_on( $sheetname );
            push @sheetnames, $sheetname;
            my @rows = ();
            my ( $row_min, $row_max ) = $worksheet->row_range;
            for my $row ( $row_min .. $row_max ){
                my @cols = ();
                my ( $col_min, $col_max ) = $worksheet->col_range;
                for my $col ( $col_min .. $col_max ){
                    my $cell = $worksheet->get_cell( $row, $col );
                    # вот тут баг - игнорируем пустые ячейки
                    # из-за этого xls-фиды, например, неправильно парсятся
                    next unless $cell;
                    my $value = $cell->value;
                    Encode::_utf8_on( $value );
                    push @cols, $value;
                }
                push @rows, join( "\t", @cols );
            }
            push @res, \@rows;
        }
    };
    if ($@) {
        unlink($file);
        die $@;
    }
    unlink($file);
    return \@res, \@sheetnames;
}

sub xlsx2array {
    my ($xls_data) = @_;

    my ($file_handle, $file) = File::Temp::tempfile("xls2array.XXXX", DIR => $Utils::Common::options->{dirs}{temp}, UNLINK => 1);
    binmode($file_handle);
    print $file_handle $xls_data;
    close($file_handle);

    my @res = ();
    eval {
        require Spreadsheet::XLSX;
        _init_xlsx_fmt2007();
     
        my $excel = Spreadsheet::XLSX->new($file);
        my $sheet = [@{$excel -> {Worksheet}}]->[0];
        for my $row ($sheet->{MinRow} .. $sheet->{MaxRow}) {
            my $val = $sheet->{Cells}[$row][$sheet->{MinCol}]->{Val};
            push(@res, $val);
        }
    };
    if ($@) {
        unlink($file);
        die $@;
    }
    unlink($file);
    return @res;
}

sub xlsx2array_allsheets {
    my ($xls_data) = @_;

    my ($file_handle, $file) = File::Temp::tempfile("xls2array.XXXX", DIR => $Utils::Common::options->{dirs}{temp}, UNLINK => 1);
    binmode($file_handle);
    print $file_handle $xls_data;
    close($file_handle);

    my $res = [];
    my $sheetnames = [];

    eval {
        ($res, $sheetnames) = _xlsxfile2array_allsheets($file);
    };
    if ($@) {
        unlink($file);
        die $@;
    }
    unlink($file);
    return $res, $sheetnames;
}

sub _xlsxfile2array_allsheets {
    my $file = shift;

    my @res = ();
    my @sheetnames = ();

    #бесплатная по памяти проверка на то, что это зипованный файл. xlsx всегда запакован строго так
    #парсер занимает память даже при неудаче
    my $nullFileName = File::Spec->devnull();
    unzip $file => $nullFileName or die "unzip failed: $UnzipError\n";

    require Spreadsheet::XLSX;
    _init_xlsx_fmt2007();

    my $excel = Spreadsheet::XLSX->new($file);
    for my $sheet ( @{$excel -> {Worksheet}} ){
        push @sheetnames, $sheet->{Name};
        my @rows = ();
        for my $row ( $sheet->{MinRow} .. $sheet->{MaxRow} ){
            my @cols = ();
            for my $col ( $sheet->{MinCol} .. $sheet->{MaxCol} ){
                push @cols, $sheet->{Cells}[$row][$col]->{Val} || '';
            }
            Encode::_utf8_on($_) for @cols;
            push @rows, join("\t", @cols );
        }
        Encode::_utf8_on($_) for @sheetnames;
        push @res, \@rows;
    }

    return \@res, \@sheetnames;
}

sub csv2array {
    my ($xls_data) = @_;
    my @res = grep {$_} split(/[;\r\n\t]+/, $xls_data);
    return @res;
}

sub csv2array_all {
    my ($csv_data) = @_;
    Encode::_utf8_on($csv_data);
    return [ split /\n/, $csv_data ];
}

sub array2xls {
    my ($arr, $sheetname) = @_;

    require Spreadsheet::WriteExcel;

    # Requires perl 5.8 or later
    open my $fh, '>', \my $str or die "Failed to open filehandle: $!";
    binmode $fh;

    my $workbook  = Spreadsheet::WriteExcel->new($fh);
    my $worksheet = $workbook->add_worksheet($sheetname);

    my $i = 0;
    #$worksheet->write( $i++, 0, $_) for @$arr;
    for my $l (@$arr){
        my $j = 0;
        my @cols = ref($l) eq 'ARRAY' ? @$l : ($l);
        for(@cols){
            s/^=+// if $_;
            s/\t/ / if $_;
        }
        Encode::_utf8_on($_);
        $worksheet->write( $i, $j++, $_) for @cols;
        $i++;
    }

    $workbook->close();

    return $str;
}

sub array2xlsx {
    my ($arr, $sheetname) = @_;

    require Excel::Writer::XLSX;

    # Requires perl 5.8 or later
    open my $fh, '>', \my $str or die "Failed to open filehandle: $!";
    binmode $fh;

    my $workbook = Excel::Writer::XLSX->new( $fh );
    my $worksheet = $workbook->add_worksheet();

    my $i = 0;
    #$worksheet->write( $i++, 0, $_) for @$arr;
    for my $l (@$arr){
        my $j = 0;
        my @cols = ref($l) eq 'ARRAY' ? @$l : ($l);
        for(@cols){
            s/^=+// if $_;
            s/\t/ / if $_;
        }
        Encode::_utf8_on($_);
        $worksheet->write( $i, $j++, $_) for @cols;
        $i++;
    }

    $workbook->close();

    return $str;
}

# Создает файл .xls с несколькими вкладками
sub array2xls_multi {
    my ($arr) = @_;   #  [ [[@list1], $title1],  [[@list2], $title2], ... ]

    require Spreadsheet::WriteExcel;

    # Requires perl 5.8 or later
    open my $fh, '>', \my $str or die "Failed to open filehandle: $!";
    binmode $fh;

    my $workbook  = Spreadsheet::WriteExcel->new($fh);
    
    my %titles;
    for my $sheet_data (@$arr) {
        my $title = $sheet_data->[1] // '';
        if ($titles{$title}) {
            my $ind = 1;
            while ($titles{"$title ($ind)"}) {
                $ind++;
            }
            $title = "$title ($ind)";
        }
        $titles{$title} = 1;
        my $worksheet = $workbook->add_worksheet($title);
        my $arr = $sheet_data->[0] // [];
        my $i = 0;
        #$worksheet->write( $i++, 0, $_) for @$arr;
        for my $l (@$arr){
            my $j = 0;
            my @cols = ref($l) eq 'ARRAY' ? @$l : ($l);
            for(@cols){
                s/^=+// if $_;
                s/\t/ / if $_;
            }
            Encode::_utf8_on($_);
            $worksheet->write( $i, $j++, $_) for @cols;
            $i++;
        }
    }

    $workbook->close();

    return $str;
}

sub xls_lines_list {
    my ($xls_data, $lsub) = @_;

    require Spreadsheet::ParseExcel;

    my ($file_handle, $file) = File::Temp::tempfile("xls2array.XXXX", DIR => $Utils::Common::options->{dirs}{temp}, UNLINK => 1);

    binmode($file_handle);
    print $file_handle $xls_data;
    close($file_handle);

    my @res = ();

    eval {
        my $parser   = Spreadsheet::ParseExcel->new();
        my $workbook = $parser->parse($file);
    
        if ( !defined $workbook ) {
            unlink ($file);
            die $parser->error(), ".\n";
        }
    
        my @ws = $workbook->worksheets();
        my $worksheet = shift @ws;
        my ( $row_min, $row_max ) = $worksheet->row_range;
        my ( $col_min, $col_max ) = $worksheet->col_range;
    
        for my $row ( $row_min .. $row_max ) {
            my @line = ();
            for my $col ( $col_min .. $col_max ){ #Собираем строку
                my $cl = $worksheet->get_cell( $row, $col );
                $line[ $col ] = $cl->value if $cl;
            }
    
            push(@res, \@line);
        }
    };
    if ($@) {
        unlink($file);
        die $@;
    }
    unlink($file);

    return \@res;
}

#Редактирование строки Excel'я
sub xls_edit_line {
    my ($xls_data, $lsub) = @_;

    require Spreadsheet::ParseExcel::SaveParser;
    require Spreadsheet::ParseExcel::SaveParser::Workbook;
    require Spreadsheet::ParseExcel::SaveParser::Worksheet;

    my ($file_handle, $file) = File::Temp::tempfile("xls2array.XXXX", DIR => $Utils::Common::options->{dirs}{temp}, UNLINK => 1);

    binmode($file_handle);
    print $file_handle $xls_data;
    close($file_handle);

    my @res = ();
    #my $parser   = Spreadsheet::ParseExcel->new();
    my $parser   = Spreadsheet::ParseExcel::SaveParser->new();
    my $workbook = $parser->Parse($file);

    if ( !defined $workbook ) {
        unlink ($file);
        die $parser->error(), ".\n";
    }

    my @ws = $workbook->worksheets();
    my $worksheet = shift @ws;
    my ( $row_min, $row_max ) = $worksheet->row_range;
    my ( $col_min, $col_max ) = $worksheet->col_range;
    for my $row ( $row_min .. $row_max ) {
        my @line = ();
        for my $col ( $col_min .. $col_max ){ #Собираем строку
            my $cl = $worksheet->get_cell( $row, $col );
            $line[ $col ] = $cl->value if $cl;
        }

        #Вызываем внешнюю функцию для каждой ячейки
        my $newline = $lsub->( $row, \@line );
        for my $col ( $col_min .. $col_max ){ #Собираем строку
            my $curval = undef;
            my $cl = $worksheet->get_cell( $row, $col );
            $curval = $cl->value if $cl;
            my $newval = $newline->[$col];
            if((defined $newval) && ( $curval ne $newval )){
                my $frm = undef;
                $frm = $cl->{FormatNo} if $cl;
                $worksheet->AddCell( $row, $col, $newval, $frm );
            }
        }
        
    }

    $workbook->SaveAs($file);

    open(F, "< $file");
    binmode(F);
    my @line = <F>;
    my $data = join('', @line);
    close(F);

    unlink($file);

    return $data;
}

1;
