#!/usr/bin/perl -w
use strict;
use utf8;

use open ':utf8';

use DBI;
use Data::Dumper;

use FindBin;
use lib "$FindBin::Bin/../lib";
use Utils::Sys qw(
    get_file_lock
    release_file_lock
    print_err
    handle_errors
    do_safely
);
use Utils::Funcs qw( sort_keys_desc sort_keys_asc );
use Getopt::Long;
use Project;
use Utils::Hosts qw(get_host_role get_curr_host);
use List::Util qw(max);
use JSON qw(to_json);

use lib "$FindBin::Bin/../wlib";
use CatalogiaMediaProject;
use BM::SolomonClient;

handle_errors();

my %opt = (
);
GetOptions(
    \%opt,
    'indc_name=s',
);

my $indc_name = $opt{indc_name} // '';

my $lock_name = "check-monrun___$indc_name";
get_file_lock($lock_name) or exit(0);
my $solomon_client = BM::SolomonClient->new();
my $host = get_curr_host();

my %indc_funcs = (

    # Проверяет количество занятых процессов FCGI (по таблице FcgiProcess)
    # Возвращает CRITICAL или WARN, если хотя бы на одном из хостов количество занятых процессов соответствует пороговому значению
    fcgi_free => sub {
        my ($proj) = @_;

        my $stm = "select * from FcgiProcess where Host = '$host'"; # Host | Pid | Login | Cmd | Act | StartTime
        $proj->log("List_SQL( $stm ) ...");
        my $list = $proj->List_SQL($stm);
        $proj->log("List_SQL( $stm ) done");
        $proj->log("list: " . to_json($list || []));

        my $fcgi_busy_count = scalar @$list;
        my $fcgi_cmdact2count = {};
        for my $el (@$list) {
            my $cmd_act = join(".", map { $_ || 'empty' } ($el->{Cmd}, $el->{Act}));
            $fcgi_cmdact2count->{$cmd_act} += 1;
        }

        $solomon_client->push_single_sensor({
            cluster => "host_info",
            service => "fcgi_statistics",
            sensor  => "states",
            labels  => {
                state => "busy",
                host   => $host,
            },
            value   => $fcgi_busy_count,
        });

        my $number_of_fcgi_processes_warn = 9; # TODO уменьшить (или увеличить количество процессов FCGI+prefprojsrv)  См. https://st.yandex-team.ru/CATALOGIA-943
        my $number_of_fcgi_processes_critical = 10;
        my $monrun_res;
        if ($fcgi_busy_count < $number_of_fcgi_processes_warn) {
            $monrun_res = { state => 0, description => "OK ($fcgi_busy_count busy processes)" };
        } else {
            my $state = ($fcgi_busy_count < $number_of_fcgi_processes_critical) ? 1 : 2;
            my $msg = join(", ", map { "$_:".$fcgi_cmdact2count->{$_} } sort_keys_desc($fcgi_cmdact2count));
            $monrun_res = { state => $state, description => "Busy processes: $fcgi_busy_count ($msg)" };
        };
        return $monrun_res;
    },
);

my $proj = CatalogiaMediaProject->new({
    no_form => 1,
    no_auth => 1,
});

check_indicator($proj, $indc_name);

release_file_lock($lock_name);
exit(0);


sub check_indicator {
    my ($proj, $indc_name) = @_;

    if (not $indc_name) {
        $proj->log("ERROR: void indc_name");
        print "2;Internal error!\n";
        return;
    }

    my $func_ref = $indc_funcs{$indc_name};
    if (not defined $func_ref) {
        $proj->log("ERROR: indc_func{$indc_name} not defined");
        print "2;Internal error!\n";
        return;
    }

    my $res = do_safely(sub{ &$func_ref($proj) }, timeout => 60, no_die => 1,);
    my $msg = $@;
    $msg =~ s/\n/  /g; # На выходе для monrun должна быть одна строка
    my $res_str = (ref($res) eq 'HASH' and defined $res->{state})  ?
                ($res->{state}.";".$res->{description})
            :  "2;Internal Error ($msg)";
    $res_str .= " [" . $proj->dates->cur_date('db_time') . "]";
    $proj->log("Result: $res_str");
    print $res_str, "\n";
    $proj->log("Checking $indc_name done");
}
