#!/usr/bin/perl -w

use strict;

my $start_time = time;

use Getopt::Long;

use utf8;
use open ':utf8';
no warnings 'utf8';
binmode(STDIN,  ":utf8");
binmode(STDOUT, ":utf8");
binmode(STDERR, ":utf8");

use List::Util qw(sum min max);
use Data::Dumper;
use POSIX qw(strftime);
use Time::Piece;

use FindBin;
use lib "$FindBin::Bin/../lib";
use Utils::Hosts qw(get_host_role get_curr_host);
use Utils::Common;
use Utils::Funcs;
use Utils::Sys qw/
    get_file_lock release_file_lock
    print_log
    print_err
    handle_errors
    time_log2unix
/;
use BM::Monitor::Utils;
use BM::SolomonClient;

# Как протестировать скрипт на локальной машине:
# 1) запустить скрипт со следующими параметрами:
#   ~/arcadia/rt-research/broadmatching$ BM_CURR_HOST=host.yandex.ru DEBUG=1 ./scripts/monitors/send-from-logs-to-graphite.pl --dbtime '2017-10-12 17:01:00' >tmp-bsc-1 2>tmp-bsc-2 &
#   где
#       - BM_CURR_HOST=host.yandex.ru - указание, что мы как будто на этой машине и обрабатываем её логи. вместо этой укажите интересующую вас машину
#       - DEBUG=1 - флаг дебага, благодаря этому флагу скрипт ничего не отправляет в Graphite
#       - dbtime '2017-10-12 17:01:00' - указание, логи за какое время обрабатываем. на момент написания инструкции работа происходит с данными за предыдущую минуту (то есть '2017-10-12 17:00:00' - '2017-10-12 17:01:00')
# 2) в директорию log подкладываем логи с таким же названием, как на той машине, которую мы указали в BM_CURR_HOST
# 3) убеждаемся, что интересующие нас строки лежат в логе с нужным называнием и попадают в указанный временной промежуток, заданный с помощью dbtime
# 4) запускаем и читаем в логах, что скрипт отправил бы в Graphite

my $DEBUG = $ENV{DEBUG};
print_err("DEBUG: $DEBUG")   if $DEBUG;

my $DATETIME_RE_STR = '\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d(?:\.\d{6})?'; # seconds or microseconds

handle_errors;

my $solomon_client = BM::SolomonClient->new();
my $curr_host = get_curr_host();
main();

exit(0);


sub main {
    my %opt = (
        dbtime => '',
    );
    GetOptions(
        \%opt,
        'dbtime=s', # (для отладки) Работать с данными из лога за эту минуту (вместо последней)
    );

    get_file_lock() or do {
        print_err("WARN: found already running script, do exit");
        exit(0);
    };

    print_err("Started");

    my $log2settings = get_log2settings() or do {
        print_err("No settings - exit");
        exit(0);
    };

    my $time = $opt{dbtime} ? time_log2unix($opt{dbtime}) : time();

    my $db_time_now = strftime("%Y-%m-%d %H:%M:00", localtime($time));
# Будем отправлять данные за предыдущую целую минуту
    my $max_time = $db_time_now;
    my $min_time = strftime("%Y-%m-%d %H:%M:00", localtime($time - 60));
    my $min_time_read = strftime("%Y-%m-%d %H:%M:00", localtime($time - 120)); # Строки в логе *почти* отсортированы по времени
# TODO fix time!
    print_err("min_time: $min_time min_time_read: $min_time_read max_time: $max_time");

    my $out = {};
    my @errors;

    for my $log_name (sort keys $log2settings) {
        print_err("log_name: $log_name");

        my $options = $Utils::Common::options;
        my $log_dir = $options->{dirs}{log};
        my $log_file = "$log_dir/$log_name";

        my $settings = $log2settings->{$log_name};

        my $data = eval { get_data_from_log($log_file, $settings, $min_time, $min_time_read, $max_time) };
        if ($@) {
            print_err("ERROR: get_data_from_log($log_file '$min_time' '$min_time_read' '$max_time') failed: $@");
            push @errors, $log_name;
            next;
        }

        for my $key (keys %$data) {
            $out->{$key} = ($out->{$key} // 0) + $data->{$key};  # Если один и тот же ключ в данных для разных файлов - суммируем. TODO
        }
    }

    output($out, time => $time, db_time => $db_time_now);

    my $duration = time - $start_time;
    my $errors_count = @errors;
    output({ "scripts.monitors.send-from-logs-to-graphite.duration" => $duration }, time => $time, db_time => $db_time_now);
    output({ "scripts.monitors.send-from-logs-to-graphite.errors_count" => $errors_count }, time => $time, db_time => $db_time_now);

    my $max_allowed_duration = 30;
    if ($duration > $max_allowed_duration) {
        print_err("ERROR: Too long: $duration");
    }

    release_file_lock();

    if ($errors_count) {
        print_err("errors_count: $errors_count");
    } else {
        print_err("send_from_logs_to_graphite_OK");
    }
    print_err("Done. Duration: $duration");
}

sub get_log2settings {
    my (%prm) = @_;

    my $host_role = $prm{host_role} // get_host_role();

    # Проверки для типов строк лога, которые могут встретиться в разных логах
    my $checks = {
        prefprojsrv => [
            {
                label => 'prefprojsrv.debug.can_read_timeout_count',
                re => qr/\tERROR: .* can't connect to prefprojsrv .*prefprojsrv can_read timeout/,
            },
            {
                label => 'prefprojsrv.debug.connection_refused_count',
                re => qr/\tERROR: .* can't connect to prefprojsrv .*Connection refused/,
            },
        ],

        log_errors => [
            {
                label => 'scripts.log_errors.count',
                re => qr/\tERROR\b/,
            },
        ],
    };

    # { 'роль_хоста или несколько ролей через пробел' => { настройки } }
    #   настройки:
    #       always_check_labels - названия метрик для Графита, которые отправлять, даже если они не встретились в логе (для отправки нулевых значений)
    #       add_check - дополнительные проверки, см. $checks
    my $hostrole2log2settings = {
        'bmapi bmapi-test bmapi-qloud bmapi-qloud-test' => {
            'fcgi-http-server.err' => {
                always_check_labels => [ qw[fcgi.begin_count fcgi.end_count fcgi.die_count scripts.yml2directinf.error_codes] ],
            },
        },

        'bmbender-front' => {
            'start_banners_server.err' => {
                always_check_labels => ['bender.cmd.bye.count', 'bender.cmd.search.count', 'bender.cmd.top.count'],
            },
        },

        ( map { $_ => {
                    'cdict-start-server.err' => {
                        ext_check => [
                            {
                                name => 'cdict',
                                regexp => qr/ADDR: .* LENGTH:(?P<reqlen>\d+) +NPHRASES:(?P<items_count>\d+) +STATE:(?P<code>\d+) +TIME:(?P<reqtime>[\d.-]+)/,
                                process => sub { loglist2stat(@_) },
                            },
                        ],
                    },
        } } qw[ bmcdict-front bmcdict-front02 ] ),

        'catmedia catalogia-media-front' => do {
            my $log2settings = {};

            $log2settings->{'prefprojsrv-restart-server.err'} = {
                always_check_labels => [qw[
                        prefprojsrv.debug.socket_recv_failed_count
                        scripts.log_errors.count
                ]],
                add_check => [
                    {
                        label => 'prefprojsrv.debug.socket_recv_failed_count',
                        re => qr/\tSocket::recv\(\) failed/,
                    },
                    @{$checks->{log_errors}},
                ],
            };
            $log2settings->{'prefprojsrv-start-server.err'} = $log2settings->{'prefprojsrv-restart-server.err'};

            $log2settings->{'fcgi-proj.err'} = {
                always_check_labels => [qw[
                        prefprojsrv.debug.can_read_timeout_count
                        prefprojsrv.debug.connection_refused_count
                        fcgi.begin_count
                        fcgi.end_count
                        scripts.log_errors.count
                ]],
                add_check => [
                    @{ $checks->{prefprojsrv} },
                    @{ $checks->{log_errors} },
                ],
            };
            $log2settings->{'subphraser-update-from-db.err'} = $log2settings->{'fcgi-proj.err'};

            $log2settings->{$_} = {
                add_check => [ @{$checks->{log_errors}} ],
                always_check_labels => [qw[ scripts.log_errors.count ]],
            } for qw[
                                prefprojsrv-start-server.log
                                prefprojsrv-restart-server.log
                                subphraser-start-server.log subphraser-start-server.err
                                subphraser-start-server-perfect.log subphraser-start-server-perfect.err
            ];

            $log2settings;
        },
    };

    # Обрабатываем ключи, состоящие из нескольких ролей
    for my $key (grep { m/\s+/ } keys %$hostrole2log2settings) {
        for my $role (grep {$_} split /\s+/, $key) {
            $hostrole2log2settings->{$role} = $hostrole2log2settings->{$key};
        }
        delete $hostrole2log2settings->{$key};
    }
    #print Dumper($hostrole2log2settings);

    my $log2settings = $hostrole2log2settings->{$host_role};
    #print Dumper($log2settings);
    return $log2settings;
}

sub get_data_from_log {
    my ($log_file, $settings, $min_time, $min_time_read, $max_time) = @_;

    print_err("get_data_from_log( $log_file '$min_time' '$min_time_read' '$max_time' )");

    my $max_line_length = 2000;
    my $good_line_re = qr/^$DATETIME_RE_STR\t\[\d+\]\t/;

    my $out = {};

    $out->{$_} //= 0    for @{ $settings->{always_check_labels} };

    # TODO check file existance?

    my @log_files = ($log_file);
    if (-f (my $f = "$log_file.1")) {  # logrotate
        push @log_files, $f;
        # неатомарно :(   TODO
    }
    print_err("log_files: @log_files");

    # Нужен timeout из-за https://st.yandex-team.ru/BSDEV-61728
    my $timeout = 120;
    my $open_time = time;
    open my $fh, "timeout --kill-after 1 $timeout tac @log_files | cut -c 1-$max_line_length |" or die("Cannot open $log_file ($!)");
    my $name2list = {};
    my $values = {};
    my $filters = {};
    while(defined (my $line = <$fh>)) {
        next if $line !~ $good_line_re;
        # TODO check bad lines count
        next if $line ge $max_time;  # TODO fix times!
        last if $line && $line lt $min_time_read;
        next if $line && $line lt $min_time;

        if ($line =~ m/\tSEND_TO_GRAPHITE:/) {
            # Строки со специальной меткой - SEND_TO_GRAPHITE
            for my $s ($line =~ m/\s(GR_OUT\S*:\S+)/g) {
                my ($type, $label, $value) = split /:/, $s;
                next unless $label;  # TODO - ?
                if ($type eq 'GR_OUT_MAX') {
                    $out->{$label} = defined $out->{$label} ? max($out->{$label}, $value) : $value;
                } elsif ($type =~ 'GR_OUT_(AVG|MED)') {
                    push @{$filters->{$label}}, $1;
                    $values->{$label} //= [];
                    push @{$values->{$label}}, $value;
                } else {
                    $out->{$label} //= 0;
                    $out->{$label} += $value;
                }
            }
        };

        for my $check (@{ $settings->{add_check} }) {
            # Строки произвольного формата, подходящие под regexp - считаем количество таких строк
            if ($line =~ $check->{re}) {
                my $label = $check->{label};
                $out->{$label} //= 0;
                $out->{$label} += 1;
            }
        }

        for my $check (@{ $settings->{ext_check} }) {
            # Строки произвольного формата, подходящие под regexp - обрабатываем массив заданной функцией
            if ($line =~ $check->{regexp}) {
                #print_log("line: $line");
                my $el = { %+ };
                push @{ $name2list->{ $check->{name} } }, $el;
            }
        }
    }
    close $fh;   # TODO  or die "Cannot close $log_file ($!)";
    if (time - $open_time >= $timeout) { # Сработал timeout - не успели прочитать данные из файла
        die "Files read timeout($timeout) (@log_files)";
    }

    for my $k (keys %{$filters}) {
        my @vls = @{$values->{$k}};
        for my $f (@{$filters->{$k}}) {
            if ($f eq 'AVG') {
                my $s = sum(@vls);
                $out->{$k} = (scalar @vls) ? ($s/(scalar @vls) ) : 0;
            } elsif ($f eq 'MED') {
                @vls = sort @vls;
                $out->{$k} = (scalar @vls) ? ($vls[$#vls/2]) : 0;
            }
        }
    }

    for my $check (@{ $settings->{ext_check} }) {
        my $name = $check->{name};
        print_err("ext_check $name process ...");
        my $list = $name2list->{$name};
        my $h = $check->{process}->($list) // {};
        my $graphite_name = $name; # Сделать отдельное поле graphite_name ?
        $h = { map { ( "$graphite_name.$_" => $h->{$_} ) } keys %$h };
        $out = { %$out, %$h };
    }

    return $out;
}

sub output {
    my ($out, %prm) = @_;
    my $output_dir = $Utils::Common::options->{dirs}{monitor_info};
    unless (-d $output_dir) {
        Utils::Sys::do_sys_cmd("mkdir -p $output_dir");
    }
    my $output_file = "$output_dir/data-from-logs";
    print_err("output_file: $output_file");
    open my $fh, ">>$output_file" or print_err("ERROR: Could not open $output_file ($!)"); # TODO exit code скрипта

    for my $key (sort keys $out) {
        my $value = $out->{$key};
        next unless defined $value;

        if ($fh) {
            print $fh join("\t", $prm{db_time}, $prm{time}, $key, $value), "\n";
        }

        BM::Monitor::Utils::graphite_client()->send_value(
            [ $key ],
            $value,
            time => $prm{time},
        );
        send_to_solomon ($key, $value, $prm{time});
    }
    if ($fh) {
        close $fh or print_err("ERROR: Could not close $output_file ($!)");
    }
}

# На входе:
#   ссылка на массив хешей, полученных обработкой строк лога regexp'ом (один запрос - одна строка лога - один элемент массива)
#   get_data => [ какие данные нужны ]
#       Возможные значения: input_data result counts_per_s counts_per_0_1_s counts_per_0_01_s
#       Если не задано, то  [qw[ input_data result ]]
# Поля в хеше (для строки лога):
#   reqlen      -  длина запроса
#   items_count -  количество элементов в запросе (если бывают запросы пачками)
#   reqtime     -  время обработки запроса
#   code        -  результат обработки запроса: 0 - OK, не 0 - ошибка.
#       Если не задано, считаем ошибкой. TODO - не учитывать undef ???
# (TODO - обработка элементов, у которых в некоторых полях - undef или '')
# На выходе - хеш для отправки в Графит и записи в файл
sub loglist2stat {
    my ($list, %prm) = @_;
    my %get_data = map { $_ => 1 } @{ $prm{get_data} || [qw[ input_data result ]] };

    my $out = {};

    if ($get_data{input_data}) {
        $out->{requests_count} = scalar @$list; # TODO убрать, оставив requests.count.total  ?
        $out->{'requests.count.total'} = scalar @$list; # TODO
        $out->{items_count} = sum( map { $_->{items_count} || 0 } @$list ) || 0;
        $out->{reqlen_total} = sum( map { $_->{reqlen} || 0 } @$list ) || 0;
    }
    if ($get_data{result}) {
        $out->{errors_count} = scalar grep { ($_->{code} // -1) != 0 } @$list;
    }

    if (@$list) {
        if ($get_data{result}) {
            $_->{'timings.per_item'} = $_->{reqtime} / $_->{items_count}   for grep { $_->{items_count} } @$list;
            $_->{'timings.per_request'} = $_->{reqtime}   for @$list;
        }

        my @keys4rate = (
            ($get_data{input_data} ? qw[ reqlen ] : ()),
            ($get_data{result} ? qw[ timings.per_item timings.per_request ] : ()),
        );
        my $stat = get_stat($list, \@keys4rate, quantiles => [qw[ 0.30 0.50 0.70 0.80 0.90 0.95 0.99 0.999 ]]);
        $out->{$_} = $stat->{$_}  for keys %$stat;

        for (
            [ 'counts_per_s',       'per_s',        '....-..-.. ..:..:(..)',        [ map { sprintf("%02s", $_) } (0 .. 59) ], ],
            [ 'counts_per_0_1_s',   'per_0_1_s',    '....-..-.. ..:..:(..\..)',     [ map { sprintf("%04.1f", $_ / 10) } (0 .. 599) ], ],
            [ 'counts_per_0_01_s',  'per_0_01_s',   '....-..-.. ..:..:(..\...)',    [ map { sprintf("%04.2f", $_ / 100) } (0 .. 5999) ], ],
        ) {
            my ($data_name, $name, $re_str, $times_base) = @$_;
            if ($get_data{$data_name}) {
                my @datetimes = map { $_->{datetime} } @$list;
                my @times = map { m/^$re_str/ } @datetimes;
                my $time2cc = { map { $_ => 0 } @$times_base };
                $time2cc->{$_}++   for @times;
                my $key = "requests.count.$name";
                my $stat = get_stat([ map { { $key => $_ } } values %$time2cc ], [ $key ], quantiles => [qw[ 0.70 0.90 0.95 0.99 0.999 ]]);
                $out->{$_} = $stat->{$_}  for keys %$stat;
            }
        }
    }

    my $convert_field_name = sub {
        my $fld = shift;
        my @keys = (
            ($get_data{input_data} ? qw[ reqlen items_count requests_count requests.count ] : ()),
            ($get_data{result} ? qw[ errors_count timings.per_item timings.per_request ] : ()),
            ($get_data{counts_per_s} ? qw[ requests.count.per_s ] : ()),
            ($get_data{counts_per_0_1_s} ? qw[ requests.count.per_0_1_s ] : ()),
            ($get_data{counts_per_0_01_s} ? qw[ requests.count.per_0_01_s ] : ()),
        );
        return unless grep { $fld =~ m/^$_\b/ } @keys;
        return $fld;
    };
    for my $fld (keys $out) {
        my $new_fld = $convert_field_name->($fld);
        if ($new_fld) {
            $out->{$new_fld} = delete $out->{$fld};
        } else {
            delete $out->{$fld};
        }
    };
    return $out;
}


sub get_stat {
    my ($list, $keys, %prm) = @_;
    my $quantiles = $prm{quantiles} // [];
    my $stat = {};
    for my $key (@$keys) {
        my @values = map { $_->{$key} } @$list;
        $stat->{ "$key.max" } = max(@values);
        $stat->{ "$key.avg" } = sum(@values) / @values   if @values;
        for my $rate (@$quantiles) {
            my ($subkey) = $rate =~ m/^0\.(.*)/;
            $stat->{ "$key.$subkey" } = Utils::Funcs::get_quantiles(\@values, $rate);
        }
    }
    return $stat;
}

sub send_to_solomon {
    my ($key, $value, $time) = @_;
    my $solomon_sensor = undef;

    my $ts_datetime = undef;
    if ($time) {
        $ts_datetime = gmtime($time)->datetime();
    }

    my @metric_data = split /\./, $key;
    return unless @metric_data;
    if ((scalar(@metric_data) == 3) && ($metric_data[0] eq "prefprojsrv")) {
        $solomon_sensor = {
            cluster     => "host_info",
            service     => "fcgi_server",
            sensor      => "prefprojsrv",
            labels      => {
                metric_name => $metric_data[2],
            },
        };
    } elsif ((scalar(@metric_data) == 3) && ($metric_data[1] eq "log_errors")) {
        $solomon_sensor = {
            cluster     => "host_info",
            service     => "logs",
            sensor      => "errors_count",
        };
    } elsif ((scalar(@metric_data) == 2) && ($metric_data[0] eq "cdict")) {
        $solomon_sensor = {
            cluster     => "host_info",
            service     => "cdict",
            sensor      => $metric_data[1],
        };
    } elsif ((scalar(@metric_data) == 4) && ($metric_data[0] eq "cdict") && ($metric_data[1] eq "timings")) {
        $solomon_sensor = {
            cluster     => "host_info",
            service     => "cdict",
            sensor      => "timings",
            labels      => {
                property  => $metric_data[2],
                item_name => $metric_data[3],
            },
        };
    } elsif ($metric_data[0] eq "catalogia-pages") {
        if (scalar(@metric_data) == 2) {
            $solomon_sensor = {
                cluster => "host_info",
                service => "catalogia-pages",
                sensor  => $metric_data[1],
            };
        } elsif ((scalar(@metric_data) == 5) && ($metric_data[1] eq "requests") && ($metric_data[2] eq "count")) {
            $solomon_sensor = {
                cluster => "host_info",
                service => "catalogia-pages",
                sensor  => "requests_counts",
                labels  => {
                    property  => $metric_data[3],
                    item_name => $metric_data[4],
                },
            };
        } elsif ((scalar(@metric_data) == 4) && ($metric_data[1] eq "timings")) {
            $solomon_sensor = {
                cluster => "host_info",
                service => "catalogia-pages",
                sensor  => "timings",
                labels  => {
                    property  => $metric_data[2],
                    item_name => $metric_data[3],
                },
            };
        } elsif ((scalar(@metric_data) == 3) && ($metric_data[1] eq "reqlen")) {
            $solomon_sensor = {
                cluster => "host_info",
                service => "catalogia-pages",
                sensor  => "reqlen",
                labels  => {
                    item_name => $metric_data[2],
                },
            };
        };
    }

    if ($solomon_sensor) {
        $solomon_sensor->{labels}->{host} = $curr_host;
        $solomon_sensor->{value} = $value;
        if ($ts_datetime) {
            $solomon_sensor->{ts_datetime} = $ts_datetime;
        }
        $solomon_client->push_single_sensor($solomon_sensor);
    }
}
