#!/usr/bin/perl -w

use strict;

my $start_time = time;

use Getopt::Long;

use utf8;
use open ':utf8';
no warnings 'utf8';
binmode(STDIN,  ":utf8");
binmode(STDOUT, ":utf8");
binmode(STDERR, ":utf8");

use Data::Dumper;
use Date::Parse qw(str2time);

use FindBin;
use lib "$FindBin::Bin/../lib";
use Utils::Sys qw[
    get_file_lock release_file_lock
    print_err
    handle_errors
];
use BM::Monitor::Utils;

handle_errors();

my %opt = (
);
GetOptions(
    \%opt,
    'help|h',
    'test',
);

if ($opt{help}) {
    print "$_\n" for (
        "send-portoctl-data-to-graphite.pl  -  Call `portoctl get self`, parse its output and send to Graphite",
        "  --test      Do not run portoctl, use test data",
    );
    exit(0);
}

my $USE_TEST_DATA = $opt{test} ? 1 : 0;
print_err("USE_TEST_DATA: $USE_TEST_DATA")   if $USE_TEST_DATA;

main();

exit(0);


sub main {
    unless ($USE_TEST_DATA) {
        get_file_lock() or do {
            # Будем запускать по крону раз в минуту. Если предыдущий скрипт еще не доработал - это плохо
            print_err("ERROR: found already running script, do exit");
            exit(0);
        };
    }
    print_err("Started");

    my $current_time = time;
    my $portoctl_output = get_portoctl_output();
    my $out = parse_portoctl_output($portoctl_output);

    for my $key (sort keys $out) {
        my $value = $out->{$key};
        $key =~ s/\s/_/;
        my $prefix = "portoctl.self.raw";
        if ($USE_TEST_DATA) {
            print_err("Output: $prefix.$key $value");
        } else {
            BM::Monitor::Utils::graphite_client()->send_value(
                [ $prefix, $key ],
                $value,
                time => $current_time,
            );
        }
    }

    unless ($USE_TEST_DATA) {
        release_file_lock();
    }
    print_err("Done.");
}

sub get_portoctl_output {
    my $cmd = '/usr/sbin/portoctl get self'; # Полный путь к portoctl - для запуска из крона
    if ($USE_TEST_DATA) {
        $cmd = "cat $Utils::Common::options->{dirs}{scripts}/tests/sample_portoctl_get_self_output.txt";
    }
    my $output = Utils::Sys::read_sys_cmd($cmd);
    return $output;
}

# На входе: строка в формате выдачи `portoctl get self`
# На выходе: хэш { "key.subkey1..." => $value } - данные, которые удалось распарсить из входной строки
sub parse_portoctl_output {
    my ($portoctl_output) = @_;
    my $res = {};

    # Для строк известных форматов - преобразует строку в число
    # Если не подходит ни под какой шаблон - возвращает undef
    my $parse = sub {
        my ($input_str) = @_;
        return unless defined $input_str;
        $input_str =~ s/(^\s*|\s*$)//g;
        my $val;
        if ($input_str =~ m/^\-?\d+(\.\d*)?$/) {
            # Число
            $val = $input_str;
        } elsif ($input_str =~ m/^(\-?\d+(?:\.\d*)?)[eE]([\-\+]\d+)s?$/) {
            # Экспоненциальная запись: 1.33673e+07s -> 13367300
            my ($n, $p) = ($1, $2);
            $val = $n * 10 ** $p;
        } elsif ($input_str =~ m/^(\d{1,2}):(\d\d):(\d\d)$/) {
            # Время (HH:MM:SS) -> секунды
            my ($h, $m, $s) = ($1, $2, $3);
            $val = $s + $m*60 + $h*60*60;
        } elsif ($input_str =~ m/^(\d+)d +(\d+):(\d\d)$/) {
            # Время (DDd HH:MM) -> секунды
            my ($d, $h, $m) = ($1, $2, $3);
            $val = $m*60 + $h*60*60 + $d*60*60*24;
        } elsif ($input_str =~ m/^....-..-.. ..:..:..$/) {
            # Время (YYYY-MM-DD HH:MM:SS) -> секунды
            my $sec = str2time($input_str);
            $val = $sec;
        } elsif ($input_str =~ m/^(\d+\.?\d*)([BKMGT])$/) {
            # (Тера|Гига|Мега|Кило|)байты -> байты
            my ($v, $l) = ($1, $2);
            my $letter2multiplier = { B => 1, K => 1024, M => 1024*1024, G => 1024*1024*1024, T => 1024*1024*1024*1024 };
            $val = int($v * $letter2multiplier->{$l});
        } elsif ( $input_str =~ m/^(\d+)c$/i) {
            # 32c -> 32 ('cpu_limit_total = 32c')
            $val = $1;
        } elsif ( $input_str =~ m/^(true|false)$/i) {
            # true/false -> 1/0
            $val = $input_str =~ m/true/i ? 1 : 0;
        }
        #print_err("  input_str: '$input_str'    val: " . ($val // 'UNDEF'));
        return $val;
    };

    # Для строк, содержащих время (YYYY-MM-DD HH:MM:SS) возвращает "возраст": разницу (current_time - input_time) (в секундах)
    my $parse_time_age = sub {
        my ($input_str) = @_;
        return unless defined $input_str;
        $input_str =~ s/(^\s*|\s*$)//g;
        my $val;
        if ($input_str =~ m/^....-..-.. ..:..:..$/) { # YYYY-MM-DD HH:MM:SS
            my $sec = str2time($input_str);
            $val = time - $sec;
        }
        #print_err("  input_str: '$input_str'    val: " . ($val // 'UNDEF'));
        return $val;
    };

    for my $line (split /\n/, $portoctl_output) {
        print_err("line: $line");
        my ($key, $val_str) = split / = /, $line, 2;
        next unless defined $val_str;
        my $key2val_str = {};
        if ($val_str =~ m/[;]/) {
            # Формат строки: "key = subkey1: value1; subkey2: value2; ..."
            for my $str (split /;\s*/, $val_str) {
                my ($subkey, $subval_str) = split /:/, $str, 2;
                $key2val_str->{"$key.$subkey"} = $subval_str;
            }
        } else {
            # Формат строки: "key = value"
            $key2val_str->{$key} = $val_str;
        }

        for my $key (sort keys %$key2val_str) {
            my $val_str = $key2val_str->{$key};
            if (defined (my $val = $parse->($val_str))) {
                #print_err("  key: $key    val: $val");
                $res->{$key} = $val;

                if (defined (my $val = $parse_time_age->($val_str))) {
                    my $key_out = $key."_age";
                    #print_err("  key_out: $key_out    val: $val");
                    $res->{$key_out} = $val;
                }
            } else {
                print_err("Could not parse value: '" . ($val_str // '') . "' line: $line");
            }
        }
    }

    return $res;
}
