#!/usr/bin/perl -w
use strict;

use utf8;
use open ':utf8';
use Data::Dumper;

use FindBin;
use IO::Handle;
use lib "$FindBin::Bin/lib";
use lib "$FindBin::Bin/wlib";
use lib "$FindBin::Bin/cpan";

use Utils::Common;
use Utils::Sys qw(
    get_file_lock
    release_file_lock
    handle_errors
    print_log
    log_msg_fmt
    do_safely
    md5int
);
use CatalogiaMediaProject;
use BM::BannersMaker::Tasks::TaskQueue;
use Utils::Hosts qw(get_curr_host);
use Utils::Sys qw(assert_curr_host);

handle_errors();

my $helpformat = "usage: $0 fork_count type";

my $fork_count = $ARGV[0];
die($helpformat) unless $fork_count;

my $type = $ARGV[1];
die($helpformat) unless $type;

assert_curr_host(allow => { role => ['bannerland', 'bannerland-preprod'] })
or die "do not run this script in non-production enviroments without BM_USE_PATCH=debug_env";

my @pids;
for my $fork_index (1..$fork_count) {
    if(my $pid = fork) {
        push @pids, $pid;
        print_log("process $fork_index/$fork_count has started with pid=$pid");
        next;
    }

    # ставим блокировку на процесс
    my $lock_file_name = "process_cronlight_tasks_${type}_$fork_index";
    if(!get_file_lock($lock_file_name)) {
        print_log("process $fork_index is already running");
        exit(0);
    }

    # перенаправление логов, чтобы выдача от разных потоков не перемешивалась
    open(CRLOLDOUT, ">&STDOUT");
    open(CRLOLDERR, ">&STDERR");
    *CRLOLDOUT = *STDOUT{IO};
    *CRLOLDERR = *STDERR{IO};
    my $prev_stdout = *CRLOLDOUT;
    my $prev_stderr = *CRLOLDERR;

    my $log_dir = $Utils::Common::options->{dirs}{'log'};
    open NCRLSTDOUT, ">> $log_dir/process_${type}_tasks_$fork_index.log" or die "Can't redirect STDOUT: $!";
    open NCRLSTDERR, ">> $log_dir/process_${type}_tasks_$fork_index.err" or die "Can't redirect STDERR: $!";
    *STDERR = *NCRLSTDERR{IO};

    select(NCRLSTDOUT); # #Если использовать не select для STDOUT, то почему-то ломается чтение в IPC::Open2, что критично для BM::LemmerTest
    STDOUT->autoflush(1);
    STDERR->autoflush(1);

    # обрабатываем таску
    process_task(prev_stderr => $prev_stderr);

    # снимаем блокировку на процесс
    release_file_lock($lock_file_name);

    exit(0);
}

for my $pid (@pids) {
    waitpid( $pid, 0 );

    if($?) {
        print_log("child $pid has failed ($?)");
    } else {
        print_log("child $pid has finished");
    }
}

exit(0);

sub process_task {
    my %opts = @_;

    # создание Project
    my $proj = CatalogiaMediaProject->new({
        load_dicts => 1,
        load_minicategs_light => 1,
        use_comptrie_subphraser => 1,
        no_auth => 1,
        no_form => 1,
        nrmsrv  => 0,
        memlog => 1,
        global_stderr => $opts{prev_stderr},
    });
    my $queue = BM::BannersMaker::Tasks::TaskQueue->new({ proj => $proj, type => $type });
    my $crontime = $proj->curtime;
    my $cronlight_id = join("_", $queue->tasks_name, $queue->worker_name, $crontime);
    $cronlight_id =~ s/ /_/g;
    
    local $SIG{'__DIE__'} = sub { print STDERR $proj->stack_trace('  STRACE_ERR   '); die(log_msg_fmt('ERROR:STRACE: '.join('', @_))); };

    # получаем слеюующую таску из очереди и ставим блокировку на нее
    my $task = $queue->get_next_task;

    # обновляем очередь, если она пуста
    if(!$task) {
        my $refresh_lock = "process_cronlight_tasks_refresh_$type";

        if(!get_file_lock($refresh_lock)) {
            print_log("tasks queue is already being refreshed");
            return;
        }

        print_log("refresh_tasks");
        eval {
            $queue->refresh_tasks;
        };
        if ($@) {
            print_log("ERROR: refresh_tasks died with '$@'");
        }
        print_log("/ refresh_tasks");
        release_file_lock($refresh_lock);

        $task = $queue->get_next_task;

        if(!$task) {
            print_log("no tasks, nothing to do");
            return;
        }
    }

    $proj->{cronlight_id} = $cronlight_id;
    $proj->{cronlight_crontime} = $crontime;

    # отдаем таску воркеру и ставим блокировку на домен
    $queue->invoke_worker($task);

    # снимаем блокировку на таску и удаляем таску
    $queue->free_last_task;
}

