#!/usr/bin/env perl

use strict;
use warnings;

=head1 NAME

publish_catalogia_to_sandbox.pl - publish catalogia dicts and scripts to Sandbox.

=head1 DESCRIPTION

Collects all specified scripts and data (from /opt/broadmatching), needed to run catalogia on MR (or any random host).
Packs it into tar archive and uploads to Sandbox.

=cut

use File::Path qw/make_path/;
use FindBin;
use lib "$FindBin::Bin/lib";
use Getopt::Long;
use IronProject;

use Utils::Common;
use BM::Bootstrap;
use BM::SandboxClient;
use Utils::Sys qw(
    get_file_lock
    release_file_lock
    wait_for_file_lock
    print_err
    do_sys_cmd
    handle_errors
);

handle_errors();

my %opt;
GetOptions(\%opt, 'bootstraps=s', 'waitlock', 'test', 'no_rsync', 'help|h', 'prod');
if ($opt{help}) {
    printf "Usage: $0 [Options]\n";
    printf "Options:\n";
    printf "  --bootstraps      list of bootstraps\n";
    printf "  --waitlock        subj\n";
    printf "  --test            don't upload to SB, only prepare bootstraps\n";
    printf "  --prod            upload with \"production=1\", 0 otherwise. Use only for production uploads\n";
    printf "  --no_rsync        not sync rsync resouces\n";
    exit(0);
}


if ($opt{waitlock}) {
    print_err("wait_for_file_lock ...");
    wait_for_file_lock()
        or die "Could not wait_for_file_lock!";
} else {
    get_file_lock()
        or exit(0);
}

my $options = $Utils::Common::options->{mr_catalogia};
my $bm_dir = $Utils::Common::options->{'dirs'}{'root'};
my $ya = '/usr/local/bin/ya';

my $work_dir = $options->{work_dir};
make_path($work_dir);

my %HOW = (
    '01_catalogia' => {
        bootstrap_name => 'catalogia_publish',
        type => 'BROADMATCH_MR_CATALOGIA_RAW',
        message => qq{BROADMATCH_MR_CATALOGIA_RAW from `hostname -f` on `date` revision `$ya tool svn </dev/null info $bm_dir | grep 'Revision' | cut -d ' ' -f2`},
    },
    '03_bm_dicts' => {
        bootstrap_name => 'bm_dicts',
        type => 'BROADMATCH_MR_CATALOGIA_EXTERNAL',
        message => qq{bm_dicts from `hostname -f` on `date`},
        attrs => { type => 'bm_dicts' },
    },
    '04_bm_scripts' => {
        bootstrap_name => 'bm_scripts',
        type => 'BROADMATCH_MR_CATALOGIA_EXTERNAL',
        message => qq{bm_scripts from `hostname -f` on `date` revision `$ya tool svn </dev/null info $bm_dir/scripts | grep 'Revision' | cut -d ' ' -f2`},
        attrs => { type => 'bm_scripts' },
    },
);

my @bootstrap_cfg = map { $HOW{$_} } sort keys %HOW;
if ($opt{bootstraps}) {
    my %use = map { $_ => 1 } split /,/, $opt{bootstraps};
    @bootstrap_cfg = grep { $use{$_->{bootstrap_name}} } @bootstrap_cfg;
}

my $proj = IronProject->new({});
my $dont_get_rsync_resources = $opt{no_rsync} // '';
$proj->log("dont_get_rsync_resources = $dont_get_rsync_resources");

for my $cfg (@bootstrap_cfg) {
    $cfg->{attrs}{production} = $opt{prod} ? '1' : '0';
    my $bootstrap = BM::Bootstrap::get_bootstrap($cfg->{bootstrap_name}, dont_get_rsync_resources=>$dont_get_rsync_resources);

    my $spec = {
        description => $cfg->{message},
        type => $cfg->{type},
        attrs => $cfg->{attrs},
    };

    if ($opt{test}) {
        $proj->log("test run, only save bootstrap $bootstrap");
    } else {
        $proj->sandbox_client->upload([$bootstrap], $spec);
    }
}

release_file_lock();
exit(0);

