#!/usr/bin/perl -w
use strict;

use utf8;
use open ':utf8';

use FindBin;
use lib "$FindBin::Bin/lib";
use lib "$FindBin::Bin/wlib";

use Data::Dumper;
use List::Util qw(max);
use Utils::Common;
use Utils::Sys qw(
    get_file_lock
    release_file_lock
    handle_errors
    print_err
    do_safely
);
use Utils::Hosts qw(
    get_curr_host
    get_hosts
);
use Utils::RestartSettings;
use POSIX qw(setsid);
use JSON qw(to_json);
use Project;
use BM::SolomonClient;

handle_errors;
get_file_lock("restart_services") or exit(0);

my @errors;

my $log_file = $Utils::Common::options->{RestartServices_params}{log_file};
my $host = get_curr_host;

print_err("host: $host");

print_err("done_cmds file: $log_file");

# загрузка лога
my %done_cmds = ();
if(open F, $log_file) {
    while(<F>) {
        chomp;
        my ($cmd, $time) = split "\t";
        next if !$cmd || !$time;
        my $prev_time = $done_cmds{$cmd} || "";
        $done_cmds{$cmd} = $time if $prev_time lt $time;
    }
    close F;
}

print_err("done_cmds: " . to_json(\%done_cmds));

my $proj = Project->new({});
my $sw_client = $proj->sw_client;

# выполнение команд
for my $cmd (keys %{$Utils::RestartSettings::settings}) {
    my $h = $Utils::RestartSettings::settings->{$cmd};

    my $hosts = $h->{hosts} || [];
    if (my $role = $h->{host_role}) {
        push @$hosts, sort (get_hosts(role => $role));
    }

    my $my_index = 0;
    
    # определение индекса текущего хоста в списке хостов
    for($my_index = 0; $my_index < @$hosts; $my_index++) {
        last if $hosts->[$my_index] eq $host;
    }
    next if $my_index >= @$hosts;
    print_err("my_index: $my_index");
    print_err("cmd: $cmd");

    my $lockname = join('_', 'restart_services',  $cmd);

    my @now = localtime time;
    my $current_time = format_time(\@now);

    my $self_alive = check_self($cmd, \@errors, 'quick');
    
    #берем лок до проверки хостов, чтобы при проверке в процессе рестарта находились только мы
    $sw_client->lock_node($lockname, waitable => 1, timeout => 60, time_between_tries=>1) or next;
    ####
    # начало эксклюзивной области
    ####

    if ($self_alive) { #если текущий хост лежит, поднимаем без проверок. Если живой - проверяем, стоит ли рестартовать
        # определение времени выполнения команды
        my $hour = $h->{start_hour} || 0;
        my $minute = $h->{start_minute} || 0;
        $minute += ($h->{restart_interval} || 60) * $my_index;
        $hour = ($hour + int($minute / 60)) % 24;
        $minute %= 60;
        my $restart_time = format_time(\@now, $hour, $minute);
        if ($current_time lt $restart_time || ($done_cmds{$cmd} && $done_cmds{$cmd} ge $restart_time)) {
            print_err("not a time for $cmd ('$current_time' '$restart_time' '$done_cmds{$cmd}')");
            $sw_client->unlock_node($lockname);
            next;
        }
    
        #проверяем, что процент живых хостов не ниже порога
        my $alive_count = $self_alive; #себя уже проверили
        my $check_host = $h->{check_host} || sub { return 1; };
        foreach my $host_to_check ( grep {$_ ne $host} @$hosts ) {
            $alive_count += $check_host->($host_to_check) ? 1 : 0;
        }
        my $total_count = scalar(@$hosts);
        my $critical_percent = 90; #TODO сделать это параметром в RestartSettings, если нужно
        if ( $alive_count/$total_count < $critical_percent / 100 ) {
            $sw_client->unlock_node($lockname);
            next;
        }
    }
    # выполнение команды
    print_err("executing $cmd on $host...");

    if (my $bash_cmd = $h->{close_under_balancer_cmd}) {
        print_err("close_under_balancer_cmd ...");
        Utils::Sys::do_sys_cmd($bash_cmd);
    }

    print_err("cmds ...");
    foreach my $bash_cmd (@{$h->{cmds} || []}) {
        Utils::Sys::do_sys_cmd($bash_cmd, no_die => 1);
    }
    Utils::Sys::do_sys_cmd("sleep 20");
    ####
    # конец эксклюзивной области
    ####
    $sw_client->unlock_node($lockname);

    check_self($cmd, \@errors) or next;

    if (my $bash_cmd = $h->{open_under_balancer_cmd}) {
        Utils::Sys::do_sys_cmd("sleep 20");
        print_err("open_under_balancer_cmd ...");
        Utils::Sys::do_sys_cmd($bash_cmd);
    }

    # запись в лог
    print_err("print to log ...");
    open F, ">> $log_file" or die("ERROR: $!");
    print F "$cmd\t$current_time\n";
    close F;

    print_err("restart_services_${cmd}_OK"); # Будем проверять в мониторинге
}

release_file_lock("restart_services");

if (@errors) {
    print_err("Done. Errors: " . join("; ", @errors));
} else {
    print_err("restart_services_OK"); # Будем проверять в мониторинге
    BM::SolomonClient->new()->set_success_script_finish("restart_services");
}

exit(0);

sub format_time {
    my ($now, $hour, $minute) = @_;

    $hour = $now->[2] if !defined($hour);
    $minute = $now->[1] if !defined($minute);

    return sprintf "%04d-%02d-%02d %02d:%02d", 
        $now->[5] + 1900, 
        $now->[4] + 1, 
        $now->[3], 
        $hour, 
        $minute;
}

sub check_self {
    my ($cmd, $errors, $quick) = @_;
    print_err("check_self ...");
    my $h = $Utils::RestartSettings::settings->{$cmd};
    if (my $sleep = $h->{check_self}{sleep_before_check}) {
        Utils::Sys::do_sys_cmd("sleep $sleep") unless $quick;
    }
    my %prm = (
        tries => 3*60*6, # 3 часа - раз в 10 с
        sleep_between_tries => 10,
        timeout => 5,
    );
    my $do_check = $h->{check_self}{cmd} ? sub {
        Utils::Sys::do_sys_cmd( $h->{check_self}{cmd},
            no_die => 1,
            no_error => 1,
            (map { $_ => $h->{check_self}{$_} // $prm{$_} } qw[ timeout ] )
        );   
    } : sub {
        $h->{check_host}->('localhost');
    };

    my $res = do_safely(
        sub {
            $do_check->() || die "do_sys_cmd failed"; # die - для tries в do_safely
        },
        no_die => 1,   
        $quick ? (tries => 3, sleep_between_tries => 1 ) : (map { $_ => $h->{check_self}{$_} // $prm{$_} } qw[ tries sleep_between_tries ]),
    );
    if ($res) {
        print_err("check_self res: $res");
        return 1;
    } else {
        print_err("WARN: check_self failed");
        push @{$errors}, "check_self($cmd) failed" unless $quick;
        return 0;
    }
}

