import argparse
import datetime
import os
import requests
import time

import irt.broadmatching.common_options
import irt.monitoring.solomon.sensors
import irt.logging


logger = irt.logging.getLogger(irt.logging.BANNERLAND_PROJECT, __name__)


def startrek_count_req(oauth_token, query):
    """
    Запрос в API Startrek-а по количеству тикетов, удовлетворяющих условию query
    :param oauth_token: OAuth-токен Startrek-а
    :param query: условие (синтаксиса Startrek-а) для фильтрации тикетов
    :return: число тикетов или None, если что-то пошло не так
    """

    headers = {
        'Content-Type': 'application/json',
        'Authorization': 'OAuth {}'.format(oauth_token),
    }

    try:
        req = requests.post(
            'https://st-api.yandex-team.ru/v2/issues/_count',
            headers=headers,
            json={'query': query}
        )
        if req.status_code == 200:
            res = req.json()
            if not isinstance(res, int):
                raise RuntimeError('Invalid type of Startrek response')
            return res
        else:
            error_message = 'Failed with getting of tickets count. Wrong response status. Expected 200, got {}'.format(req.status_code)
            logger.error(error_message)
            logger.debug('Response: %s', req.text)
            raise RuntimeError(error_message)

    except Exception as err:
        logger.exception('Unknown error: %s', err)
        raise


def get_opened_ticket_count(oauth_token, day_ago=None):
    """
    Получаем количество незакрытых тикетов, которое было акутально к <days_ago> дней назад (или на текущий момент)
    :param oauth_token: OAuth-токен Startrek-а
    :param days_ago: количество дней назад. Если не указан - то считаем чиселки по состоянию на текущий момент
    :return: искомое число незакрытых тикетов
    """

    if day_ago is None:
        str_date = 'now()'
    else:
        datetime_ago = datetime.datetime.now() - datetime.timedelta(days=day_ago)
        str_date = datetime_ago.strftime('%Y-%m-%d')

    logger.info('Calculating result for datetime: %s', str_date)
    logger.info('Getting all created tickets')
    created_tickets = startrek_count_req(oauth_token, 'Queue: SUPBL AND Created: <= {}'.format(str_date))
    time.sleep(0.1)
    logger.info('Getting all closed tickets')
    resolved_tickets = startrek_count_req(oauth_token, 'Queue: SUPBL AND Status: Закрыт AND Resolved: <= {}'.format(str_date))

    return created_tickets - resolved_tickets


def main():
    argparser = argparse.ArgumentParser(description='Count startrek statistic for SUPBL Queue')
    argparser.add_argument('-d', '--days_ago', type=int, default=None, help='Count of days until today for which statistics is calculated')
    args = argparser.parse_args()
    days_ago = args.days_ago
    if (days_ago is not None) and (days_ago < 1):
        error_message = 'Invalid argument for days ago. Expected None or greater then zero. Got: {}'.format(days_ago)
        logger.error(error_message)
        raise RuntimeError(error_message)

    logger.info('Getting robot Oauth token for Startrek')
    tokens_folder = os.path.join(irt.broadmatching.common_options.get_options()['secrets']['server_dir'], 'tokens')
    token_file = os.path.join(tokens_folder, 'startrek_oauth_token')
    oauth_token = ''
    try:
        with open(token_file, 'r') as f:
            oauth_token = f.read().strip()
    except Exception as err:
        logger.exception('Error while getting Startrek OAuth token from %s: %s', token_file, err)
        raise

    logger.info('Pushing stats to Solomon')
    solomon_client = irt.monitoring.solomon.sensors.SolomonAgentSensorsClient()
    solomon_sensor_template = {
        'cluster': 'startrek',
        'service': 'tickets',
        'sensor': 'not_resolved',
        'labels': {'queue': 'SUPBL'},
    }

    if days_ago is None:
        tickets_count = get_opened_ticket_count(oauth_token)
        solomon_client.push_single_sensor(**dict(solomon_sensor_template, value=tickets_count))
    else:
        for day_ago in range(1, days_ago + 1):
            tickets_count = get_opened_ticket_count(oauth_token, day_ago)

            ts_datetime = datetime.date.today() - datetime.timedelta(day_ago - 1)
            ts_datetime = datetime.datetime.combine(ts_datetime, datetime.time(0, 0))
            ts_datetime += datetime.timedelta(0, time.timezone)

            solomon_client.push_single_sensor(**dict(
                solomon_sensor_template,
                value=tickets_count,
                ts_datetime=ts_datetime.isoformat()
            ))


if __name__ == '__main__':
    main()
