#!/usr/bin/perl -w

use strict;

use utf8;
use open ':utf8';
no warnings 'utf8';
binmode(STDIN,  ":utf8");
binmode(STDOUT, ":utf8");
binmode(STDERR, ":utf8");

use IO::Socket;
use IO::Handle;

use FindBin;
use lib "$FindBin::Bin/../lib";
use Project;
use Utils::Common;
use Utils::Sys;
use File::Copy qw/ /;
use Getopt::Long;

Utils::Sys::handle_errors();

my $prepare_comptrie_file = '';
my $upload_to_sandbox = '';
GetOptions(
    'prepare-comptrie-file' => \$prepare_comptrie_file,
    'upload-to-sandbox'     => \$upload_to_sandbox,
) or die($!);

# на некоторых машинах не разворачиваем атомы
my $role = Utils::Hosts::get_host_role(); 
my $dont_expand_atoms = ($role eq "catmedia" || $role eq "catalogia-media-gen" || $role eq "catalogia-media-front");

my $proj = Project->new({
    load_dicts => 1,
    load_minicategs => 1,
    load_languages => ["tr", "en"],
    dont_expand_atoms => $dont_expand_atoms
});

my $temp_dir = $Utils::Common::options->{'dirs'}{'temp'};
my $temp_file = Utils::Sys::get_tempfile('subphraser_init_data_temp', DIR => $temp_dir, UNLINK => 1);
open F, "> $temp_file" or die("can't open $temp_file for writing");
for my $lang ($proj->get_languages_list) {
    for my $layer ($lang->get_layers_list) {
        $proj->log("writing phrases from ".$layer->name);

        # свойства
        for my $prop (grep{$layer->{$_}} qw(dont_sort max_words_dist)) {
            print F join("\t", "propset", $layer->name, $prop, $layer->{$prop})."\n"; 
        }

        # фразы
        print F join("\t", "add", $layer->name, $_, $layer->get_text_data($_))."\n" for $layer->get_phrases;

        # словарные формы фраз
        for my $categ (sort keys %{$layer->orighash}) {
            my %raw;

            for my $snorm (sort keys %{$layer->orighash->{$categ}}) {
                my $orig = $layer->orighash->{$categ}{$snorm};

                print F join("\t", "setorig", $layer->name, $categ, $snorm, join(",", @$orig)) . "\n";
                $raw{$_}++ for @$orig;
            }

            print F join("\t", "setorig", $layer->name, $categ, "%RAW", join(",", sort keys %raw)) . "\n";
        }
    }
}
close F;

my $fn = $proj->subphrases_client->init_data_file;
File::Copy::move($temp_file, $fn) or die($!);

if ($prepare_comptrie_file) {
    my $subphrases_client = $proj->subphrases_client;
    Utils::Sys::make_comptrie_subphraser(
        input => $fn,
        output => $subphrases_client->comptrie_file,
        temp_dir => $subphrases_client->temp_dir,
    );
    my $ya = $Utils::Common::options->{ya_path};
    
    if ($upload_to_sandbox) {
        my $sandbox_params = $Utils::Common::options->{'sandbox_params'};
        my $sandbox_token = $proj->sandbox_client->get_token();
        my $command = join(' ',
            "$ya upload",
            $subphrases_client->comptrie_file,
            '--type=' . $subphrases_client->sandbox_resource,
            "--user=$sandbox_params->{'user'}",
            "--token='$sandbox_token'",
            "--owner=$sandbox_params->{'owner'}",
            '--do-not-remove',
        );
        my $tries = 5;
        my $sleep_between_tries = 120;
        Utils::Sys::do_safely(
            sub { Utils::Sys::do_sys_cmd($command); },
            tries               => $tries,
            sleep_between_tries => $sleep_between_tries,
        );
    }
}

$proj->log("all done!");
