# -*- coding: utf-8 -*-

# ! Обратите внимание, что запуск обновления Service-тикетов происходит в нашей плейбуке (IRT/ansible-playbooks).
# В случае правок имени, локации скрипта, его команд и/или аргументов пожалуйста, не забудьте сделать соответствующие
# правки и в плейбуке. Подробности можно почитать на страничке IRT в Вики, а также уточнить у fawkes@ или modest@ .

import argparse
import os
import shutil
import tempfile

import ticket_parser2.api.v1

import irt.broadmatching.common_options
import irt.logging
import irt.monitoring.solomon.sensors


logger = irt.logging.getLogger(irt.logging.BANNERLAND_PROJECT, __name__)

# Хешик соотвтествия имени доменного окружения и его атрибутов в Паспорте
DOMAIN_TO_TVMATTRS = {
    'yandex.ru': {
        'blackbox_env': ticket_parser2.api.v1.BlackboxEnv.Prod,
        'env_num': {
            'prod': 222,
            'mimino': 239,
        }
    },
    'yandex-team.ru': {
        'blackbox_env': ticket_parser2.api.v1.BlackboxEnv.ProdYateam,
        'env_num': {
            'prod': 223,
        }
    },
}

# Словарик, где именам TVM-приложений для Паспорта ставятся в соответствие их атрибуты для получения Service-тикетов
# Имена приложений в этом словарике  прописаны для удобства, как атрибут, передаваемый при запуске настоящего скрипта
# То есть эти имена (ключи словарика) не являются фактичпескими атрибутами TVM-приложений
TVM_APP_ATRS = {
    'bmfront': {
        'app_id': 2010542,
        'auth_domains': ['yandex-team.ru'],
        'env_mode': 'prod',
    },
    'catmedia': {
        'app_id': 2012276,
        'auth_domains': ['yandex.ru'],
        'env_mode': 'prod',
    },
    'catmedia-dev': {
        'app_id': 2012476,
        'auth_domains': ['yandex.ru'],
        'env_mode': 'mimino',
    },
}


# Обновляет Service-тикет для выбранного доменного окружения
def update_service_ticket(auth_domain, app_id, app_secret, env_mode, tvm_folder):

    tvm_client = ticket_parser2.api.v1.TvmClient(ticket_parser2.api.v1.TvmApiClientSettings(
        self_client_id=app_id,
        enable_service_ticket_checking=True,
        enable_user_ticket_checking=DOMAIN_TO_TVMATTRS[auth_domain]['blackbox_env'],
        self_secret=app_secret,
        dsts={'dst': DOMAIN_TO_TVMATTRS[auth_domain]['env_num'][env_mode]},
    ))

    if tvm_client.status != ticket_parser2.api.v1.TvmClientStatus.Ok:
        error_message = 'Failed with getting TVM-client for domain: {}'.format(auth_domain)
        logger.error(error_message)
        raise RuntimeError(error_message)

    service_ticket = tvm_client.get_service_ticket_for('dst')

    fd, temp_path = tempfile.mkstemp()
    os.write(fd, service_ticket)
    os.close(fd)
    os.chmod(temp_path, 0644)
    shutil.move(temp_path, os.path.join(tvm_folder, '{}_service_ticket'.format(auth_domain.replace('.', '_'))))

    logger.info('Successfull update Service-Ticket for: %s', auth_domain)


def main():
    # Принимаем из аргументной строки имя TVM-приложения, атрибуты которого мы хотим извлечь
    arg_parser = argparse.ArgumentParser()
    arg_parser.add_argument('-n', '--tvm_name', required=True)
    args = arg_parser.parse_args()
    tvm_name = args.tvm_name

    # Проверяем, существует ли такое TVM-приложение и получаем его ID
    if tvm_name not in TVM_APP_ATRS:
        error_message = 'Unknown TVM-app name: {}. Expecting one of: {}'.format(tvm_name, TVM_APP_ATRS.keys())
        logger.error(error_message)
        raise RuntimeError(error_message)
    app_id = TVM_APP_ATRS[tvm_name]['app_id']
    env_mode = TVM_APP_ATRS[tvm_name]['env_mode']

    # Получаем папку с секретами/тикетами TVM. Если такая папка отсутствует, создаём её
    try:
        tvm_folder = os.path.join(irt.broadmatching.common_options.get_options()['secrets']['server_dir'], 'tvm')
        if not os.path.isdir(tvm_folder):
            os.makedirs(tvm_folder)
    except Exception as exception:
        logger.exception('Failed with getting folder of TVM secrets with exception: %s', exception)
        raise

    # Проверяем существование ID и секрета заданного TVM-приложения
    app_secret_file = os.path.join(tvm_folder, 'tvm_{}_secret'.format(tvm_name))
    if not os.path.isfile(app_secret_file):
        error_message = 'File `{}` doesn\'t exist.'.format(app_secret_file)
        logger.error(error_message)
        raise RuntimeError(error_message)

    # Считываем секрет TVM-приложения
    with open(app_secret_file, 'r') as token_file:
        app_secret = token_file.read().strip()

    # Обновляем Service-тикеты для заданного приложения и соответствующих ему доменных окружений для авторизации
    success_count = 0
    for auth_domain in TVM_APP_ATRS[tvm_name]['auth_domains']:
        try:
            update_service_ticket(auth_domain, app_id, app_secret, env_mode, tvm_folder)
            success_count += 1
        except Exception as exception:
            logger.exception('Failed to update Service-Ticket for %s: %s', auth_domain, exception)

    # Проверяем, что все тикеты успешно обновились - добавляем строчку в логе
    if success_count == len(TVM_APP_ATRS[tvm_name]['auth_domains']):
        logger.info('ALL_SERVICE_TICKETS_OK')
        solomon_client = irt.monitoring.solomon.sensors.SolomonAgentSensorsClient()
        solomon_client.set_success_script_finish('update_service_tickets')


if __name__ == '__main__':
    main()
