#!/usr/bin/perl
use strict;

use utf8;
use open ":utf8";

use FindBin;
use lib "$FindBin::Bin/lib";
use Utils::Common;
use Utils::Words;
use Utils::Sys qw/
    get_file_lock
    release_file_lock
    handle_errors
    dir_files
    modtime
    md5int
    load_json
/;
use Project;
use BM::Phrase;

use File::Temp qw(tempdir);
use List::Util qw(minstr);
use Time::Local qw(timegm);
use POSIX qw(strftime);
use DBI;
use Data::Dumper;
use JSON;

handle_errors();
get_file_lock() 
    or die "update_catmedia_db.pl is already runned!";

my @used_languages = qw(ru en tr);
my $proj = Project->new({
    load_dicts      => 1,
    load_minicategs => 1,
    load_languages  => \@used_languages
});
$proj->catalogia_media_dbh->do_connect;
my $dbh = $proj->catalogia_media_dbh->{dbh};

# /kostyl options
CreatePrefilterListTable('PrefilterList');

my @insert = ();
my $counter = 1;
$proj->log("1. NewPrefilterList...");
dbh_do("drop table if exists NewPrefilterList");
CreatePrefilterListTable('NewPrefilterList');
disable_keys("NewPrefilterList");

# inserts into PrefilterList table
open (my $prefilters_fh, '<', $Utils::Common::options->{dict_prefilters}) or die($!);
while (my $line = <$prefilters_fh>) {
    chomp($line);
    push @insert, '(' . join(',', (map {$dbh->quote($_)} ($line =~ /([^\t]+)\t([^\t]+)\t(.*)/))) . ')';
}
close $prefilters_fh or die($!);
dbh_do('INSERT INTO NewPrefilterList (Language, Pattern, ReplacingPattern) VALUES ' . join(',', @insert));
# /inserts into PrefilterList table

enable_keys("NewPrefilterList");
dbh_do("alter table NewPrefilterList disable keys");

create_category_comments_table("CategoryComments");

#
dbh_do('drop table if exists NewCatalogiaVirtualBrief');
create_virtual_catalogia_table('NewCatalogiaVirtualBrief');
create_virtual_catalogia_table('CatalogiaVirtualBrief');

dbh_do('drop table if exists NewCatalogiaVirtualDict');
create_virtual_catalogia_translation_table('NewCatalogiaVirtualDict');
create_virtual_catalogia_translation_table('CatalogiaVirtualDict');

CreateHiddenPhrasesTable('HiddenPhrases');
CreateUserListTable("Users");
create_banners_categs_diff_table("BannersCategoriesDiff");
create_test_banners("TestBanners");
CreatePhraseListCache("PhraseListCache");
dbh_do("drop table if exists DataCache");
create_data_cache_table("DataCache");

$proj->log('update_catmedia_db ...');

# подготовка таблиц

dbh_do("drop table if exists NewSiblings");
create_siblings_table("NewSiblings");
create_siblings_table("Siblings");

dbh_do("drop table if exists NewSynCells");
create_syn_cells_table("NewSynCells");
create_syn_cells_table("SynCells");

dbh_do("drop table if exists NewCatalogiaDict");
create_catalogia_translation_table("NewCatalogiaDict");
create_catalogia_translation_table("CatalogiaDict");

dbh_do("drop table if exists NewCatalogiaBrief");
create_catalogia_table("NewCatalogiaBrief");
create_catalogia_table("CatalogiaBrief");

dbh_do("drop table if exists NewCatalogiaFlags");
create_flags_table("NewCatalogiaFlags");
create_flags_table("CatalogiaFlags");

dbh_do("drop table if exists NewCatalogiaFlagsDescription");
create_flags_description_table("NewCatalogiaFlagsDescription");
create_flags_description_table("CatalogiaFlagsDescription");

CreateCatPhrases('CatalogiaPhrases');
CreateModerateTable('CatalogiaModerate');

# Virtual categories
$proj->log('virtual categories');
$proj->log("2. NewCatalogiaVirtualBrief...");
disable_keys('NewCatalogiaVirtualBrief');
my @virtual_categ_list = $proj->categs_tree->get_virtual_categs_list;
@insert = ();
foreach (@virtual_categ_list) {
    push @insert, "(" . 
            join(
                ',',
                map {
                    $dbh->quote($_)
                } ($proj->categs_tree->get_virtual_categ_id($_), join(',', $proj->categs_tree->get_virtual_categ_flags($_)))
            ) . ")";
    if ( $#insert > 5_000 ) {
        dbh_do('insert into NewCatalogiaVirtualBrief (CatID, Flags) values ' . join(',', @insert));
	@insert = ();
    }
}
if ( @insert ) {
    dbh_do('insert into NewCatalogiaVirtualBrief (CatID, Flags) values ' . join(',', @insert));
}
enable_keys('NewCatalogiaVirtualBrief');

dbh_do('drop table if exists DoneCatalogiaVirtualBrief');
dbh_do('rename table CatalogiaVirtualBrief to DoneCatalogiaVirtualBrief, NewCatalogiaVirtualBrief to CatalogiaVirtualBrief');

@insert = ();
$counter = 1;
disable_keys('NewCatalogiaVirtualDict');
$proj->log("3. NewCatalogiaVirtualDict...");
foreach my $current_language ($proj->get_languages_list) {
    foreach (@virtual_categ_list) {
        my @phrase_list = $current_language->get_virtual_phrases($_);
        push @insert, "(" . 
                join (
                    ',',
                    $counter++,
                    map {
                        $dbh->quote($_)
                    } ( $proj->categs_tree->get_virtual_categ_id($_),
                        $current_language->name,
                        $current_language->virtual_from_ru($_),
                        join(',', @phrase_list))
                ) . ")";
    }
    if ( $#insert > 5_000 ) {
        dbh_do('insert into NewCatalogiaVirtualDict (ID, CatID, Language, CategoryName, CategoryPhrases) values ' . join(',', @insert));
        @insert = ();

    }
}

if ( @insert ) {
    dbh_do('insert into NewCatalogiaVirtualDict (ID, CatID, Language, CategoryName, CategoryPhrases) values ' . join(',', @insert));
}
enable_keys('NewCatalogiaVirtualDict');

dbh_do('drop table if exists DoneCatalogiaVirtualDict');
dbh_do('rename table CatalogiaVirtualDict to DoneCatalogiaVirtualDict, NewCatalogiaVirtualDict to CatalogiaVirtualDict');

# сиблинги
@insert = ();
$proj->log("siblings");
my $siblings = $proj->categs_tree->get_siblings_hash;
disable_keys('NewSiblings');
$proj->log("4. NewSiblings...");
for my $name (keys %$siblings) {
    my $id = md5int($name);

    push @insert, "(" . join(",", map{$dbh->quote($_)} ($id, $name, "ru", join("/", sort @{$siblings->{$name}}))) . ")";
    
    if ( $#insert > 5_000 ) {
        dbh_do("insert into NewSiblings (GroupID, GroupName, Language, Categories) values " . join(',', @insert));
        @insert = ();
    }
}

if ( @insert ) { 
    dbh_do("insert into NewSiblings (GroupID, GroupName, Language, Categories) values " . join(',', @insert));
    @insert = ();
}

enable_keys('NewSiblings');
dbh_do("drop table if exists DoneSiblings");
dbh_do("rename table Siblings to DoneSiblings, NewSiblings to Siblings");
$proj->log("/ siblings");

# categories text fields
$proj->log('updating CategoriesTextFields');
create_categories_text_fields_table('CategoriesTextFields');
dbh_do('drop table if exists NewCategoriesTextFields');
create_categories_text_fields_table('NewCategoriesTextFields');
fill_categories_text_fields_table('NewCategoriesTextFields');
$proj->log('/ updating CategoriesTextFields');

# синонимы
$proj->log("fill syn_cells");
@insert = ();
disable_keys('NewSynCells');
$proj->log("6. NewSynCells...");
for my $lang (keys %Utils::Words::mainsynlang) {
    my $syns = $Utils::Words::mainsynlang{$lang};
    my $clusters = {};

    my $norm = $proj->generated_dicts->{norm};
    my $info_file = $norm->{info}{"syn_$lang"};
    if(!$info_file) {
        $proj->log("no syns for $lang");
        next;
    }
    if(!-f $info_file) {
        $proj->log("ERROR: $info_file doesn't exist");
        next;
    }
    
    my $data = load_json($info_file);
    my $index = (keys %$data > 2) ? $data : $data->{neighbour};    # костыль для перехода на новый формат файла $info_file    TODO remove this and use         my $index = $data->{neighbour};    

    for my $w (keys %$syns) {
        push @{$clusters->{$syns->{$w}} ||= []}, $w;
    }
    
    for my $w (keys %$clusters) {
        my $words = $clusters->{$w};
        next if @$words < 2;

        # список пар, порождающих кластер
        my @pairs;
        for my $w1 (sort @$words) {
            for my $w2 (sort grep{$w1 lt $_} keys %{$index->{$w1}}) {
                push @pairs, "$w1:$w2";
            }
        }

        push @insert, "(".join(",", map{$dbh->quote($_)} (md5int($w), $w, $lang, join(",", @$words), join(",", @pairs))).")";
        if ( $#insert > 5_000 ) {
            dbh_do("insert into NewSynCells (MainSynID, MainSyn, Language, Syns, Pairs) values " . join(',', @insert));
            @insert = ();
        };
    }
}

if ( @insert ) {
   dbh_do("insert into NewSynCells (MainSynID, MainSyn, Language, Syns, Pairs) values " . join(',', @insert));
   @insert = ();
}

enable_keys('NewSynCells');
$proj->log("/ fill syn_cells");

# заполнение таблицы
my %all_flags;
my ( @insert1, @insert2, @insert3 ) = ( (), (), () );
disable_keys('NewCatalogiaBrief');
disable_keys('NewCatalogiaDict');
disable_keys('NewCatalogiaFlags');
$proj->log("7.,8.,9.. NewCatalogiaBrief, NewCatalogiaDict, NewCatalogiaFlags...");

for my $categ ($proj->categs_tree->get_minicategs_list()) {
    next if $categ =~ /^[\!]/;

    my $id          = $proj->categs_tree->get_minicateg_id($categ);
    my $parent      = $proj->categs_tree->get_minicateg_parent($categ);
    my $parent_id   = $proj->categs_tree->get_minicateg_id($parent) || "0";
    my $is_atom     = $categ =~ /^\./;
    my @flags       = keys %{$proj->categs_tree->get_minicateg_flags($categ)};
   
    push @insert1, "(".join(", ", map{$dbh->quote($_)} ($id, $parent_id, join(",", @flags), $is_atom)).")";
    if ( $#insert1 > 5_000 ) {
        dbh_do("insert into NewCatalogiaBrief (CatID, ParentID, Flags, IsAtom) values " . join(',', @insert1));
        @insert1 = ();
    }
   
    for my $lang (@used_languages) {
        my $phrases     = join ",", $proj->get_language($lang)->get_category_raw_phrases($categ);
        my $minus       = join ",", $proj->get_language($lang)->get_minicateg_antiwords($categ);

        push @insert2, "(".join(", ", map{$dbh->quote($_)} ($id, $proj->get_language($lang)->category_from_ru($categ), $phrases, $minus, $lang)).")";
        if ( $#insert2 > 5_000 ) {
            dbh_do("insert into NewCatalogiaDict (CatID, CategoryName, CategoryPhrases, MinusWords, Language) values " . join(',', @insert2));
            @insert2 = ();
        }
    }

    for my $flag (@flags) {
        $all_flags{$flag}++;

        push @insert3, "(" . join(",", map{$dbh->quote($_)} ($id, $flag)).")";
        if ( $#insert3 > 5_000 ) {
            dbh_do("insert into NewCatalogiaFlags (CatID, Flag) values " . join(',', @insert3));
            @insert3 = ();
        }
    }
}

if ( @insert1 ) {
    dbh_do("insert into NewCatalogiaBrief (CatID, ParentID, Flags, IsAtom) values " . join(',', @insert1));
}
if ( @insert2 ) {
    dbh_do("insert into NewCatalogiaDict (CatID, CategoryName, CategoryPhrases, MinusWords, Language) values " . join(',', @insert2));
}
if ( @insert3 ) {
    dbh_do("insert into NewCatalogiaFlags (CatID, Flag) values " . join(',', @insert3));
}

enable_keys('NewCatalogiaBrief');
enable_keys('NewCatalogiaDict');
enable_keys('NewCatalogiaFlags');

my @flags = $proj->categs_tree->get_flags_list();
@insert = ();
disable_keys('NewCatalogiaFlagsDescription');
$proj->log("10. NewCatalogiaFlagsDescription...");
for my $flag (@flags) {
    push @insert, "(".join(",", map{$dbh->quote($_)} ($flag, $proj->categs_tree->get_flag_description($flag))).")";
    if ( $#insert > 5_000 ) {
        dbh_do("insert into NewCatalogiaFlagsDescription (Flag, Description) values " . join(',', @insert));
        @insert = ();
    }
}

if ( @insert ) {
    dbh_do("insert into NewCatalogiaFlagsDescription (Flag, Description) values " . join(',', @insert));
}
enable_keys('NewCatalogiaFlagsDescription');

# обновление таблицы
dbh_do("drop table if exists DonePrefilterList");
dbh_do("rename table PrefilterList to DonePrefilterList, NewPrefilterList to PrefilterList");

dbh_do("drop table if exists DoneSynCells");
dbh_do("rename table SynCells to DoneSynCells, NewSynCells to SynCells");

dbh_do("drop table if exists DoneCatalogiaDict");
dbh_do("rename table CatalogiaDict to DoneCatalogiaDict, NewCatalogiaDict to CatalogiaDict");

dbh_do("drop table if exists DoneCatalogiaBrief");
dbh_do("rename table CatalogiaBrief to DoneCatalogiaBrief, NewCatalogiaBrief to CatalogiaBrief");

dbh_do("drop table if exists CatalogiaFlags");
dbh_do("rename table NewCatalogiaFlags to CatalogiaFlags");

dbh_do("drop table if exists CatalogiaFlagsDescription");
dbh_do("rename table NewCatalogiaFlagsDescription to CatalogiaFlagsDescription");

dbh_do('drop table if exists DoneCategoriesTextFields');
dbh_do('rename table CategoriesTextFields to DoneCategoriesTextFields');
dbh_do('rename table NewCategoriesTextFields to CategoriesTextFields');

$dbh->disconnect;
release_file_lock();
$proj->log('update_catmedia_db done!');
exit(0);

sub dbh_do {
    my ($cmd) = @_;

    $dbh->do($cmd) or die("ERROR: query = '$cmd', errstr = '$DBI::errstr'")
}

sub create_category_comments_table {
    my ($table) = @_;

    dbh_do "create table if not exists `$table`(
        `ID` int not null auto_increment,
        `CatID` char(20) NOT NULL DEFAULT '0',
        `Login`   char(255) not null default '',
        `Comment` text not null,
        `UpdateTime` timestamp not null default current_timestamp,
        primary key (`ID`)
    ) ENGINE=MyISAM DEFAULT CHARSET=utf8";
}

sub create_banners_categs_diff_table {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `ID` bigint(20) unsigned NOT NULL auto_increment,
                `Login`   char(255) not null default '',
                `UpdateTime` timestamp not null default current_timestamp,
                `Comment` text not null,
                `NumBanners` integer not null default 10000,
                `PhrasesFileName` text not null,
                `Phrases` longblob not null,
                `State` char(20) NOT NULL DEFAULT 'New',
                `Language` char(5) NOT NULL DEFAULT 'ru',
                `Result` longblob,
                `DataType` char(10) NOT NULL DEFAULT 'banner',
                `DiffType` char(10) NOT NULL DEFAULT 'phrases',
                PRIMARY KEY (`ID`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8 ";
}

sub create_siblings_table {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `GroupID` bigint(20) unsigned NOT NULL,
                `GroupName` text NOT NULL,
                `Language` char(5) NOT NULL DEFAULT 'ru',
                `Categories` text NOT NULL,
                PRIMARY KEY (`GroupID`, `Language`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8 ";
}

sub create_syn_cells_table {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `MainSynID` bigint(20) unsigned NOT NULL,
                `MainSyn` char(100) NOT NULL DEFAULT '0',
                `Language` char(5) NOT NULL DEFAULT 'ru',
                `Syns` text NOT NULL,
                `Pairs` text NOT NULL,
                PRIMARY KEY (`MainSynID`, `Language`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8 ";
}

sub create_data_cache_table {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `DataKey` char(40) NOT NULL DEFAULT '0',
                `DataValue` longblob NOT NULL,
                `UpdateTime` timestamp not null default current_timestamp,
                PRIMARY KEY (`DataKey`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8 ";
}

sub create_catalogia_table {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `CatID` char(20) NOT NULL DEFAULT '0',
                `ParentID` char(20) NOT NULL DEFAULT '0',
                `Flags` text NOT NULL,
                `IsAtom` bool NOT NULL,
                PRIMARY KEY (`CatID`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8 ";
}

sub create_catalogia_translation_table {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `CatID` char(20) NOT NULL DEFAULT '0',
                `CategoryName` char(255) NOT NULL DEFAULT '',
                `CategoryPhrases` mediumtext NOT NULL,
	            `Language`   char(2) NOT NULL DEFAULT 'ru',
                `MinusWords` text NOT NULL,
                `PhraseListID` char(20) DEFAULT NULL, 
                PRIMARY KEY (`CatID`, `Language`),
                KEY `Language_CategoryName` (`Language`, `CategoryName`),
                KEY `Language_CatID` (`Language`, `CatID`),
                KEY `CategoryName` (`CategoryName`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8 ";
}

sub create_virtual_catalogia_table {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `CatID` char(20) NOT NULL DEFAULT '0',
                `Flags` text NOT NULL,
                PRIMARY KEY (`CatID`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8 ";
}

sub create_virtual_catalogia_translation_table {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `ID` int(10) NOT NULL AUTO_INCREMENT,
                `CatID` char(20) NOT NULL DEFAULT '0',
                `Language` char(2) NOT NULL DEFAULT 'ru',
                `CategoryName` char(255) NOT NULL DEFAULT '',
                `CategoryPhrases` mediumtext NOT NULL,
                PRIMARY KEY (`ID`),
                UNIQUE KEY (`CatID`, `Language`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8";
}

sub CreatePhraseListCache {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `PhraseListID` char(20) NOT NULL,
                `Phrases` text NOT NULL,
                `UpdateTime` timestamp not null default current_timestamp,
                `AccessTime` timestamp not null default current_timestamp,
                `IsComposite` tinyint(4) NOT NULL DEFAULT '0',
                PRIMARY KEY(`PhraseListID`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8 ";
}

sub create_flags_table {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `CatID` char(20) NOT NULL,
                `Flag` char(50) NOT NULL,
                PRIMARY KEY(`CatID`, `Flag`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8 ";
}

sub create_flags_description_table {
    my ($table) = @_;

    dbh_do "CREATE TABLE if not exists `$table` (
                `Flag` char(50) NOT NULL,
                `Description` text NOT NULL,
                PRIMARY KEY(`Flag`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8 ";
}

sub CreateModerateTable {
	my ($table ) = @_;
	dbh_do("CREATE TABLE if not exists `$table` (
	  `LogID`   bigint(20) not null auto_increment,
	  `CatID`   char(20) NOT NULL DEFAULT '',
	  `Command` char(50) NOT NULL DEFAULT '',
	  `Value`   char(255) not null default '',
	  `Login`   char(255) not null default '',
          `UpdateTime` timestamp not null default current_timestamp,
	  `ModerationStatus` char(255) not null default '',
	  `Comment` char(255) not null default '',
	  primary key (LogID)
	  ) ENGINE=MyISAM DEFAULT CHARSET=utf8
	  ");
}

sub create_test_banners {
    my ($table) = @_;
	dbh_do("CREATE TABLE if not exists `$table` (
      `SetID` bigint(20) unsigned NOT NULL,
      `BannerID` int(10) unsigned NOT NULL,
      `Title` text NOT NULL,
      `Body` text NOT NULL,
      `Phrases` text NOT NULL,
      `Mode` char(255) NOT NULL DEFAULT '',
      `ManualCategory` text NOT NULL,
      `InitialCategories` text NOT NULL, 
      `PhrasesCategories` text NOT NULL, 
      `Lang` char(255) NOT NULL DEFAULT '',
	  primary key (SetID, BannerID, Lang)
	  ) ENGINE=MyISAM DEFAULT CHARSET=utf8
    ");
}

sub CreateCatPhrases {
	my ($table ) = @_;
	dbh_do("CREATE TABLE if not exists `$table` (
          `CatID`   char(50) NOT NULL DEFAULT '',
          `Language`   char(2) NOT NULL DEFAULT 'ru',
          `InitialPhraseID` bigint(20) unsigned NOT NULL DEFAULT '0',
          `InitialPhrase` text not null,
          `CreateTime` datetime DEFAULT NULL,
          `Login`   char(255) not null,
          `LastLogin`   char(255) not null,
          `Status` char(255) NOT NULL DEFAULT 'New',
          `PreStatus` char(255) NOT NULL DEFAULT 'New',
          `LastMedia` char(255) NOT NULL,
          `MediaStatus` char(255) NOT NULL DEFAULT 'New',
          `MediaPreStatus` char(255) NOT NULL DEFAULT 'New',
          `MediaComment` text NOT NULL,
          `Comment` text NOT NULL,
          `UserComment` text,
          `Action` char(20) NOT NULL DEFAULT 'Add',
          `UpdateTime` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          `Importance` int(11),
          PRIMARY KEY (`CatID`,`InitialPhraseID`,`Action`,`Language`),
          KEY `InitialPhraseID` (`InitialPhraseID`),
          KEY `Status` (`Status`,`CatID`,`InitialPhraseID`,`Language`,`Action`),
          KEY `Action` (`Action`),
          KEY `Login` (`Login`),
          KEY `LastLogin` (`LastLogin`),
          KEY `CatID` (`CatID`),
          KEY `UpdateTime` (`UpdateTime`),
          KEY `CreateTime` (`CreateTime`)
      ) ENGINE=MyISAM DEFAULT CHARSET=utf8
	");
}

sub CreateUserListTable {
    my ($table) = @_;
    dbh_do("CREATE TABLE if not exists `$table` (
            `uid` int(10) DEFAULT '0',
            `Login` char(255) NOT NULL DEFAULT '',
            `NumVisits` int(11) NOT NULL DEFAULT '0',
            `LastVisit` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `state` char(255) NOT NULL DEFAULT '',
            `role` char(255) NOT NULL DEFAULT '',
            `DirectCampaignsJSON` longtext,
            `DirectCampaignsDate` datetime DEFAULT NULL,
            `EMail` varchar(255) DEFAULT '',
            PRIMARY KEY (`Login`))");
}

sub CreateHiddenPhrasesTable {
    my ($table) = @_;
    dbh_do("CREATE TABLE if not exists `$table` (
            `CategoryId` varchar(20) NOT NULL,
            `Language` varchar(100) NOT NULL,
            `Phrase` varchar(100) NOT NULL,
            PRIMARY KEY (`CategoryId`, `Language`, `Phrase`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8");
}

sub CreatePrefilterListTable {
    my ($table) = @_;
    dbh_do("CREATE TABLE if not exists `$table` (
            `ID` int(11) NOT NULL AUTO_INCREMENT,
            `Language`          varchar(20) NOT NULL,
            `Pattern`           text,
            `ReplacingPattern`  varchar(100),
            PRIMARY KEY (`ID`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8 ");
    
}

sub create_categories_text_fields_table {
    my $table = shift;
    dbh_do("
        CREATE TABLE if not exists `$table` (
        `ID` int(10) unsigned NOT NULL AUTO_INCREMENT,
        `CatID` char(50) NOT NULL DEFAULT '',
        `Language` char(2) NOT NULL DEFAULT 'ru',
        `KeyID` bigint(20) unsigned NOT NULL DEFAULT 0,
        `Key` text NULL DEFAULT '',
        `Value` text NOT NULL DEFAULT '',
        `UpdateTime` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP on update CURRENT_TIMESTAMP,
        PRIMARY KEY (`ID`),
        UNIQUE KEY (`CatID`, `Language`, `KeyID`)
        ) ENGINE=MyISAM DEFAULT CHARSET=utf8");
}

sub fill_categories_text_fields_table {
    my $table = shift;

    my $dict_file = $Utils::Common::options->{dirs}{dicts} . '/dict_categories_text_fields';
    my @values = ();
    open (my $filehandle, '<', $dict_file) or die($!);
    while (my $json_text = <$filehandle>) {
        chomp($json_text);
        push @values, JSON::from_json($json_text);
    }
    close $filehandle;
    my @insert = ();
    for my $value (@values) {
        my $key_id = Utils::Sys::md5int($value->{Key});;
        push @insert, '(' . join(',', (map {$dbh->quote($value->{$_})} qw/CatID Language Key Value UpdateTime/), $key_id) . ')';
    }
    dbh_do("insert into $table (CatID, Language, `Key`, Value, UpdateTime, KeyID) values " . join(',', @insert)) if @insert;
}

sub disable_keys {
    my ( $table_name ) = @_;
    dbh_do("alter table $table_name disable keys");
}

sub enable_keys {
    my ( $table_name ) = @_;
    dbh_do("alter table $table_name enable keys");
}
