#!/usr/bin/perl -w
use strict;

use FindBin;
use lib "$FindBin::Bin/lib";
use File::Slurp qw(read_file);
use Getopt::Long;
use HTTP::Request;
use IronProject;
use JSON qw(to_json from_json);
use LWP::UserAgent;
use Utils::Common;
use Utils::Sys qw(print_err handle_errors read_sys_cmd);

Utils::Sys::handle_errors();


my $MAX_WAIT = 60 * 60 * 2;

my %opt = (
    "resource_type" => "BROADMATCH_MR_CATALOGIA_RAW",
    "project" => 'bm',
);

sub parse_arguments {
    GetOptions(
        \%opt,
        'resource_type|resource-type=s',
        'resource_attrs|resource-attr=s%',
        'project=s',
        'application=s',
        'environment=s',
        'component=s@',
        'fetch',
        'help',
    ) or die "$!";

    my $help_message = join("\n",
        "\t--help: print this text",
        "\t--resource-type: type of resource to fetch from sandbox. (default BROADMATCH_MR_CATALOGIA_RAW)",
        "\t--resource-attr: key=valye that will be passed to sandbox to filter resources",
        "\t--project: name of project in qloud",
        "\t--application: name of application in qloud",
        "\t--environment: name of environment in qloud",
        "\t--component: name of qloud app component to update. May be specified multiple times. (default 'backend')",
        "\t--fetch: fetch qloud environment before updating resources",
    );
    if ($opt{help}) {
        print "$help_message\n";
        exit 0;
    }

    $opt{component} //= ['backend'];
    $opt{'resource_attrs'} //= {};
    die "Please supply --application" unless $opt{application};
    die "Please supply --environment" unless $opt{environment};
}

sub fetch_resource {
    # Fetch fresh resource from sandbox.
    # Return the latest one found
    my ($proj, $resource_type, $resource_attrs) = @_;
    my $resources = $proj->sandbox_client->get_resource_info(
        type => $resource_type,
        limit => 1,
        attrs => to_json($resource_attrs),
        any_attr => 'false', # require all attributes to match
    )->{'items'};

    unless (@$resources) {
        die "ERROR: Got no resources of type: $resource_type";
    }
    return $resources->[0];
}

sub fetch_current_environment {
    # Fetch current environment from qloud and store it in environments-json. This will overwrite any changes in svn.
    my ($bm_path, $env_name) = @_;
    my $env_path = "$bm_path/scripts/qloud/environments-json/$env_name.json";
    my $qloud_url = "https://qloud-ext.yandex-team.ru/api/v1/environment/dump/$env_name";
    my $ua = LWP::UserAgent->new;
    my $token = get_qloud_token();
    my $res = $ua->get($qloud_url, "Authorization" => "OAuth $token");

    unless ($res->is_success) {
        die "Could not get $env_name from qloud: " . $res->status_line;
    }

    my $env_obj = from_json($res->decoded_content);
    _dump_env($env_path, $env_obj);
    print_err("Fetched current environment for $env_name to $env_path");
}

sub update_qloud_environment {
    # Update $components in $env_name environment. Fails if environment does not contain indicated components. Otherwise
    # updates resource ids and comment. Writes result to environments-json/ dir.
    my ($bm_path, $env_name, $resource, $components) = @_;

    my $env_path = "$bm_path/scripts/qloud/environments-json/$env_name.json";
    my $env_obj = from_json(read_file($env_path));

    # filter correct components
    my %component_names = map { $_ => 1 } @$components;
    my @found = grep {$component_names{$_->{componentName}}} @{$env_obj->{components}};
    unless (scalar @found == scalar keys %component_names) {
        die "Requested to update " . join(',', keys %component_names) . " but env only contains " . join(',', map {$_->{componentName}} @{$env_obj->{components}});
    }

    my $comment = '';
    my $has_changes = 0;
    for my $component (@found) {
        # В ql_resources останутся те записи из sandboxResources у которых localName совпадет с file_name в sandbox
        my @ql_resources = grep {$_->{localName} eq $resource->{file_name}} @{$component->{sandboxResources}};
        unless (@ql_resources) {
            die "Component '$component->{componentName}' doesn't contain resource '$resource->{file_name}'";
        }
        if (scalar @ql_resources > 1) {
            die "Component '$component->{componentName}' has multiple resources $resource->{file_name}. Don't know which one to update.";
        }
        my $ql_resource = $ql_resources[0];
        # если id совпали — не надо обновлять ни ресурс ни комментарий. А то будет лишний коммит и лишняя версия env'а
        unless ($ql_resource->{id} eq $resource->{id}) {
            $comment .= "Updating resource $resource->{file_name} from $ql_resource->{id} to $resource->{id} in $component->{componentName} ";
            $ql_resource->{id} = int $resource->{id};
            $has_changes = 1;
        }
    }
    unless ($has_changes) {
        print_err("Nothing to update in $env_name");
        return;
    }
    print_err($comment);
    $env_obj->{comment} = $comment if $comment;

    _dump_env($env_path, $env_obj);
    print_err("Wrote changes for $env_name to $env_path");
}

sub _dump_env {
    # pretty print in predictable way.
    my ($file_name, $env_obj) = @_;
    open my $fh, ">", "$file_name" or die "Couldn't open $file_name for writing: $!";
    print $fh to_json($env_obj, {canonical =>1, pretty => 1, utf8 => 1});
    close $fh;
}

sub commit_qloud_environment {
    # commit $env_name.json to svn
    my ($bm_path, $env_name) = @_;

    my $env_path = "scripts/qloud/environments-json/$env_name.json";
    my $svn = get_svn_cmd();
    my $command = join(" && ",
        "set -x",
        "[ -d $bm_path ]",
        "echo 'svn commit qloud env $env_name'",
        "cd $bm_path",
        "$svn commit -m 'Auto update $env_name commit SKIP_CHECK' '$env_path'",
        "echo 'svn commit done'",
    );
    my $out = read_sys_cmd($command) or die "can't run '$command': $!";
    print_err($out);
}

sub push_qloud_environment {
    # POST new environment to qloud. Returns version of the new env if there were changes. Otherwise, returns undef.
    my ($bm_path, $env_name) = @_;
    my $env_path = "$bm_path/scripts/qloud/environments-json/$env_name.json";
    my $url = "https://qloud-ext.yandex-team.ru/api/v1/environment/upload/return-header?skipIfNoChange=true";

    my $json_str = read_file($env_path);

    my $req = HTTP::Request->new(POST => $url);
    $req->content_type('application/json');
    $req->content($json_str);
    my $token = get_qloud_token();
    $req->header("Authorization" => "OAuth $token");

    my $ua = LWP::UserAgent->new;
    my $res = $ua->request($req);

    unless ($res->is_success) {
        die "Could not upload new version of $env_name to qloud '$url': " . $res->status_line;
    }
    print_err("Uploaded $env_name to qloud. Got " . $res->status_line);

    if ($res->code() eq '204') {
        print_err("Nothing changed in $env_name");
        return;
    }
    my $resp = from_json($res->decoded_content);
    print_err("Upoaded $env_name. New version is $resp->{version}");
    return $resp->{version};
}

sub wait_for_qloud_env {
    # Waits for $env_name of $env_version to become DEPLOYED. Waits from $MAX_WAIT seconds,
    # Fails if gets a different version of environment during deployment (e.g. there is a new depoyment ongoing).
    my ($env_name, $env_version) = @_;
    my $qloud_url = "https://qloud-ext.yandex-team.ru/api/v1/environment/stable/$env_name";
    my $ua = LWP::UserAgent->new;
    my $token = get_qloud_token();
    my $start = time();
    while (1) {
        my $res = $ua->get($qloud_url, "Authorization" => "OAuth $token");
        unless ($res->is_success) {
            die "Could not fetch $env_name from qloud '$qloud_url': " . $res->status_line;
        }
        my $resp = from_json($res->decoded_content);
        if ($resp->{version} ne $env_version) {
            die "Was expecting version $env_version, but got $resp->{version}. Aborting";
        }
        print_err("Environment $env_name has status $resp->{status}");
        last if ($resp->{status} eq 'DEPLOYED');
        if (time() - $start > $MAX_WAIT) {
            die "Environment $env_name did not get to DEPLOYED in $MAX_WAIT sec.";
        };
        sleep 30;
    }
}

sub get_svn_cmd {
    my $ya_path = $Utils::Common::options->{ya_path};
    return "$ya_path tool svn </dev/null";
}

sub get_qloud_token {
    my $token_path = "$Utils::Common::options->{dirs}->{secrets}" . "/tokens/qloud_oauth_token";
    my $token = read_file($token_path);
    chomp $token;
    return $token;
}

sub main {
    parse_arguments();

    my $proj = IronProject->new({});
    my $resource = fetch_resource($proj, $opt{'resource_type'}, $opt{'resource_attrs'});
    my $env_name = join(".", $opt{project}, $opt{application}, $opt{environment});

    my $bm_path = $Utils::Common::options->{RedButton_params}->{svnup_prm}->{bm_path};

    if ($opt{fetch}) {
        fetch_current_environment($bm_path, $env_name);
    }
    update_qloud_environment($bm_path, $env_name, $resource, $opt{component});
    commit_qloud_environment($bm_path, $env_name);
    my $env_version = push_qloud_environment($bm_path, $env_name);
    if ($env_version) {
        wait_for_qloud_env($env_name, $env_version);
    }
}

main();
1;
