#!/usr/bin/perl -w
use strict;
use utf8;

use JSON qw(to_json from_json);
use Getopt::Long;

use FindBin;
use lib "$FindBin::Bin/../lib";

use Utils::Sys qw(
    get_file_lock
    load_json handle_errors
    print_err
);
use Project;
use Deploy;


handle_errors();
main();
exit(0);


sub main {
    my %args;
    GetOptions(\%args, 'file=s', 'conf=s', 'help|h');

    if ($args{help} or (!$args{file} and !$args{conf})) {
        printf "Usage: $0 [--file CONFIG_FILE] [--conf CONFIG]\n";
        printf "Broadmatching deploy with config given in file or in command line\n";
        printf "Config file is json-encoded hash, with keys:\n";
        printf "  'command', 'hosts', 'macro', 'paths', ...\n";
        printf "Examples:\n";
        printf "  {'macro': 'lightcat'}  # svn up, subphraser prepare data, suphraser stop on hosts: catalogia01[def]\n\n";
        printf "For debug:\n";
        printf "  DEBUG=1 $0 ...\n";
        printf "  you must provide 'revision' key in config\n";
        exit($args{help} ? 0 : 1);
    }

    my $options;
    if ($args{file}) {
        $options = load_json($args{file}) or die "Can't load config: $!";
    } else {
        $options = from_json($args{conf});
    }
    my $proj = Project->new;
    my $main_pid = $$;

    # Проверяем, не запущен ли деплой с такими же параметрами.
    # Это не достаточная проверка, но защищает от задублированных нажатий
    # (У некоторых пользователей Mac нажатия красной кнопки "дублируются". Есть предположение, что это связано с антивирусом, см. https://st.yandex-team.ru/IRT-1736)
    # Не разрешаем одновременный запуск макроса даже с разными параметрами!
    my @keys4lock = grep { !m/webusername/ and $_ ne 'macro_par' } sort keys %$options;
    my $lock_name = join("_", "deploy", (map {( $_, $options->{$_} )} @keys4lock));
    $lock_name =~ s/[^a-z0-9\-\.]/_/ig;
    print_err("get_file_lock( $lock_name ) ...");
    get_file_lock($lock_name)
        or die "Could not get_file_lock( $lock_name ). Perhaps the same deploy is already run";

    my $options_save = Deploy::run_deploy($proj, $options);

    # Делаем DbSave только если pid совпадает с начальным. Нужно, т.к. могли сделать fork, а в дочернем процессе не дойти до exit из-за die, который перехватывается в do_safely
    if ($$ != $main_pid) {
        return;
    }

    Deploy::DbSave($proj, $options_save);

    print_err("Done!");
}
