#!/usr/bin/perl -w
use strict;

use utf8;
use open ":utf8";

binmode STDIN, ':utf8';
binmode STDOUT, ':utf8';

use Getopt::Long;
use Data::Dumper;

use FindBin;
use lib "$FindBin::Bin/../lib";

use Utils::Sys qw(
    save_json
    load_json
    dir_files
    rsync
    log_msg_fmt
    log_time_fmt
    uniq
    print_log
    print_err
    do_safely
    handle_errors
);
use Utils::Hosts qw(get_hosts get_host_role get_curr_host get_host_info);
use Utils::Funcs qw(expand_names);

use Parallel::ForkManager;

select STDERR; $|++;
select STDOUT; $|++;

handle_errors();
#handle_errors(DIE => {stack_trace => 1});

my %opt = (
);
GetOptions(
    \%opt,
    'help',
    'list',
    'fork=i',
);

if ($opt{help}) {
    print join("\n",
        "Usage: do_remote_cmd.pl HOSTS_DESCRIPTION COMMAND [OPTIONS]",
        "  HOSTS_DESCRIPTION can be:",
        "    Comma-separated list of hosts or strings for expand_names() ('{...}' format in expand_names() is not supported)",
        "    Hash of parameters for get_host()",
        "    Array of hashes of parameters for get_host() (combine by 'OR')",
        "Options:",
        "  --help                   Print this help and exit",
        "  --list                   only print list of hosts",
        "  --fork forks_number      run cmd in parall",

        qq!Examples:!,
        qq!  do_remote_cmd.pl bmfront01f 'df -h /'!,
        qq!  do_remote_cmd.pl 'bmgen02[ih]' 'df -h /'!,
        qq!  do_remote_cmd.pl 'bmgen02[ih],bmfront01f' 'df -h /'!,
        qq!  do_remote_cmd.pl 'bmgen01[gh],bmfront01f' 'df -h /'!,
        qq!  do_remote_cmd.pl '{ role => "bmgen02" }' 'df -h /'!,
        qq!  do_remote_cmd.pl '[ {role => "bmgen02"}, { role => "bmfront" } ]' 'df -h /'!,

    ), "\n";
    exit(0);
}

my $hosts_prm_str = shift @ARGV
    or die "Void hosts_prm_str!";
my $remote_cmd = shift @ARGV;
if (!$opt{list} and !$remote_cmd) {
    die "Void remote_cmd!";
}

my @hosts;

if ($hosts_prm_str =~ m/^[a-z0-9_\-\.\[\],]+$/i) {
    # Строки для expand_names, через запятую
    # TODO accept the "{...}" format for expand_names()
    push @hosts, map { expand_names($_) }  split /,/, $hosts_prm_str;
} else {
    # Параметры для get_hosts или строки для expand_names, через запятую
    my $hosts_prm;
    eval "\$hosts_prm = $hosts_prm_str";
    if ($@ or not $hosts_prm) {
        die "Bad hosts_prm_str (\$hosts_prm = $hosts_prm_str) ($@)";
    }

    if (ref($hosts_prm) eq 'ARRAY'  or  ref($hosts_prm) eq 'HASH') {
        for my $hosts_prm_hash (ref($hosts_prm) eq 'ARRAY'  ?  @$hosts_prm  :  ($hosts_prm)) {
            push @hosts, get_hosts(%$hosts_prm_hash);
        }
        #print Dumper([map {[ $_, get_host_info($_) ]}  @hosts]);
        @hosts = grep { not ((get_host_info($_) || {})->{is_alias}) }  @hosts;
    } else {
        push @hosts, map { expand_names($_) }  split /,/, $hosts_prm;
    }

    @hosts = sort (uniq(@hosts));
}

unless (@hosts) {
    print_err("ERROR: Void hosts list");
    exit(1);
}

print_err("hosts: @hosts");
if ($opt{list}) {
    print $_."\n" for @hosts;
    exit(0);
}
print_err("remote_cmd: ( $remote_cmd )");

my @hosts_with_errors;

my $parall = $opt{fork};
die "number of forks should be > 0, got: $parall" if (defined $parall) and ($parall < 1);

my $pm = new Parallel::ForkManager($parall) if $parall;
$pm->run_on_finish(
    sub { my ($pid, $exit_code, $ident) = @_;
        push @hosts_with_errors, $ident if $exit_code != 0;
    }
) if $parall;

for my $host (@hosts) {
    $pm->start($host) and next if $parall;
    print_err("Host: $host ...");
    my $res = do_safely( sub {
            my $cmd = get_bash_cmd_at_host($host, $remote_cmd);
            print_err("cmd: $cmd");
            open my $fh, "$cmd |"  or do {
                print_err("ERROR: Can not open ($!)");
                return;
            };
            while (<$fh>) {
                print join("\t", "[$host]", log_time_fmt(), $_);
            }
            close $fh or do {
                print_err("ERROR: Can not close ($!)");
                return;
            };
            return 1;
        },
        no_die => 1,
        # timeout => 20,
    ) || do {
        print_err("ERROR at host: $host");
        if ($parall) {
            $pm->finish(1);
        } else {
            push @hosts_with_errors, $host;
            next;
        }
    };
    print_err("Host: $host done");
    $pm->finish(0) if $parall;
}
$pm->wait_all_children if $parall;

print_err("All done " . (@hosts_with_errors ? "with errors at ( ".join(" ", @hosts_with_errors)." )" : "- OK!"));

exit(@hosts_with_errors ? 1 : 0);


sub get_bash_cmd_at_host {
    my ($host, $command) = @_;
    $command =~ s/'/'"'"'/g;
    $command =~ s/"/\\"/g;
    $command =~ s/\$/\\\$/g;
    $command = qq[ssh -o StrictHostKeyChecking=no $host "/bin/bash -c 'set -o pipefail; $command '"];
    return $command;
}
