#!/usr/bin/perl -w
use strict;
use FindBin;
use Getopt::Long;

use lib "$FindBin::Bin/../lib";
use Project;

use Utils::Sys qw(
    print_err
    handle_errors
);

use lib "$FindBin::Bin/../wlib";
use CatalogiaMediaProject;
use User;

use utf8;
use open ":utf8";
binmode STDIN, ':utf8';
binmode STDOUT, ':utf8';

select STDERR; $|++;
select STDOUT; $|++;

handle_errors();

main();
exit(0);

sub print_help {
    print "To make actions over the phrase.\n";
    print "Options:\n";
    print "    --file FILE --login LOGIN --approve_with_comment COMMENT\n";
    print "Data is tab separated from file or STDIN:\n";
    print "    'categ \\t Action \\t Phrases \\t User_comment \\t Lang'. Where:\n";
    print "        categ            Name of category;\n";
    print "        Action           'Dell' or 'Add;'\n";
    print "        Phrases          set of phrases, separated by ',';\n";
    print "        User_comment     comment for modification;\n";
    print "        Lang             language. Or nothing, if nothing Lang is 'ru';\n";
}

#Преобразовываем категроию, это может быть либо название категории либо id.
#Возвращаем id
sub get_cat_id {
    my ($proj, $categ_or_id) = @_;

    # Сейчас пробелы удалять нельзя, т.к. есть лишние пробелы в названиях категорий - https://st.yandex-team.ru/CATALOGIA-1154 
    #$categ_or_id =~ s/(^\s+|\s+$)//g;

    my $res;

    if (my $cat_id = $proj->categs_tree->get_minicateg_id($categ_or_id)) {
        $res = $cat_id;
    } elsif (my $categ = $proj->categs_tree->get_minicateg_by_id($categ_or_id)) {
        $res = $categ_or_id;
    }

    return $res;
}


sub main {

    my ($help, $file, $login, $approve_with_comment);
    GetOptions('h|help' => \$help, 'file=s' => \$file, 'login=s' => \$login, 'approve_with_comment=s' => \$approve_with_comment);

    if ($help) {
        print_help();
        return;
    }

    my $file_handle;
    if (defined $file) {
        open $file_handle, "<", $file or die "Could not open $file ($!)";
    } else {
        $file_handle = *STDIN;
    }

    unless (defined $login) {
        die "You must define login";
    }

    my $proj = CatalogiaMediaProject->new({
           no_form => 1,
           no_auth => 1,
           load_dicts => 1,
           load_minicategs_light => 1,
           use_comptrie_subphraser => 1,
           use_sandbox_categories_suppression_dict => 1,
    });
    $proj->categs_tree->never_write_categs_cache(1);
    $proj->categs_tree->never_read_categs_cache(1);

    my $res = $proj->List_SQL("select role from Users where Login = \"$login\"");
    die "Login '$login' is uncorrect (should be \@yandex.ru login)" unless(@$res);

    $proj->login($login); # Логин на yandex.ru
    $proj->{user}  = User->new({
        proj    => $proj,
        login   => $login,
        %{$Utils::Common::options->{'auth_users'}},
    });

    my $line_number = 1;

    while (my $line = <$file_handle>) {
        chomp $line;
        my ($categ_or_id, $action, $phrases, $user_comment, $lang) = split("\t", $line);
        $proj->log("line $line_number: '$categ_or_id' '$action' '$phrases' '$user_comment' '$lang'");
        my %hash_result;

        my $cat_id = get_cat_id($proj, $categ_or_id)
            or die "Categ '$categ_or_id' is incorrect (line: $line_number)";
        $hash_result{cat_id} = $cat_id;

        $action //= '';
        if ($action ne "Delete" and $action ne "Add") {
            die "Bad action: '$action'. Allowed actions: 'Delete', 'Add'. (line: $line_number)";
        }
        $hash_result{action} = $action;

        $hash_result{phrases} = $phrases || die "Void phrases (line: $line_number)"; 
        $hash_result{user_comment} = $user_comment || die "Void user_comment (line: $line_number)"; 

        unless (grep { $lang eq $_ } qw[ ru en tr ]) {
            die "Bad lang: $lang. (line: $line_number)";
        }
        $hash_result{lang} = $lang;

        make_action($proj, \%hash_result, "$approve_with_comment");

        $proj->log("line $line_number - done");

        $line_number++;
    }
    return 1;
}

sub make_action {
    my ($proj, $hash, $approve_with_comment) = @_;

    my $cat_id = $hash->{cat_id}; # 'Id Категории выполнения Action'
    my $categ = $proj->categs_tree->get_minicateg_by_id($cat_id);

    my $action = $hash->{action};
    my $comment = undef;
    my $user_comment = $hash->{user_comment};
    my $lang = $hash->{lang};

    my $phrase_list = $proj->phrase_list([split(/,/, $hash->{phrases})]);

    my @result;

    for my $phr (@$phrase_list) {
        push @result, {
            CatID           => $cat_id,
            Language        => $lang,
            InitialPhrase   => $phr->text,
            snorm_phr       => (($phr->text !~ /\[/) ? $phr->snorm_phr : "<i>атом</i>"),
            Category        => $categ,
            Action          => $action,
        };
    }

    $proj->edit_user_phrases_warnings(\@result);

    for my $phr (@result) {
        my $phrase = $phr->{InitialPhrase};
        if (@{$phr->{Warnings}}) {
            my $warnigs = join("\t", @{$phr->{Warnings}});
            $proj->log("ERROR! '$action' '$phrase' $cat_id ($categ) Warnings: '$warnigs'");
            next;
        }
        $proj->log("$action user_phrase $phrase");

        $proj->add_user_phrase( $cat_id, $phrase, $lang, $action, $comment, $user_comment);
        my $phrase_id = $proj->get_user_phrase_id($phrase);

        $proj->log("$cat_id, $phrase_id, $action, $lang, $approve_with_comment");

        if ($approve_with_comment) {
            $proj->log("Approve ...");
            $proj->set_user_phrase_status_by_id(
                $cat_id,
                $phrase_id,
                $action,
                $lang,
                "Accepted",
                $approve_with_comment,
            );
        }
        $proj->log("phrase_id: $phrase_id");
    }
    $proj->log("Done");
}

