#!/usr/bin/perl -w

use strict;
use utf8;
no warnings qw/utf8/;
use v5.10;

#use File::Basename;
#use File::Path;
use Data::Dumper;
use Getopt::Long;

use Pid::File::Flock qw/:auto/;
use POSIX qw/strftime/;
use Parallel::ForkManager;

my @LOG_ROOT = ();

my %DAYS_BACK = ();             # log_name_prefix => N
my $DEFAULT_N_DAYS_BACK = 0;
my %EMPTIFY = ();               # log_name_prefix => N
my $DEFAULT_N_EMPTIFY = 7;

# количество процессов для архивирования логов. Один процесс - один лог
# If you specify 0, then NO fork will be done; this is good for debugging purposes.
my $NUM_PROCESSES = 0;

GetOptions(
    "help|h"      => sub {
        say "Архивирование логов с суффиксом YYYYMMDD";
        say "Обязательная (множественная) опция --logroot - директория с логами";
        say "Множественные опции:";
        say "   --days-back log_name_prefix=N       не архивировать логи с префиксом log_name_prefix последние N дней (по умолчанию N=$DEFAULT_N_DAYS_BACK для всех логов)";
        say "   --emptify log_name_prefix=N         удалять логи с префиксом log_name_prefix старше N дней (по умолчанию N=$DEFAULT_N_EMPTIFY для всех логов)";
        say "Необязательные опции:";
        say "   --num-processes K                   архивировать в K процессов";
        exit 0;
    },
    'logroot=s' => \@LOG_ROOT,
    'days-back=s' => \%DAYS_BACK,
    'emptify=s' => \%EMPTIFY,
    'num-processes=i' => \$NUM_PROCESSES,
)  or die("Error in command line arguments\n");

unless (scalar @LOG_ROOT) {
    say 'no --logroot specified, see --help';
    exit 0;
}

# получаем список файлов в директориях с логами
my @files = ();
for my $cur_log_root (@LOG_ROOT) {
    opendir(DIR, $cur_log_root) || die "Can't open dir $cur_log_root: $!";

    # валидные имена файлов в директории (включая заархивированные)
    my @names = sort grep {-f "$cur_log_root/$_"} grep {/^([\w\.\-]+)$/} readdir(DIR);

    # разбиваем файлы на группы по префиксам
    my %groups = map {my ($group, $date) = parse_log_filename($_); ($group, 1)} @names;

    # задаем значения по умолчанию для групп
    $EMPTIFY{$_} //= $DEFAULT_N_EMPTIFY for keys %groups;
    $DAYS_BACK{$_} //= $DEFAULT_N_DAYS_BACK for keys %groups;

    push @files, map {[$cur_log_root, $_]} @names;
    closedir(DIR);
}

# проверка значений опций
if (grep {$_ < 0} values %EMPTIFY) {
    die "error: some 'emptify' values are negative";
}
if (grep {$_ < 0} values %DAYS_BACK) {
    die "error: some 'days-back' values are negative";
}

# работа с логами
my $pm = new Parallel::ForkManager($NUM_PROCESSES);
for my $pair (@files) {
    $pm->start and next;
    # не пытаться удалять лок при смерти дочернего процесса с gzip
    Pid::File::Flock->abandon;

    my ($dir, $file) = @$pair;
    my $path = "$dir/$file";
    my ($group, $date) = parse_log_filename($file);

    # удаляем старые логи
    my $emptify = $EMPTIFY{$group};
    my $emptify_day_border = strftime("%Y%m%d", localtime(time-$emptify*24*60*60));
    if ($date < $emptify_day_border) {
        unlink($path) || die "Can't unlink $path: $!";
        next;
    }

    # пакуем новые логи
    next if $file =~ /\.gz$/;

    my $days_back = $DAYS_BACK{$group};
    my $days_back_day_border = strftime("%Y%m%d", localtime(time-$days_back*24*60*60));
    if ($date < $days_back_day_border) {
        # на всякий случай посылаем сигнал процессам, которые могут писать в логи
        my $lsof_out = `lsof $path | awk '{print \$2}'`; chomp $lsof_out;
        my @pids = grep {/^\d+$/} split(/\n/, $lsof_out);
        kill('HUP', $_) for @pids;

        my $arc_name = "$dir/$file.gz";
        system("gzip -c $path > $arc_name.tmp && mv $arc_name.tmp $arc_name") == 0
            or die "Can't gzip $path: $!";
        unlink($path) || die "Can't unlink $path: $!";
    }
}

sub parse_log_filename {
    # извлечение группы и даты из имени лога
    # дата -- обязательно суффикс YYYYMMDD[.gz]
    my $log_filename = shift;

    my $fname = pop [split "/", $log_filename];
    $fname =~ /^(.+)(\d{8})(.gz)?$/;

    return ($1, $2);
}
