#!/usr/bin/perl

# "посекундный мониторинг", копипаст + творческое переосмысление админского скрипта mysql-monitor, которым реально мониторится mysql
# не предназначен для поджигания никаких проверок в juggler или чём-то таком, это просто программа для записи лога, который потом можно почитать
# отвечает на вопрос "как себя чувствовал и чем занимался mysql в такой-то момент времени"
# помимо собственно значений мониторинга печатает время, затраченное на одну итерацию получения метрик, если оно большое - это показатель того, что тормозит не какой-то запрос, а mysql в целом

use strict;
use warnings;

use Getopt::Long;

use YAML;
use Carp;
use DBI;
use File::Path qw/mkpath/;
use File::Slurp;
use Time::Local;
use Time::HiRes;

use FindBin;
use lib "$FindBin::Bin/../lib";

use Utils::Sys qw(print_err get_file_lock);

use utf8;
use open ':std' => ':utf8';

my $CONF_DIR = '/etc/mysql-monitor.d';
my $TIMEOUT = 600;

# порог срабатывания на медленные запросы
my $SLOW_THRESHOLD = 60;

# перезапускаться раз в
my $WORK_TIME = 86400;  # seconds

# спать между итерациями
my $SLEEP_INTERVAL = 1.0;  # seconds


my $run_once;
my $keep_running;
GetOptions(
    "run-once" => \$run_once,
    "keep-running" => \$keep_running,
) or die "Error in command line arguments";


if ($run_once) {
    run_once();
} elsif ($keep_running) {
    keep_running();
} else {
    die "provide some parameter";
}

sub keep_running {
    get_file_lock("mysql-monitor-persecond") or exit(0);

    print_err "Started";
    
    # мы перезапускаем скрипт на каждом шаге, чтобы гарантированно перепрочитать конфиги, переоткрыть соединения, etc, etc
    my $started = time();
    while(time() < $started + $WORK_TIME) {
        system "perl $0 --run-once";
        Time::HiRes::sleep($SLEEP_INTERVAL);
    }

    print_err "Done";
}

sub run_once {
    # цикл по всем конфигам
    opendir(my $dh, $CONF_DIR) || die "Can't open dir $CONF_DIR: $!";
    for my $file (sort grep {!/\./} readdir($dh)) {
        my $started = Time::HiRes::time();
        my $status = eval {
            local $SIG{ALRM} = sub {die "TIMEOUT"};
            alarm($TIMEOUT);
            process_config($file);
        };
        alarm(0);
        if ($@) {
            $status = {error => "$@"};
        }
        my $elapsed = sprintf "%.3f", Time::HiRes::time() - $started;
        my $state = ($elapsed < 1) ? "ok" : ($elapsed < 3) ? "bad" : ($elapsed < 10) ? "very bad" : "awfully bad";
        my $msg_prefix = "[$elapsed secs, $state]";

        my @results = map { "$msg_prefix $_" } split m/\n/s, YAML::Dump($status);
        print_err($_) for @results;
    }
    closedir($dh) || die "Can't close $CONF_DIR: $!";
}

sub process_config {
    my ($config_name) = @_;

    # читаем, проверяем конфиг
    my $conf = read_config("$CONF_DIR/$config_name");

    # читаем пароль, если есть
    $conf->{password} = '';
    eval {
        open(my $pwd_file, '<', "/etc/mysql/${config_name}-root") || die "Can't open file";
        chomp($conf->{password} = <$pwd_file> || '');
    };

    # коннектимся в базу
    my $dbh = DBI->connect("DBI:mysql:;mysql_socket=$conf->{socket};mysql_enable_utf8=1", "root", $conf->{password}, {RaiseError => 1, PrintError => 0});

    my %status;

    # данные о сессиях в бд
    my $processes = $dbh->selectall_arrayref("show processlist", {Slice => {}});
    $status{queries_cnt} = @$processes;
    $status{queries_active} = grep {$_->{Command} eq 'Query'} @$processes;
    $status{queries_sleep} = grep {$_->{Command} eq 'Sleep'} @$processes;
    $status{queries_connect} = grep {$_->{Command} eq 'Connect'} @$processes;
    $status{slaves_num} = grep {$_->{Command} eq 'Binlog Dump'} @$processes;
    $status{queries_slow} = grep {$_->{Command} eq 'Query' && $_->{Time} >= $SLOW_THRESHOLD} @$processes;

    # статистическая информация о работе бд
    my $mysql_status = $dbh->selectall_hashref("show status", "Variable_name");
    $status{queries_qps} = sprintf('%.2f', $mysql_status->{Queries}->{Value} / $mysql_status->{Uptime}->{Value});

    # информация о репликации
    my $slave_status = $dbh->selectrow_hashref("show slave status");
    $status{has_slave} = 1;
    $status{has_slave} = 0 if not defined $slave_status;
    $status{slave_running} = $slave_status
        && $slave_status->{Slave_IO_Running} && $slave_status->{Slave_IO_Running} eq 'Yes'
        && $slave_status->{Slave_SQL_Running} && $slave_status->{Slave_SQL_Running} eq 'Yes'
        ? 1 : 0
        ;
    $status{slave_behind} = $slave_status ? $slave_status->{Seconds_Behind_Master} || 0 : 0;

    # проверка возраста самой долгой транзакции
    my $innodb_trx_exists = !! @{$dbh->selectcol_arrayref("DESC INFORMATION_SCHEMA.INNODB_TRX")};
    if ($innodb_trx_exists) {
        ($status{trx_max_age}) = $dbh->selectrow_array("SELECT ifnull(max(now() - trx_started), 0) trx_max_age FROM INFORMATION_SCHEMA.INNODB_TRX");
    } else {
        $status{trx_max_age} = 0;
    }

    # все запросы
    my @queries = map {
        $_->{Info} //= "";
        $_->{State} //= "";
        $_->{Time} //= "unknown";
        $_->{Rows_sent} //= "unknown";
        $_->{Rows_examined} //= "unknown";
        
        $_->{Info} =~ s/\s+/ /gs;
        
        "$_->{Command} from $_->{Host} ($_->{Time} sec): '" . substr($_->{Info}, 0, 200) . "', State: '$_->{State}', rows sent: $_->{Rows_sent}, rows examined: $_->{Rows_examined}"
    } grep {
        $_->{Command} ne 'Sleep'
    } @{ $dbh->selectall_arrayref("show full processlist", { Slice => {} }) };

    $status{processlist} = \@queries;

    # он жив!
    $status{alive} = 1;

    return \%status;
}

sub read_config {
    my ($file) = @_;
    my $conf = YAML::LoadFile($file);
    if (!exists $conf->{socket}) {
        die "Error in $file: socket is not defined";
    }
    return $conf;
}

