(function() {

    /**
     * Блок дерева регионов
     */
    utils.o_o.decl({ className: 'show-regions', baseClass: 'spinner' }, {
        init: function() {
            var that = this;

            // target - id модального окна
            if (this.params.target) {
                this._initModal();
            }

            this.addSpinner(this.elem('.show-regions__wrap'));

            this.__self._getRegions()
                .then(function(data) {
                    that._buildTree(data);
                    that._init(data);

                    that.removeSpinner(that.elem('.show-regions__wrap'));

                    that._treeReady = true;

                    if (that._needSetAfterLoad) {
                        that.setRegions(that._needSetAfterLoad);
                        that._needSetAfterLoad = null;
                    }
                });
        },

        /**
         * Подписывается на кнопки «Сохранить», «Уточнить»
         * @private
         */
        _initModal: function() {
            var that = this,
                target = this.params.target; // id модального окна

            this._modal = $('#' + target);

            // кнопки «Уточнить»
            $(document)
                .on('click', '.show-regions-button__open_type_' + target, function() {
                    that._relatedTarget = null;
                    that._setRegionsFromButton($(this));
                });

            // кнопка «Сохранить»
            $('.show-regions-modal__save', this._modal)
                .on('click', function() {
                    that._saveModalValue();
                });
        },

        /**
         * Объект с элементами show_region_button
         */
        _relatedTarget: null,

        /**
         * Записывает id-регионов в скрытый инпут и обновляет строку выбранных регионов
         * @private
         */
        _saveModalValue: function() {
            if (this._treeReady && this._currentData && this._relatedTarget) {
                // инпут
                this._relatedTarget.input.val(this._currentData.result.join());

                // Читаемая строка
                this._relatedTarget.readable.html(this.elem('.show-regions__readable').html());

                this._modal.modal('hide');
            }
        },

        /**
         * Флаг, true когда дерево построено
         */
        _treeReady: false,

        /**
         * Строит и добавляет дерево в DOM
         * @param {Object[]} regions - массив регионов
         * @param {String} regions.regionID - id региона
         * @param {String} regions.region - читабельное название региона
         * @param {Array} [regions.regionChildren] - массив дочерних регионов
         * @private
         */
        _buildTree: function(regions) {
            this.elem('.show-regions__wrap').append($(this._getBranch(regions, true)));
        },

        /**
         * Возвращает плоскую структуру регионов для поиска
         * @param {Object[]} data - массив регионов
         * @param {String} data.regionID - id региона
         * @param {String} data.region - читабельное название региона
         * @param {Array} [data.regionChildren] - массив дочерних регионов
         * @returns {Array}
         * @private
         */
        _getRegionsForAutocomplete: function(data) {
            var result = [];
                forEachBranch = function(regions) {
                    regions.forEach(function(r) {
                        result.push({
                            data: r.regionID,
                            value: r.region
                        });

                        if (r.regionChildren && r.regionChildren.length) {
                            forEachBranch(r.regionChildren);
                        }
                    });
                };

            forEachBranch(data);

            return result;
        },

        /**
         * Строит ветку регионов
         * @param {Object[]} regions - массив регионов
         * @param {String} regions.regionID - id региона
         * @param {String} regions.region - читабельное название региона
         * @param {Array} [regions.regionChildren] - массив дочерних регионов
         * @param {Boolean} [isRoot] - флаг, true когда это первый уровень
         * @returns {string}
         * @private
         */
        _getBranch: function(regions, isRoot) {
            var wrap = $('<ul class="show-regions__list"></ul>');

            regions.forEach(function(r) {
                var hasChildren = r.regionChildren && r.regionChildren.length;

                wrap.$li([
                    $().$span(
                        hasChildren ? isRoot ? '-' : '+' : '',
                        hasChildren ?
                            { 'class': 'show-regions__toggle', onclick: utils.onclickParams({ id: r.regionID }) } :
                            { 'class': 'show-regions__gap' }),

                    $().$label([

                        $().$input({
                            type: 'checkbox',
                            autocomplete: 'off',
                            'data-name': r.region,
                            'data-id': r.regionID,
                            'class': isRoot ? 'show-regions__main-checkbox' : ''
                        }),

                        $().$span(r.region, { 'class': 'show-regions__text' })

                    ], { 'class': 'show-regions__label' }),

                    !!hasChildren &&
                        $().$div([
                            this._getBranch(r.regionChildren)
                        ], {
                            id: 'children_' + r.regionID,
                            'class': 'show-regions__children',
                            style: isRoot ? '' : 'display: none'
                        })

                ], { 'class': 'show-regions__item' });
            }, this);

            return wrap;
        },

        /**
         * Находит составляющие кнопки, проставляет регионы
         * @param {jQuery} button
         * @private
         */
        _setRegionsFromButton: function(button) {
            var parent = button.closest('.show-regions-button'),
                input = $('.show-regions-button__input', parent);


            if (this._treeReady) {
                this.setRegions(input.val());
            } else {
                this._needSetAfterLoad = input.val();
            }

            this._relatedTarget = {
                input: input,
                parent: parent,
                readable: $('.show-regions-button__readable', parent)
            }
        },

        /**
         * Значение выбранных регионов, если открыли модальное окно раньше чем загрузилось дерево
         */
        _needSetAfterLoad: null,

        /**
         * Текущее состояние дерева
         */
        _currentData: null,

        /**
         * Инициализация построенного дерева
         * @param data
         * @private
         */
        _init: function(data) {
            var that = this;

            // дерево чекбоксов
            // плагин: http://uoziod.github.io/deep-checkbox/
            this.domElem.deepcheckbox({
                readableListTarget: '.show-regions__readable',
                labelExceptBefore: ' (исключая ',
                labelNothingIsSelected: 'Ничего не выбрано',
                onChange: function(items, except, result) {
                    that._currentData = {
                        items: items,
                        except: except,
                        result: result
                    };
                }
            });

            // скрыть/открыть ветку
            this.elems('.show-regions__toggle')
                .on('click', function() {
                    that.elem('#children_' + this.onclick().id).toggle();

                    $(this).html($(this).html() == '+' ? '-' : '+');
                });


            // поиск
            // плагин: https://github.com/devbridge/jQuery-Autocomplete
            this.elem('.show-regions__search-input').autocomplete({
                lookup: this._getRegionsForAutocomplete(data),
                onSelect: function (suggestion) {
                    var checkbox = that.elem("input[data-id='" + suggestion.data + "']");

                    that
                        ._openBranch(checkbox.closest('.show-regions__children'))
                        ._scrollTo(checkbox)
                        ._highlight(checkbox.next(), suggestion.value);
                },
                onSearchComplete: function() {
                    that._removeHighlight();
                }
            });
        },

        /**
         * Текущий выделенный элемент
         */
        _highlightedElem: null,

        /**
         * Удаляет выделение с элемента
         * @private
         */
        _removeHighlight: function() {
            if (this._highlightedElem) {
                this._highlightedElem.removeClass('highlighted');
                this._highlightedElem = null;
            }
        },

        /**
         * Ставит элементу выделение
         * @param {jQuery} textElem
         * @returns {o_o}
         * @private
         */
        _highlight: function(textElem) {
            this._removeHighlight();

            this._highlightedElem = textElem.addClass('highlighted');

            return this;

        },

        /**
         * Скролит к выбранному элементу
         * плагин: https://github.com/flesler/jquery.scrollTo
         * @param item
         * @returns {o_o}
         * @private
         */
        _scrollTo: function(item) {
            this.elem('.show-regions__wrap').scrollTo && this.elem('.show-regions__wrap').scrollTo(
                this.elem('.show-regions__wrap').scrollTop() + item.position().top - 200);

            return this;
        },

        /**
         * Показывает всю ветку от переданного элемента
         * @param {jQuery} elem
         * @returns {o_o}
         * @private
         */
        _openBranch: function(elem) {
            var item = elem.closest('.show-regions__item'),
                children,
                toggler;

            if (item.length) {
                children = item.children('.show-regions__children');
                toggler = item.children('.show-regions__toggle');

                if (children.length) {
                    children.toggle(true);
                    toggler.html('-');
                }

                this._openBranch(item.parent());
            }

            return this;
        },

        /**
         * Очищает дерево
         * @private
         */
        _resetTree: function() {
            this.elems('.show-regions__main-checkbox')
                .each(function(index, checkbox) {
                    $(checkbox)
                        .prop('checked', false)
                        .change();
                });
        },

        /**
         * Выставляет регионы
         * @param {String} regions - строка с id через запятую
         */
        setRegions: function(regions) {
            if (typeof regions == 'string') {
                regions = regions
                    .replace(/\s/g, '')
                    .split(',');
            }

            this._resetTree();

            regions.forEach(function(id) {
                var regionId = +id,
                    isChecked = regionId >= 0,
                    checkbox;

                if (!isChecked) {
                    regionId = regionId * -1;
                }

                checkbox = this.elem("input[data-id='" + regionId + "']");
                checkbox
                    .prop('checked', isChecked)
                    .change(); // триггерим change

                this._openBranch(checkbox.closest('.show-regions__children'));
            }, this);
        }
    }, {

        /**
         * Получает регионы по ручке
         * @returns {Deffered}
         * @private
         */
        _getRegions: function() {
            var deferred = $.Deferred();

            utils.catAjax({
                data: { action: 'getRegionsTree' },
                success: function(data) {
                    deferred.resolveWith(this, [[data.regionsTree]]);
                },
                error: function() {
                    console.log(arguments);
                }
            }, this);

            return deferred.promise();
        }

    });

})();
