package CatalogiaMediaProject;

use utf8;
use open ':utf8';

use std;
use base qw(Project);

use BaseForm;
use Data::Dumper;
use File::Basename;
use URI::Escape;
use WebCommon;
use JSON::XS;
use Encode;
use IO::Socket;
use HTML::Entities;

use DBList;
use Cmds::Base;
use Cmds::Prototypes;
use Cmds::Fields;
use Cmds::Interface;
use Cmds::SetsBanners;
use Cmds::AdminInterface;
use Cmds::RedButton;
use Cmds::BroadMatch;
use Cmds::Dicts;
use Cmds::ShowPhrases;
use Cmds::SearchCategs;
use Cmds::DataBases;
use Cmds::Market;
use Cmds::AnalysePhraseList;
use Cmds::Categs;
use Cmds::Users;
use Cmds::MediaGroups;
use Cmds::VideoDirectGroups;
use Cmds::InterestsMapping;
use Cmds::Tagging;
use Cmds::Moderation;
use Cmds::CategoryAjaxHandle;
use Cmds::DeprecatedWordsChecker;
use Cmds::Feeds;
use Cmds::Pages;
use Cmds::DynBanners;

use CategoryInterface;

use Macros;

use Cmds::PhraseBlock;

use Cmds::Crons;

use QTests;

use Text::Iconv;
use List::Util qw(min);

use ObjLib::FileLogger;
use Utils::XLS qw(xls2array xlsx2array csv2array array2xls xls2array_allsheets xlsx2array_allsheets);
use Utils::Sys qw(md5int u2w mem_usage print_err uniq);
use Utils::Hosts qw(get_curr_host);

use User;
use UserStates;

use BlackboxSimple;

use FindBin;
use lib "$FindBin::Bin/../../lib";

use States;

########################################################
#Доступ к полям
########################################################

__PACKAGE__->mk_accessors(qw(
    form
    template_name
    login
    user
    user_states
    real_user
    real_login
    macros
    category_interface
));
#    role

########################################################
# Инициализация
########################################################

sub init {
    my ($self) = @_;

    $self->{nrmsrv} = 1 unless defined $self->{nrmsrv};
    $self->{dbgroups} = 1 unless defined $self->{dbgroups};

    $self->timelogbeg("Project init");
    $self->SUPER::init;
    $self->timelogend("Project init");

    # форма запроса
    unless($self->{no_form}){
        $self->parse_form;
    }

    # аутентификация
    my $no_auth = int($self->{no_auth});
    if(!$no_auth) {
        $self->do_auth;
    } else {
        $self->{login} = "bmclient"; # для скриптов без веб-интерфейса
        $self->{user} = undef; # для скриптов без веб-интерфейса
    }

    # менеджер статусов
    $self->user_states(UserStates->new({
        proj => $self,
        %{$Utils::Common::options->{'auth_users'}},
        # граф статусов
        states => States->new({
            proj => $self,
            current_type => 'Users',
        }),
    }));

    # база данных
    $self->dbh($self->connect($Utils::Common::options->{catalogia_media_db_inf}));

    # обновляем время последнего захода пользователя
    if(!$no_auth) {
        WebCommon::UpdateVisitTime($self->dbh, $self->login);
    }

    # название шаблона
    my $template_name = basename($0);
    $template_name =~ s/\.pl$/\.tmpl/g;
    $self->{template_name} = $template_name;
    unless($self->{no_form}){
        print "Content-type: text/html\n\n" unless $self->{'indcmd'};
    }

    # Category interface
    $self->{category_interface} = CategoryInterface->new({proj => $self});

    my $objs_data = {
        data_cache => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'DataCache',
            id_field => 'Key',
        }),
        categories => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'CatalogiaBrief',
            id_field => 'CatId',
        }),
        categories_dict => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'CatalogiaDict',
            id_field => 'CatId',
        }),
        user_phrases => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'CatalogiaPhrases',
            id_field => 'InitialPhraseId',
        }),
        check_moderation => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'CheckModeration',
        }),
        flags => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'CatalogiaFlags',
        }),
        flags_description => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'CatalogiaFlagsDescription',
        }),
        tags_description => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'CatalogiaTagsDescription',
            id_field => 'Tag',
        }),
        categories_tags => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'CategoriesTags',
        }),
        log_table => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'CatalogiaModerate',
            id_field => 'LogID'
        }),
        phrase_list_cache => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'PhraseListCache',
            id_field => 'PhraseListID'
        }),
        phrase_list_action_log => DataSource::Elem->new({
            %{$self->dsih('catalogia_media_dbh')},
            db_table => 'PhraseListActionLog',
            id_field => 'ID'
        }),
        syn_cells => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'SynCells',
        }),
        test_banners => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'TestBanners',
        }),
        banners_categories_diff_meta => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'BannersCategoriesDiffMeta',
            id_field => "ID"
        }),
        banners_categories_diff_input => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'BannersCategoriesDiffInput',
            id_field => "ID"
        }),
        banners_categories_diff_results => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'BannersCategoriesDiffResults',
            id_field => "ID"
        }),
        hidden_phrases => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'HiddenPhrases',
        }),
        category_comments => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'CategoryComments',
            id_field => "ID"
        }),
        category_description => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'CategoryDescription',
        }),
        category_subscriptions  => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'CategorySubscriptions',
        }),
        catalogia_users => DataSource::Elem->new({
            %{$self->dsih},
            db_table => 'Users',
            id_field => 'Login',
        }),
        subscriptions_filters   => DataSource::Elem->new({
            %{$self->dsih},
            db_table    => 'SubscriptionsFilters',
            id_field    => 'ID',
        }),
        categories_text_fields => DataSource::Elem->new({
            %{$self->dsih},
            db_table    => 'CategoriesTextFields',
            id_field    => 'ID',
        }),
        domain_flags => DataSource::Elem->new({
            %{$self->dsih},
            db_table    => 'CatalogiaDomainFlags',
            id_field    => 'ID',
        }),
    };

    # Макросы для веб-страниц
    $self->{macros} = Macros->new({ proj => $self });

    $BM::PhraseCategs::compute_phrase_analysis = 1; # Для просмотра в веб-интерфейсе. TODO: выставлять только для веб-интерфейса (не для скриптов, использующих CatalogiaMediaProject.pm)

    $self->mk_accessors(keys %$objs_data);
    $self->set($_, $objs_data->{$_}) foreach keys %$objs_data;
}

sub post_init {
    my $self = shift;

    return if $self->host_role eq 'bmfront';

    $self->deprecated_words_checker;
    $self->SUPER::post_init;
}

########################################################
#Методы
########################################################

sub host_role {
    my ($self) = @_;
    #$self->dd($ENV{HTTP_HOST});
    #return 'drf';
    return 'drf' if $ENV{HTTP_HOST} && ( $ENV{HTTP_HOST} =~ /drf\./i);
    return 'catmedia' if $ENV{HTTP_HOST} && ( $ENV{HTTP_HOST} =~ /catmedia\./i);
    return 'catless' if $ENV{HTTP_HOST} && ( $ENV{HTTP_HOST} =~ /catmass\./i);
    return 'catless' if $ENV{HTTP_HOST} && ( $ENV{HTTP_HOST} =~ /catless\./i);
    return $self->SUPER::host_role;
}

sub role {
    my ($self) = @_;
    #$self->dd($self->stack_trace);
    return $self->user ? $self->user->role : 'user';
}

sub dblist {
    my ($self, $prms) = @_;
    return DBList->new( { proj => $self, ( $prms ? %$prms : () ) } );
}

sub do_auth {
    my ($self) = @_;
    $self->log("do_auth...");
    $self->timelogbeg("CheckCatalogiaAuth");

    #Для определенных логинов разрешаем доступ без авторизации
    if($self->form && $self->form->{cmd} && ($self->form->{cmd} =~ /^noauth_/ ) ){
        $self->{login} = '__noauth';
        $self->{user}  = User->new({
            proj    => $self,
            login   => '__noauth',
            %{$Utils::Common::options->{'auth_users'}},
        });
        $self->log("__noauth ".$self->form->{cmd});
        return;
    }

    $self->{real_login} = '';
    # Для скачивания с --user-agent='bm_download' логин берем из поля externallogin и проверяем подпись в поле sign
    if ($ENV{HTTP_USER_AGENT} eq "bm_download" and $self->form->{externallogin}) {
        $self->log("do_auth: HTTP_USER_AGENT: " . $ENV{HTTP_USER_AGENT} . " (" . $self->form->{'externallogin'} . ") ");
        $self->{real_login} = $self->check_sign(query => $ENV{QUERY_STRING}) ? $self->form->{externallogin} : '';
        #$self->{real_login} = $self->form->{externallogin};
        $self->log("Auth with externallogin: " . $self->{real_login} . " " . $ENV{SERVER_NAME} . " " . $ENV{QUERY_STRING});
    }

    $self->{real_login} = WebCommon::CheckCatalogiaAuth()    if not $self->{real_login};
    $self->log("CheckCatalogiaAuth done. real_login=" . $self->{real_login});
    $self->{real_user}  = User->new({
        proj    => $self,
        login   => $self->{real_login},
        %{$Utils::Common::options->{'auth_users'}},
    });

    if($self->{real_user}->admin_role && $self->viewoptions->{'changeuserlogin'}) {
        $self->{login} = $self->viewoptions->{'changeuserlogin'};
        $self->{user}  = User->new({
            proj    => $self,
            login   => $self->{login},
            %{$Utils::Common::options->{'auth_users'}},
        });
    } else {
        $self->{login} = $self->{real_login};
        $self->{user} = $self->{real_user};
    }

    $self->user->log_visit;
    $self->timelogend("CheckCatalogiaAuth");
    $self->log("do_auth done. Login=" . $self->{login} . " real_login=" . $self->{real_login} . " url=" . $ENV{HTTP_HOST} . $ENV{REQUEST_URI});
}

sub parse_form {
    my ($self) = @_;
    $self->form(BaseForm->new);
    $self->form->parse;
    my $form = $self->form;
    $self->{'viewoptions'} = {};
    $self->{'viewoptions'} = $self->_parse_viewoptions($form->getlast('viewoptionsstr')) if $form->{viewoptionsstr};
    $self->{'viewoptions'}{'lang'} = $form->{'lang'} || 'ru' unless $self->{'viewoptions'}{'lang'};
}

# вернуть ответ
# vars       ссылка на хэш параметров
sub show_template {
    my ($self, $vars) = @_;
    my $result = {
        login           => $self->login,
        role            => $self->role,
        host_role       => $self->host_role,
        is_moderator    => $self->is_moderator,
        %{$vars}
    };
    my $template = WebCommon::CatalogiaTemplate();

    $template->process($self->template_name, $result)
    or die $template->error();
}

sub show_cmd_template {
    my ($self, $vars) = @_;
    $self->timelogbeg('show_cmd_template');
    my $result = {
        login           => ( $self->user ? $self->user->login : $self->login ) // "undefined_guest",
        role            => $self->role,
        host_role       => $self->host_role,
        is_moderator    => $self->is_moderator,
        template_name   => $vars->{'template'},
        host            => (get_curr_host =~ m/(.*)\.yandex\.(ru|net)$/)[0] // undef,
        pid             => $$,
        %{$vars}
    };

    $self->_add_functions($result);

    #$Template::Config::STASH = 'Template::Stash'; #Вариант решения баги в Template::Stash::XS
    my $template = WebCommon::CatalogiaTemplate();
    eval { $template->process($vars->{'template'}, $result)
    or die $template->error(); };
    if($template->error()){
        print $template->error();
    }
    $self->timelogend('show_cmd_template');
}

# выполнить редирект (для обработки POST-запросов)
sub do_redirect {
    my ($self, $url) = @_;
    $self->log("do_redirect $url");

    $url ||= uri_escape("http://".$ENV{SERVER_NAME}.$ENV{REQUEST_URI});

    #$self->dd($url);
    #Добавляем лог времени для действий с редиректами
    if( $url =~ /^\/|^(ind.pl)?\?/){
        $self->timelogend("make_cmd data");
        $url .= '?' unless $url =~ /\?/;
        $url .= '&prevtimelog='.$self->timeloginf->{"make_cmd data"}{'=ttime'};
    }

    $self->delete_cmdlog;
    print "Status: 302 Found\n";
    print "Location: $url\n\n";
    #exit(0);
    goto EXIT; #нужно, чтобы убрать прерывание работы программы под nginx'ом
}

sub _perldata2js {
    my ($data) = @_;
    my $text = '';
    return $data =~ /^[0-9\.]+$/ ? $data : '"'.$data.'"' unless ref($data);
    return '['.join(', ', map { _perldata2js($_) } @$data ).']' if ref($data) eq 'ARRAY';
    return '{'.join(', ',  map { '"'.$_.'": '. _perldata2js($data->{$_}) } keys %$data ).'}' if ref($data) eq 'HASH';
    return 'CODE';
}

sub perldata2js {
    my $self = shift;
    return _perldata2js(@_);
}

sub _check_rights {
    my ($self, $list) = @_;
    my $rights = $self->user->rights;
    $list = [ grep { $_->{'rights'} ? ( grep { $rights->{$_} } split( /\s+|,/, $_->{'rights'})  ) : 1 } @$list ];
    $_->{sublist} = $self->_check_rights($_->{sublist}) for grep { $_->{sublist} } @$list;
    return $list;
}

sub _add_functions {
    my ($self, $vars) = @_;
    my $rf = $self->proxy_ref;

    my $h = $self->macros->get_macros($rf, $vars);
    $vars->{$_} = $h->{$_} for keys %$h;
}

sub is_moderator {
    my ($self) = @_;

    return $self->role eq "admin";
}

sub cache_data {
    my ($self, $key, $value) = @_;
    my $encoded = encode_json($value);

    $self->dbh->do("replace into DataCache (DataKey, DataValue) values (".$self->dbh->quote($key).",".$self->dbh->quote($encoded).")")
    or die($DBI::errstr);
}

sub get_cached_data {
    my ($self, $key) = @_;

    my $data = $self->data_cache->Get("DataKey", $key);

    if($data) {
        my $json = JSON->new->allow_nonref;
        #return $json->decode(decode_utf8($data->{DataValue}));
        return decode_json($data->{DataValue});
    }

    return undef;
}

sub get_test_banners {
    my ($self) = @_;

    return $self->test_banners->List();
}

sub search_syn_cells {
    my ($self, $text, $lang) = @_;

    $lang ||= "ru";

    return $self->List_SQL("
        select
            *
        from
            ".$self->syn_cells->db_table."
        where
            Language = ? and
            Syns like ?
    ", [$lang, "%$text%"]);
}

sub get_syn_cell {
    my ($self, $id, $lang) = @_;

    $lang ||= "ru";

    return $self->syn_cells->List({ Language => $lang, MainSynID => $id })->[0];
}

sub get_syn_cells_list {
    my ($self, $lang) = @_;

    $lang ||= "ru";

    return $self->syn_cells->List({ Language => $lang }, "", "", "MainSyn");
}

sub is_syn_pair_deleted {
    my ($self, $pair, $lang) = @_;

    $lang ||= "ru";

    return $self->user_phrases->List([
        [Action => "DeleteSynPair"],
        [InitialPhrase => $pair],
        [Language => $lang],
        [Status => "!=" => "Declined"],
        [Status => "!=" => "Done"]
    ])->[0];
}

sub is_flag_deleted {
    my ($self, $cat_id, $flag) = @_;

    return $self->user_phrases->List([
        [Action => "DeleteFlag"],
        [InitialPhrase => $flag],
        [CatID => $cat_id],
        [Status => "!=" => "Declined"],
        [Status => "!=" => "Done"]
    ])->[0];
}


sub get_categories_list {
    my ($self) = @_;
    return $self->categories->List({IsAtom => 0});
}

#Кэшируемый на период сессии словарь категорий
sub get_cached_categories_list :TABLECACHE('CatalogiaPhrases', 'catalogia_media_dbh', 'Rows') {
    my ($self) = @_;

    my $lst1 = $self->categories_dict->List({}, [qw{ CatID CategoryName Language }]); #Берём данные из таблицы со списком категорий
    my $lst2 = $self->dbtable('CatalogiaPhrases')->List({ Action => 'AddCategory' }, [qw{ CatID InitialPhrase }]); #Берём добавленные категории с модерации
    $_->{CatID} =~ s/:.*// for @$lst2; #В номере на модерации через : указывается ещё и родитель - удаляем

    #Оставляем только часть записей для отладки
    #$_ = [ @$_[0 .. 50] ] for $lst1, $lst2;

    my $mm = {
        ( map { $_->{CatID}.' _ '.$_->{Language} => $_->{CategoryName} } @$lst1 ),
        ( map { $_->{CatID}.' _ '.'ru'           => $_->{InitialPhrase} } @$lst2 ),
    };

    my $mm2 = {
        ( map { $_->{CategoryName} => $_->{CatID} } @$lst1 ),
        ( map { $_->{InitialPhrase} => $_->{CatID} } @$lst2 ),
    };

    return { catid2catname => $mm, catname2catid => $mm2, }
}

sub get_cached_parents_ids_hash : TABLECACHE('CatalogiaPhrases', 'catalogia_media_dbh', 'Rows') {
    my ($self) = @_;

    my $main_categs_tree = $self->List_SQL("
        select CatID, ParentID
        from " . $self->categories->db_table . "
    ");

    my $added_categs = $self->List_SQL("
        select CatID
        from " . $self->user_phrases->db_table . "
        where
            Action = 'AddCategory'
            and Status in ('New', 'Accepted')
    ");

    my $changed_parents = $self->List_SQL("
        select CatID, InitialPhrase as ChangedParentID
        from " . $self->user_phrases->db_table . "
        where
            Action = 'ChangeParent'
            and Status in ('New', 'Accepted')
    ");

    my $catid2parent_id = {
        map { $_->{CatID} => $_->{ParentID} } @$main_categs_tree
    };

    for my $added_categ (@$added_categs) {
        my ($catid, $parent_id) = split(m/:/, $added_categ->{CatID});
        $catid2parent_id->{$catid} = $parent_id if defined($catid) && defined($parent_id);
    }

    for my $changed_parent (@$changed_parents) {
        $catid2parent_id->{$changed_parent->{CatID}} = $changed_parent->{ChangedParentID};
    }

    return $catid2parent_id;
}


sub get_virtual_categories_list {
    my ($self) = @_;
    return $self->List_SQL("
        select distinct
            CatID, CategoryName
        from
            CatalogiaVirtualDict
        where
            Language = 'ru'
    ");
}

sub get_categories_list_lang {
    my ($self, $lang) = @_;
    return $self->categories_dict->List({Language => $lang});
}

sub get_atoms_list {
    my ($self, $lang) = @_;

    $lang ||= "ru";

    my $atoms = $self->List_SQL("
        select
            b.CatID CatID,
            ParentID,
            Flags,
            IsAtom,
            CategoryName,
            CategoryPhrases
            from (
                select *, ? as Language from
                " . $self->categories->db_table . "
                where IsAtom = 1
            ) b
        inner join
            " . $self->categories_dict->db_table . "
            using (CatID, Language)
        order by
            CategoryName
    ", [$lang]);

    $self->update_categories_parents($atoms);

    push @$atoms, $_ for grep{$_->{IsAtom}} @{$self->get_new_categories};

    return $atoms;
}

sub get_flags_list {
    my ($self) = @_;

    return $self->flags_description->List();
}

sub get_flags_list_full {
    my ($self) = @_;

    #имитация outer join-а, которая работает на порядок быстрее, чем outer join
    my $SQL = "
        select
            Flag,
            max(Description) as Description,
            max(NumCategs) as NumCategs
        from
            (
                select
                    Flag,
                    '' as Description,
                    count(CatID) as NumCategs
                from
                    CatalogiaFlags
                group by Flag
            union
                select
                    Flag,
                    Description,
                    0 as NumCategs
                from
                    CatalogiaFlagsDescription
            ) xx
        group by Flag
        order by Flag
    ";
    return $self->List_SQL($SQL);
}

sub get_flag_categories {
    my ($self, $flag, $lang) = @_;

    $lang ||= "ru";

    return $self->List_SQL("
        select
            CatID,
            CategoryName
        from
            ".$self->flags->db_table."
        left join
            ".$self->categories->db_table." as cb
            using(CatID)
	left join
	    ".$self->categories_dict->db_table." as cdict
	    using(CatID)
        where
            Flag = ".$self->dbh->quote($flag)." and cdict.Language = ".$self->dbh->quote($lang)."
        order by
            CategoryName
        ");
}

sub get_flag {
    my ($self, $flag) = @_;

    return $self->flags_description->Get("Flag", $flag);
}

sub get_virtual_category_full {
    my ($self, $cat_id, $lang) = @_;

    $lang ||= 'ru';
    my $query = $self->List_SQL("
        select
            CatalogiaVirtualDict.CatID,
            CatalogiaVirtualDict.CategoryName,
            CatalogiaVirtualDict.CategoryPhrases,
            CatalogiaVirtualBrief.Flags
        from
            CatalogiaVirtualDict
        inner join
            CatalogiaVirtualBrief
        using
            (CatID)
        where
            CatID = ? and Language = ?
    ", [$cat_id, $lang]);
    return $query->[0];
}

sub get_category_full {
    my ($self, $id, $lang) = @_;

    $lang ||= "ru";

    my $res = $self->dbhlist->{catalogia_media_dbh}->List_SQL("
        select
            b.CatID CatID,
            ParentID,
            CategoryName,
            CategoryPhrases,
            MinusWords,
            Language,
            IsAtom,
            group_concat(f.Flag) Flags
        from
            CatalogiaBrief b
        inner join
            CatalogiaDict d
            using (CatID)
        left join
            ".$self->flags->db_table." f
            using (CatID)
        where
                CatID = ? and Language = ?
        group by CatID
    ", [ $id, $lang ]);

    my $h = $res->[0];

    if(!$h) {
        $h = $self->get_new_category($id, $lang);
    }

    if($h) {
        $self->update_category_parent($h);
        $h->{IsDeleted} = $self->is_category_deleted($id);
    }

    return $h;
}

sub update_category_parent {
    my ($self, $h) = @_;
    $self->update_categories_parents([$h]);
}

sub update_categories_parents {
    my ($self, $arr) = @_;
    if (!defined $self->{updated_categories_parents}) {
        my $res = $self->List_SQL("
            select
                *
            from
                CatalogiaPhrases
            where
                Action = 'ChangeParent' and
                Status != 'Declined' and
                Status != 'Done'
            order by CreateTime
        ");
        #ордер нужен, чтобы в хронологическом порядке присвоился последний родитель
        $self->{updated_categories_parents} = {};
        foreach my $row (@$res) {
            $self->{updated_categories_parents}->{$row->{CatID}} = $row->{InitialPhrase};
        }
    }
    foreach my $elem ( @$arr ) {
        $elem->{ParentID} = $self->{updated_categories_parents}->{$elem->{CatID}} if exists $self->{updated_categories_parents}->{$elem->{CatID}};
    }
}

sub get_category_comments {
    my ($self, $id) = @_;

    return $self->category_comments->List2(filter => { CatID => $id }, order_by => "-UpdateTime");
}

sub get_new_categories {
    my ($self) = @_;
    my $categs = [];

    my $res = $self->dbhlist->{catalogia_media_dbh}->List_SQL("
        select
            *
        from
            CatalogiaPhrases
        where
            Action = 'AddCategory' and
            Status != 'Declined' and
            Status != 'Done'
    ");

    for my $item (@$res) {
        push @$categs, $self->decode_new_category($item);
    }

    return $categs;
}

sub decode_new_category {
    my ($self, $item) = @_;
    my ($cat_id, $parent_id) = split ":", $item->{CatID};

    my $h = {
        CatID           => $cat_id,
        ParentID        => $parent_id,
        CategoryName    => $item->{InitialPhrase},
        CategoryPhrases => "",
        MinusWords      => "",
        Language        => $item->{Language},
        Flags           => "",
        IsAtom          => ($item->{InitialPhrase} =~ /^\./)
    };

    $self->update_category_parent($h);

    return $h;
}

sub get_new_category {
    my ($self, $id, $lang) = @_;
    my $res = $self->dbhlist->{catalogia_media_dbh}->List_SQL("
        select
            *
        from
            CatalogiaPhrases
        where
            Action = 'AddCategory' and
            CatID like ? and
            Status != 'Declined' and
            Status != 'Done'
    ", ["$id:%"]);

    for my $item (@$res) {
        my $categ = $self->decode_new_category($item);
        my $cat_id = $categ->{CatID};
        next if !$cat_id || $cat_id ne $id;

        return $categ;
    }

    return undef;
}

sub get_category {
    my ($self, $id, $lang) = @_;

    $lang ||= "ru";

    my $h = $self->categories_dict->List({"Language" => $lang, "CatID" => $id})->[0];

    if($h) {
        $self->update_category_parent($h);
    } else {
        $h = $self->get_new_category($id, $lang);
    }

    return $h;
}

sub get_category_name {
    my ($self, $cat_id, $lang) = @_;

    $lang ||= 'ru';

    #Предотвращаем массовые хождения в базу за названиями
    return $self->get_cached_categories_list->{catid2catname}{$cat_id.' _ '.$lang};

    my $category = $self->categories_dict->List({Language => $lang, CatID => $cat_id})->[0];
    return $category->{CategoryName} if $category;

    my $added_category = $self->List_SQL("
        select
            InitialPhrase
        from
            " . $self->user_phrases->db_table . "
        where
            Action = 'AddCategory' and
            CatID like ? and
            Status != 'Declined' and Status != 'Done'
    ", ["$cat_id:%"])->[0];
    return $added_category->{InitialPhrase} if $added_category;

    return undef;
}

sub get_category_id {
    my ($self, $category_name) = @_;

    return $self->get_cached_categories_list->{catname2catid}{$category_name};
}

sub get_parent_id {
    my ($self, $cat_id) = @_;

    return $self->get_cached_parents_ids_hash->{$cat_id};
}

sub get_category_by_name {
    my ($self, $name, $lang) = @_;

    #используется BINARY, чтобы включить у селекта чувствительность к регистру
    my $rows = $self->categories_dict->List([
        [CategoryName => "=BINARY" => $name],
        [Language => "ru"],
    ]);

    my $categ = @$rows ? $rows->[0] : undef;

    if(!$categ) {
        my $items = $self->user_phrases->List([
            [Action => "AddCategory"],
            [InitialPhrase => "=BINARY" => $name],
            [Status => "!=" => "Declined"],
            [Status => "!=" => "Done"],
        ]);

        $categ = $self->decode_new_category($items->[0]) if @$items;
    }

    if($categ && $lang && $lang ne "ru") {
        $categ = $self->get_category($categ->{CatID}, $lang);
    }

    return $categ;
}

sub get_category_children {
    my ($self, $id) = @_;

    my $result = $self->categories->List({"ParentID" => $id});
    push @$result, grep{$_->{ParentID} eq $id} @{$self->get_new_categories};

    # учитываем изменения родительских категорий
    my $changed = $self->user_phrases->List([
        [Action => "ChangeParent"],
        [Status => "!=" => "Declined"],
        [Status => "!=" => "Done"],
    ]);

    my $changed_ids = { map{$_->{CatID} => $_} @$changed };
    $result = [ grep{!$changed_ids->{$_->{CatID}}} @$result ];
    push @$result, $self->get_category($_->{CatID}) for grep{$_->{InitialPhrase} eq $id} @$changed;

    return $result;
}

sub get_minicategs_subtrees {
    my ($self, $name) = @_;
    #my @res = ($name);
    my @res = ();
    my @arr = ($name);
    while( @arr ){ #Перебираем получение детей до самого глубокого уровня
        #Получаем список номеров категорий
        my $lst = [map {$_->{CatID}} map {@$_} $self->categories_dict->List2( filter => { CategoryName => \@arr }, gfields => ['CatID']) ];
        #Получааем список номеров детей
        my $chlds = [ map {$_->{CatID}} map {@$_} $self->categories->List2( filter => { ParentID => $lst }, gfields => ['CatID', ]) ];
        #Получаем названия детей
        @arr = map { $_->{CategoryName} } map {@$_} $self->categories_dict->List2( filter => { CatID => $chlds, Language => 'ru' }, gfields => ['CatID', 'CategoryName']);
        push(@res, @arr);
    }
    return @res;
    #$self->dd($name, \@res);
}

# получить список фраз категории
# опции:
# hier              дописывать фразы всех потомков
# user_phrases      дописывать пользовательские фразы
sub get_category_phrases {
    my ($self, $id, %opts) = @_;
    $opts{lang} ||= "ru";

    my @ids;
    my @categs = ($self->get_category($id, $opts{lang}));
    my @phrases;

    while(@categs) {
        my $categ = pop @categs;
        push @phrases, grep{$_} split ",", $categ->{CategoryPhrases};
        push @ids, $categ->{CatID};

        # обходим потомков, если нужно
        if($opts{hier}) {
            push @categs, @{$self->get_category_children($categ->{CatID})};
        }
    }

    # пользовательские фразы
    if($opts{user_phrases}) {
        push @phrases, map{$_->{InitialPhrase}} @{$self->user_phrases->List({"CatID" => \@ids})};
    }

    return \@phrases;
}

sub log_action {
    my ($self, $cat_id, $cmd, $value) = @_;

    $self->log_table->Add({
        CatID               => $cat_id,
        Command             => $cmd,
        Value               => $value || " ",
        Login               => $self->login,
        ModerationStatus    => "New"
    });
}

sub get_user_phrases_history {
    my ($self, $cat_id) = @_;

    return $self->user_phrases->List2(order_by => "-UpdateTime");
}

sub get_user_category_name {
    my ($self, $cat_id, $lang) = @_;
    my @list = sort{$a->{UpdateTime} cmp $b->{UpdateTime}} @{$self->user_phrases->List([
        [CatID => $cat_id],
        [Language => $lang],
     	[Action => "RenameCategory"],
        [Status => "!=" => "Declined"],
        [Status => "!=" => "Done"]
    ])};

    return @list ? $list[-1]->{InitialPhrase} : "";
}

sub get_user_flags {
    my ($self, $cat_id) = @_;

    return [map{$_->{InitialPhrase}} @{$self->user_phrases->List([
        [CatID => $cat_id],
     	[Action => "AddFlag"],
        [Status => "!=" => "Declined"],
        [Status => "!=" => "Done"]
    ])}];
}

sub get_user_syn_pairs {
    my ($self, $lang) = @_;

    return $self->user_phrases->List([
     	[Language => $lang],
        [Action => "AddSynPair"],
        [Status => "!=" => "Declined"],
        [Status => "!=" => "Done"]
    ]);
}

sub get_user_antiwords {
    my ($self, $cat_id, $lang) = @_;

    return $self->user_phrases->List([
        [CatID => $cat_id],
     	[Language => $lang],
        [Action => "AddAntiword"],
        [Status => "!=" => "Declined"],
        [Status => "!=" => "Done"]
    ]);
}

sub get_user_phrases {
    my ($self, $cat_id, $lang) = @_;

    return $self->user_phrases->List([
        [CatID => $cat_id],
     	[Language => $lang],
        [Status => "!=" => "Declined"],
        [Status => "!=" => "Done"]
    ]);
}

sub is_category_deleted {
    my ($self, $cat_id) = @_;

    # кэшируем список удалённых категорий
    if(! defined $self->{deleted_categories}) {
        my $cts = $self->user_phrases->List([
            [Action => "DeleteCategory"],
            [Status => "!=" => "Declined"],
            [Status => "!=" => "Done"]
        ]);

        $self->{deleted_categories} = { map{$_->{CatID} => $_} @$cts };
    }

    return $self->{deleted_categories}{$cat_id};
}

sub get_accepted_user_phrases {
    my ($self)  = @_;

    return $self->user_phrases->List([
        [Status => "Accepted"],
    ]);
}

sub get_moderation_media_phrases {
    my ($self) = @_;

    return $self->dbhlist->{catalogia_media_dbh}->List_SQL("
        select
            *,
            CONCAT(
                IF((cm.CatID LIKE 'adict%'), '_Dict', IF((cm.CatID REGEXP '^a|^__homonym|^rgn'),'_Atom',
                    IF( (Action != 'AddPrefilter' OR Action != 'DeletePrefilter'), '', 'Categ')
                )),
                IF( (Action = 'Add' OR Action = 'Delete' ), 'AddDel',
                    IF( (Action = 'AddPrefilter' OR Action = 'DeletePrefilter' ), 'Prefilter', Action )
                )
            ) GroupAction
        from ".$self->user_phrases->db_table." cm
        left join
            CheckModeration
        using(CatID, Language, InitialPhraseID, Action)
        where
            Status = 'New' and MediaStatus = 'New'
        order by
            Importance desc, Action
    ");

    return $self->user_phrases->List([
        [Status => "New"]
    ]);
}
sub get_moderation_phrases {
    my ($self) = @_;

    return $self->dbhlist->{catalogia_media_dbh}->List_SQL("
        select
            *,
            CONCAT(
                IF((cm.CatID LIKE 'adict%'), '_Dict', IF((cm.CatID REGEXP '^a|^__homonym|^rgn'),'_Atom',
                    IF( (Action != 'AddPrefilter' OR Action != 'DeletePrefilter'), '', 'Categ')
                )),
                IF( (Action = 'Add' OR Action = 'Delete' ), 'AddDel',
                    IF( (Action = 'AddPrefilter' OR Action = 'DeletePrefilter' ), 'Prefilter', Action )
                )
            ) GroupAction
        from ".$self->user_phrases->db_table." cm
        left join
            CheckModeration
        using(CatID, Language, InitialPhraseID, Action)
        where
            Status = 'New' and MediaStatus != 'Declined'
        order by
            Importance desc, Action
    ");

    return $self->user_phrases->List([
        [Status => "New"]
    ]);
}

sub get_user_phrase_id {
    my ($self, $phrase) = @_;
    my $text = "".$phrase;
    $text =~ s/\s+/ /g;
    $text =~ s/^\s+|\s+$//g;
    return int(md5int($text));
}

sub add_category_description {
    my ($self, $cat_id, $lang, $login, $description) = @_;
    my $added_value = {
        CatID       => $cat_id,
        Language    => $lang,
        Login       => $login,
        Description => $description,
        CreateTime  => $self->curtime,
    };
    $self->category_description->Add([$added_value]);
}

sub get_category_description {
    my ($self, $cat_id, $lang) = @_;
    my $query = $self->List_SQL("
        select
            Description
        from
            " . $self->category_description->db_table . "
        where
            CatID = ? and Language = ?
        order by
            CreateTime desc
        limit
            1
    ", [$cat_id, $lang]);
    return @$query ? $query->[0]{Description} : '';
}

sub remove_atoms_from_text {
    my ($self, $text) = @_;

    $text =~ s/([\[\/])\.[^\]\/]+/$1/g;
    $text =~ s/\/\/+/\//g;
    $text =~ s/\[\//\[/g;
    $text =~ s/\/\]/\]/g;
    $text =~ s/\[\]//g;

    return $text;
}

sub retrieve_all_atoms_from_text {
    my ($self, $text) = @_;

    my @atoms = split /([\[\/])([^\]\/]+)/, $text;
    return [] if @atoms <= 1;

    my @data = ();
    my $current_length = 0;
    for (my $split_index = 0; $split_index < @atoms; $split_index += 3) {
        $current_length += length($atoms[$split_index]);
        $current_length += length($atoms[$split_index + 1]) if $split_index + 1 < @atoms;
        if ($split_index + 2 < @atoms) {
            my $atom = $atoms[$split_index + 2];
            print "atom = $atom\n";
            my $atom_length = length($atom);
            push @data, {
                start   => $current_length,
                length  => $atom_length,
                atom    => $atom,
            };
            $current_length += $atom_length;
        }
    }
    return \@data;
}

sub retrieve_atoms_from_text {
    my ($self, $text) = @_;

    my @atoms = split /([\[\/])(\.[^\]\/]+)/, $text;
    return [] if @atoms <= 1;

    my @data = ();
    my $current_length = 0;
    for (my $split_index = 0; $split_index < @atoms; $split_index += 3) {
        $current_length += length($atoms[$split_index]);
        $current_length += length($atoms[$split_index + 1]) if $split_index + 1 < @atoms;
        if ($split_index + 2 < @atoms) {
            my $atom = $atoms[$split_index + 2];
            my $atom_length = length($atom);
            my $category = $self->get_category_by_name($atom);
            push @data, {
                start   => $current_length,
                length  => $atom_length,
                catid   => $category->{CatID},
                atom    => $atom,
            } if $category;
            $current_length += $atom_length;
        }
    }
    return \@data;
}

sub update_subphraser {
    my ($self, $cat_id, $phrase, $action, $lang, $status) = @_;

    my $is_add = ($action eq "Add" && ($status eq "New" || $status eq "Accepted")) ||
        ($action eq "Delete" && $status eq "Declined");
    my $is_del = ($action eq "Add" && $status eq "Declined") ||
        ($action eq "Delete" && ($status eq "New" || $status eq "Accepted"));

    if($is_add || $is_del) {
        my $phr_obj = $self->get_language($lang)->phrase($phrase);
        my $categ = $self->get_category($cat_id);

        $self->log("update_subphraser add_subphraser_category");
        if($is_add) {
            $phr_obj->add_subphraser_category($categ->{CategoryName});
        } elsif($is_del) {
            $phr_obj->delete_subphraser_category($categ->{CategoryName});
        }
        $self->log("/ update_subphraser add_subphraser_category");
    }
}

sub update_all_subphrasers {
    my ($self, $cat_id, $phrase, $action, $lang, $status) = @_;

    # отражение изменений в subphraser-ax
    # kostyl technology
    $self->log("add_user_phrase: update my subphraser");
    $self->update_subphraser($cat_id, $phrase, $action, $lang, $status);
    $self->log("add_user_phrase: / update my subphraser");
    my $projsrv_cl = $self->projsrv_cl;
    my $my_host = get_curr_host;
    for my $host (@{$Utils::Common::options->{subphrasers_hosts}}) {
        next if $host =~ /$my_host/ || $my_host =~ /$host/;
        my $try = 1;
        do {
            eval {
                $self->log("add_user_phrase: attempt $try");
                $self->log("add_user_phrase: update $host subphraser");
                $self->{projsrv_cl} = undef;
                $self->projsrv_cl({
                    host            => $host,
                    client_timeout  => $Utils::Common::options->{PrefProjSrv_params}{client_timeout},
                });
                $self->update_subphraser($cat_id, $phrase, $action, $lang, $status);
                $self->log("add_user_phrase: / update $host subphraser");
            };
            my $error = "$@";
            $error =~ s/\s/ /g;
            $self->log("ERROR: '$error' during add_user_phrase") if $@;
        } while ( $@ && $try++<3 );
    }
    # возвращаем параметры projsrv_cl
    $self->{projsrv_cl} = $projsrv_cl;
    # / kostyl technology
}

sub compute_phrase_importance {
    my ($self, $text, $lang) = @_;

    my $phrase = $self->get_language($lang)->phrase($text);
    my @banner_ids = $self->banners_bender->find_ids_atoms($phrase, 10 * 1000);
    return scalar(@banner_ids);
}

#сброс закэшированного списка фраз категории
sub reset_category_phrase_block {
    my ($self, $cat_id, $lang) = @_;
    my $phrase_block = Cmds::PhraseBlock->new({
       proj     => $self,
       cat_id   => $cat_id,
       lang     => $lang,
    });
    $phrase_block->reset_phrase_list_id;
}

sub add_user_phrase {
    my ($self, $cat_id, $phrase, $lang, $action, $comment, $user_comment) = @_;
    my $phrase_id = $self->get_user_phrase_id($phrase);
    my $add_action = 1;
    $action ||= "Add";
    $comment ||= "";

    $self->log("add_user_phrase: $cat_id / '$phrase'");

    # kostyl
    if (grep {$action eq $_} qw(Add Delete)) {
        $self->reset_category_phrase_block($cat_id, $lang);
    }
    # /kostyl

    my $cat_name = $self->get_category_name($cat_id, $lang);
    # Генерим дифф на каждое добавленное антислово, если это не словарь
    if ($action eq 'AddAntiword' && $cat_name !~ /^[.]/) {
        my $dbt_meta = $self->dbtable('BannersCategoriesDiffMeta');
        my $dbt_input = $self->dbtable('BannersCategoriesDiffInput');

        $dbt_meta->Add({
            Login           => $self->login,
            Comment         => "Автоматически сгенерированный дифф для антиворда '$phrase' на категорию '$cat_name'",
            DiffType        => 'actions',
            DataType        => 'banner',
            NumBanners      => 1000000,
            InputFileName   => 'dummy',
            Language        => $lang,
        });

        my $last_id = $dbt_meta->get_last_id();

        $dbt_input->Add({
            ID              => $last_id,
            Input           => JSON::to_json([{CatID            => $cat_id,
                                               Language         => $lang,
                                               InitialPhraseID  => $phrase_id,
                                               InitialPhrase    => $phrase,
                                               Action           => $action,}]),
        });
    }

    # принудительное удаление табов
    $phrase =~ s/[\t\n\r]/ /g unless grep {$action eq $_} qw(AddPrefilter DeletePrefilter);

    # добавление/удаление фразы
    if (grep{$_ eq $action} (qw/Add Delete AddPrefilter DeletePrefilter AddFlag DeleteFlag AddNephew DeleteNephew/)) {
        my $anti_action;
        $anti_action = 'Delete'             if $action eq 'Add';
        $anti_action = 'Add'                if $action eq 'Delete';
        $anti_action = 'DeletePrefilter'    if $action eq 'AddPrefilter';
        $anti_action = 'AddPrefilter'       if $action eq 'DeletePrefilter';
        $anti_action = 'DeleteFlag'         if $action eq 'AddFlag';
        $anti_action = 'AddFlag'            if $action eq 'DeleteFlag';
        $anti_action = 'AddNephew'          if $action eq 'DeleteNephew';
        $anti_action = 'DeleteNephew'       if $action eq 'AddNephew';

        my $where = "where
                Status != 'Done' and Status != 'Declined' and
                Action = '". $anti_action ."' and
                CatID = ".$self->dbh->quote($cat_id)." and
                InitialPhraseID = ".$self->dbh->quote($phrase_id)." and
                Language = ".$self->dbh->quote($lang);
        my $prev = $self->user_phrases->List_SQL("select * from CatalogiaPhrases $where");

        if(@$prev) {
            $self->user_phrases->Do_SQL("delete from CatalogiaPhrases $where");
            $add_action = grep{$_->{Login} ne $self->{login}} @$prev;
        }
    }

    if($add_action) {
        my $phinf = {
                Action          => $action,
                CatID           => $cat_id,
                InitialPhrase   => $phrase,
                InitialPhraseID => $phrase_id,
                Language        => $lang,
                Login           => $self->login,
                LastLogin       => $self->login,
                Comment         => $comment,
                UserComment     => $user_comment,
                CreateTime      => $self->curtime,
        };
        $self->user_phrases->Add( [$phinf],  { replace => 1} );
        # Изменения в маркерах проходят мимо модерации, для маркеров ставим сразу Accepted
        if ( $cat_id =~ /^amarker/ ) {
            $self->set_user_phrase_status_by_id(
                $cat_id,
                $phrase_id,
                $action,
                $lang,
                "Accepted",
                "автопринятие для маркеров"
            );
        }
        #спам-фразы редактируются модерацией директа и проходят мимо нашей модерации
        if ($action eq 'AddWideSpamPhrase' ) {
            $self->set_user_phrase_status_by_id(
                $cat_id,
                $phrase_id,
                $action,
                $lang,
                "Accepted",
                "автопринятие спам-фраз"
            );
        }
    }

    # отражение изменений в subphraser-ax
    $self->update_all_subphrasers($cat_id, $phrase, $action, $lang, "New");

    $self->log_action($cat_id, "AddPhrase", $phrase);
}

sub set_user_phrase_done_status_by_id {
    my ($self, $cat_id, $phrase_id, $action, $lang, $status, $comment) = @_;

    return if $status ne 'Done';

    $self->Do_SQL("
        update
            ". $self->user_phrases->db_table. "
        set
            ".(defined($comment) ? "Comment = ".$self->dbh->quote($comment)."," : "" )."
            Status = ".$self->dbh->quote($status).",
            PreStatus = " . $self->dbh->quote($status) . "
        where
            CatID = ".$self->dbh->quote($cat_id)." and
            Language = ".$self->dbh->quote($lang)." and
            Action = ".$self->dbh->quote($action)." and
            InitialPhraseID = ".$self->dbh->quote($phrase_id)
    );
}

sub set_user_phrase_status_by_id {
    my ($self, $cat_id, $phrase_id, $action, $lang, $status, $comment) = @_;
    my $item = $self->user_phrases->Get(
        InitialPhraseID     => $phrase_id,
        CatID               => $cat_id,
        Language            => $lang,
        Action              => $action
    );
    my $phrase = $item->{InitialPhrase};

    # если нужно, откатываем изменения subphraser
    $self->update_all_subphrasers($cat_id, $phrase, $action, $lang, $status) if $status eq 'Declined';

    # kostyl
    if (grep {$action eq $_} qw(Add Delete)) {
        $self->reset_category_phrase_block($cat_id, $lang);
    }
    # /kostyl

    $self->Do_SQL("
        update ".$self->user_phrases->db_table."
        set
            ".(defined($comment) ? "Comment = ".$self->dbh->quote($comment)."," : "" )."
            LastLogin = ".$self->dbh->quote($self->login).",
            Status = ".$self->dbh->quote($status).",
            PreStatus = " . $self->dbh->quote($status) . "
        where
            CatID = ".$self->dbh->quote($cat_id)." and
            Language = ".$self->dbh->quote($lang)." and
            Action = ".$self->dbh->quote($action)." and
            InitialPhraseID = ".$self->dbh->quote($phrase_id)
    );

    $self->log_action($cat_id, "Phrase$status", $phrase);
}

sub set_user_phrase_status {
    my ($self, $cat_id, $phrase, $action, $lang, $status, $comment) = @_;
    my $phrase_id = $self->get_user_phrase_id($phrase);

    if ($self->user->rights->{right_moderate}) {
        $self->set_user_phrase_status_by_id($cat_id, $phrase_id, $action, $lang, $status, $comment);
    } elsif ($self->user->rights->{right_moderate_media}) {
        $self->set_user_phrase_media_status_by_id($cat_id, $phrase_id, $action, $lang, $status, $comment);
    } else {
        # TODO no rights
    }
}

sub set_user_phrase_media_status_by_id {
    my ($self, $cat_id, $phrase_id, $action, $lang, $media_status, $media_comment) = @_;

    my $user_phrases_table = $self->user_phrases->db_table;
    $self->Do_SQL("
        update
            $user_phrases_table
        set
            MediaStatus = ?, MediaPreStatus = ?, LastMedia = ?, MediaComment = ?
        where
           CatID = ? and Language = ? and Action = ? and InitialPhraseID = ?
    ", [$media_status, $media_status, $self->login, $media_comment, $cat_id, $lang, $action, $phrase_id]);

    $self->set_user_phrase_status_by_id($cat_id, $phrase_id, $action, $lang, $media_status, $media_comment) if $media_status eq 'Declined';
}

sub get_initial_phrase_by_id {
    my ($self, $cat_id, $phrase_id, $lang, $action) = @_;
    my $item = $self->user_phrases->Get(
        InitialPhraseID     => $phrase_id,
        CatID               => $cat_id,
        Language            => $lang,
        Action              => $action
    );

    return $item ? $item->{InitialPhrase} : "";
}

sub cmdslist {
    return Cmds::Base::_cmds_list();
}

sub cmdshlist {
    return Cmds::Base::_cmds_h_list();
}

sub getcmdh { #Получить хэш настроек для команд, описанных как CMDH
    my ($self, $name) = @_;
    return $self->cmdshlist->{$name}->($self, {});
}

sub cronslist {
    return Cmds::Base::_crons_list();
}

sub cronlight_tasks { return Cmds::Base::_specprojfuncs_hash()->{'CRONLIGHTTASKS'}; }

sub cronlight_workers { return Cmds::Base::_specprojfuncs_hash()->{'CRONLIGHTWORKER'}; }


sub _qtestslist { #Возвращаем массив проверок в зависимости от хоста
    my ($self) = @_;
    my $qtl = ObjLib::Attr::_qtests_list(); #Список всех возможных проверок
    my $hostrules = ObjLib::Attr::_qtests_rules()->{$self->host_role} || [];
    return map { [ $_, $qtl->{$_} ] } @$hostrules; #Возвращаем массив функций проверки
}

sub qtests {
    my ($self) = @_;
    my @list = $self->_qtestslist;
    print "QTESTS ".$self->host_role."\n";
    for my $qtel (@list){
        my ($name, $qt) = @$qtel;
        my $res = $qt->($self);
        print "STATUS $name " . $res->{status} . "\n";
    }
}

sub cmdloggerfile {
    my ($self) = @_;
    my $logdir = $self->options->{'cmdlog'};
    -d $logdir || mkdir $logdir;
    my $logfile = $logdir."/cmdlog$$";
    return $logfile;
}

sub create_cmdlog {
    my ($self) = @_;
    my $logfile = $self->cmdloggerfile;
    $self->log("cmdlogger create $logfile");
    $self->{cmdlogger} = ObjLib::FileLogger->new({log_file => $logfile});
    return $self->{cmdlogger};
}

sub delete_cmdlog {
    my ($self) = @_;
    $self->{cmdlogger} = '';
    my $logfile = $self->cmdloggerfile;
    unlink($logfile);
    $self->log("cmdlogger delete $logfile");
}

sub cmdlogger {
    my ($self) = @_;
    $self->create_cmdlog unless $self->{cmdlogger};
    return $self->{cmdlogger};
}

sub cmdlog {
    my ($self, $text) = @_;
    my $dbg = 1;
    $self->log('cmdlog '.$text) if $dbg;
    $self->cmdlogger->info($text);
}

sub make_cmd {
    my ($self) = @_;

    my $debugtimeinf = "\n";
    $debugtimeinf .= "begin ".$self->curtime."\n";

    $DataSource::Elem::long_requests_logging_min_duration = 1;
    $self->log("long_requests_logging_min_duration: " . $DataSource::Elem::long_requests_logging_min_duration);

    my $time_begin = time;

    $self->timelogclear;

    $self->timelogbeg("make_cmd data");

    $self->create_cmdlog;

    $self->cmdlog("form");

    # сброс кэша
    delete $self->{deleted_categories};
    delete $self->{updated_categories_parents};

    my $form = $self->form;
    my $vars = { form => $form };

    $self->begin_session;

    #$vars->{'timelog'} = 1;
    $vars->{'timelog'} = 1 if $form->{'timelog'};

    $vars->{'show_debug_mode_warnings'} = $self->options->{'show_debug_mode_warnings'};

    if($self->user) {
        $vars->{UserEMail} = $self->user->user_inf->{EMail};
        $vars->{rights} = $self->user->rights;
    }

    # На bmfront-е доступ определяется доступом к балансеру "bmfront.bm.yandex-team.ru" и авторизацией по логинам "yandex-team.ru"
    if (! $vars->{rights}->{right_login}
        &&  $self->host_role ne 'bmfront'
    ) {
        print "Content-type: text/html; charset=UTF-8\n\n";
        print 'Ограничение прав';
        goto EXIT; #нужно, чтобы убрать прерывание работы программы под nginx'ом
    }

    $vars->{viewoptions} = $self->viewoptions;

    $self->current_lang($vars->{viewoptions}{lang}); #Принудительно устанавливаем дефолтный язык

    my @regions = grep{$_} split "_", ($vars->{viewoptions}{region} || "");
    $self->current_region([@regions]); # устанавливаем регион для получения частот

    #Если не указан cmd - пробуем подставить дефолтный
    if((!$form->{cmd}) && $self->user){
        $form->{cmd} = $self->user->default_cmd;
    }

    #Временный фикс редиректа
    if( ($self->host_role eq 'drf') && ($form->{cmd} eq 'edit_phrase_list')){
        $self->do_redirect('/ind.pl?cmd='.$self->user->default_cmd);
    }

    if( $form->{cmd} ){
        $debugtimeinf .= "beg cmd ".$self->curtime."\n";
        my $cmdh = $self->cmdslist;

        $self->cmdlog("cmd=".$form->{cmd} . " Login=".($self->login // '') );

        if( defined $cmdh->{$form->{cmd}} ){
            $self->timelogbeg('cmd '.$form->{cmd});
            #$cmdh->{$form->{cmd}}->($self, $vars);
            my $err = $self->catch_exception(sub { $cmdh->{$form->{cmd}}->($self, $vars); });
            if($err){
                $err = join("\n",
                    "Error at " . eval{ get_curr_host } . " [$$]",
                    $err,
                );
                print "Content-type: text/html\n\n";
                print "<h3>Internal Error 1939</h3><pre>$err</pre>";
                $self->log("ERROR: " . join(" ",
                        "cmd processing failed",
                        "(" . ($form->{cmd} // '') ." " . ($form->{act} // '') . ")",
                        (grep{$_} split /\n/, $err)[-1],
                ));
                $self->log("err: " . join(" // ", split /\n/, $err ));
                $self->log("Login=" . ($self->login // ''));
                $self->log("ENV: " . join(", ", map { "'$_' => '" . $ENV{$_} . "'" } sort keys %ENV));
                my $body = join("\n",
                    "cmd=" . ($form->{cmd} // ''),
                    "Login=" . ($self->login // ''),
                    "real_login=" . $self->{real_login},
                    "url=" . $ENV{HTTP_HOST} . $ENV{REQUEST_URI} . "",
                    "\nError:\n" . (length($err) <= 10000  ?  $err  :  substr($err, 0, 10000) . "  ...  [truncated]"),
                    "\nENV:\n" . Dumper(\%ENV),
                );
                if ($ENV{HTTP_HOST} =~ /^(cat|drf)/) {
                    $self->log("Sending error...");
                    $self->SendMail({
                        from => 'no_reply@yandex-team.ru',
                        to => 'bm-dev@yandex-team.ru',
                        subject => "Error in web-interface at " . $ENV{HTTP_HOST},
                        body => $body,
                    });
                    $self->log("Sent");
                }
                return;
            }
            $self->timelogend('cmd '.$form->{cmd});
        }
        $debugtimeinf .= "end cmd ".$self->curtime."\n";
    }

    $self->cmdlog("result");

    $vars->{viewoptionsstr} = $self->_join_viewoptions($vars->{viewoptions}||{});
    $self->timelogend("make_cmd data");
    $self->timelogbeg("make_cmd template");
    if($vars->{'_return_xls'}){
        print "Content-type: application/vnd.ms-excel\n";
        print "Content-disposition: attachment; filename=".$vars->{'_return_xls'}."\n";
        print "\n";
        print $vars->{text};
        goto EXIT; #нужно, чтобы убрать прерывание работы программы под nginx'ом
    }elsif($vars->{'_return_real_xls'}){
        binmode STDOUT, ':bytes';
        my $file = $vars->{'_return_real_xls'};
        print "Content-type: application/vnd.ms-excel\n";
        print "Content-disposition: attachment; filename=".$file."\n";
        print "\n";
        print $vars->{text};
        goto EXIT; #нужно, чтобы убрать прерывание работы программы под nginx'ом
    }elsif($vars->{'_return_textfile'}){
        binmode STDOUT, ':bytes';
        my $file = $vars->{'_return_textfile'};
        #print "Content-type: application/vnd.ms-excel\n";
        print "Content-disposition: attachment; filename=".$file."\n";
        print "\n";
        print $vars->{text};
        goto EXIT; #нужно, чтобы убрать прерывание работы программы под nginx'ом
    }elsif($vars->{'_return_feed'}){
        binmode STDOUT, ':bytes';
        my $file = $vars->{'_return_feed'};
        print "Content-type: application/vnd.ms-excel\n";
        print "Content-disposition: attachment; filename=".$file."\n";
        print "\n";
        print $vars->{text};
        goto EXIT; #нужно, чтобы убрать прерывание работы программы под nginx'ом
    }elsif($vars->{'_return_cp1251_xls'}){
        binmode STDOUT, ':bytes';
        print "Content-type: application/vnd.ms-excel\n";
        print "Content-disposition: attachment; filename=".$vars->{'_return_cp1251_xls'}."\n";
        print "\n";
        print u2w($vars->{text});
        goto EXIT; #нужно, чтобы убрать прерывание работы программы под nginx'ом
    }elsif($vars->{'_return_text'}){
        #Encode::_utf8_off($vars->{text});
        #*STDOUT->syswrite("Content-type: text/html; charset=UTF-8\nVary: Accept-Encoding\n\n");
        #*STDOUT->syswrite($vars->{text});
        #Encode::_utf8_off($vars->{text}) if $self->{fcgi_utf8_fix};
        #binmode STDOUT, ':bytes';
        binmode STDOUT, ':bytes';
        Encode::_utf8_off($vars->{text});
        print "Content-type: text/html; charset=UTF-8\n\n";
        print $vars->{text};
        return;
    }elsif($vars->{_return_json}){
	print "Content-type: application/json; charset=UTF-8\n\n";
        $vars->{text} =~ s/^\{/\{"жжжжстатус":3,/g; # this is a spike to kill FCGI utf-8 known bug
        print $vars->{text};
        return;
    }elsif( $vars->{'Content-type'} ){
        print "Content-type: ".$vars->{'Content-type'}."\n\n";
        print $vars->{text};
        goto EXIT; #нужно, чтобы убрать прерывание работы программы под nginx'ом
    }elsif(! $vars->{template}){
        print "Content-type: text/html; charset=UTF-8\n\n";

        print '<pre>'.$vars->{text}.'</pre>' if $vars->{text};
    }elsif( $vars->{template} ){
        print "Content-type: text/html; charset=UTF-8\n\n";

        $self->cmdlog("show_cmd_template beg");

        $debugtimeinf .= "beg tmpl ".$self->curtime."\n";
        $self->show_cmd_template($vars);
        $debugtimeinf .= "end tmpl ".$self->curtime."\n";
        #$self->dd($debugtimeinf);# if $vars->{form}->{debugtmpllog};
        $self->cmdlog("show_cmd_template end");
    }
    $self->timelogend("make_cmd template");
    print '<pre>'.$self->timelogreport.'</pre>' if $vars->{'timelog'};
    EXIT:
    $self->delete_cmdlog;
    $self->end_session;
    my $mem = int( mem_usage() / (1024 * 1024));
    my $time_elapsed = time - $time_begin;
    $self->log("EXIT  (time elapsed:$time_elapsed mem:$mem)  Login=" . ($self->login // '') . " url=" . $ENV{HTTP_HOST} . $ENV{REQUEST_URI} . "");
    return 1;
}

sub make_cron {
    my ($self, $cronname) = @_;

    my $cronh = $self->cronslist;

    if( defined $cronh->{$cronname} ){
        $self->timelogbeg('cron '.$cronname);
        my $err = $self->catch_exception(sub { $cronh->{$cronname}->($self); });
        if($err){
            print_err("ERROR: cron $cronname: $err");
            return;
        }
        $self->timelogend('cron '.$cronname);
    }
}

sub xls2arr {
    my ($self, $data, $filename) = @_;
    my ($res, $shls) = (undef, undef);
    if($filename =~ /\.xls$/){
       eval {
           ($res, $shls) = xls2array_allsheets($data);
       };
       if( $@ ){
           #$self->dd("ERROR".$@);
           eval {
               ($res, $shls) = xlsx2array_allsheets($data);
           };
           if( $@ ){
               return undef;
           }
           return $res;
       }
       return $res;
    }elsif($filename =~ /\.xlsx$/){
        ($res, $shls) = xlsx2array_allsheets($data);
        return $res;
    }
    return undef;
}

sub load_phl_from_form_file {
    my ($self, $filefield) = @_;
    my $form = $self->form;
    my $tt = $form->{$filefield};
    if($form->{"${filefield}_filename"} =~ /\.xls$/){
       # $tt = join("\n", map { $dch->text2utf8($_) } xls2array($tt));
       eval {
           $tt = join(",", xls2array($tt));
       };
       if( $@ ){
           my $dch = $self->detect_charset;
           $tt = $dch->text2utf8($tt);
           $tt = join(",", csv2array($tt));
       }
    }elsif($form->{"${filefield}_filename"} =~ /\.xlsx$/){
        $tt = join(",", xlsx2array($tt));
        my $dch = $self->detect_charset;
        $tt = $dch->text2utf8($tt);
        Encode::_utf8_on($tt);
    }elsif($form->{"${filefield}_filename"} =~ /\.csv$/){
        my $dch = $self->detect_charset;
        $tt = $dch->text2utf8($tt);
    #print "Content-type: text/html\n\n";
    #print "$tt\n";
    #exit;
        $tt = join(",", csv2array($tt));
    }else{ #Для всех остальных ожидаем, что это просто текстовый файл
        $tt =~ s/\n/,/g;
    }
    $tt = $self->detect_charset->text2utf8($tt);

    my $phl = $self->phrase_list($tt);
    return $phl;
}

sub phl2xls {
    my ($self, $phl) = @_;
    my @arr = map {$_->text} @$phl;
    return array2xls(\@arr);
}

sub save_phrase_list {
    my ($self, $phl) = @_;
    $self->log("save_phrase_list ...");
    $self->log("phl count: " . $phl->count);

    $self->log("big_sql_data...");
    $self->big_sql_data;

    my $partsize = 20000;

    if ( $phl->count > $partsize ) {
        my $parent_cache_id = $phl->cache_id;
        my @child_cache_ids;
        #делим список на чанки, каждый чанк сохраняем отдельно. Список всех чанков сохраняем в виде отдельного phraselist-а. Достаем из базы аналогично.
        foreach my $chunk ( $phl->split_by_count($partsize) ) {
            push @child_cache_ids, $chunk->cache_id;
            $self->_save_phrase_list_chunk($chunk);
        }
        $self->phrase_list_cache->Add(
            {
                PhraseListID    => $parent_cache_id,
                Phrases         => join( ",", @child_cache_ids ),
                IsComposite     => 1,
            },
            { replace => 1}
        );
    }
    else {
        $self->_save_phrase_list_chunk($phl);
    }

    $self->log("save_phrase_list done");
}

sub _save_phrase_list_chunk {
    my ($self, $phl) = @_;

    $self->log("txt...");
    my $txt = join(",", map{$_->text} $phl->phrases);
    #$self->log("txt: $txt");

    $self->log("encode...");
    $txt = Encode::encode('utf-8', $txt);
    $self->log("_utf8_on");
    Encode::_utf8_on($txt);

    #MySQL понимает только трехбайтовый юникод, поэтому обрежем все 4+-байтовые символы
    $txt =~ s/[\N{U+10000}-\N{U+7FFFFFFF}]/ /g;

    $self->log("cache_id...");
    my $cache_id = $phl->cache_id;
    $self->log("Add...");
    $self->phrase_list_cache->Add(
        {
            PhraseListID    => $cache_id,
            Phrases         => $txt,
        },
        { replace => 1}
    );
}

sub get_phrase_list_text {
    my ($self, $id) = @_;

    my $phlinf = $self->phrase_list_cache->Get("PhraseListID", $id);
    return undef unless $phlinf;

    my $text;
    my @ids_to_update = ($id);
    if ( $phlinf->{IsComposite} ) {
        my @child_cache_ids = split /,/, $phlinf->{Phrases};
        my @chunk_texts;
        for my $child_cache_id ( @child_cache_ids ) {
            push @chunk_texts, $self->get_phrase_list_text( $child_cache_id );
            push @ids_to_update, $child_cache_id;
        }
        $text = join ',', @chunk_texts;
    }
    else {
        $text = $phlinf->{Phrases};
    }

    $self->update_phrase_list_access_time(\@ids_to_update);

    return $text;
}

sub update_phrase_list_access_time {
    my ($self, $ids) = @_;
    return undef unless $ids && scalar(@$ids);

    $self->phrase_list_cache->Do_SQL("
        update " . $self->phrase_list_cache->db_table . "
        set AccessTime = now()
        where PhraseListID in ( " . join(", ", map { "'$_'" } @$ids) . " )
    ");
}

sub get_phrase_list {
    my ($self, $id) = @_;
    my $text = $self->get_phrase_list_text($id);
    return $self->phrase_list unless $text;
    return $self->phrase_list($text);
}

sub debug_head {
    my ($self) = @_;
    unless( $self->{dbgprints}++ ){
        print "Content-type: text/html; charset=UTF-8\n\n<pre>\n\n\n</pre>";
    }
}

sub debug_print {
    my ($self, $text) = @_;
    $self->debug_head;
    print "<pre>\n$text\n</pre>";
}

sub debug_dumper {
    my ($self, $data, %prm) = @_;
    $self->debug_head;
    my $text = $self->dump_lite($data);
    $text = HTML::Entities::encode_entities($text) if $prm{encode_html};
    print "<pre>\n$text\n</pre>";
}

sub dd { my $self = shift; $self->debug_dumper(\@_) }
sub ddhtml { my $self = shift; $self->debug_dumper(\@_, encode_html => 1) }
sub ddst { my $self = shift; $self->debug_dumper([@_, $self->stack_trace]) }

sub add_phrase_list_action {
    my ($self, $data) = @_;
    $data->{'login'} = $self->login unless defined $data->{'login'};
    $data->{'date'} = $self->dates->cur_date('db_time');
    return $self->phrase_list_action_log->Add(
        $data,
        { replace => 1}
    );
}

sub viewoptions {
    my ($self) = @_;
    return $self->{'viewoptions'};
}

sub _parse_viewoptions {
    my ($self, $text) = @_;

    my %res = ();

    foreach (split(/\|/, $text)) {
        my ($name, $value) = split('_', $_, 2);
        $res{$name} = $value if $name;
    };

    return \%res;
};

sub _join_viewoptions {
    my ($self, $h) = @_;
    my $str = join '|', map { $_."_".$h->{$_} } keys %$h;
    return $str;
};

sub search_categories {
    my ($self, $text, $lang) = @_;

    $lang ||= 'ru';
    my $stext = $text;
    my $trstext = $self->phrase($stext)->poor_layout;
    my $list = $self->dbtable('CatalogiaDict')->List({ search => {
        'CatID LIKE' => '%'.$stext.'%',   'CategoryName LIKE' => '%'.$stext.'%',
        'CatID  LIKE' => '%'.$trstext.'%', 'CategoryName  LIKE' => '%'.$trstext.'%',  #Наличие пробелов важно, чтобы не склеились ключи
    }, Language => $lang, }, [qw{CatID CategoryName}]);
    $list = [ sort { $a->{'CatID'} cmp $b->{'CatID'} }  grep { $_->{'CatID'} !~ /^(\.|__homonym_)/ } @$list ];

    return $list;
}

sub search_categs {
    my ($self, $text, $lang) = @_;

    $lang ||= $self->current_lang;

    my $stxt = '%'.$text.'%';
    my $list = $self->List_SQL("
        select
            CatID,
            ParentID,
            CategoryName,
            IsAtom,
            group_concat(fl.Flag) Flags
        from
            CatalogiaBrief
        left join
            CatalogiaDict
            using(CatID)
        left join
            CatalogiaFlags fl
            using(CatID)
        where
                not IsAtom
            and
                Language = ?
            and
                (     CategoryName LIKE ?
                   or
                      CatID LIKE ?
                )
        group by
            CatID
        ", [ $lang, $stxt, $stxt ]);

    return $list;
}

#Возвращает соответствие медийных групп и категорий Каталогии
#Данные берутся из базы
sub categ2mediagroup_hash {
    my ($self) = @_;

    if(!$self->{categ2mediagroup_hash}) {
        my $h = {};

        push @{$h->{$_->{categ}} ||= []}, $_->{mediagroup} for map {@$_} $self->dbtable('categs_mediagroups')->List;

        $self->{categ2mediagroup_hash} = $h;
    }

    return $self->{categ2mediagroup_hash};
}

sub levenshtein_distance {
    my ($self, $source, $destination, $insert_cost, $delete_cost, $replace_cost) = @_;

    $insert_cost ||= 1;
    $delete_cost ||= 1;
    $replace_cost ||= 1;

    my $source_length = length $source;
    my $destination_length = length $destination;

    my @distance;
    my @path;
    $distance[0][0] = 0;
    for (my $source_preffix = 1; $source_preffix <= $source_length; ++$source_preffix) {
        $distance[$source_preffix][0] = $source_preffix * $delete_cost;
        $path[$source_preffix][0] = 'delete';
    }
    for (my $destination_preffix = 1; $destination_preffix <= $destination_length; ++$destination_preffix) {
        $distance[0][$destination_preffix] = $destination_preffix * $insert_cost;
        $path[0][$destination_preffix] = 'insert';
    }
    for (my $source_preffix = 1; $source_preffix <= $source_length; ++$source_preffix) {
        for (my $destination_preffix = 1; $destination_preffix <= $destination_length; ++$destination_preffix) {
            # replace or match
            my $add = substr($source, $source_preffix - 1, 1) eq substr($destination, $destination_preffix - 1, 1) ? 0 : $replace_cost;
            $distance[$source_preffix][$destination_preffix] = $distance[$source_preffix - 1][$destination_preffix - 1] + $add;
            $path[$source_preffix][$destination_preffix] = $add ? 'replace' : 'match';
            # insert
            if ($distance[$source_preffix][$destination_preffix] > $distance[$source_preffix][$destination_preffix - 1] + $insert_cost) {
                $distance[$source_preffix][$destination_preffix] = $distance[$source_preffix][$destination_preffix - 1] + $insert_cost;
                $path[$source_preffix][$destination_preffix] = 'insert';
            }
            # delete
            if ($distance[$source_preffix][$destination_preffix] > $distance[$source_preffix - 1][$destination_preffix] + $delete_cost) {
                $distance[$source_preffix][$destination_preffix] = $distance[$source_preffix - 1][$destination_preffix] + $delete_cost;
                $path[$source_preffix][$destination_preffix] = 'delete';
            }
        }
    }
    my $source_preffix = $source_length;
    my $destination_preffix = $destination_length;
    my $editing_sequence = '';
    while ($source_preffix || $destination_preffix) {
        if ($path[$source_preffix][$destination_preffix] eq 'insert') {
            $editing_sequence = 'i' . $editing_sequence;
            --$destination_preffix;
        } elsif ($path[$source_preffix][$destination_preffix] eq 'delete') {
            $editing_sequence = 'd' . $editing_sequence;
            --$source_preffix;
        } else {
            $editing_sequence = ($path[$source_preffix][$destination_preffix] eq 'match' ? 'm' : 'r') . $editing_sequence;
            --$source_preffix;
            --$destination_preffix;
        }
    }
    return {
        distance            => $distance[$source_length][$destination_length],
        editing_sequence    => $editing_sequence,
    };
}

sub levenshtein_transformation {
    my ($self, $source, $destination, $insert_cost, $delete_cost, $replace_cost) = @_;

    my $editing_sequence = $self->levenshtein_distance($source, $destination, $insert_cost, $delete_cost, $replace_cost)->{'editing_sequence'};
    my $source_index = 0;
    my $destination_index = 0;
    my @source = split //, $source;
    my @destination = split //, $destination;
    my @result = ();
    for (split //, $editing_sequence) {
        if ($_ eq 'm') {
            push @result, [$source[$source_index], 'm'];
            ++$source_index;
            ++$destination_index;
        } elsif ($_ eq 'r') {
            push @result, [$destination[$destination_index], 'r'];
            ++$source_index;
            ++$destination_index;
        } elsif ($_ eq 'i') {
            push @result, [$destination[$destination_index], 'i'];
            ++$destination_index;
        } elsif ($_ eq 'd') {
            push @result, [$source[$source_index], 'd'];
            ++$source_index;
        } else {
        }
    }
    my @packed_result = ();
    my $previous_action = '';
    for (@result) {
        if ($previous_action ne $_->[1]) {
            $previous_action = $_->[1];
            push @packed_result, ['', $previous_action];
        }
        $packed_result[-1][0] .= $_->[0];
    }
    return \@packed_result;
}

sub get_tracking_updates_phrases {
    my ($proj, $cat_id, $lang, $initial_phrase_id, $action, $algorithm) = @_;
    $proj->log("get_tracking_updates_phrases ($cat_id, $lang, $initial_phrase_id, $action, $algorithm) ...");

    my $phrase_info = $proj->List_SQL("
        select
            *
        from
            " . $proj->user_phrases->db_table . "
        where
            CatID = ? and Action = ? and Language = ? and InitialPhraseID = ?
    ", [$cat_id, $action, $lang, $initial_phrase_id])->[0];

=h
    my $candidates_raw = $proj->List_SQL("
        select
            *
        from
            " . $proj->user_phrases->db_table . "
        where
            Action = 'Add' and Status != 'Done' and
            ABS(TIMESTAMPDIFF(HOUR, CreateTime, ?)) < 3
    ", [$phrase_info->{CreateTime}]);
=cut
    my $anti_action = $action eq 'Add' ? 'Delete' : 'Add';
    my $candidates_raw = $proj->List_SQL("
        select
            *
        from
            " . $proj->user_phrases->db_table . "
        where
            Action = ? and Status != 'Done' and
            ABS(TIMESTAMPDIFF(HOUR, CreateTime, ?)) < 3
    ", [$anti_action, $phrase_info->{CreateTime}]);

    my $h = $proj->get_language($phrase_info->{Language})->phrase($phrase_info->{InitialPhrase})->normwordshash;
    my @candidates = ();
    $proj->log("candidates ...");
    for my $candidate (@$candidates_raw) {
        my @norm_words = $proj->get_language($candidate->{Language})->phrase($candidate->{InitialPhrase})->normwords;
        push @candidates, $candidate if grep {$h->{$_}} @norm_words;
    }
    $proj->log("candidates: " . scalar @candidates);

    my $func = sub {
        my ($str) = shift;
        my $result = {};
        for (split //, $str) {
            ++$result->{$_};
            ++$result->{total};
        }
        return $result;
    };

    $proj->log("LevenshteinDistance ...");
    $_->{LevenshteinDistance} = $proj->levenshtein_distance($phrase_info->{InitialPhrase}, $_->{InitialPhrase}) for @candidates;
    $proj->log("grep candidates ...");
    @candidates = grep {my $res = $func->($_->{LevenshteinDistance}{editing_sequence}); $res->{r} <= 5 && $res->{m} * 3 >= $res->{total}} @candidates;
    $proj->log("sort candidates ...");
    @candidates = sort {$a->{LevenshteinDistance}{distance} <=> $b->{LevenshteinDistance}{distance}} @candidates;
    unshift @candidates, $phrase_info;

    $proj->log("get_tracking_updates_phrases ($cat_id, $lang, $initial_phrase_id, $action, $algorithm) donyye");
    return \@candidates;
}

sub get_image_nephews {
    my ($self, $cat_id, $lang) = @_;

    my $cat_name = $self->get_category_name($cat_id);
    my %h = map {$_ => 1} $self->categs_tree->get_image_nephews($cat_name);
    my @added_nephews = @{$self->List_SQL("
        select
            InitialPhrase
        from
            " . $self->user_phrases->db_table . "
        where
            CatID = ? and Status in ('New', 'Accepted') and Action = 'AddNephew'
    ", [$cat_id])};
    @added_nephews = map {$self->get_category_name($_->{InitialPhrase})} @added_nephews;
    $h{$_} = 1 for @added_nephews;
    my @deleted_nephews = @{$self->List_SQL("
        select
            InitialPhrase
        from
            " . $self->user_phrases->db_table . "
        where
            CatID = ? and Status in ('New', 'Accepted') and Action = 'DeleteNephew'
    ", [$cat_id])};
    @deleted_nephews = map {$self->get_category_name($_->{InitialPhrase})} @deleted_nephews;
    $h{$_} = 0 for @deleted_nephews;
    my @cat_id_nephews = sort map {$self->get_category_by_name($_, 'ru')->{CatID}} grep {$h{$_}} keys %h;
    return map {
        {catID => $_, catName => $self->get_category_name($_, $lang),}
    } @cat_id_nephews;
}

sub get_domain_nephews {
    my ($self, $cat_id, $lang) = @_;

    my $cat_name = $self->get_category_name($cat_id);
    my %h = map {$_ => 1} $self->categs_tree->get_domain_nephews($cat_name);
    my @added_nephews = @{$self->List_SQL("
        select
            CatID
        from
            " . $self->user_phrases->db_table . "
        where
            InitialPhrase = ? and Status in ('New', 'Accepted') and Action = 'AddNephew'
    ", [$cat_id])};
    @added_nephews = map {$self->get_category_name($_->{CatID})} @added_nephews;
    $h{$_} = 1 for @added_nephews;
    my @deleted_nephews = @{$self->List_SQL("
        select
            CatID
        from
            " . $self->user_phrases->db_table . "
        where
            InitialPhrase = ? and Status in ('New', 'Accepted') and Action = 'DeleteNephew'
    ", [$cat_id])};
    @deleted_nephews = map {$self->get_category_name($_->{CatID})} @deleted_nephews;
    $h{$_} = 0 for @deleted_nephews;
    my @cat_id_nephews = sort map {$self->get_category_by_name($_, 'ru')->{CatID}} grep {$h{$_}} keys %h;
    return map {
        {catID => $_, catName => $self->get_category_name($_, $lang),}
    } @cat_id_nephews;
}

sub deprecated_words_checker {
    my ($self) = shift;

    $self->{deprecated_words_checker} = Cmds::DeprecatedWordsChecker->new({
        proj    => $self,
    }) if !$self->{deprecated_words_checker};

    return $self->{deprecated_words_checker};
}

sub _edit_user_phrases_warnings_light {
    my ($proj, $phrases) = @_;

    for my $phrase (@$phrases) {
        my $warning = {};
        my $critical_warning = 0;
        if ($phrase->{Action} eq 'Add') {
            # синтаксис скобок
            if (!$proj->phrase($phrase->{InitialPhrase})->check_brackets_syntax) {
                $warning->{'Неверный синтаксис скобок'} = 1;
                $critical_warning = 1;
            }
            # неправильные символы внутри угловых скобок
            if(my ($bad_char) = $phrase->{InitialPhrase} =~ /<[^>\/,]*([\/,])/) {
                $warning->{"Недопустимый символ '$bad_char' внутри угловых скобок"} = 1;
                $critical_warning = 1;
            }
        }
        $phrase->{Warnings} = [keys %$warning];
        $phrase->{CriticalWarning} = $critical_warning;
    }

    return $phrases;
}

sub edit_user_phrases_warnings {
    my ($proj, $phrases, %options) = @_;

    $options{check_deprecated_words} //= 1;
    my $check_deprecated_words = $options{check_deprecated_words};

    return $proj->_edit_user_phrases_warnings_light($phrases) if $options{light};

    return [] if !@$phrases;

    my $lang = "ru";
    my %all_categories = ();
    my %deleted_snorm_phrases_hash = ();
    my %added_snorm_phrases_hash = ();
    for my $phrase (@$phrases) {
        $all_categories{$_} = 1 for $proj->get_language($phrase->{Language})->phrase($phrase->{InitialPhrase})->get_minicategs;
        if ($phrase->{Action} eq 'Delete') {
            my $snorm_text = $proj->get_language($phrase->{Language})->phrase($phrase->{InitialPhrase})->snorm_phr_uniq;
            $deleted_snorm_phrases_hash{$snorm_text} = 1;
        }
        elsif ($phrase->{Action} eq 'Add') {
            my $snorm_text = $proj->get_language($phrase->{Language})->phrase($phrase->{InitialPhrase})->snorm_phr_uniq;
            push @{$added_snorm_phrases_hash{$snorm_text}}, $phrase->{InitialPhrase};
        }

    }
    my @all_categories = sort {$a->{CategoryName} cmp $b->{CategoryName}} map {$proj->get_category_by_name($_, $lang)} keys %all_categories;
    for my $phrase (@$phrases) {
        my $phrase_object = $proj->get_language($phrase->{Language})->phrase($phrase->{InitialPhrase});
        if(!$phrase->{ID}) {
            $phrase->{ID} = 'phr_' . $proj->get_user_phrase_id($phrase->{InitialPhrase});
        }
        $phrase->{SnormPhrase} = $phrase->{InitialPhrase} !~ /\[/ ?
                                 $phrase_object->snorm_phr :
                                 "<i>атом</i>";
        $phrase->{CurrentCategories} = join "/", map {$proj->get_language($lang)->category_from_ru($_)} $phrase_object->get_minicategs;
        if(!$phrase->{Categories}) {
            $phrase->{Categories} = [ map {{CatID => $_->{CatID}, CategoryName => $_->{CategoryName}}} grep {$_->{CatID} ne $phrase->{CatID}} @all_categories ];
        }
    }
    my %antiwords_hash = ();
    for my $phr (@$phrases) {
        my $action = $phr->{Action};
        my $lang = $phr->{Language};
        my $cat_id = $phr->{CatID};
        my $category_name = $proj->get_category_name($cat_id);
        my $cat_name = $proj->get_category($cat_id, 'ru')->{CategoryName};
        my $phrase = $proj->get_language($lang)->phrase($phr->{InitialPhrase});

        my $warning = {};
        my $critical_warning = 0; # 1 => $warning содержит предупреждение, при котором фразу нельзя добавлять
                                  # 0 => в противном случае
        if ($action eq 'Add') {
            my $is_marker = $cat_id =~ /^amarker/;

            # синтаксис скобок
            if (!$phrase->check_brackets_syntax) {
                $warning->{'Неверный синтаксис скобок'} = 1;
                $critical_warning = 1;
            }

            if (my ($bad_symbol) = $phrase->text =~ /([^\w -<>{}\[\]\/\.,])/) {
                $warning->{"Недопустимый символ '$bad_symbol' для правила категории"} = 1;
                $critical_warning = 1;
            }

            # неправильные символы внутри угловых скобок
            if(my ($bad_char) = $phrase->text =~ /<[^>\/,]*([\/,])/) {
                $warning->{"Недопустимый символ '$bad_char' внутри угловых скобок"} = 1;
                $critical_warning = 1;
            }

            # предупреждение, что синтаксис поиска с фиксацией словоформ тут неуместен (такая фраза не будет работать)
            if( $phrase->text =~ /(\W|^)!\w/ ) {
                $warning->{"Фиксация словоформ восклицательным знаком не поддерживается"} = 1;
                $critical_warning = 1;
            }

            # именованные атомы
            my @atoms = $phrase->text =~ /\[([^\]]+)\]/g;
            my $has_non_empty_atoms = 0;
            for my $atom (@atoms) {
                my $is_empty = $atom =~ /\/$/;
                for my $categ_name (split "/", $atom) {
                    if($categ_name =~ /^\s*$/) {
                        $is_empty = 1;
                    } elsif($categ_name =~ /^\./) {
                        my $categ = $proj->get_category_by_name($categ_name, $lang);
                        if (!$categ) {
                            $warning->{"Именованный атом \"$categ_name\" не существует"} = 1;
                            $critical_warning = 1;
                        }
                        elsif ( $categ->{CatID} =~ /^amarker/ ) {
                            $warning->{"Использовать маркеры в качестве атомов запрещается"} = 1;
                            $critical_warning = 1;
                        }
                    }
                }
                $has_non_empty_atoms = 1 if !$is_empty;
            }

            # фраза не может быть пустой
            if(!$has_non_empty_atoms) {
                my $text_without_atoms = $phrase->text;
                $text_without_atoms =~ s/(\[[^\]]+\])/ /g;
                if(!$proj->phrase($text_without_atoms)->norm_phr) {
                    $warning->{"Фраза состоит из атомов, каждый из которых может быть пустым"} = 1;
                    $critical_warning = 1;
                }
            }
            # нельзя добавлять фразы с атомами в именованные атомы
            if ($phrase->text =~ /[\[\]]/ && $category_name =~ /^\./) {
                $warning->{'Добавление фразы с атомами в именованный атом недопустимо'} = 1;
                $critical_warning = 1;
            };

            # катализаторы
            my @catalysts = $phrase->text =~ /\{([^\}]+)\}/g;
            for my $categ_name (@catalysts) {
                $categ_name =~ s/^hier\s+//g;
                my $categ = $proj->get_category_by_name($categ_name, $lang);
                if (!$categ) {
                    $warning->{"Категория \"$categ_name\" не существует"} = 1;
                    $critical_warning = 1;
                }
            }

            # порно
            $warning->{"Порно"} = 1 if $phrase->is_porno_phrase;

            # широкая фраза
            $warning->{"Широкая фраза (" . ($phrase->widephr ? "содержится в словаре широких фраз" : "все слова широкие") .")"} = 1
                if $phrase->is_wide_phrase;

            # добавлена больше, чем в одну категорию
            my @contains_in = sort $phrase->get_exact_minicategs;
            # в случае атома интересует только категория, в которую добавляем
            @contains_in = grep {$_ eq $category_name} @contains_in if $category_name =~ /^\./;
            my $categories = $phrase->text =~ /\{|\}/ ? '' : join("/", @contains_in);
            if ($categories && !$deleted_snorm_phrases_hash{$phrase->snorm_phr_uniq}) {
                $warning->{"Уже содержится в категориях ($categories)"} = 1;
                $critical_warning = 1 if !$proj->user->rights->{right_moderate};
            }
            #повторы при добавлении списка фраз
            my $snorm_text = $phrase->snorm_phr_uniq;
            if ( exists $added_snorm_phrases_hash{$snorm_text} && scalar @{$added_snorm_phrases_hash{$snorm_text}} > 1 ) {
                my $once = 0;
                #если пользователь добавил несколько полностью одинаковых фраз, покажем ему список с правильным их количеством (все, кроме текущей)
                my $same_phrases = join ',', grep { $_ ne $phr->{InitialPhrase} || $once++ } @{$added_snorm_phrases_hash{$snorm_text}};
                $warning->{"Имеет одинаковый снорм с другими добавляемыми фразами: $same_phrases" } = 1;
                $critical_warning = 1 if !$proj->user->rights->{right_moderate};
            }

            # проверка на совместимость с антисловами
            my $no_named_atoms_phrase = $proj->get_language($lang)->phrase($phrase->erase_named_atoms);
            my $antiwords_phrase_list = $proj->phrase_list($proj->category_interface->get_antiwords($cat_id, $lang, 'apply_moderation'));
            my $suffers_from = $antiwords_phrase_list->search_subphrases_in_phrase($no_named_atoms_phrase);
            if (@$suffers_from) {
                $warning->{"Данная фраза противоречит антисловам: " . join(', ', map{"\"" . $_->text . "\""} @$suffers_from)} = 1;
                $critical_warning = 1;
            }
            # запрещенные слова
            if ($check_deprecated_words) {
                my $deprecated_words = $proj->deprecated_words_checker->find_deprecated_words($phrase->text, $cat_id, $lang);
                if (@$deprecated_words && $category_name !~ /^\.region/) {
                    my $deprecated_words_str = join '/', (sort @$deprecated_words);
                    $warning->{"Данная фраза содержит недопустимую широкую подфразу \"$deprecated_words_str\""} = 1;
                    unless($is_marker) { #для маркеров делаем исключение
                        $critical_warning = 1 if !$proj->user->rights->{right_moderate} ;
                    }
                }
            }
            # замена разлиных форм дефисов и тире на минус
            if ($phr->{InitialPhrase} =~ s/‒|–|—|―|‐/-/g) { # выглядит так, будто это все одно и то же тире, но это обман
                $warning->{"Выполнена принудительная замена тире на минус"} = 1;
            }
            # "висячий" минус
            # человек, написавший "купить слона - хобот", либо опечатался, либо не понимает что делает
            # но мы разрешаем правила вида "{Услуги - коммерческая аренда} Москва", минусы в именах категорий это ОК
            my $initial_phrase_without_categ_refs = $phr->{InitialPhrase} =~ s/\{[^\{\}]+\}//rg;
            if ($initial_phrase_without_categ_refs =~ m/-\s+/) {
                $warning->{"Фраза содержит 'висячий' минус"} = 1;
                $critical_warning = 1;
            }

            my @minuswords = $proj->get_language($phr->{'Language'})->phrase($phr->{'InitialPhrase'})->minuswords;
            if (@catalysts && @minuswords) {
                $warning->{"Минус слова не поддерживаются при использовании синтаксиса фигурных скобок"} = 1;
                $critical_warning = 1;
            }
            #проверки для маркеров
            if ( $is_marker ) {
                #в маркерах запрещается вся специальная разметка, кроме плюс-минус-слов
                if ( $phr->{InitialPhrase} =~ /[^\w +-]/ ) {
                    $warning->{"Фраза содержит недопустимый синтаксис"} = 1;
                    $critical_warning = 1;
                }
                my $initialphr = $proj->phrase(lc $phr->{InitialPhrase});
                my %initialhash = map { $_ => 1 } split /\s/, $initialphr->text();
                my @plusworddiff = grep {$_} map {$_ =~ s/\s//gr } map {"\"$_\""} grep {!exists $initialhash{$_}} split /\s/, $initialphr->text_delete_pluses_except_stop_words();

                if ( @plusworddiff ) {
                    $warning->{"Фраза содержит избыточные плюс-слова, не являющиеся стоп-словами: " . join(',',@plusworddiff) } = 1;
                    $critical_warning = 1;
                }
            }
        } elsif ($action eq 'AddAntiword') {
            if ($antiwords_hash{$phrase->snorm_phr_uniq}) {
                $warning->{"Данное антислово уже есть в этом списке"} = 1;
                $critical_warning = 1;
            }
            $antiwords_hash{$phrase->snorm_phr_uniq} = 1;
            if (grep {$phrase->text eq $_} @{$proj->category_interface->get_antiwords($cat_id, $lang, 'apply_moderation')}) {
                $warning->{'Данное антислово уже содержится в этой категории'} = 1;
                $critical_warning = 1;
            }
            my @category_phrases = @{$proj->category_interface->get_phrases($cat_id, $lang, 'apply_moderation')};
            my $antiword_phrase_list = $proj->phrase_list([$phrase]);
            my @suffered_phrases = ();
            for my $category_phrase (@category_phrases) {
                my $no_named_atoms_category_phrase = $proj->get_language($lang)->phrase($category_phrase)->erase_named_atoms;
                my $phrase_obj = $proj->get_language($lang)->phrase($no_named_atoms_category_phrase);
                push @suffered_phrases, $category_phrase if $antiword_phrase_list->search_subphrases_in_phrase($phrase_obj);
            }
            if (@suffered_phrases) {
                $warning->{"Данному антислову противоречат следующие слова категории: " . join(', ', map {"\"" . $_ . "\""} @suffered_phrases)} = 1;
                $critical_warning = 1;
            }
        }

        $phr->{Warnings} = [keys %$warning];
        $phr->{CriticalWarning} = $critical_warning;
        if (!@{$phr->{Warnings}}) {
            $phr->{Priority} = -1;
        } elsif (!$phr->{CriticalWarning}) {
            $phr->{Priority} = 0;
        } else {
            $phr->{Priority} = 1;
        }
    }
    $phrases = [sort {$a->{Priority} <=> $b->{Priority}} @$phrases];

    return $phrases;
}


# Преобразует $form в строку запроса типа 'act_data=List&cids=3305823&cmd=camp_stat&indicator=Shows%2C%20Clicks%2C%20Cost%2C%20CTR'
# На входе:
#   $form   Объект BaseForm или хэш
#   sign    1/0  Добавить в querystring подпись
sub form2querystring {
    my ($self, $form, %prm) = @_;
    return if not $form;
    return if ref($form) ne 'BaseForm'  and  ref($form) ne 'HASH';
    my @keys = grep {$_ ne '__arrayref' and $_ ne '__origin'} sort keys %$form;
    @keys = grep {$_ ne 'sign'} @keys   if $prm{sign};
    my $str = join("&", (map { URI::Escape::uri_escape_utf8($_) . "=" . URI::Escape::uri_escape_utf8($form->{$_}) } @keys));
    if ($prm{sign}) {
        $str .= "&sign=" . $self->get_sign(query => $str);
    }
    return $str;
}


# Преобразует $form в строку запроса
# На входе:
#   $form           Объект BaseForm или хэш
#   sign            1/0  Добавить в querystring подпись
#   http_host       Если не задано, используется $ENV{HTTP_HOST}
#   script_name     Если не задано, используется $ENV{SCRIPT_NAME}
sub form2url {
    my ($self, $form, %prm) = @_;
    return if not $form;
    return if ref($form) ne 'BaseForm'  and  ref($form) ne 'HASH';
    my $query_string = $self->form2querystring($form, sign => $prm{sign});
    my $url = join("",
        $prm{http_host} // $ENV{HTTP_HOST},
        $prm{script_name} // $ENV{SCRIPT_NAME},
        "?" . $query_string,
    );
    return $url;
}


# Создает подпись для строки запроса
# my $sign = $proj->get_sign(query => $query);   $query .= "&sign=$sign"
sub get_sign {
    my ($self, %prm) = @_;
    my $query = $prm{query} // '';
    my $host_role = $self->host_role;
    my $key = "HelloCatmediaWorld!";
    my $res = md5int(join(" ", $host_role, $query, $key));
    $self->log(join(", ", "get_sign", $host_role, $query, $res));
    return $res;
}

# Проверка подписанного url
# my $res = $proj->check_sign(query => $ENV{QUERY_STRING}) ? 'OK' : '';
sub check_sign {
    my ($self, %prm) = @_;
    my $query = $prm{query} // '';
    $query =~ m/&sign=(\w*)/;
    my $sign = $1  //  return;
    my $query_init = $query;
    $query_init =~ s/&sign=$sign//g;
    my $sign_correct = get_sign($self, query => $query_init)  //  return;
    my $res = ($sign eq $sign_correct) ? 1 : 0;
    return $res;
}

#Метод, который вызывается перед началом новой сессии
sub begin_session {
    my ($self) = @_;

    $self->{_session_cache} = {};

}

#Метод, который вызывается по завершению обработки сессии
sub end_session {
    my ($self) = @_;

}

##################################################################################
# Работа с ClientID
##################################################################################

sub uids2logins {
    my ($self, $uids) = @_;
    return [] unless $uids && @$uids;

    my $text = BlackboxSimple::userinfo_request("uid", join(',', @$uids));
    $text =~ s/\n//g;
    my $res = {};
    for my $t ( split('<user ',$text) ){
         my ($uid, $login);
         $uid=$1 if $t=~/<uid[^>]*>([^<]*)<\/uid>/;
         $login=$1 if $t=~/<login[^>]*>([^<]*)<\/login>/;
         $res->{$uid} = $login if $uid && $login;
    }
    return $res;
}

sub _logins2uids {
    my ($self, $logins) = @_;
    return {} unless $logins && @$logins;

    my $text = BlackboxSimple::userinfo_request("login", join(',', @$logins));
    $text =~ s/\n//g;
    #$self->dd(["logins2uids", $text ]);
    my $res = {};
    for my $t ( split('<user ',$text) ){
         my ($uid, $login);
         $uid=$1 if $t=~/<uid[^>]*>([^<]*)<\/uid>/;
         $login=$1 if $t=~/<login[^>]*>([^<]*)<\/login>/;
         $res->{$login} = $uid if $uid && $login;
    }
    return $res;
}

sub logins2uids {
    my ($self, $logins) = @_;
    return {} unless $logins && @$logins;
    my $res = {};
    for my $l (@$logins){
        my $cres = $self->_logins2uids([$l]);
        $res->{$_} = $cres->{$_} for keys %$cres;
    }
    return $res;
}

sub logins2userinf {
    my ($self, $logins) = @_;
    my $mpng =  {}; #соответствие измененных логинов и исходных
    my @narr = ();
    for my $l (@$logins){
        my ($login1, $login2) = ($l, $l);
        $login1 =~ s/\./-/g;
        $login2 =~ s/\-/./g;
        $mpng->{$_} = $l for $login1, $login2, $l;
        push(@narr, $l, $login1, $login2);
    }
    my $dbres = $self->direct_users->List({Login => \@narr});
    my %dblogins = map { $mpng->{$_->{Login}} => 1 } @$dbres; #фильтр для исходных логинов
    $_->{Login} = $mpng->{$_->{Login}} for @$dbres;
    my @newlogins = grep { ! $dblogins{$_} } @$logins;
    return @$dbres unless @newlogins;

    my @res = ();
    while(@newlogins){
        my @arr = splice(@newlogins, 0, 10);
        my $l2u = $self->logins2uids(\@arr);
        my @curres = map {{ Login => $_, uid => $l2u->{$_}  }} keys %$l2u;
        $self->direct_users->Add(\@curres, { replace => 1 }, );
        push(@res, @curres);
    }

    return @$dbres, @res;
}

sub uids2userinf {
    my ($self, $ids) = @_;
    my $dbres = $self->direct_users->List({uid => $ids});
    my %dbids = map { $_->{uid} => 1 } @$dbres;
    my @newids = grep { ! $dbids{$_} } @$ids;
    return @$dbres unless @newids;

    @newids = uniq @newids;
    my @res = ();
    while(@newids){
        my @arr = splice(@newids, 0, 10);
        my $u2l = $self->uids2logins(\@arr);
        my @curres = map {{ uid => $_, Login => $u2l->{$_}  }} keys %$u2l;
        $self->direct_users->Add(\@curres, { replace => 1 });
        push(@res, @curres);
    }

    return @$dbres, @res;
}

sub get_uid2login_hash {
    my ($self, $ids) = @_;
    my @arr = $self->uids2userinf($ids);
    #$self->dd(\@arr, $ids);
    return { map { $_->{uid} => $_->{Login} } @arr };
}

sub get_login2uid_hash {
    my ($self, $logins) = @_;
    my @arr = $self->logins2userinf($logins);
    #$self->dd(['get_login2uid_hash', \@arr, $logins]);
    return { map { $_->{Login} => $_->{uid} } @arr };
}

1;
