package CategoryInterface;

use strict;
use utf8;
use open ':utf8';
use Data::Dumper;
use POSIX qw(strftime);
use Utils::Sys;

use base qw(ObjLib::ProjPart);

sub get_flags {
    my ($self, $cat_id) = @_;

    my $proj = $self->proj;

    my $flags_str = ($proj->categories->Get($cat_id) || {})->{'Flags'};
    my %flags_hash = map {$_ => 1} split /,/, $flags_str;

    return [ keys %flags_hash ];
}

sub add_phrase_to_categ {
    my ($self, $phrase, $categ) = @_;
    my $proj = $self->proj;
    my $categ_id = $proj->categs_tree->get_minicateg_id($categ);

    return if !$categ_id;

    $proj->add_user_phrase($categ_id, $phrase->text, $phrase->language, "Add");
}

sub add_phrase {
    my $self = shift;
    my $category = shift;
    my $text = shift;
    my $lang = shift;

    my $proj = $self->proj;
    $lang //= 'ru';
    $category = $proj->get_category_id($category) || $category;
    $proj->add_user_phrase($category, $text, $lang, 'Add');
}

sub delete_phrase {
    my $self = shift;
    my $category = shift;
    my $text = shift;
    my $lang = shift;

    my $proj = $self->proj;
    $lang //= 'ru';
    $category = $proj->get_category_id($category) || $category;
    $proj->add_user_phrase($category, $text, $lang, 'Delete');
}

sub add_flag {
    my ($self, $cat_id, $flag) = @_;
    
    my $proj = $self->proj;
    $proj->add_user_phrase(
        $cat_id,
        $flag,
        'ru',
        'AddFlag'
    );
}

sub delete_flag {
    my ($self, $cat_id, $flag) = @_;
    
    my $proj = $self->proj;
    $proj->add_user_phrase(
        $cat_id,
        $flag,
        'ru',
        'DeleteFlag'
    );
}

sub get_categories_containing_flag {
    my ($self, $flag, $apply_moderation) = @_;

    my $proj = $self->proj;
    my $categories = $proj->categories->List([
        [Flags => 'regexp' => "(^|,)$flag(\$|,)"],
    ]);

    return [ sort map {$_->{CatID}}  @$categories ] if !$apply_moderation;

    my %cat_ids_hash = map {$_->{CatID} => 1}  @$categories;
    my $add_flag_users_phrases = $proj->user_phrases->List([
        [Action => 'AddFlag'],
        [Status => ['New', 'Accepted']],
        [InitialPhrase => $flag],
    ]);
    my $del_flag_users_phrases = $proj->user_phrases->List([
        [Action => 'DeleteFlag'],
        [Status => ['New', 'Accepted']],
        [InitialPhrase => $flag],
    ]);
    $cat_ids_hash{$_} = 1 for map {$_->{CatID}} @$add_flag_users_phrases;
    $cat_ids_hash{$_} = 0 for map {$_->{CatID}} @$del_flag_users_phrases;

    return [ sort grep {$cat_ids_hash{$_}} keys %cat_ids_hash ];
}

sub get_users_tags {
    my ($self, $cat_id) = @_;

    my $proj = $self->proj;
    my $users_tags = $proj->user_phrases->List([
        [CatID => $cat_id],
        [Status => ['New', 'Accepted']],
        [Action => 'AddFlag'],
        [InitialPhrase => 'like' => 'tag_%'],
    ]);
    $_->{InitialPhrase} =~ s/^tag_// for @$users_tags;

    return [ map {$_->{InitialPhrase}} @$users_tags ];
}

sub get_dict_tags {
    my ($self, $cat_id, $apply_moderation) = @_;

    my $proj = $self->proj;
    my $dict_flags_str = ($proj->categories->Get($cat_id) || {})->{'Flags'};
    my @dict_tags = map {s/^tag_//; $_} grep {/^tag_/} split /,/, $dict_flags_str;

    return \@dict_tags if !$apply_moderation;
    
    my $deleted_tags = $proj->user_phrases->List([
        [CatID => $cat_id],
        [Status => ['New', 'Accepted']],
        [Action => 'DeleteFlag'],
        [InitialPhrase => 'like' => 'tag_%'],
    ]);
    $_->{InitialPhrase} =~ s/^tag_// for @$deleted_tags;

    my %tags_hash = map {$_ => 1} @dict_tags;
    $tags_hash{$_} = 1 for @{$self->get_users_tags($cat_id)};
    $tags_hash{$_->{InitialPhrase}} = 0 for @$deleted_tags;

    return [ grep {$tags_hash{$_}} keys %tags_hash ];
}

sub get_tags {
    my ($self, $cat_id) = @_;

    my @tags = @{$self->get_users_tags($cat_id)};
    push @tags, @{$self->get_dict_tags($cat_id, 1)};

    return \@tags;
}

sub _get_phrases_from_db {
    my ($self, $cat_id, $lang) = @_;

    my $proj = $self->proj;
    my $category_info = $proj->categories_dict->List([
        [CatID => $cat_id],
        [Language => $lang],
    ])->[0];
    my $joined_dict_phrases = ($category_info || {})->{'CategoryPhrases'};
    my @dict_phrases = map {$_->text} @{$proj->phrase_list({phrases_text => $joined_dict_phrases})};

    return [sort @dict_phrases];
}

sub _get_phrases_from_subphraser {
    my ($self, $cat_id, $lang) = @_;

    my $proj = $self->proj;
    my $cat_name = $proj->get_category_name($cat_id);

    return $proj->get_language($lang)->phrase($cat_name)->get_category_phrases;
}

sub get_phrases {
    my ($self, $cat_id, $lang, $apply_moderation, %parameters) = @_;
    
    $parameters{source} //= 'subphraser';

    my $dict_phrases;
    if ($parameters{source} eq 'subphraser') {
        $dict_phrases = $self->_get_phrases_from_subphraser($cat_id, $lang);
    } elsif ($parameters{source} eq 'db') {
        $dict_phrases = $self->_get_phrases_from_db($cat_id, $lang);
    } else {
        # TODO unknown source 
    }

    return $dict_phrases if !$apply_moderation;
    
    my $proj = $self->proj;
    my @dict_phrases = @$dict_phrases;
    my %phrases_hash = map {$_ => 1} @dict_phrases;
    my $added_phrases = $proj->user_phrases->List([
        [CatID      => $cat_id],
        [Language   => $lang],
        [Action     => 'Add'],
        [Status     => ['New', 'Accepted']],
    ]);
    $phrases_hash{$_->{InitialPhrase}} = 1 for @$added_phrases;
    my $deleted_phrases = $proj->user_phrases->List([
        [CatID      => $cat_id],
        [Language   => $lang],
        [Action     => 'Delete'],
        [Status     => ['New', 'Accepted']],
    ]);
    $phrases_hash{$_->{InitialPhrase}} = 0 for @$deleted_phrases;

    return [sort grep {$phrases_hash{$_}} keys %phrases_hash];
}

sub get_antiwords {
    my ($self, $cat_id, $lang, $apply_moderation) = @_;

    my $proj = $self->proj;
    my $category_info = $proj->categories_dict->List([
        [CatID      => $cat_id],
        [Language   => $lang],
    ])->[0];
    my $joined_dict_antiwords = ($category_info || {})->{'MinusWords'};
    my @dict_antiwords = map {$_->text} @{$proj->phrase_list($joined_dict_antiwords)};

    return [sort @dict_antiwords] if !$apply_moderation;

    my %antiwords_hash = map {$_ => 1} @dict_antiwords;
    my $added_antiwords = $proj->user_phrases->List([
        [CatID      => $cat_id],
        [Language   => $lang],
        [Action     => 'AddAntiword'],
        [Status     => ['New', 'Accepted']],
    ]);
    $antiwords_hash{$_->{InitialPhrase}} = 1 for @$added_antiwords;

    return [sort grep {$antiwords_hash{$_}} keys %antiwords_hash];
}

sub update_flags_tagging_entry {
    my ($self, $bn, $entry) = @_;
    my $proj = $self->proj;
    my @categs = $bn->get_minicategs;
    my @flags = $bn->get_catalogia_flags;
    my $ctphs = [ map {{ category => $_->[0], phrase => $_->[1], }} map { @$_ } $bn->get_categs_phrases ];
    my $phrases = $bn->can('phrases') ? (join ", ", $bn->phrases) : ($bn->{all_phrases} || $bn->{phrases});
    my $new_data = {
        CategoriesString => join('/', @categs),
        CategPhrases => $proj->serial($ctphs),
        bid          => $bn->id,
        Flags       => join(",", @flags),
        Data        => $proj->serial([{
            map {
                if ( $_ eq 'phrases' ) {
                    'phrases' => $phrases;
                }
                else {
                    $_ => ''.$bn->{$_}
                }
            } qw{title title_extension body phrases url} }]),
        PrefilteredText   => ''.$bn->preprocess_title_body,
    };

    if(defined($entry->{Flags})) {
        my $timestamp = strftime("%Y-%m-%d", localtime);

        $new_data->{OldFlags} = $entry->{OldFlags} . "\n$timestamp\t" . $entry->{Flags};
    }

    $entry->{$_} = $new_data->{$_} for keys %$new_data;
}

sub create_flags_tagging_set {
    my ($self, $bnl, $set_id) = @_;
    my $proj = $self->proj;

    # баннеры
    my @result = ();
    for my $bn ( @$bnl ) {
        next if !$bn;

        my $entry = { SetID => $set_id };

        $self->update_flags_tagging_entry($bn, $entry);

        push @result, $entry;
    }
    my $dbt_tsb = $proj->dbtable('FlagsTaggingElems');
    $dbt_tsb->Add(\@result);
}

sub get_text_field_value {
    my ($self, $cat_id, $field, $lang) = @_;

    my $proj = $self->proj;
    $cat_id = $proj->get_category_id($cat_id) || $cat_id;
    $lang ||= 'ru';
    my $field_id = Utils::Sys::md5int($field);

    return $proj->categories_text_fields->List([
        [CatID      => $cat_id],
        [Language   => $lang],
        [KeyID      => $field_id],
    ])->[0]{'Value'};
}

sub add_text_field {
    my ($self, $cat_id, $field, $value, $lang) = @_;

    my $proj = $self->proj;
    $cat_id = $proj->get_category_id($cat_id) || $cat_id;
    $lang ||= 'ru';
    my $field_id = Utils::Sys::md5int($field);

    my $added_item = {
        CatID       => $cat_id,
        Language    => $lang,
        KeyID       => $field_id,
        Key         => $field,
        Value       => $value,
    };

    $proj->categories_text_fields->Add($added_item, {replace => 1});
}

1;
