package Cmds::CategoryAjaxHandle;

use strict;
use utf8;
use open ":utf8";

use base qw(Cmds::Base);

use JSON qw(to_json from_json);
use CatalogiaMediaProject;
use Utils::Array;
use Cmds::PhraseBlock;
use Cmds::NephewRelationBlock;
use Cmds::HierarchicalRelation;
use Cmds::Categs;
use URI::Escape qw(uri_escape_utf8 uri_unescape);

sub category_ajax_api : CMD {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    my $action = $form->{action};
    my $login = $proj->login;
    $proj->log("category_ajax_api: action=\"$action\", login=\"$login\"");
    if ($action eq 'setTranslate') {
        $vars->{text} = set_translate($proj, $vars);
    } elsif ($action eq 'changeParent') {
        $vars->{text} = change_parent($proj, $vars);
    } elsif ($action eq 'getFlagsList') {
        $vars->{text} = get_flags_list($proj, $vars);
    } elsif ($action eq 'setCategoryFlags') {
        $vars->{text} = set_category_flags($proj, $vars);
    } elsif ($action eq 'setCategoryDescription') {
        $vars->{text} = set_category_description($proj, $vars);
    } elsif ($action eq 'addCategoryComment') {
        $vars->{text} = add_category_comment($proj, $vars);
    } elsif ($action eq 'deleteCategoryComment') {
        $vars->{text} = delete_category_comment($proj, $vars);
    } elsif ($action eq 'editCategoryComment') {
        $vars->{text} = edit_category_comment($proj, $vars);
    } elsif ($action eq 'getCategoryChildren') {
        $vars->{text} = get_category_children($proj, $vars);
    } elsif ($action eq 'applyVirtualCategories') {
        $vars->{text} = apply_virtual_categories($proj, $vars);
    } elsif ($action eq 'handlePhraseBlock') {
        $vars->{text} = handle_phrase_block($proj, $vars);
    } elsif ($action eq 'getBannerSamples') {
        $vars->{text} = get_banner_samples($proj, $vars);
    } elsif ($action eq 'handleRelationBlock') {
        $vars->{text} = handle_relation_block($proj, $vars);
    } elsif ($action eq 'searchCategories') {
        $vars->{text} = search_categories($proj, $vars);
    } elsif ($action eq 'handleCategorySubscription') {
        $vars->{text} = handle_category_subscription($proj, $vars);
    } elsif ($action eq 'createCategoryTag') {
        $vars->{text} = create_category_tag($proj, $vars);
    } elsif ($action eq 'handleCategoryTags') {
        $vars->{text} = handle_category_tags($proj, $vars);
    } elsif ($action eq 'handlePhraseList') {
        $vars->{text} = handle_phrase_list($proj, $vars);
    } elsif ($action eq 'deleteCategory') {
        $vars->{text} = delete_category_with_user_comment($proj, $vars);
    } else {
        $vars->{text} =  JSON::to_json({
            status  => 'FAILED',
            comment => 'unknown action',
        });
    }
    $vars->{_return_json} = 1;
    $proj->log("/ category_ajax_api: action=\"$action\", login=\"$login\"");
}

# изменение родителя категории
sub change_parent {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    # id категории
    my $cat_id = $form->{catid};
    # id нового родителя
    my $parent_cat_id = $form->{parent_catid};

    my $parent_category = $proj->get_category_full($parent_cat_id);
    my $comment;
    # категория не существует
    $comment = "Category \"$parent_cat_id\" does not exist" if !$parent_category;
    # родитель самого себя
    $comment = "Category \"$cat_id\" can not be a parent of itself" if $cat_id eq $parent_cat_id;

    my $tree_upwalk_category = $proj->get_category_full($parent_category->{ParentID});
    while ($tree_upwalk_category) {
        # новый родитель является потомком
        if ($tree_upwalk_category->{CatID} eq $cat_id) {
            $comment = "Category \"$parent_cat_id\" is a child of category \"$cat_id\"";
            last;
        }
        $tree_upwalk_category = $proj->get_category_full($tree_upwalk_category->{ParentID});
    }

    my $user_comment = $form->{user_comment};

    $proj->add_user_phrase(
        $cat_id,
        $parent_cat_id,
        'ru',
        'ChangeParent',
        $comment,
        $user_comment
    ) if !$comment;

    my $result = {};
    $result->{status} = $comment ? 'FAILED' : 'OK';
    $result->{comment} = $comment if $comment;

    my $json_result = JSON::to_json($result);
    return $json_result;
}

#
sub delete_category_with_user_comment {
    my ($proj, $vars) = @_;

    my $catid = $proj->form->{id} // $proj->form->{catid};
    my $user_comment = $proj->form->{user_comment} // '';

    my $comment;
    if (!$catid) {
        $comment = "Category \"$catid\" does not exists!";
    }
    elsif (scalar @{$proj->get_category_children($catid)} > 0) {
        $comment = "Category \"$catid\" has children and cannot be deleted";
    }

    my $catname = $proj->get_category($catid, 'ru')->{CategoryName};
    $proj->add_user_phrase($catid, $catname, "ru", "DeleteCategory", $comment, $user_comment) if (!$comment);

    my $result = {};

    $result->{status} = $comment ? 'FAILED' : 'OK';
    $result->{comment} = $comment if $comment;

    my $json_result = JSON::to_json($result);
    return $json_result;
}

# изменение перевода категории
sub set_translate {
    my ($proj, $vars) = @_;

    my $form = $proj->{form};
    # id категории
    my $cat_id = $form->{catid};
    # новое имя для категории
    my $new_cat_name = $form->{new_name} || "";
    # язык, для которого изменяем имя
    my $lang = $form->{lang} || "ru";

    $proj->add_user_phrase(
        $cat_id,
        $new_cat_name,
        $lang,
        'RenameCategory'
    );

    return JSON::to_json({status => 'OK'});
}

# получение списка всех имеющихся флагов с описанием
sub get_flags_list {
    my ($proj, $vars) = @_;
    # получаем список из базы

    $proj->log("get_flags_list_full method");
    my @flags_list_full = @{$proj->get_flags_list_full};
    $proj->log("/ get_flags_list_full method");
    my $flags = [
        map{
            {
                flag        => $_->{Flag},
                description => $_->{Description} || '',
            }
        } @flags_list_full
    ];

    return JSON::to_json({
        status  => 'OK',
        result  => $flags,
    });
}

# выставление флагов категории
sub set_category_flags {
    my ($proj, $vars) = @_;

    my $form = $proj->{form};
    # id категории
    my $cat_id = $form->{catid};
    # комментарий пользователя
    my $user_comment = $form->{comment};
    # список добавляемых флагов
    my @flags_to_add = split /, /, $form->{'add[]'};
    @flags_to_add = uniq_array(@flags_to_add);
    # список удаляемых флагов
    my @flags_to_del = split /, /, $form->{'delete[]'};
    @flags_to_del = uniq_array(@flags_to_del);
    # хэш добавляемых флагов
    my $flags_to_add_hash = { map{$_ => 1} @flags_to_add };
    # хэш удаляемых флагов
    my $flags_to_del_hash = { map{$_ => 1} @flags_to_del };
    # убираем флаги, которые и добавляются, и удаляются
    @flags_to_add = grep{!$flags_to_del_hash->{$_}} @flags_to_add;
    @flags_to_del = grep{!$flags_to_add_hash->{$_}} @flags_to_del;

    # удаляем пробелы в начале и в конце комментария
    $user_comment =~ s/(^\s+|\s+$)//g;

    # запрещаем изменения технических флагов без комментария (кроме флага asocial, см. CATALOGIA-992#1519904138000)
    my @technical_flag_changes = grep { $_ !~ m/^_/ && $_ ne 'asocial' } (@flags_to_add, @flags_to_del);
    if (scalar(@technical_flag_changes) && !$user_comment) {
        return JSON::to_json({
            status    => 'FAILED',
            comment   => 'no user comment provided',
        });
    }

    # добавляем флаги
    $proj->add_user_phrase(
        $cat_id,
        $_,
        'ru',
        'AddFlag',
        undef,
        $user_comment
    ) for @flags_to_add;
    # удаляем флаги
    $proj->add_user_phrase(
        $cat_id,
        $_,
        'ru',
        'DeleteFlag',
        undef,
        $user_comment
    ) for @flags_to_del;
    # получаем список флагов из словарей
    my @dictionary_flags = map {
        {
            flag        => $_,
            isDeleted   => $proj->is_flag_deleted($cat_id, $_) ? 1 : 0,
        }
    } split /,/, $proj->phrase_list->web_get_category_full($cat_id, 'ru')->{'Flags'};
    # получаем список флагов с модерации
    my $user_flags = $proj->get_user_flags($cat_id);
    return JSON::to_json({
        status          => 'OK',
        dictionaryFlags => \@dictionary_flags,
        userFlags       => $user_flags,
    });
}

# редактирование информации о категории
sub set_category_description {
    my ($proj, $vars) = @_;

    my $form = $proj->{form};
    # id категории
    my $cat_id = $form->{catid};
    # язык, для которого выставляется описание категории
    my $lang = $form->{lang} || 'ru';
    # описание категории
    my $description = $form->{description} || '';
    $proj->add_category_description($cat_id, $lang, $proj->login, $description);
    return JSON::to_json({status => 'OK'});
}

# добавление комментария к категории
sub add_category_comment {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    # id категории
    my $cat_id = $form->{catid};
    # комментарий
    my $comment = $form->{comment};
    $proj->category_comments->Add({
       CatID    => $cat_id,
       Comment  => $comment,
       Login    => $proj->login,
    });
    my $comment_id = $proj->List_SQL("select last_insert_id()")->[0]{'last_insert_id()'};
    return JSON::to_json({
        status          => 'OK',
        commentID       => $comment_id,
        addCommentTime  => $proj->category_comments->Get($comment_id)->{'UpdateTime'},
    });
}

# удаление комментария к категории
sub delete_category_comment {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    # id комментария
    my $comment_id = $form->{comment_id};
    # я не боюсь SQL инъекций!
    $proj->Do_SQL("
        delete
            from " . $proj->category_comments->db_table . "
        where
            " . $proj->category_comments->id_field . "= ?
    ", [$comment_id]);
    return JSON::to_json({status => 'OK'});
}

# редактирование комментария к категории
sub edit_category_comment {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    # id комментария
    my $comment_id = $form->{comment_id};
    # новый комментарий
    my $comment = $form->{comment};
    $proj->category_comments->Edit($comment_id, {Comment => $comment});
    return JSON::to_json({
        status          => 'OK',
        editCommentTime => $proj->category_comments->Get($comment_id)->{'UpdateTime'},
    });
}

# получение списка детей категории
sub get_category_children {
    my ($proj, $vars) = @_;

    my $form= $proj->form;
    # id категории
    my $catid = $form->{catid} || '0';
    # язык, на котором возвращаются названия категорий
    my $lang = $form->{lang} || 'ru';
    my $children = $proj->get_category_children($catid);
    my $children_info = [];
    for my $child_id (map{$_->{CatID}} @$children) {
        my $child_info = $proj->get_category($child_id, $lang);
        push @$children_info, {catid => $child_id, categoryName => $child_info->{CategoryName}};
    }
    return JSON::to_json({
        status      => 'OK',
        children    => $children_info,
    });
}

# применить виртуальные категории
sub apply_virtual_categories {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    # id категории
    my $cat_id = $form->{catid};
    # список виртуальных категорий
    my @virtual_cat_ids = split /, /, $form->{'apply[]'};
    # список режимов: применить к категории (ApplyVirtual) или
    # применить к категории вместе с ее потомками (ApplyVirtualFull)
    my @modes = split /, /, $form->{'mode[]'};
    return JSON::to_json({
        status  => 'FAILED',
        comment => 'apply[] and mode[] have different sizes',
    }) if (@virtual_cat_ids != @modes);
    for (my $index = 0; $index < @virtual_cat_ids; ++$index) {
        my $virtual_cat_id = $virtual_cat_ids[$index];
        my $mode = $modes[$index];
        $proj->add_user_phrase($cat_id, $virtual_cat_id, 'ru', $mode);
    }
    return JSON::to_json({status => 'OK'});
}

sub get_phrase_block {
    my ($phrase_block_id, $proj, $cat_id, $lang) = @_;
    if ($phrase_block_id eq 'phraseBlock') {
        return Cmds::PhraseBlock->new({
            proj    => $proj,
            cat_id  => $cat_id,
            lang    => $lang,
        });
    }
}

# работа с блоком фраз
sub handle_phrase_block {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    # id блока фраз
    my $phrase_block_id = $form->{phraseBlockID};
    # id категории
    my $cat_id = $form->{catid};
    # язык
    my $lang = $form->{lang};
    # список фраз на добавление
    # kostyl again :(
    $form->{add} =~ s/\n/,/g;
    # $form->{add} =~ s/\+/ /g; # +13см vs 13см
    $form->{add} =~ s/&lt;/</g;
    $form->{add} =~ s/&gt;/>/g;
    $form->{add} =~ s/&quot;/"/g;
    my @added_phrases = @{$proj->phrase_list($form->{add})->perl_array};
    # /kostyl
    # список фраз на удаление
    my @deleted_phrases = @{$proj->phrase_list($form->{'delete[]'})->perl_array};
    my $phrase_block = get_phrase_block($phrase_block_id, $proj, $cat_id, $lang);
    # $phrase_block->add_phrase($_) for @added_phrases;
    # $phrase_block->delete_phrase($_) for @deleted_phrases;
    my $result = {};
    $result->{editingPhrasesWarnings} = $phrase_block->editing_phrases_warnings(\@added_phrases, \@deleted_phrases);
    # добавление/удаление фраз
    if ($form->{approvedPhrases}) {
        my $origin_form = $proj->form->{__origin};
        my $approved_phrases = JSON::from_json($origin_form->{approvedPhrases});
        for my $item (@$approved_phrases) {
            my $cat_id = $item->{catid};
            my $lang = $item->{lang};
            my $phrase_block = get_phrase_block($phrase_block_id, $proj, $cat_id, $lang);
            $phrase_block->add_phrase(
                HTML::Entities::decode_entities($_->{text}),
                HTML::Entities::decode_entities($_->{comment})) for @{$item->{add}};
            $phrase_block->delete_phrase(
                HTML::Entities::decode_entities($_->{text}),
                HTML::Entities::decode_entities($_->{comment})) for @{$item->{delete}};
        }
    }
    # обработка получения фраз
    if ($form->{getBlockPhrases}) {
        # параметры пагинации
        my $page_size = $form->{pageSize};
        my $page_index = $form->{pageIndex};
        $result->{getBlockPhrases} = $phrase_block->get_block_phrases($page_size, $page_index);
        $result->{totalBlockPhrasesCount} = $phrase_block->get_block_phrase_list->count;
    }
    # обработка поиска фраз
    my $search_query = $form->{searchBlockPhrases};
    $result->{searchBlockPhrases} = $phrase_block->search_block_phrases($search_query) if $search_query;
    $result->{status} = 'OK';
    return JSON::to_json($result);
}

sub get_relation_block {
    my ($relation_block_id, $proj, $cat_id) = @_;
    return Cmds::NephewRelationBlock->new({
        cat_id  => $cat_id,
        proj    => $proj,
    }) if $relation_block_id eq 'nephewRelationBlock';
}

sub handle_relation_block {
    my ($proj, $vars) = @_;

    my $lang = $vars->{viewoptions}{lang};
    my $form = $proj->form;
    # id блока отношений
    my $relation_block_id = $form->{relationBlockID};
    # id категории
    my $cat_id = $form->{catid};
    my $relation_block = get_relation_block($relation_block_id, $proj, $cat_id);
    my $result = {};
    # обработка получения отношений, в которых участвует категория cat_id
    # domain - отношения вида (x, cat_id)
    # image - отношения вида (cat_id, x)
    if ($form->{'addRelation[]'}) {
        my @cat_ids = split /, /, $form->{'addRelation[]'};
        $relation_block->add_relation(@cat_ids);
    }
    if ($form->{'deleteRelation[]'}) {
        my @cat_ids = split /, /, $form->{'deleteRelation[]'};
        $relation_block->delete_relation(@cat_ids);
    }
    if ($form->{getRelation}) {
        $result->{domain} = [ $relation_block->get_domain_categories($lang) ];
        $result->{image} = [ $relation_block->get_image_categories($lang) ];
    }
    $result->{status} = 'OK';
    return JSON::to_json($result);
}

sub get_banner_samples {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    # id категории
    my $cat_id = $form->{catid};
    # язык, по дефолту русский
    my $lang = $form->{lang} || 'ru';
    # количество баннеров, по дефолту 3
    my $count = $form->{count} || 3;
    return JSON::to_json({
        status  => 'FAILED',
        comment => 'no category id',
    }) if !$cat_id;
    my $active_only = $form->{active_only};
    my @banner_ids = split /,/, $proj->random_banners_client->k_random_banners($count, {lang => [$lang,], categ_ids => [$cat_id,], $active_only ? (active_flag=>[1,]) : ()  });
    my @banner_samples = ();
    for my $banner_id (@banner_ids) {
        my $banner = $proj->bf->get_banner_by_id($banner_id);
        push @banner_samples, {title => $banner->title, body => $banner->body, id => $banner_id, active_flag => $banner->active_flag, url => $banner->url};
    }
    return JSON::to_json({
        status          => 'OK',
        bannerSamples   => \@banner_samples,
    });
}

sub search_categories {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    # текст поиска
    my $text = $form->{text};
    # язык поиска, по дефолту русский
    my $lang = $form->{lang} || 'ru';
    return JSON::to_json({
        status      => 'OK',
        categories  => $proj->search_categories($text, $lang),
    });
}

sub handle_category_subscription {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    # id категории
    my $cat_id = $form->{catid};

    $proj->log("handle_category_subscription database part");
    $proj->category_subscriptions->Add(
        {CatID => $cat_id, Login => $proj->login, SubscriptionType => 'Single'},
        {ignore => 1}
    ) if ($form->{subscribeSingle});

    $proj->category_subscriptions->Add(
        {CatID => $cat_id, Login => $proj->login, SubscriptionType => 'Full'},
        {ignore => 1}
    ) if ($form->{subscribeFull});

    $proj->category_subscriptions->DelList(
        {CatID => $cat_id, Login => $proj->login, SubscriptionType => 'Single'}
    ) if ($form->{unsubscribeSingle});

    $proj->category_subscriptions->DelList(
        {CatID => $cat_id, Login => $proj->login, SubscriptionType => 'Full'}
    ) if ($form->{unsubscribeFull});
    $proj->log("/ handle_category_subscription database part");

    # список подписчиков
    my @subscribers = ();
    if ($form->{getSubscribers}) {
        my %subscribers_hash = map {
            $_ => 1
        } map {
            $_->{Login}
        } @{$proj->category_subscriptions->List([[CatID => $cat_id]])};
        my $parent_id = $proj->get_parent_id($cat_id);
        while ($parent_id) {
            my @current_subscribers = map {
                $_->{Login}
            } @{$proj->category_subscriptions->List([[CatID => $parent_id], [SubscriptionType => 'Full']])};
            $subscribers_hash{$_} = 1 for @current_subscribers;
            $parent_id = $proj->get_parent_id($parent_id);
        }
        @subscribers = keys %subscribers_hash;
    }

    # список родительских категорий, на поддеревья которых подписан пользователь
    my @subscribed_parents = ();
    if ($form->{getSubscribedParents}) {
        my $parent_id = $proj->get_parent_id($cat_id);
        while ($parent_id) {
            my $subscription = $proj->category_subscriptions->List({CatID => $parent_id, Login => $proj->login, SubscriptionType => 'Full'});
            push @subscribed_parents, $parent_id if @$subscription;
            $parent_id = $proj->get_parent_id($parent_id);
        }
    }
    @subscribed_parents = map {$proj->get_category_name($_)} @subscribed_parents;

    return JSON::to_json({
        status              => 'OK',
        subscribers         => \@subscribers,
        subscribedParents   => \@subscribed_parents,
    });
}

sub create_category_tag {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    my $tag = $form->{tag};
    my $description = URI::Escape::uri_unescape($form->{description});

    return JSON::to_json({
        status  => 'FAILED',
        comment => "tag's name must start with 'tag_'",
    }) if $tag !~ /^tag_/;

    return JSON::to_json({
        status  => 'FAILED',
        comment => "tag '$tag' already exists",
    }) if $proj->tags_description->Get($tag);

    $proj->tags_description->Add({Tag => $tag, Description => $description});

    return JSON::to_json({
        status => 'OK',
    });
}

sub handle_category_tags {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    my $result = {};
    my $cat_id = $form->{catid};
    my $categories_tags = $proj->categories_tags->List();
    my $hierarchical_relation = Cmds::HierarchicalRelation->new({
        proj => $proj,
        data => [map {[$_->{CatID}, $_->{Tag}, $_->{Type}]} @$categories_tags],
    });
    if ($form->{getTagsInfo}) {
        $result->{categoryInfluentialTags} = $hierarchical_relation->get_influential_properties($cat_id);
        $result->{categoryTags} = $hierarchical_relation->get_properties($cat_id);
        $result->{tagsList} = $proj->tags_description->List();
    }
    if ($form->{addTags}) {
        my $added_tags = JSON::from_json($form->{addTags});
        $proj->categories_tags->Add({
            CatID   => $cat_id,
            Tag     => $_->[0],
            Type    => $_->[1],
        }) for @$added_tags;
    }
    if ($form->{deleteTags}) {
        my $deleted_tags = JSON::from_json($form->{deleteTags});
        $proj->categories_tags->DelList({
            CatID   => $cat_id,
            Tag     => $_->[0],
            Type    => $_->[1],
        }) for @$deleted_tags;
    }

    return JSON::to_json({
        status => 'OK',
        %{$result},
    });
}

sub handle_phrase_list {
    my ($proj, $vars) = @_;

    my $origin_form = $proj->form->{__origin};
    my $text = $origin_form->{text};
    $text =~ s/\n/,/g;
    my $phrase_list_id = $origin_form->{phraseListID};
    my $readonly = $origin_form->{readonly};
    my $method = $origin_form->{method};
    my $max_lines_threshold = $origin_form->{maxLinesThreshold} + 0 || 20_000;
    my @arguments = $origin_form->{arguments} ? @{JSON::from_json($origin_form->{arguments})} : ();

    my $phrase_list = $readonly ? $proj->get_phrase_list($phrase_list_id) : $proj->phrase_list($text);
    my $resultant_phrase_list = $phrase_list->$method(@arguments);
    my $resultant_phrase_list_id = $proj->save_phrase_list($resultant_phrase_list);
    my @resultant_texts = $max_lines_threshold < $resultant_phrase_list->count ?
                          splice(@{$resultant_phrase_list->perl_array}, 0, $max_lines_threshold) :
                          @{$resultant_phrase_list->perl_array};

    return JSON::to_json({
        status              => 'OK',
        text                => join("\n", @resultant_texts),
        phraseListID        => '' . $resultant_phrase_list->cache_id,
        readonly            => $resultant_phrase_list->count > $max_lines_threshold,
        maxLinesThreshold   => $max_lines_threshold,
    });
}

1;
