package Cmds::DynBanners;

use strict;

use utf8;
use open ':utf8';

use base qw(Cmds::Base);

use List::Util qw(sum max min maxstr);
use JSON qw(to_json from_json);
use Data::Dumper;
use POSIX qw(strftime);
use Time::HiRes qw/gettimeofday tv_interval/;
use Time::Local qw(timelocal);
use Encode;
use Utils::Hosts qw(get_curr_host get_hosts get_host_role);
use Utils::Sys qw(uniq do_safely md5int log_msg_fmt);
use URI::Escape;
use Storable qw(dclone);
use BM::Filter;
use BM::BannersMaker::Tasks::TaskQueueSettings;
use Utils::Sys qw{uncompressdata uncompressfile get_tempfile};
use Utils::Array qw{in_array};

use Encode qw{_utf8_on _utf8_off};

sub has_target_specurls_only {
    my ($target) = @_;
    return 0 unless $target->{Condition};
    if (ref($target->{Condition}) eq 'HASH') {
        return $target->{Condition}->{type} eq 'URL_prodlist';
    }
    if (ref($target->{Condition}) eq 'ARRAY') {
        for my $cond(@{$target->{Condition}}) {
            return 1 if $cond->{type} eq 'URL_prodlist';
        }
    }
    return 0;
}

sub has_task_specurls_only {
    my ($task) = @_;
    return 0 unless (ref($task->{Resource}) eq 'HASH');
    return 0 unless $task->{Resource}->{Targets};
    if (ref($task->{Resource}->{Targets}) eq 'HASH') {
        return has_target_specurls_only($task->{Resource}->{Targets});
    }
    if (ref($task->{Resource}->{Targets}) eq 'ARRAY') {
        for my $target (@{$task->{Resource}->{Targets}}) {
            return 0 unless has_target_specurls_only($target);
        }
        return 1;
    }
    return 0;
}


# Группируем таски
sub dyngrptasks : CRONLIGHTTASKS {
    my $proj = shift;
    my $source_table = shift;

    my $dummy_task = $proj->dyntask({}); # фиктивный объект для получения списка тасок
    my $source_tasks = $dummy_task->get_all_tasks_from_yt_table($source_table);
    my $grp_tasks = {};
    my $info = {};

    $proj->log("Starting to handle of all downloaded tasks");
    my $tasks_counter = 0;
    for my $task (@$source_tasks){
        if ($tasks_counter % 1000 == 0) {
            $proj->log("$tasks_counter has been handled in 'dyngrptasks'");
        }
        $tasks_counter += 1;

        my $dmn = $task->{Resource}{Domain};
        $dmn = $proj->site($task->{Resource}{FeedUrl})->domain if ( !$dmn && $task->{Resource}{FeedUrl} );
        $dmn =~ s/^www.//;

        my ($grp_key, $grlog_dir, $distribution_key);
        my $worker_threads = $proj->options->{dyn_domain_workers_number}->{$dmn} || 1;

        # для крупных клиентов можно настроить разбиение его тасок на более чем одну групповую, но запускаем мы их всё равно на одном хосте
        my $secondary_key = 1 + md5int($task->{GroupExportIDs}[0]) % $worker_threads;

        if (defined $task->{Resource}{FeedUrl}) {
            # таски с FeedUrl нет смысла держать строго на одном хосте для всего домена, распределяем их по FeedUrl
            $grp_key = $task->{Resource}{FeedUrl};
            $distribution_key = $grp_key;
            $grlog_dir = "grplog_feedurl";
            if ($worker_threads > 1) {
                $grp_key.= "_$secondary_key";
                $grlog_dir.= "_$secondary_key";
            }
        } elsif (has_task_specurls_only($task)) {
            $distribution_key = $dmn;
            $grp_key = $dmn."_specurls";
            $grlog_dir = "grplog_specurls";
            if ($worker_threads > 1) {
                $grp_key.= "_$secondary_key";
                $grlog_dir.= "_$secondary_key";
            }
        } else {
            # "обычные" таски собираем все в одну группу, чтобы обходчик не обходил одно и то же по многу раз
            $grp_key = $dmn;
            $distribution_key = $grp_key;
            $grlog_dir = "grplog";
        }

        $grp_tasks->{$grp_key} ||= [];
        push(@{$grp_tasks->{$grp_key}}, $task);

        $info->{$grp_key} ||= {};
        $info->{$grp_key}->{domain} = $dmn;
        $info->{$grp_key}->{grlog_dir} = $grlog_dir;
        $info->{$grp_key}->{distribution_key} = $distribution_key;
    }
    my @res = map { { domain => $info->{$_}->{domain},
                      tasks => $grp_tasks->{$_},
                      lock_value => $_,
                      distribution_key => $info->{$_}->{distribution_key},
                      grlog_dir => $info->{$_}->{grlog_dir}} } keys %$grp_tasks;

    $proj->log("Returning the result of 'dyngrptasks'");
    return \@res;
}

my $_is_host_alive = undef;
my $_alive_host_count = 0;

#####
# Поиск работоспособного резервного хоста
# 1) Детерминирован, зависит только от $checksum
# 2) Обеспечивает равномерное распределение тасков с поломанного хоста на рабочие
# 3) Корректен
#
sub find_alive_reserve_host {
    my ($checksum) = @_;
    die "There is no alive hosts!" if (!$_is_host_alive || scalar(grep {$_} @$_is_host_alive) == 0);

    my $hosts_cnt = scalar(@$_is_host_alive);
    my ($tries, $max_md5_tries) = (0, 5);

    # пытаемся найти живой хост, беря пробы md5(md5(md5(...)))
    # этот этап обеспечивает более-менее равномерное распределение тасков с поломанного хоста на рабочие (диффузия)
    my $target_index = $checksum % $hosts_cnt;
    while (!$_is_host_alive->[$target_index] && $tries < $max_md5_tries) {
        $checksum = md5int($checksum);
        $target_index = $checksum % $hosts_cnt;
        $tries++;
    }

    if ($_is_host_alive->[$target_index]) {
        return $target_index;
    } else {
       # если не получилось пробами md5 найти живой хост, то грубо идем вперед по списку от последней пробы
       # этот этап обеспечивает корректность работы алгоритма (если ответ есть - он будет найден)
       for(my $shift = 1; $shift < $hosts_cnt; $shift++) {
            my $ind = ($target_index + $shift) % $hosts_cnt;
            if ($_is_host_alive->[$ind]) {
                return $ind;
            }
        }
    }

    # до сюда управление не должно дойти, т.к. есть хотя бы один рабочий хост
    die "ERROR in find_alive_reserve_host - no host found!";
    return -1;
}

sub get_last_level_domain {
    my $hostname = shift;
    $hostname =~ s/(\..+)//;
    return $hostname;
}

sub check_task_host {
    my ($proj, $task_checksum, $host_index) = @_;

    if(!$_is_host_alive) {
        my @hosts = get_production_hosts();
        my @alive_hosts;
        my $host_role = get_host_role();
        my $locke_locks_path = $proj->options->{alive_yt_locke_path};
        my $yt_token_path = $proj->options->{yt_client_params}{params}{token_path};
        local $ENV{YT_PROXY} = 'locke';
        local $ENV{YT_TOKEN_PATH} = $yt_token_path;
        my $cmd = "yt list --format=json $locke_locks_path/$host_role";
        my $alive_hosts_json = $proj->read_sys_cmd($cmd, timeout => 10, no_die => 1, no_error => 1, shell => 1);
        eval {@alive_hosts = @{JSON::from_json($alive_hosts_json)} if $alive_hosts_json;};
        $proj->log("ERROR: locke yt list does not work or return zero hosts") if !@alive_hosts;

        my %alive_hosts = map{get_last_level_domain($_) => 1} @alive_hosts;

        $proj->log(scalar(keys %alive_hosts) . " hosts are alive");

        $_is_host_alive = [ map{0} @hosts ];
        $_is_host_alive->[$_] = ++$_alive_host_count for grep{$alive_hosts{$hosts[$_]}} 0..$#hosts;

        # если живых хостов нет, то сломалась ручка
        if(!$_alive_host_count) {
            $proj->log("no alive hosts found; work like all hosts alive");
            $_is_host_alive = [ map{1} @hosts ];
        } else {
            $proj->log("host_index=$_ ($hosts[$_]) is dead") for grep{!$_is_host_alive->[$_]} 0..$#hosts ;
        }
    }

    my $target_host_index = $task_checksum % @$_is_host_alive;
    # если исходный хост жив...
    if ($_is_host_alive->[$target_host_index]) {
        # то все ок
        return 1 if $target_host_index == $host_index;
    } else {
        # ... иначе ищем рабочий резервный хост
        my $reserve_host_index = find_alive_reserve_host($task_checksum);
        if ($reserve_host_index == $host_index) {
            my @hosts = get_production_hosts();
            $proj->log("take task ($task_checksum) from $hosts[$target_host_index]");
            return 1;
        }
    }

    return 0;
}

sub get_production_hosts {
    my @hosts = sort { $a cmp $b } map { get_last_level_domain($_) } get_hosts(role => get_host_role());
    return @hosts;
}

sub dyngrptasks_local :CRONLIGHTTASKS {
    my $proj = shift;
    my $source_table = shift;

    $proj->log("Starting dyngrptasks with source table $source_table");
    my $alltasks = dyngrptasks($proj, $source_table);

    $proj->log("Starting the receiving of local dyn tasks for current host");
    return get_local_tasks($proj, $alltasks, 'dyn');
}

sub dyngrpworker : CRONLIGHTWORKER {
    my ($proj, $task) = @_;
    my $tsk = $proj->dyngrptask($task);
    $tsk->make_task;
}

#Получение списка тасков для перфоманс-баннеров
sub perftasks : CRONLIGHTTASKS {
    my $proj = shift;
    my $source_table = shift;
    my $tsk = $proj->perftask({}); #Фиктивный объект, чтобы получить доступ к методам получения списка

    my $all_tasks = $tsk->get_all_tasks_from_yt_table($source_table);
    # фильтруем по OrderTag != turbo_ecom, остальное реком
    @$all_tasks = grep { !$_->{OrderTags} or $_->{OrderTags} !~ /turbo_ecom/} @$all_tasks;

    return $all_tasks;
}

sub perftasks_local :CRONLIGHTTASKS {
    my $proj = shift;
    my $source_table = shift;
    my $alltasks = perftasks($proj, $source_table);

    return get_local_tasks($proj, $alltasks, 'perf');
}

sub perfworker :CRONLIGHTWORKER {
    my ($proj, $task) = @_;
    my $tsk = $proj->perftask($task);
    $tsk->make_task;
}


sub get_local_tasks {
    my $proj = shift;
    my $alltasks = shift;
    my $task_type = shift;

    my @hosts = get_production_hosts();
    my $curr_host = get_curr_host();

    my $temp_dir = $proj->options->{"${task_type}_banners_dirs"}->{temp_dir};

    $proj->log("Creating folder $temp_dir unless it already exists");
    $proj->do_sys_cmd("mkdir -p " . $temp_dir) unless -d $temp_dir;

    my $logfile_json = "$temp_dir/tasks_all.json";
    $proj->log("Writing info to $logfile_json");
    open(FJ, "> $logfile_json");
    print FJ $proj->d2j($alltasks);
    close(FJ);

    my $logfile = "$temp_dir/tasks_all.dump";
    $proj->log("Starting to fill $logfile");
    open(FAL, "> $logfile");
    print FAL $proj->curtime." tasks: ".@$alltasks."\n";

    $proj->log("Starting to fill tasks_local.dump and tasks_local.json");
    my $logfile2 = "$temp_dir/tasks_local.dump";
    open(FF, "> $logfile2");

    my $logfile3 = "$temp_dir/tasks_local.json";
    open(FFF, "> $logfile3");

    my $res = '';

    my $get_task_checksum_lambda;
    if ($task_type eq 'dyn') {
        $get_task_checksum_lambda = sub { md5int($_->{distribution_key}) };
    } else {
        $get_task_checksum_lambda = sub { md5int($_->{Resource}{FeedUrl}) };
    }

    $proj->log("Entering to the cycle for all hosts to get for local tasks");
    for my $host_index (0..$#hosts) {
        my $host_re = $hosts[$host_index];

        $proj->log("Starting to get tasks for host_index=$host_index and host_re=$host_re");
        my $tasks = [ grep{check_task_host($proj, $get_task_checksum_lambda->($_), $host_index)} @$alltasks ];
        $proj->log("Finishing of getting the tasks");

        $proj->log("Adding tasks for host_index=$host_index to tasks_all.dump");
        for my $el ( map {[ $host_index, $host_re, $_ ]} @$tasks ){
            print FAL Dumper($el);
        }
        $proj->log("Finishing to write to tasks_all.dump for host_index=$host_index");

        if($curr_host =~ /$host_re/) {
            $proj->log("host_index=$host_index is this host's index. Starting to fill tasks_local.dump and tasks_local.json");
            #Печатаем в общий лог факт получения данных от БК
            print FF $proj->curtime." tasks: ".@$tasks."\n";
            print FF Dumper($tasks);

            print FFF $proj->json_obj->encode($tasks);
            $proj->log("Finishing to fill tasks_local.dump and tasks_local.json");

            $res = $tasks;
        }
    }

    $proj->log("Starting to close file descriptors in get_local_tasks");
    close(FAL);
    close(FF);
    close(FFF);

    $proj->log("Returning the result of get_local_tasks");
    return $res if $res;

    die("$curr_host: local ${task_type}tasks not found");
}


#######################################################################################
# Интерфейс
#######################################################################################

sub TasksReport :CMDH {
    my ($proj, $vars) = @_;
    return {
        NN => 1,
        idfield => 'TaskID',
        readonly => 1,
        rights => 'right_view_dynbanners_tasks',
        default_field_params => { shlist => 1, edlist => 0, },
        fields => [
             {
                 grp => [
                      ( map { {  name => $_, title => $_, } }
                      qw{OrderID BannerIDs ParentExportIDs host} ),
                      {  name => 'firsttime', title => 'FirstTime:', },
                 ],
             },
        ],
        grfield => 'cronlight_id',
        grfield_reverse => 1,
        filters => [
            { field => "cronlight_id",     title => "cronlight_id",      },
            { field => "OrderID",          title => "OrderID",           },
            { field => "BannerIDs",        title => "BannerIDs",         },
            { field => "ParentExportIDs",  title => "ParentExportIDs",   },
        ],
        order_by => 'cronlight_id DESC',
    };
}

sub BaseTasksQueue :CMDH {
    my ($proj, $vars) = @_;

    return {
        rights              => 'right_view_dynbanners_tasks',
        default_field_params    => { shlist => 1 },
        fix_sql_problem     => 1,
        readonly            => 1,
        idfield             => "ID",
        fields              => [
            { name  => "ID", shlist => 0 },
            { name  => "Domain", inlinefilter => {}, },
            { name  => "OrderID", inlinefilter => {}, },
            { name  => "Host", },
            { name  => "TaskFile", },
            { name  => "State", },
        ],
        filters             => [
            { field => "OrderID", title=> "OrderID", like => 1, },
            { field => "Domain", title=> "Domain", like => 1, },
            { field => "Host", title => "Host", grp => 1 },
            { field => "State", title=> "State", grp => 1 },
        ],
        pager               => { name => 'p', cc => 100, },
    };
}

sub DynTasksQueue :CMDH {
    my ($proj, $vars) = @_;

    return {
        base                => "BaseTasksQueue",
        title               => "Обработка динамических баннеров",
        dbhname             => "bannerland_dbh",
        table               => "DynTasksQueue",
    };
}

sub PerfTasksQueue :CMDH {
    my ($proj, $vars) = @_;

    return {
        base                => "BaseTasksQueue",
        title               => "Обработка перформанс-баннеров",
        dbhname             => "bannerland_dbh",
        table               => "PerfTasksQueue",
    };
}

sub get_technical_metrics {
    my ($proj, $entries, $result, $show_fields) = @_;
    my $task_times = {};
    my $task_types = {};

    for my $entry (@$entries) {
        my $date = $entry->{Date};
        my $type = $entry->{TaskType} || "";
        my $day = ( $result->{$date} ||= { Date => $date } );

        $task_types->{$type}++;

        if($entry->{State} eq "Done") {
            push @{$day->{$type . "Durations"} ||= []}, $entry->{Duration};

            if($entry->{DebugInfo}) {
                my $debug_info;

                eval { $debug_info = JSON::from_json($entry->{DebugInfo}); };

                my $result_count = $debug_info->{ResultCount} // $debug_info->{BannerPhraseCount} // $debug_info->{BannerCount};
                if($type ne "Tskv") {
                    $day->{$type . "EmptyCount"}++ if !$result_count;
                    $day->{$type . "ResultCount"} += $result_count if $result_count;

                    my $domain_data = (($day->{$type . "Domains"} ||= {})->{$entry->{Domain}} ||= { Count => 0, EmptyCount => 0 });
                    $domain_data->{Count}++;
                    $domain_data->{EmptyCount}++ if !$result_count;
                }

                if($debug_info) {
                    push @{$day->{LoopDurations} ||= []}, $debug_info->{wait_time} if defined($debug_info->{wait_time});
                    if (defined($debug_info->{memory_usage_mb}) && (!defined($day->{MaxMem}) || $day->{MaxMem} < $debug_info->{memory_usage_mb})) {
                        $day->{MaxMem} = $debug_info->{memory_usage_mb};
                    }
                    $day->{BadDomains}++ if defined($debug_info->{bad_domain});
                    $day->{TooLongTasks}++ if defined($debug_info->{too_long_task});
                    $day->{NoOffersCount}++ if defined($debug_info->{OfferCount}) && !$debug_info->{OfferCount};
                    $day->{AllFilteredOffersCount}++ if defined($debug_info->{FilteredOfferCount}) && !$debug_info->{FilteredOfferCount} && $debug_info->{OfferCount};
                }
            }

            push @{$task_times->{$entry->{Domain} . $entry->{TaskID}} ||= []}, [$entry->{TimeStamp}, $day];
        } elsif($entry->{State} eq "Timeout") {
            $day->{$type . "Timeouts"}++;
            $proj->log("timeout ".$entry->{Date}." ".$entry->{TaskID});
        } elsif($entry->{State} eq "Failed") {
            $day->{$type . "Fails"}++;
            $proj->log("fail ".$entry->{Date}." ".$entry->{TaskID});
        } elsif($entry->{State} eq "StartedAgain") {
            $day->{$type . "StartedAgain"}++;
            $proj->log("started again ".$entry->{Date}." ".$entry->{TaskID});
        }
    }

    for my $day (values %$result) {
        my $loops = $day->{LoopDurations} || [];
        $loops = [sort{$a <=> $b} @$loops];
        delete $day->{LoopDurations};
        my $size_loops = scalar(@$loops);
        # 99 перцентиль, потому что есть выбросы (клиент выключает и включает таску через полгода)
        $day->{MaxLoopTime} = $size_loops ? $loops->[$size_loops * 99 / 100] : 0;
        $day->{AvgLoopTime} = @$loops ? sum(@$loops) / $size_loops : 0;
        $day->{Loop50PercTime} = $size_loops ? $loops->[$size_loops / 2] : 0;
        $day->{Loop90PercTime} = $size_loops ? $loops->[$size_loops * 9 / 10] : 0;

        $day->{BadDomains} ||= 0;
        $day->{MaxMem} ||= 0;
        $day->{TooLongTasks} ||= 0;
        $day->{NoOffersCount} ||= 0;
        $day->{AllFilteredOffersCount} ||= 0;
        $day->{TaskEmptyCount} ||= 0;

        for my $type (keys %$task_types) {
            my $durations = $day->{$type . "Durations"} || [];
            $durations = [sort{$a <=> $b} @$durations];
            delete $day->{$type . "Durations"};
            my $n = @$durations;

            $day->{$type . "Count"} = $n;
            $day->{$type . "AvgTime"} = $n ? sum(@$durations) / $n : 0;
            $day->{$type . "MaxTime"} = max(@$durations) || 0;
            $day->{$type . "50PercTime"} = $n ? $durations->[$n / 2] : 0;
            $day->{$type . "90PercTime"} = $n ? $durations->[$n * 9 / 10] : 0;
            $day->{$type . "Timeouts"} ||= 0;
            $day->{$type . "Fails"} ||= 0;

            if($type ne "Tskv") {
                my $task_domains = $day->{$type . "Domains"} || {};
                delete $day->{$type . "Domains"};

                $day->{$type . "AvgResultCount"} = $n ? $day->{$type . "ResultCount"} / $n : 0;
                $day->{$type . "DomainCount"} = scalar(keys %$task_domains);
                $day->{$type . "EmptyDomainCount"} = 0;

                for my $domain (keys %$task_domains) {
                    my $data = $task_domains->{$domain};

                    if($data->{Count} > 0 && $data->{Count} == $data->{EmptyCount}) {
                        $day->{$type . "EmptyDomainCount"}++;
                    }
                }
            }
        }
    }

    my $day = (values(%$result))[0];
    for my $type (sort keys %$task_types) {
        for my $field (qw(Count AvgTime MaxTime Timeouts Fails StartedAgain DomainCount EmptyDomainCount EmptyCount AvgResultCount ResultCount)) {
            next if !defined($day->{$type . $field});
            my $h = { name => $type . $field };
            $h->{showmacro} = "format_number_2" if $field =~ /Avg/;
            $h->{showmacro} = "boldtext" if $field eq "Count";
            push @$show_fields, $h;
        }
    }

    push @$show_fields, (
        { name => "AvgLoopTime", showmacro => "format_number_2" },
        { name => "MaxLoopTime" },
        { name => "NoOffersCount" },
        { name => "AllFilteredOffersCount" },
        { name => "TooLongTasks" },
        { name => "BadDomains" },
        { name => "MaxMem" },
    );
}

sub get_growth_metrics {
    my ($proj, $entries, $result) = @_;

    for my $entry (@$entries) {
        my $date = $entry->{Date};
        my $day = ($result->{$date} ||= { Date => $date });

        $day->{$_} += $entry->{$_} for qw(Shows Clicks Cost);

        if($entry->{Shows}) {
            ($day->{Domains} ||= {})->{$entry->{Domain}}++;
            ($day->{Orders} ||= {})->{$entry->{OrderID}}++;
        }
    }

    for my $day (values %$result) {
        $day->{DomainCount} = scalar(keys %{$day->{Domains} || {}});
        delete $day->{Domains};
        $day->{OrderCount} = scalar(keys %{$day->{Orders} || {}});
        delete $day->{Orders};
        $day->{CostPerDomain} = $day->{DomainCount} ? $day->{Cost} / $day->{DomainCount} : 0;
        $day->{CTR} = $day->{Shows} ? 100.0 * $day->{Clicks} / $day->{Shows} : 0;
        $day->{CPC} = $day->{Clicks} ? $day->{Cost} / $day->{Clicks} : 0;
        $day->{CPM} = $day->{Shows} ? 1000.0 * $day->{Cost} / $day->{Shows} : 0;
        delete $day->{Clicks};
        delete $day->{Shows};
    }
}

sub _apply_timeres {
    my ($data, $timeres) = @_;

    if($timeres eq "months") {
        $_->{Date} = substr($_->{Date}, 0, 7) for @$data;
    }
}

sub get_tasks_monitor_cmdh {
    my ($proj, $vars, %opts) = @_;
    my @show_fields;
    my $result = {};
    my $timeres = $proj->form->{timeres} || "days";
    my $type = $opts{type};
    my $translation = $opts{translation} || {};

    if($opts{monitor_data}) {
        _apply_timeres($opts{monitor_data}, $timeres);
        get_technical_metrics($proj, $opts{monitor_data}, $result, \@show_fields);
    }

    if($opts{statistics_data}) {
        _apply_timeres($opts{statistics_data}, $timeres);
        get_growth_metrics($proj, $opts{statistics_data}, $result);

        push @show_fields, (
            { name => "DomainCount", showmacro => "boldtext" },
            { name => "Cost", showmacro => "bscost" },
            { name => "CostPerDomain", showmacro => "bscost", },
            { name => "OrderCount", },
            { name => "CTR", showmacro => "format_number_2" },
            { name => "CPC", showmacro => "bscost" },
            { name => "CPM", showmacro => "bscost" }
        );
    }

    $_->{title} = $translation->{$_->{name}} for @show_fields;

    return {
        title               => $opts{title},
        rights              => 'right_view_dynbanners_tasks',
        default_field_params    => { shlist => 1 },
        readonly            => 1,
        cache_getlistflt => {  # Нужно для того, чтобы работали графики
            cache_time => 1,
            table_prefix => $timeres,
        },
        topmenu => [
            {
                title => "Разрешение",
                sublist => [
                    { title => "Дни", url => $ENV{REQUEST_URI}, add_prm => { timeres => "days" }, },
                    { title => "Месяцы", url => $ENV{REQUEST_URI}, add_prm => { timeres => "months"  }, },
                ],
            },
            {
                title => "Дни",
                sublist => [
                    { title => "3", url => $ENV{REQUEST_URI}, add_prm => { count_days => "3" }, },
                    { title => "7", url => $ENV{REQUEST_URI}, add_prm => { count_days => "7" }, },
                    { title => "14", url => $ENV{REQUEST_URI}, add_prm => { count_days => "14" }, },
                    { title => "30", url => $ENV{REQUEST_URI}, add_prm => { count_days => "30" }, },
                ],
            }
        ],
        fields              => [
            { name => "Date", },
            @show_fields
        ],
        action_onlist => sub {
            my ($proj, $el) = @_;
            $el->{"gfld$_"} = $el->{$_}   for map {$_->{name}} @show_fields;   # kostyl for charts     TODO
        },
        getlistflt          => sub {
            my ($self, %prm) = @_;

            return [map{$result->{$_}} sort{$b cmp $a} keys %$result];
        },
        #filters => [
        #    { field => "Date", type => "date2" },
        #],
        charts => {
            date_field => "Date",
            default_chart_params => {
                legend => 1,
                chartdiv_style => 'height: 700px;',
            },
            list => [
                {
                    chartdiv_style => 'height: 700px;',
                    type => "multilines",
                    #axes => 1,
                    flist => [ map{ { name => $_->{name}, value => $_->{name} } } @show_fields ],
                    legend => 1,
                    selectlist => { map {$_ => $_} @show_fields },
                },
            ],
        },
    };
}

sub aggregate_dyn_tasks_monitor {
    my ($proj, $last_days) = @_;

    $last_days ||= 1;

    my $table = $BM::BannersMaker::Tasks::TaskQueueSettings::types->{dyn}{monitor_table_name};
    my $days_query = $last_days + 3; # плюс три дня, чтобы захватить старт обхода

    my $monitor_data = _convert_monitor_entries($proj->bannerland_dbh->List_SQL("select * from $table WHERE Time BETWEEN CURDATE() - INTERVAL $days_query DAY AND NOW()"));
    my $statistics_data = $proj->bannerland_dbh->List_SQL("
        select
            date(Date) Date,
            Domain,
            OrderID,
            Shows,
            Clicks,
            Cost
        from DynTasksDirectSummary WHERE DATE BETWEEN CURDATE() - INTERVAL $days_query DAY AND NOW();
    ");

    my @show_fields;
    my $result = {};
    get_technical_metrics($proj, $monitor_data, $result, \@show_fields);
    get_growth_metrics($proj, $statistics_data, $result);

    my $db = $proj->dbtable('DynTasksMonitorAggr', 'Date', 'bannerland_dbh');
    my @dates = ();
    my @add_data = ();
    my $curr_date = $proj->dates->cur_date('db');
    for my $i (0..$last_days) {
        my $date = $proj->dates->next_n_date(-$i, 'db', 'db', $curr_date);
        push @dates, $date;
        push @add_data, $result->{$date};
    }

    $db->DelList({Date => \@dates});
    $db->Add(\@add_data);
}

sub DynTasksMonitor :CMDH {
    my ($proj, $vars) = @_;

    my $fields = [

        {name => 'Date', title => 'Дата'},

        {name => 'TaskCount', title => 'Кол-во обработ. тасок', showmacro => 'boldtext'},
        {name => 'MaxMem', title => 'Макс. память (МБ)', },
        {name => 'TaskTimeouts', title => 'Таймауты при обработке таски'},
        {name => 'TooLongTasks', title => 'Долгие таски'},
        {name => 'TaskFails', title => 'Крэши тасок'},
        {name => 'TaskStartedAgain', title => 'Таски без завершения'},
        {name => 'TaskDomainCount', title => 'Количество доменов с ДО'},
        {name => 'TaskEmptyDomainCount', title => 'Доменов с пустыми тасками'},
        {name => 'TaskEmptyCount', title => 'Пустые таски'},
        {name => 'NoOffersCount', title => 'Кол-во тасок без офферов'},
        {name => 'AllFilteredOffersCount', title => 'Тасок, где всё отфильтровалось'},
        {name => 'TaskAvgResultCount', title => 'Ср. кол-во фраз в таске'},
        {name => 'TaskResultCount', title => 'Всего фраз', showmacro => 'mlns'},

        {name => 'TskvCount', title => 'Обходов сайтов', showmacro => 'boldtext'},
        {name => 'TskvTimeouts', title => 'Таймауты при обходе'},
        {name => 'TskvFails', title => 'Крэши при обходе'},
        {name => 'TskvStartedAgain', title => 'Обходы без завершения'},
        {name => 'BadDomains', title => 'Плохие домены'},
    ];

    return {
        title                => 'Мониторинг динамических тасок',
        rights               => 'right_view_dynbanners_tasks',
        default_field_params => { shlist => 1 },
        readonly             => 1,
        dbhname              => 'bannerland_dbh',
        table                => 'DynTasksMonitorAggr',
        fields               => $fields,
        order_by             => '-Date',
        charts => {
            date_field => "Date",
            default_chart_params => {
                legend => 1,
                chartdiv_style => 'height: 700px;',
            },
            list => [
                { dtvalue => 'TaskCount', },
                { dtvalue => 'TaskDomainCount', },
                { dtvalue => 'TaskEmptyDomainCount', },
                { dtvalue => 'TaskEmptyCount', },
                { dtvalue => 'TaskAvgResultCount', },
                { dtvalue => 'TaskResultCount', },

                { dtvalue => 'TskvCount', },
                { dtvalue => 'NoOffersCount', },
                { dtvalue => 'AllFilteredOffersCount', },
            ],
        },
        toptext => qq[

        <span style="color: #ff6600;">Общее: вся статистика считается по таскам и обходам, завершенным
        в указанную дату, причем они могут быть начаты за пару дней до этого.</span>

        <ul>
            <li><b>Кол-во обработ. тасок</b> — количество именно отдельных обработанных тасок
            (а не групповых), не учитывая tskv.</li>
            <li><b>Макс. память</b> — максимальное количество памяти в МБ, используемой
            одной таской (из заверешенных в этот день). С учетом текущей конфигурации
            баннерлэндов безопасное значение до 20000.</li>
            <li><b>Таймауты при обработке таски</b> — количество жестких таймаутов
            (без отсылки в БК) за день.</li>
            <li><b>Долгие таски</b> — количество тасок, которые сгенерировали баннеры
            не по всем офферам, чтобы уложиться в таймаут (отправка в БК произошла).
            Не так плохо как крэш или таймаут, но мы не досыпаем баннеров (по крупным клиентам),
            поэтому надо ускорять генерацию. чтобы было меньше долгих тасок.</li>
            <li><b>Крэши тасок</b> — задачи которые экстренно завершились,
            причем могла завершиться и вся групповая таска.</li>
            <li><b>Таски без завершения</b> — странные ситуации, когда таска началась, но при этом нет
            ни крэша, ни таймаута, ни правильного окончания.</li>
            <li><b>Количество доменов с ДО</b> — количество доменов, по которым
            были завершены таски в этот день.</li>
            <li><b>Домены с пустыми тасками</b> — количество доменов, у которых все завершившиеся
            таски за этот день не сгенерировали баннеров.</li>
            <li><b>Пустые таски</b> — количество всех тасок, которые завершились в этот день
             и не сгенерировали баннеров.</li>
            <li><b>Кол-во тасок без офферов</b> — количество тасков, у которых на вход
            пришло 0 офферов.</li>
            <li><b>Тасок, где всё отфильтровалось</b> — количество тасок, у которых все офферы отфильтровались.</li>
            <li><b>Ср. кол-во фраз в таске</b> — среднее количество фраз во всех завершенных</li>
            <li><b>Всего фраз</b> — количество баннерофраз во всех тасках, которые
            завершились за день (в миллионах).</li>
            <li><b>Обходов сайтов</b> — количество всех сгенерированных tskv</li>
            <li><b>Таймауты при обходе</b> — количество жестких таймаутов при обходе
            (tskv не сгенерировался).</li>
            <li><b>Крэши при обходе</b> — обходы которые экстренно завершились,
            причем могла завершиться и вся групповая таска</li>
            <li><b>Обходы без завершения</b> — странные ситуации, когда обход начался, но при этом нет
            ни крэша, ни таймаута, ни правильного окончания.</li>
            <li><b>Плохие домены</b> — количество тасков с плохими доменами
            (из словарей dyn_bad_domains и dyn_bad_domains_2level).</li>
        </ul>
        ],
    };
}

sub DynTasksMonitorTime :CMDH {
    my ($proj, $vars) = @_;

    my $fields = [

        {name => 'Date', title => 'Дата'},
        {name => 'TaskAvgTime', title => 'Ср.время обработки таски', showmacro => 'seconds_to_hms'},
        {name => 'Task50PercTime', title => 'время таски(50%)', showmacro => 'seconds_to_hms'},
        {name => 'Task90PercTime', title => 'время таски(90%)', showmacro => 'seconds_to_hms'},
        {name => 'TaskMaxTime', title => 'время таски(100%)', showmacro => 'seconds_to_hms'},

        {name => 'TskvAvgTime', title => 'Ср.время обхода сайта', showmacro => 'seconds_to_hms'},
        {name => 'Tskv50PercTime', title => 'время обхода сайта(50%)', showmacro => 'seconds_to_hms'},
        {name => 'Tskv90PercTime', title => 'время обхода сайта(90%)', showmacro => 'seconds_to_hms'},
        {name => 'TskvMaxTime', title => 'время обхода сайта(100%)', showmacro => 'seconds_to_hms'},

        {name => 'AvgLoopTime', title => 'Ср.время ожидания переобхода', showmacro => 'seconds_to_hms'},
        {name => 'Loop50PercTime', title => 'время ожидания переобхода(50%)', showmacro => 'seconds_to_hms'},
        {name => 'Loop90PercTime', title => 'время ожидания переобхода(90%)', showmacro => 'seconds_to_hms'},
        {name => 'MaxLoopTime', title => 'время ожидания переобхода(99%)', showmacro => 'seconds_to_dh'},

    ];

    return {
        title                => 'Мониторинг времени динамических баннеров',
        rights               => 'right_view_dynbanners_tasks',
        default_field_params => { shlist => 1 },
        readonly             => 1,
        dbhname              => 'bannerland_dbh',
        table                => 'DynTasksMonitorAggr',
        fields               => $fields,
        order_by             => '-Date',
        charts => {
            date_field => "Date",
            default_chart_params => {
                legend => 1,
                chartdiv_style => 'height: 700px;',
            },
            list => [
                map { { dtvalue => $_->{name} } } grep { $_->{name} ne 'Date' } @$fields
            ],
        },
    };
}

sub DynTasksFailsMonitor : CMDH {
    my ($proj, $vars) = @_;

    my $state_to_name = {
        'Done' => 'ОК',
        'Failed' => 'Fail',
        'StartedAgain' => 'Некорректный перезапуск'
    };

    my $table = $BM::BannersMaker::Tasks::TaskQueueSettings::types->{dyn}{monitor_table_name};
    my $add_days = 2; # чтобы получить информацию о всех запусках задач, которые завершились в нужный период
    my $days = $proj->form->{count_days} || 3;
    my $data = _convert_monitor_entries($proj->bannerland_dbh->List_SQL("select * from $table WHERE Time BETWEEN CURDATE() - INTERVAL @{[$days + $add_days]} DAY AND NOW()"));

    my $report_threshold = timelocal(0, 0, 0, (localtime)[3..5]) - 3600 * 24 * $days;
    my $domain_task_error = {};
    for my $entry (@$data) {
        if ($entry->{TimeStamp} >= $report_threshold) {
            my $id = $entry->{Domain}.$entry->{TaskID};
            if ($entry->{State} ne "Done") {
                $domain_task_error->{$id} = 1;
            }
        }
    }

    my %fields = ();
    my $domains = {};
    for my $entry (@$data) {
        my $id = $entry->{Domain}.$entry->{TaskID};
        if ($domain_task_error->{$id}) {
            $fields{$entry->{Date}} = 1;
            my $d = $domains->{$id} ||= {Domain => $entry->{Domain}, TaskID => $entry->{TaskID}};
            if (not $d->{$entry->{Date}} or $entry->{State} ne 'Done') {
                $d->{$entry->{Date}} = $state_to_name->{$entry->{State}};
            }
         }
    }

    my @fields = sort keys %fields;
    @fields = @fields[$add_days..scalar(@fields)];

    return {
        title => "Ошибки обработки задач и сайтов",
        rights => 'right_view_dynbanners_tasks',
        readonly => 1,
        default_field_params => { shlist => 0 },
        fields => [{name => 'Domain'}, {name => 'TaskID'}, map{{name=>$_}} @fields],
        getlistflt => sub {
            my ($self, %prm) = @_;
            return [map{$domains->{$_}} sort keys %$domains];
        },
        topmenu => [
            {
                title => "Дни",
                sublist => [
                    { title => "3", url => $ENV{REQUEST_URI}, add_prm => { count_days => "3" }, },
                    { title => "7", url => $ENV{REQUEST_URI}, add_prm => { count_days => "7" }, },
                    { title => "14", url => $ENV{REQUEST_URI}, add_prm => { count_days => "14" }, },
                    { title => "30", url => $ENV{REQUEST_URI}, add_prm => { count_days => "30" }, },
                ],
            }
        ]
    };
}

sub PerfTasksMonitor :CMDH {
    my ($proj, $vars) = @_;

    my $days = $proj->form->{count_days} || 3;

    return get_tasks_monitor_cmdh(
        $proj, $vars,
        title           => "Мониторинг перфоманс-баннеров",
        monitor_data    => _convert_monitor_entries($proj->bannerland_dbh->List_SQL("select * from PerfTasksMonitorEntries WHERE Time BETWEEN CURDATE() - INTERVAL $days DAY AND CURDATE()")),
        type            => "perf",
    );
}

sub DynTasksSummaryBase :CMDH {
    my ($proj, $vars) = @_;
    my $now = time;
    return {
        title               => "Статистика по динамическим баннерам",
        rights              => 'right_view_dynbanners_tasks',
        default_field_params    => { shlist => 1 },
        readonly            => 1,
        fields              => [
            { name  => "Date", showmacro => 'datetime2date', inlinefilter => { group => 1 }, },
            { name  => "Domain", inlinefilter => { group => 1 }, },
            { name  => "OrderID", inlinefilter => { group => 1 }, },
            { name  => "BannerID", inlinefilter => { group => 1 }, },
            { name  => "Shows", showmacro => "format_number", dbtype => "integer" },
            { name  => "Clicks", showmacro => "format_number", dbtype => "integer" },
            { name  => "Cost", showmacro => "format_number", dbtype => "integer", showmacro => 'bscost', },
        ],
        filters             => [
            { field => "Date", title => "Дата", type => "date2" },
            { field => "Domain", title => "Domain", use_other_filters => 1, },
            { field => "OrderID", title => "OrderID", use_other_filters => 1, },
        ],
        default_filter      => {
            from_Date => strftime("%F", localtime $now - 7 * 86400), # I don't care of conversion of time
            to_Date => strftime("%F", localtime $now),
        },
        stat_fields         => [ qw(Shows Clicks Cost),
            {name => 'Bounces', stattype => 'Bounces', shlist => 0 },
            {name => 'SessionNum', stattype => 'SessionNum', shlist => 0 },
            {name => 'BounceRatio', title => 'BounceRatio', stgrptype => '#Bounces#/#SessionNum#*100', showmacro => 'format_number_2',},
            {name => 'PrGoodMultiGoal', title => 'PrGoodMultiGoal', showmacro => 'format_number_2' } ],
        order_by            => "-Date,Domain,OrderID,BannerID",
        pager => { name => 'p', cc => 1000, },
    };
}

sub DynTasksDirectSummary :CMDH {
    my ($proj, $vars) = @_;
    return {
        base => 'DynTasksSummaryBase',
        dbhname => 'bannerland_dbh',
        table => 'DynTasksDirectSummary',
        pager => { name => 'p', cc => 100, },
        data_import_export  => {
            list => [qw(Date Domain Shows Clicks Cost CTR CPM CPC)],
            export => { use_grp => 1, filtered => 1 },
            no_import => 1,
        },
        charts => {
            date_field => "Date",
            default_chart_params => {
                legend => 1,
                chartdiv_style => 'height: 700px;',
            },
            list => [
                { dtvalue => 'Shows', },
                { dtvalue => 'Clicks', },
                { dtvalue => 'Cost', },
                { dtvalue => 'CTR', },
                { dtvalue => 'CPM', },
                { dtvalue => 'CPC', },
            ],
        },
    };
}

sub PerfTasksDirectSummary :CMDH {
    my ($proj, $vars) = @_;
    return {
        base   => 'DynTasksSummaryBase',
        title  => "Статистика по смарт-баннерам",
        dbhname => 'bannerland_dbh',
        table  => 'PerfTasksDirectSummary',
        pager => { name => 'p', cc => 1000, },
    };
}

sub DynTasksReport :CMDH {
    my ($proj, $vars) = @_;
    return {
        base => 'TasksReport',
        title => 'Динамические баннеры',
        rights => 'right_view_dynbanners_tasks',
        dbhname => 'bannerland_dbh',
        table => 'DynTasks',
        joinedfilter => 1,
        fields => [
#               (
#                  map { {  name => $_, title => $_,  } }
#                  qw{ Domain }
#               ),
             {  showmacro => 'dyn_task_get_files TaskID end', width => 20, },
             {  name => 'FunnelInfo', showsubprojel => sub {
                     my ($proj, $el) = @_;
                     return get_funnel('dyn', $proj, $el);
                 },
             },
             {
                  title => 'Geo/Body',
                  grp => [
                      { name => 'Domain', showmacro => 'boldtext', },
                      ( map { {  name => $_, } }
                      qw{Geo Body host}, ),
#                      { name => 'addurlparams', title => 'UrlParams', inline => 1, edlist => 1, }
                  ],
                  hide_empty_line => 1,
             },
             {
                 title => 'Обработка',
                 grp => [
                     {
                         name => 'work',
                         ftype => 'select',
                         width => 100,
                         selectlist => [
                                 { name => 'worked',    value => 'work',   default => 1,  color => '#00FF00', },
                                 { name => 'stopped',   value => '', color => '#cd261b', textcolor => '#FFFFFF', },
                             ],
#                         inline => 1,
                         edlist => 1,
                     },
#                     ( map { {  name => $_, 'showmacro' => ( $_ =~ /Cost/ ? 'bscost' : 'format_number'), 'align' => 'right', } }
#                      qw{Shows Clicks Cost}, ),
                 ],
             },
             {
                 title => 'cron/begin/end',
                 grp => [
                      map { {  name => $_, showmacro => 'space2nbsp', } }
                      qw{ crontime begin end }
                 ],
             },
             {
                 title => 'prev_begin/prev_end',
                 grp => [
                      map { {  name => $_, } }
                      qw{ prev_crontime prev_begin prev_end }
                 ],
             },
        ],
        filters => [
            { field => "Domain", title => "Domain", like => 1,  },
        ],
        extlists => [
            {  cmd => 'get_bs_orderid_stat', using => 'TaskID', addelemparams => { BannerIDs => 'BannerIDs', } },
        ],
        pager => { name => 'p', cc => 100, },
    };
}

sub DynTasksReportFile :CMD {
    my ($proj, $vars) = @_;

    my $form = $vars->{form};
    my $act = $form->{act};
    my $TaskID = $form->{TaskID};

    my $dbt = $proj->dbtable('DynTasks', 'TaskID', 'bannerland_dbh');
    my $task = $dbt->Get($TaskID);
    my $host = $task->{host};
#    my $domain = $task->{Domain};

    my $domain_dir_name = $task->{domain_dir_name};
    # оставляем костыль, пока в базе не обновятся записи в поле domain_dir_name по каждой таске
    unless ($domain_dir_name) {
        my $raw_domain = $task->{Domain};
        $raw_domain = $proj->site($task->{FeedUrl})->domain if ( !$raw_domain && $task->{FeedUrl} );
        $domain_dir_name = BM::BannersMaker::Tasks::TaskUtils::domain_to_dir($raw_domain);
    }

    my $beg = BM::BannersMaker::Tasks::TaskUtils::filebegtime($task->{begin});

    my $fileend;
    if ($act eq 'task_err') {
        $fileend = "_task.err";
    } else {
        $vars->{template} = 'Lists/message.tmpl';
        $vars->{title} = 'Wrong param act';
        $vars->{text} = "$act is unknown value for &act";
        return;
    }

    my $filename = get_tempfile('tmp', UNLINK => 1);
    chmod 0777, $filename;
    $proj->do_sys_cmd("scp -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no $host:/opt/broadmatching/temp/dyn_banners/$domain_dir_name/$TaskID/$beg$fileend $filename");
    $vars->{'_return_textfile'} = $domain_dir_name."_$beg$fileend";
    $vars->{text} = $proj->file($filename)->text;
}

sub get_bs_orderid_stat :CMDH { #Проблема с дырками
    my ($proj, $vars) = @_;
    my $form = $proj->form;
    my $task_id = $form->{TaskID};
    my $dbt = $proj->dbtable('DynTasks', 'TaskID', 'bannerland_dbh');
    $dbt = $proj->dbtable('PerfTasks', 'TaskID', 'bannerland_dbh') if $form->{perf};
    my $task = $dbt->Get($task_id);
    $vars->{text} = $task->{stat};
    my @arr = split "\n", $vars->{text};
    my @flds = split "\t", shift @arr;
    s/#// for @flds;
    my @res = ();
    for my $str (@arr){
        my %h = ();
        @h{@flds} = split("\t", $str);
        next if $form->{BannerIDs} ne $h{BannerID};
        push(@res, \%h);
    }

    #Группировка
    my $resh = {};
    for my $el (@res){
        my $kk = $el->{UpdateTime};
        my $gel = $resh->{$kk} ||= {};
        #$gel->{$_} = $el->{$_} for qw { UpdateTime BannerID TargetType ContextType DeviceType };
        $gel->{$_} = $el->{$_} for qw { UpdateTime };
        $gel->{$_} += $el->{$_} for qw { Shows Clicks Cost SessionNum GoalsNum };
    }
    @res = values %$resh;


#$proj->dd('ddddd', Dumper($task));
    #$vars->{text} = $proj->page($url)->tt;

    #$proj->dd(\@flds);

    #            qw { UpdateTime BannerID PhraseID TargetType ContextType DeviceType Shows Clicks Cost CostCur FirstPageShows FirstPageClicks FirstPageSumPosShows FirstPageSumPosClicks SessionNum SessionDepth SessionCost GoalsNum MaxHostTime }
    return {
        getlistflt => sub { return \@res },
        default_field_params => { shlist => 1, edlist => 0, },
        readonly => 1,
        fields => [
            (  map { { name => $_, } }
                #qw { UpdateTime BannerID PhraseID TargetType ContextType DeviceType }
                #qw { UpdateTime BannerID TargetType ContextType DeviceType }
                qw { UpdateTime }
            ),
            (  map { { name => $_, 'showmacro' => ( $_ =~ /Cost/ ? 'bscost' : 'format_number'), 'align' => 'right', } }
                #qw { Shows Clicks Cost SessionNum SessionDepth SessionCost GoalsNum }
                qw { Shows Clicks Cost SessionNum GoalsNum }
            ),
        ],
        #group_by => 'UpdateTime',
        order_by => 'UpdateTime desc',
    };
}

sub PerfTasksDmns :CMDH {
    my ($proj, $vars) = @_;
    return {
#        base => 'TasksReport',
        NN => 1,
        idfield => 'domain',
        readonly => 1,
        rights => 'right_view_dynbanners_tasks',
        title => 'Перфоманс-баннеры',
#        dbhname => 'bannerland_dbh',
#        table => 'PerfTasks',
        getlistflt => sub {
            my ($tinf, %prms) = @_;
            my $where = '';
            if(%{$prms{filter}}){
                if($prms{filter}{OrderID} && $prms{filter}{OrderID} =~ /^\d+$/){
                    $where .= ' OrderID='.$prms{filter}{OrderID}.' ';
                    delete($prms{filter}{OrderID});
                }
                if($prms{filter}{'FeedUrl LIKE '}){
                    $where = $where." AND " if $where;
                    $prms{filter}{'FeedUrl LIKE '} =~ s/'//g;
                    $where .= " FeedUrl LIKE '".$prms{filter}{'FeedUrl LIKE '}."' ";
                    delete($prms{filter}{'FeedUrl LIKE '});
                }

                my $proj = $tinf->proj;
                $where = " where $where " if $where;
                my $sql = 'select domain, count(distinct OrderID) OrderIds, OrderID, FeedUrl, min(begin) minbeg, max(begin) maxbeg, group_concat(distinct CONCAT(st_task, "=", IF(st_status, "", st_status))) startrack from PerfTasks '.$where.' group by domain';
                my $lst = $proj->bannerland_dbh->List_SQL($sql);
                return $lst;
            }
            return [];
        },
        default_field_params => { shlist => 1, edlist => 0, },
        fields => [
             { name => 'domain', title => 'domain', inlinefilter => {},  },
             { name => 'OrderIds', title => 'OrderIds', inlinefilter => {}, },
             { name => 'minbeg', title => 'minbeg', inlinefilter => {}, },
             { name => 'maxbeg', title => 'maxbeg', inlinefilter => {}, },
             { name => 'startrack', title => 'startrack', inlinefilter => {}, showsubel => sub { my ($el, $f) = @_; return join ", ", map { $_->[0] !~ /^OLDTASK/ ? '<a href=https://st.yandex-team.ru/'.$_->[0].' target=_blank>'.$_->[0].'&nbsp;('.$_->[1].')</a>' : $_->[0] } map {[split '=', $_]} sort split(',', $el->{'startrack'}); } },
        ],
#        filters => [
#            { field => "FeedUrl", title => "FeedUrl", like => 1,  },
#        ],
        filters => [
            { field => "domain", title => 'Domain', like => 1, },
            { field => "OrderID", title => 'OrderID', },
            { field => "GroupExportIDs", title => 'GroupExportIDs', },
            { field => "FeedUrl", like => 1, title => 'FeedUrl', like => 1, },
        ],
        extlists => [
            {  cmd => 'PerfTasksOrderIDs', using => 'domain', },
        ],
        pager => { name => 'p', cc => 100, },
#        group_by => '',
    };
}

sub PerfTasksOrderIDs :CMDH {
    my ($proj, $vars) = @_;
    return {
#        base => 'TasksReport',
        NN => 1,
        idfield => 'domain',
        readonly => 1,
        rights => 'right_view_dynbanners_tasks',
        title => 'Перфоманс-баннеры',
#        dbhname => 'bannerland_dbh',
#        table => 'PerfTasks',
        getlistflt => sub {
            my ($tinf, %prm) = @_;
            my $proj = $tinf->proj;
            my $domain = $proj->form->{domain};
            my $where_condition; # DYNSMART-968
            my $args = [];
            if ($domain) {
                $where_condition = "where domain = ?";
                push @$args, $domain;
            } else {
                $where_condition = "where domain is Null";
            }
            my $lst = $proj->bannerland_dbh->List_SQL("select OrderID, count(distinct TaskID) TaskIDs, min(begin) minbeg, max(begin) maxbeg, group_concat(distinct host) hosts from PerfTasks $where_condition group by OrderID", $args);
            return $lst;
        },
        default_field_params => { shlist => 1, edlist => 0, },
        fields => [
             { name => 'OrderID', title => 'OrderID', addform => 1, inlinefilter => {}, },
             { name => 'TaskIDs', title => 'TaskIDs', inlinefilter => {}, },
             { name => 'minbeg', title => 'minbeg', inlinefilter => {}, },
             { name => 'maxbeg', title => 'maxbeg', inlinefilter => {}, },
             { name => 'hosts', title => 'hosts', inlinefilter => {}, },
        ],
#        filters => [
#            { field => "FeedUrl", title => "FeedUrl", like => 1,  },
#        ],
        extlists => [
            {  cmd => 'PerfTasksReport', addelemparams => { OrderID => 'OrderID', } },
        ],
#        group_by => '',
    };
}

sub get_pocket_info {
    my $proj = shift;
    my $dir = shift;
    my $yt_client = $proj->yt_client;

    my %par = @_;
    my $cluster = 'hahn';  # TODO определять кластер, где происходит генерация
    my $attrs = $yt_client->get_attribute($dir, '');  # get all attributes
    my %res = (
        url => "https://yt.yandex-team.ru/$cluster/navigation?path=$dir",
        start_time => $attrs->{start_time},
        svn_revision => $attrs->{svn_revision},
        last_started_step => $attrs->{bmyt_last_step},
    );
    if ($par{generated}) {
        $res{tasks} = $yt_client->get_attribute("$dir/tasks.final", "row_count");
        $res{offers} = $yt_client->get_attribute("$dir/tasks_and_offers", "row_count");
        $res{phrases} = $yt_client->get_attribute("$dir/generated_banners.final", "row_count");
        for my $wrk (qw(full_state funnel)) {
            my $attr = "archive_worker_$wrk";
            $res{$wrk} = $attrs->{$attr};
        }
        $res{end_time} = $attrs->{end_time};
    }
    return \%res;
}

sub perf_yt_overview :CMD {
    my ($proj, $vars) = @_;

    my $yt_client = $proj->yt_client;
    # на бете проблема: бета работает без fcgi, от пользователя www-data,
    # у которого нет прав на чтение токена; можно при тестировании переоределить token_path, например:
    # $yt_client->{params}{token_path} = '/home/malykhin/yt_plato';  # for catalogia-media-dev02e

    my $mk_dir = '//home/bannerland/perf/make_banners';

    $vars->{running_fields} = [qw(start_time svn_revision last_started_step)];

    my @validator_pockets_info;
    my $validator_done_info = get_pocket_info($proj, "$mk_dir/validator/run.prev");
    $validator_done_info->{pocket} = "validator_done";
    push @validator_pockets_info, $validator_done_info;

    my $validator_info = get_pocket_info($proj, "$mk_dir/validator/run");
    $validator_info->{pocket} = "validator_running";
    push @validator_pockets_info, $validator_info;
    $vars->{validator_pockets_info} = \@validator_pockets_info;

    my $current_pocket_dir = "$mk_dir/current";
    my $current_pocket_info = {};
    if ($yt_client->exists($current_pocket_dir)) {
        $current_pocket_info = get_pocket_info($proj, $current_pocket_dir);
    } else {
        $current_pocket_info = {};
    }
    $current_pocket_info->{pocket} = 'current';
    my @running_pockets_info = ($current_pocket_info);
    $vars->{running_pockets_info} = \@running_pockets_info;

    my $arc_dir = "$mk_dir/archive";
    my @pockets = sort { $b cmp $a } @{$yt_client->list($arc_dir) // []};
    my $num_pockets = 5;
    $vars->{generated_count} = $num_pockets;
    my @recent_pockets = splice(@pockets, 0, $num_pockets);
    my @generated_pockets_info;
    for my $pocket (@recent_pockets) {
        my $pocket_info = get_pocket_info($proj, "$arc_dir/$pocket", generated => 1);
        $pocket_info->{pocket} = $pocket;
        push @generated_pockets_info, $pocket_info;
    }
    $vars->{generated_pockets_info} = \@generated_pockets_info;
    $vars->{fields} = [qw(start_time end_time svn_revision tasks offers phrases full_state funnel)];

    my $bad_dir = "$mk_dir/bad_run_archive";
    my @failed_pockets = sort { $b cmp $a } @{$yt_client->list($bad_dir) // []};
    my $num_failed_pockets = 2;
    $vars->{failed_count} = $num_failed_pockets;
    my @recent_failed_pockets = splice(@failed_pockets, 0, $num_failed_pockets);
    my @failed_pockets_info;
    for my $pocket (@recent_failed_pockets) {
        my $pocket_info = get_pocket_info($proj, "$bad_dir/$pocket");
        $pocket_info->{pocket} = $pocket;
        push @failed_pockets_info, $pocket_info;
    }
    $vars->{failed_pockets_info} = \@failed_pockets_info;

    $vars->{template} = 'BannerLand/perf_yt_overview.tmpl';
}

sub get_funnel {
    my $task_type = shift;
    my $proj = shift;
    my $el = shift;

    my $row_delim = "<br>\n";

    my $t = $el->{FunnelInfo};
    $t =~ s/\n/$row_delim/g;
    # получаем YT воронку!
    my $task_id = $el->{TaskID};

    $task_id =~ s/_\d+_\d+$//; # удаляем инфу о чанке, для YT

    my $funnel;
    my $cache = ($proj->{__cmdh_make_cache__} //= {});
    if ($cache->{yt_funnel}->{$task_id}) {
        $funnel = $cache->{yt_funnel}->{$task_id};
    } else {
        $funnel = get_yt_funnel($task_type, $proj, $task_id, $row_delim);
        $cache->{yt_funnel}->{$task_id} = $funnel;
    }

    if ($funnel) {
        if ($funnel =~ m/RealTime generation/) {
            $t = join($row_delim, "FUNNEL:", $funnel);
        } else {
            $t = join($row_delim, "IRON_FUNNEL:", $t, "YT_FUNNEL:", $funnel);
        }
    }
    return $t;
}

sub get_yt_funnel {
    my $task_type = shift;
    my $proj = shift;
    my $task_id = shift;
    my $row_delim = shift;

    my $res = $proj->dt_proxy_client->do_select(
        table => get_tasks_info_yt_tablepath($task_type),
        fields => [ 'value', 'pocket', 'update_time' ],
        condition => "task_id = '$task_id' and property = 'funnel_str'",
    );
    if ($res and $res->[0]) {
        my $update_time = $res->[0]->{update_time};
        my $pocket = $res->[0]->{pocket};
        my $funnel_str = $res->[0]->{value};

        return join($row_delim,
            "update_time: $update_time",
            "pocket: $pocket",
            join($row_delim, split("\n" ,$funnel_str)),
        );
    }
    return;
}

sub get_tasks_info_yt_tablepath {
    my $task_type = shift;
    return "//home/bannerland/$task_type/dyntables/${task_type}_tasks_info";
}

sub PerfTasksReport :CMDH {
    return {
        NN => 1,
        idfield => 'TaskID',
        readonly => 1,
        rights => 'right_view_dynbanners_tasks',
        default_field_params => { shlist => 1, edlist => 0, },
        title => 'Перфоманс-баннеры',
        dbhname => 'bannerland_dbh',
        table => 'PerfTasks',
        fields => [
             { name => 'OrderID', addform => 1, shlist => 0, },
             {
                 grp => [
                      { name => 'TaskID', title => 'TaskID', },
                      ( map { {  name => $_, title => $_, } }
                      qw{OrderID BannerIDs ParentExportIDs} ),
                      {  name => 'firsttime', title => 'FirstTime:', },
                 ],
             },
             {  name => 'FunnelInfo', showsubprojel => sub {
                     my ($proj, $el) = @_;
                     return get_funnel('perf', $proj, $el);
                 },
             },
             {
                  grp => [
                      ( map { {  name => $_, title => $_, } }
                      qw{FeedUrl Geo Body host} ),
                      { name => 'addurlparams', title => 'UrlParams', inline => 1, edlist => 1, }
                  ],
                  hide_empty_line => 1,
             },
             {
                 title => 'cron/begin/end',
                 grp => [
                      map { {  name => $_, showmacro => 'space2nbsp', } }
                      qw{ crontime begin end currentstatus }
                 ],
             },
             {
                 title => 'prev_begin/prev_end',
                 grp => [
                      map { {  name => $_, showmacro => 'space2nbsp', } }
                      qw{ prev_crontime prev_begin prev_end }
                 ],
             },
        ],
        filters => [
            { field => "cronlight_id",     title => "cronlight_id",      },
            { field => "OrderID",          title => "OrderID",           },
            { field => "GroupExportIDs", title => "GroupExportIDs",  },
            { field => "BannerIDs",        title => "BannerIDs",         },
            { field => "ParentExportIDs",  title => "ParentExportIDs",   },
            { field => "FeedUrl", title => "FeedUrl", like => 1,  },
        ],
        #order_by => 'cronlight_id DESC',
        order_by => 'TaskID desc',
        pager => { name => 'p', cc => 100, },
#        extlists => [
#            {  cmd => 'get_bs_orderid_stat', using => 'TaskID', addelemparams => { BannerIDs => 'BannerIDs', }, addparams => { perf => 1, }, },
#        ],
    };
}

sub PerfParamMapping :CMDH {
    my ($proj, $vars) = @_;

    return {
        idfield             => 'ID',
        title               => "Маппинг параметров для PerfBanners",
        dbhname             => 'bannerland_dbh',
        table               => "PerfParamMapping",
        rights              => 'right_view_dynbanners_tasks',
        fix_sql_problem     => 1,
        default_field_params    => { shlist => 1 },
        readonly            => 0,
        fields              => [
            { name  => "type", inlinefilter => { group => 1 }, },
            { name  => "src",  },
            { name  => "desr", },
        ],
#        default_inlinefilter_params => {fields => ["Date"]},
        filters             => [
        ],
    };
}

sub DynDomains :CMDH {
    my ($proj, $vars) = @_;

    my $cattypes = [
                        { name => 'Нет', value => '',   default => 1,  },
                        { name => 'Товарная категория', value => 'goods',  color => '#00FF00', },
                        { name => 'Строительство и ремонт',    value => 'build',  color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Одежда', value => 'wear',   color => '#00FF00', },
                        { name => 'Мебель', value => 'furn',   color => '#00FF00', },
                        { name => 'Парфюмерия',         value => 'parfume', color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Отели',              value => 'hotels',  color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Авиабилеты',         value => 'hotels',  color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Автозапчасти',       value => 'carrepl',  color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Медицина',           value => 'med',  color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Секс-шопы',          value => 'sexshop',  color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Шины/Диски',         value => 'tires',  color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Другое',             value => 'other',  color => '#cd261b', textcolor => '#FFFFFF', },
    ];

    my $errros = [
                        { name => 'Нет', value => '',   default => 1,  },
                        { name => 'Всё хорошо', value => 'goods',   default => '',  color => '#00FF00', },
                        { name => 'Сайт на JS или Flash', value => 'badjs', color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Одностраничный сайт',  value => 'onepage', color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Мало товаров',         value => 'toosmall',  color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Проблемная категория', value => 'badcat',  color => '#cd261b', textcolor => '#FFFFFF', },
                        { name => 'Проблема со ссылками',  value => 'badurls',  color => '#cd261b', textcolor => '#FFFFFF', },
    ];

    return {
        idfield             => 'Domain',
        readonly            => 1,
        dbhname             => 'bannerland_dbh',
        table               => 'DynDomains',
        title               => "Анализ доменов Динамических баннеров",
        default_field_params    => { shlist => 1 },
        fields => [
            { name => 'Domain', showmacro => 'exturl', },
            {  name => 'status', title => '',  shlist => 1, ftype => 'select', width=>50, mass_change => 1, iwonchange_reload_extlists => 1,
               rights => 'right_edit_dyndomains_admin',
               selectlist => [
                       { name => 'unk',    value => '',     default => 1,  color => '#CCCCCC', },
                       { name => 'gd',     value => 'good', color => '#00FF00', },
                       { name => 'wrn',    value => 'warn', color => '#FFFF00', },
                       { name => 'bad',    value => 'bad', color => '#cd261b', textcolor => '#FFFFFF', },
                   ],
               inline => 1,
               edlist => 1,
            },
            {  name => 'status_assessor', title => '',  shlist => 1, ftype => 'select', width=>50, mass_change => 1, iwonchange_reload_extlists => 1,
               rights => 'right_edit_dyndomains_assessor',
               selectlist => [
                       { name => 'unk',    value => '',     default => 1,  color => '#CCCCCC', },
                       { name => 'gd',     value => 'good', color => '#00FF00', },
                       { name => 'wrn',    value => 'warn', color => '#FFFF00', },
                       { name => 'bad',    value => 'bad', color => '#cd261b', textcolor => '#FFFFFF', },
                   ],
               inline => 1,
               edlist => 1,
            },
            { name => 'host', },
            { name => 'tskv_time', inlinefilter => {}, showmacro => 'space2nbsp', },
            { name => 'tskv_size', title=>'tskv', inlinefilter => {}, },
            { name => 'tree_size', title=>'tree', inlinefilter => {}, },
            { name => 'MCost', wshowmacro => 'bscost', inlinefilter => {}, },
            {
                name => 'categ',
                rights => 'right_edit_dyndomains_admin',
                ftype => 'select',
                width => 200,
                showmacroel => 'show_list_inline_list_select_field',
                selectlist => $cattypes,
                inline => 1,
                edlist => 1,
            },
            {
                name => 'problem',
                rights => 'right_edit_dyndomains_admin',
                ftype => 'select',
                width => 200,
                showmacroel => 'show_list_inline_list_select_field',
                selectlist => $errros,
                inline => 1,
                edlist => 1,
            },
            {
                name => 'categ_assessor',
                rights => 'right_edit_dyndomains_assessor',
                ftype => 'select',
                width => 200,
                showmacroel => 'show_list_inline_list_select_field',
                selectlist => $cattypes,
                inline => 1,
                edlist => 1,
            },
            {
                name => 'problem_assessor',
                rights => 'right_edit_dyndomains_assessor',
                ftype => 'select',
                width => 200,
                showmacroel => 'show_list_inline_list_select_field',
                selectlist => $errros,
                inline => 1,
                edlist => 1,
            },
            { name => 'Login',  shlist => 1, disable_edit => 1, autoedit => sub { $proj->{login} }, },
        ],
        action_before_edit => sub { my ($proj, $id, $h) = @_; $h->{Login} = $proj->{login}; },
        filters => [
            { field => "Domain",           title => "Domain",  like => 1, },
            { field => "categ",            title => "categ",   grp => 1,  },
            { field => "problem",          title => "problem", grp => 1,  },
            { field => "login",            title => "login", grp => 1,  },
        ],
        order_by => 'MCost desc',
        pager => { name => 'p', cc => 50, },
    };
}

sub PerfUtmProblemDomains :CMDH {
    my ($proj, $vars) = @_;

    return {
        idfield             => 'ID',
        title               => "Домены, для которых нужно удалять UTM-метки из урлов",
        dbhname             => 'bannerland_dbh',
        table               => "PerfUtmProblemDomains",
        rights              => 'right_view_dynbanners_tasks',
        fix_sql_problem     => 1,
        compact_inline_add => 1,
        default_field_params    => { shlist => 1 },
        fields              => [
            { name  => "domain",  },
            { name => 'Login',  shlist => 1, disable_edit => 1, autoedit => sub { $proj->{login} }, },
            { name  => "date", title => 'Last modification time', autoedit => sub { $proj->dates->cur_date('db_time') }, },
        ],
        del_mes => "Вы уверены, что хотите удалить данную строку?",
#        default_inlinefilter_params => {fields => ["Date"]},
        filters             => [
        ],
    };
}

sub PerfPremiumGoodsDomains :CMDH {
    my ($proj, $vars) = @_;

    return {
        base                => 'PerfUtmProblemDomains',
        title               => "Домены, у которых только премиум-товары",
        dbhname             => 'bannerland_dbh',
        table               => "PerfPremiumGoodsDomains",
    };
}

sub PerfAllowCustomPhrases  :CMDH {
    my ($proj, $vars) = @_;

    return {
        idfield             => 'OrderID',
        title               => "Заказы с кастомными фразами из фида",
        dbhname             => 'bannerland_dbh',
        table               => "PerfAllowCustomPhrases",
        rights              => 'right_view_dynbanners_tasks',
        fix_sql_problem     => 1,
        compact_inline_add => 1,
        default_field_params    => { shlist => 1 },
        fields              => [
            { name  => "OrderID",  },
            { name  => "MaxPhrasesPerOffer",  },
            { name  => "Login",  shlist => 1, disable_edit => 1, autoedit => sub { $proj->{login} }, },
            { name  => "date", title => 'Last modification time', autoedit => sub { $proj->dates->cur_date('db_time') }, },
            { name  => "OnlyClientPhrases",  ftype => 'checkbox'},
        ],
        order_by => "OrderID",
        action_before_add => sub {
            my ($proj, $data) = @_;
            $data->{OrderID} = 0 unless ($data->{OrderID} =~ /^\d+$/);
            $proj->bannerland_dbh->Do_SQL("delete from PerfAllowCustomPhrases where OrderID = '$data->{OrderID}'");
        },
        Moderate => {
            Del => sub {
                my ($self, $id) = @_;

                my $tbl_review = $self->proj->dbtable("PerfAllowCustomPhrases", undef, "bannerland_dbh");
                $tbl_review->Do_SQL("delete from PerfAllowCustomPhrases where OrderID = '$id'");
            },
        },
        del_mes => "Вы уверены, что хотите удалить данную строку?",
    };
}

sub PerfCustomTemplates  :CMDH {
    my ($proj, $vars) = @_;

    return {
        idfield             => 'ID',
        title               => "Управление кастомными шаблонами генерации",
        dbhname             => 'bannerland_dbh',
        table               => "PerfCustomTemplates",
        rights              => 'right_view_dynbanners_tasks',
        fix_sql_problem     => 1,
        compact_inline_add => 1,
        default_field_params    => { shlist => 1, edlist => 1, },
        fields              => [
            { name  => "OrderIDOrDomain", title => 'OrderID/Domain', ftype => 'select',
                selectlist => [
                    { name => 'OrderID',    value => 'orderid',   default => 1, },
                    { name => 'Domain',   value => 'domain', },
                ],
            },
            { name  => "OrderIDOrDomainValue", title => 'OrderID/Domain value', },
            { name  => "AdvType", },
            { name  => "Variables", title => 'Variables', ftype => 'bigtextarea', },
            { name  => "PhraseTemplates", title => 'Phrase templates', ftype => 'bigtextarea', },
            { name  => "TitleTemplates", title => 'Title templates', ftype => 'bigtextarea', },
            { name  => "Login", disable_edit => 1, autoedit => sub { $proj->{login} }, edlist => 0, },
            { name  => "Date", title => 'Время изменения', disable_edit => 1, autoedit => sub { $proj->dates->cur_date('db_time') }, edlist => 0, },
        ],
        order_by => "OrderIDOrDomainValue",
        action_before_add => sub {
            my ($proj, $data) = @_;
            $data->{ID} = 0 unless ($data->{ID} =~ /^\d+$/);
            $proj->bannerland_dbh->Do_SQL("delete from PerfCustomTemplates where ID = '$data->{ID}' or (OrderIDOrDomain = '$data->{OrderIDOrDomain}' and OrderIDOrDomainValue = '$data->{OrderIDOrDomainValue}' and AdvType = '$data->{AdvType}')");
        },
        Moderate => {
            Del => sub {
                my ($self, $id) = @_;
                $self->proj->bannerland_dbh->Do_SQL("delete from PerfCustomTemplates where ID = '$id'");
            },
        },
        del_mes => "Вы уверены, что хотите удалить данную строку?",
    };
}

sub timestr_to_timestamp {
    my ($date, $time) = split(' ', shift);
    my ($year, $month, $day) = split('-', $date);
    my ($hour, $min, $sec) = split(':', $time);

    return timelocal($sec, $min, $hour, $day, $month - 1, $year);
}

sub _convert_monitor_entries {
    my ($entries) = @_;

    my @processed_entries = ();
    my %unprocessed_entries = ();

    my %host_and_domain_to_TaskID = ();

    for my $item (@{$entries}) {
        if ($item->{State} eq "Fail") {
            my $host_and_domain_name = $item->{Host} . $item->{Domain};
            $item->{TaskID} = $host_and_domain_to_TaskID{$host_and_domain_name};
            delete $host_and_domain_to_TaskID{$host_and_domain_name};
        }

        my $currentID = $item->{TaskID} . $item->{Domain};

        if ($item->{State} eq "FalseStart") {
            delete $unprocessed_entries{$currentID};
            next;
        }

        if (exists($unprocessed_entries{$currentID})) {
            my $state;
            if ($item->{State} eq "Start") {
                $state = "StartedAgain";
            } elsif ($item->{State} eq "Fail") {
                $state = "Failed";
            } else {
                $state = "Done";
            }

            my $start_timestamp = timestr_to_timestamp($unprocessed_entries{$currentID}->{Time});
            my $finish_timestamp = timestr_to_timestamp($item->{Time});
            my $duration = $finish_timestamp - $start_timestamp;
            my ($date, $time) = split(' ', $item->{Time});

            my %new_entry = (
                TaskID     => $item->{TaskID},
                TaskType   => $item->{TaskID} eq "tskv" ? "Tskv" : "Task",
                Date       => $date,
                TimeStamp  => $finish_timestamp,
                Duration   => $duration,
                Domain     => $item->{Domain},
                Host       => $item->{Host},
                DebugInfo  => $item->{DebugInfo},
                State      => $state
            );
            push(@processed_entries, \%new_entry);
            delete $unprocessed_entries{$currentID};
        }

        if ($item->{State} eq "Start") {
            $unprocessed_entries{$currentID} = $item;
            my $host_and_domain_name = $item->{Host} . $item->{Domain};
            $host_and_domain_to_TaskID{$host_and_domain_name} = $item->{TaskID};
        }
    }

    return \@processed_entries;
}

1;
