package Cmds::Interface;

use strict;

use utf8;

use base qw(Cmds::Base);
use Encode;
use List::Util qw(min max maxstr);
use Text::Iconv;
use Data::Dumper;
use URI::Escape;
use Utils::Sys qw(format_number h2sa uniq);
use Time::HiRes qw(gettimeofday tv_interval);
use Utils::Sys qw(md5int do_safely);
use Utils::Array qw(array_intersection);
use JSON qw(to_json from_json);
use List::Util 'shuffle';
use CatalogiaMediaProject;
use BM::ContextSyns::ContextLinks;
use Utils::XLS qw(array2xls);

use open ":utf8";

sub categories_text_fields : CMDH {
    my ($proj, $vars) = @_;

    return {
        title   => 'Categories text fields',
        table   => 'CategoriesTextFields',
        idfield => 'ID',
        default_field_params => { shlist => 1},
        default_filter => {Language => $vars->{viewoptions}{lang}},
        action_before_add => sub {
            my ($proj, $data) = @_;

            $data->{KeyID} = md5int($data->{Key});
            $data->{CatID} = $proj->get_category_id($data->{CatID}) || $data->{CatID};
        },
        fields => [
            {
                name        => 'CatID',
                title       => 'Категория',
                showmacro   => 'catid2link',
            },
            {
                name    => 'Key',
                title   => 'Поле',
            },
            {
                name    => 'Value',
                title   => 'Значение',
            },
            {
                name            => 'UpdateTime',
                title           => 'Обновлено',
                disable_edit    => 1,
                disable_add     => 1,
            },
        ],
        pager => { name => 'p', cc => 100, },
    };
}

sub biwords_list : CMDH {
    return {
        title       => 'Biwords List',
        readonly    => 1,
        getlistflt => sub {
            my ($self, %prm) = @_;

            my $proj = $self->proj;
            my $res = $proj->phrase()->get_biwords_dict_with_norm_and_snorm;
            return $res;
        },
        fields => [
            { name => "text", shlist => 1, },
            { name => "norm", shlist => 1, },
            { name => "snorm", shlist => 1, },
        ],
        pager => {
            name => 'p',
            cc => 60,
        },
        search => { fields => [ 'text', 'norm', 'snorm' ], name => 'text', title => 'Search' },
        order_by => 'text',
    };
}

sub cmdslist : CMD {
    my ($proj, $vars) = @_;
    $vars->{text} .= "$_\n" for keys %{ $proj->cmdslist };
}

sub get_banner_info : CMD {
    my ($proj, $vars) = @_;

    my $lang = $vars->{viewoptions}{lang};
    $lang ||= "ru";
    $proj->current_lang($lang);

    $vars->{title} = 'Banner analyze';
    $vars->{template} = 'get_banner_info.tmpl';

    my $bnr;

    if ( $proj->form->{bid} ) {
        $vars->{bid} = $proj->form->{'bid'};
        my $bid = $proj->form->{'bid'};
        $bnr = $proj->bf->get_banner_by_id($bid);
    }
    elsif ( $proj->form->{bs_banner_id} ) {
        $vars->{bs_banner_id} = $proj->form->{'bs_banner_id'};
        my $bsid = $proj->form->{'bs_banner_id'};
        $bnr = $proj->bf->get_banner_by_bsid($bsid);
    }
    elsif ( $proj->form->{'banner_title'} || $proj->form->{'banner_title_extension'} || $proj->form->{'banner_body'} || $proj->form->{'banner_phrases'} || $proj->form->{'banner_url'} ) {
        $vars->{banner_title} = $proj->form->{'banner_title'};
        $vars->{banner_title_extension} = $proj->form->{'banner_title_extension'};
        $vars->{banner_body} = $proj->form->{'banner_body'};
        $vars->{banner_phrases} = $proj->form->{'banner_phrases'};
        $vars->{banner_url} = $proj->form->{'banner_url'};
        $bnr = $proj->bf->psbanner({
            proj            => $proj,
            title           => $proj->form->{'banner_title'},
            title_extension => $proj->form->{'banner_title_extension'},
            body            => $proj->form->{'banner_body'},
            phrases         => $proj->form->{'banner_phrases'},
            url             => $proj->form->{'banner_url'},
            lang            => $lang
        });
    }

    return unless $bnr;

    my %analyze = ();

    my @info = ();
    push @info, { key => $_, value => ref($bnr->$_) eq 'ARRAY'? join ',', @{$bnr->$_} : join ',', $bnr->$_ } foreach (grep {$bnr->can($_)} qw( title title_extension body phrases url id bs_banner_id campaign_id uid active_flag is_moderated ));

    #чтобы посмотреть флаги фраз
    $bnr->moderate_filter_phrase_list($bnr->phrase_list);

    $analyze{phrases} = [ map {{
        text  => $_->text,
        flags => join("+", @{$_->{moderate_filterreason}}),
        diez_flags => join ", ", sort $proj->categs_tree->get_catalogia_flags($bnr->extend_diez_phr($_)->get_minicategs)
    }} $bnr->phrases ];

    $analyze{categs_total} = {method => $bnr->get_minicategs_method, categs => join('/', $bnr->get_minicategs ) };

    $analyze{processed_text} = [
        { method => 'title+body без префильтра', text => $bnr->title_body_phr =~ s/#//gr , categs => join '/', $bnr->title_body_phr->get_minicategs },
        { method => 'title+body с префильтром', text => $bnr->preprocess_title_body =~ s/#//gr, categs => join '/', $bnr->preprocess_title_body->get_minicategs },
        { method => 'joined phrases без префильтра', text => $bnr->join_phrases_for_categorization, categs => join '/', $bnr->join_phrases_for_categorization->get_minicategs },
    ];

    my @categs_methods = ();
    push @categs_methods, {method => 'Внешняя категоризация', categs => join('/', $bnr->external_minicategs ? @{$bnr->external_minicategs} : () ) };
    push @categs_methods, {method => 'Домен', categs => join('/', $bnr->language->pages_categories->get_static_domain_categs($bnr->urlpage) ) };
    push @categs_methods, {method => 'Prefiltered title+body', text => $bnr->preprocess_title_body =~ s/#//gr, categs => join('/', $bnr->get_title_body_minicategs ) };
    push @categs_methods, {method => 'Фразы', text => $bnr->join_phrases_for_categorization, categs => join('/', $bnr->get_phrases_minicategs ) };

    $analyze{categs_methods} = \@categs_methods;

    $analyze{info} = \@info;
    $analyze{flags_analysis} = $bnr->get_catalogia_flags_analysis;

    $vars->{Analyze} = \%analyze;
}

sub get_phrase_info : CMD {
    my ($proj, $vars) = @_;

    my $lang = $vars->{viewoptions}{lang};
    $proj->current_lang($lang);

    $vars->{title} = 'Phrase analyze';
    $vars->{template} = 'get_phrase_info_new.tmpl';

    my $flds_titles_descr = [
        [ 'norm_phr',                    'Нормализованная фраза', '' ],
        [ 'snorm_phr',                   'Снормализованная фраза', '' ],
        [ 'correct_phr',                 'Исправление опечаток', '' ],
        [ 'minicategs',                  'Категоризация', '' ],
        [ 'cdict_minicategs',            'Категоризация хвостиков', '' ],
        [ 'production_minicategs',       'Категоризация продакшена', '' ],
        [ 'core_subphrase',              'Категорийное ядро фразы', '' ],
        [ 'moderation_flags',            'Флаги модерации', 'get_banner_catalogia_flags: рассматриваем фразу как текст баннера и возвращаем флаги модерации' ],
        [ 'badphrsreason',               'bad', '' ],
        [ 'filter_phrase_list',          'фильтры фраз (BM)', '' ],
        [ 'moderate_filter_phrase_list', 'фильтры фраз (модерация)', ''],
        [ 'search_count',                'Количество запросов с подфразой', '' ],
        [ 'search_query_count',          'Количество запросов', '' ],
        [ 'is_misspell',                 'Наличие опечаток', '' ],
    ];

    $vars->{phr_fields} = [ map { $_->[0] } @$flds_titles_descr ];
    $vars->{fields_titles} = { map { $_->[0] => $_->[1] } @$flds_titles_descr };
    $vars->{fields_descriptions} = { map { $_->[0] => $_->[2] } @$flds_titles_descr };

    # Получаем текст, который без замены символов(",<,> etc) на  html кодировки. Далее используем этот текст почти везде, кроме $vars->{phrase_text}
    my $text = $proj->form->{__origin}{'phrase_text'} || '';

    $vars->{prefilter} = $proj->form->{prefilter};

    my $get_lemmas = sub {
        my $word = shift;
        my @gram = ();
        my $data = $proj->lemmer_test->analyze($word);
        for (split /\n/, $data) {
            next if $_ eq '';
            next if $_ eq $word;
            next if $_ =~ /====/;
            my ($lem, $quality, $sp, $lang) = (split / /)[0, 2, 3, 8];
            $sp =~ s/,.*//;
            push @gram, [ $lem, $lang, $quality, $sp ];
        }
        return @gram;
    };

    if ($text eq '') {
        return;
    }

    $vars->{phrase_text} = $text;

    my %analyze;
    $analyze{text} = $text;
    if ($vars->{prefilter}) {
        ($text, my $prefilters) = @{$proj->get_language($lang)->phrase($text)->get_banner_prefiltered_text}{qw/text debug/};
        foreach my $match (@{$prefilters}) {
            my $status = exists $match->{all} ? 'all' : 'worked';
            push @{$vars->{prefilters}{$status}}, "'$match->{regex}' => { '$match->{pattern}' => '$match->{replace}' } ";
        }
        #вычисляем префильтры, которые не сработали, кладем их отдельно
        my %worked = map { $_ => 1 } @{$vars->{prefilters}{worked}};
        push @{$vars->{prefilters}{missed}}, grep {!exists $worked{$_}} @{$vars->{prefilters}{all}};
    }

    # Создаем экземпляр объекта фраза из $text
    my $phr_obj = $proj->phrase($text);

    $analyze{prefiltered} = $text;
    $analyze{norm_phr} = $phr_obj->norm_phr;
    $analyze{snorm_phr} = $phr_obj->snorm_phr;
    $analyze{search_count} = $phr_obj->get_search_count;
    $analyze{search_query_count} = $phr_obj->get_search_query_count;
    ($analyze{categs_test_info}, my $analysis) = $phr_obj->get_minicategs_test_info_analyzer("<br/>");
    $analyze{flags_analysis} = $phr_obj->get_banner_catalogia_flags_analysis;
    $analysis->[$_]{'data'} = Dumper($analysis->[$_]{'data'}) for (0..$#{$analysis});
    $vars->{phrase_analysis} = $analysis;
    $analyze{correct_phr} = $phr_obj->correct_phr;
    $analyze{is_misspell} = $phr_obj->is_misspell;
    $analyze{atoms_info} = Dumper($phr_obj->search_atoms);
    my ($brand, $model) = $phr_obj->parse_fast;
    $analyze{brand} = $brand;
    $analyze{moderation_flags} = join(',', $phr_obj->get_banner_catalogia_flags());
    $analyze{regions} = to_json([ $phr_obj->get_regions ], { pretty => 1 } );
    $analyze{core_subphrase} = $phr_obj->get_core_subphrase;

    $proj->cmdlog("get_phrase_info badphrsreason");

    $analyze{'badphrsreason'} = $proj->phrase($text)->badphrsreason;

    my $filter_phl = $proj->phrase_list([$phr_obj]);
    my $empty_bnr = $proj->bf->lbanner({});
    $empty_bnr->filter_phrase_list($filter_phl, check_all => 1);
    $analyze{filter_phrase_list} = join ',', map {@{$_->{filterreason}}} @$filter_phl;
    $empty_bnr->moderate_filter_phrase_list($filter_phl, check_all => 1);
    $analyze{moderate_filter_phrase_list} = join ',', map {@{$_->{moderate_filterreason}}} @$filter_phl;

    for my $type (qw(minicategs cdict_minicategs production_minicategs)) {
        my $method = "get_$type";
        my @ctg = $proj->phrase($text)->$method;
        $analyze{$type} = [ grep{$_} map{ $proj->get_category_by_name($_, $lang) } @ctg ];
    }

    $proj->cmdlog("get_phrase_info clean_minicategs_subphrases_hash");

    do {
        my $h = $phr_obj->orig_minicategs_subphrases_hash;

        push @{$analyze{minicategs_subphrases}}, [$_,
            [ grep{$_} map{ $proj->get_category_by_name($_, $lang) } keys %{$h->{$_}}]] for sort keys %$h;
    };

    $proj->cmdlog("get_phrase_info wordsinf");

    # $proj->dd($vars->{phrase_analysis});
    $analyze{words} = [];
    require Utils::Words;
    for my $word (grep { !/^-/ } Utils::Words::text2words($text, $lang)) {
        my $phr_obj_word = $proj->phrase($word);
        my $inf = $phr_obj_word->word_analyze;

        my @gram = $get_lemmas->($word);
        my $leminfo = join(' | ', map { my ($lem, @info) = @$_; $lem.'('.join(',', @info).')' } @gram);
        $inf->{lemmas} = $leminfo;
        $inf->{search_count} = $phr_obj_word->get_search_count;
        push @{$analyze{words}}, $inf;
    }

    $proj->cmdlog("get_phrase_info end");

    $vars->{Analyze} = \%analyze;
}

sub get_text_info : CMD {
    my ($proj, $vars) = @_;
    return unless $proj->user->rights->{right_moderate};

    $vars->{title} = 'Text analysis';
    $vars->{template} = 'get_text_info.tmpl';

    my $text = $proj->form->{text} || '';
    return if $text eq '';

    my %analyze;
    my %banner_info;
    if ($text =~ /^\d+$/) {
        # возможно, это BannerID
        my $bnr = $proj->bf->get_banner_by_id($text);
        $banner_info{bnr} = $bnr if $bnr;
    }
    if (defined $banner_info{bnr}) {
        my $bnr = $banner_info{bnr};
        $analyze{banner_id} = $bnr->id;
        $text = $bnr->concat_title_body;
    }
    $analyze{text} = $text;

    my $split_sentences = $proj->form->{sentences};
    my $with_structures = $proj->form->{structures};
    my $clarifs = $proj->form->{clarifications};

    my $phr = $proj->phrase($text);
    my $sentences;
    if ($split_sentences) {
        # разбиваем на предложения
        $sentences = $phr->sentences;
    } else {
        # рассматриваем текст как одно предложение
        $sentences = $proj->phrase_list([$phr->text]);
    }
    my @sentences_with_grammar;
    for my $sentence_phr (@$sentences) {
        my @arr;
        push @arr, $sentence_phr->text;
        if ($with_structures) {
            push @arr, join("  ", map {"[$_]"} map {$sentence_phr->word_grammar_info($_)} $sentence_phr->words);
        }
        push @sentences_with_grammar, \@arr;
    }
    $analyze{sentences} = \@sentences_with_grammar;  #[map {$_->text} @$sentences];

    if ($clarifs) {
        my $clarifs = $phr->clarifications;
        $analyze{clarifications} = [@$clarifs];
    }

    $vars->{Analyze} = \%analyze;
}

sub cancel_diff_task : CMD {
    my ($proj, $vars) = @_;
    my $task_id = $proj->form->{task_id};
    $task_id =~ s/\D//g;
    my $task = $proj->banners_categories_diff_meta->Get($task_id);

    if($task_id and $task->{Login} eq $proj->login && $task->{State} eq "New") {
        $proj->Do_SQL("delete from BannersCategoriesDiffMeta where ID=$task_id");
        $proj->Do_SQL("delete from BannersCategoriesDiffInput where ID=$task_id");
        $proj->Do_SQL("delete from BannersCategoriesDiffResults where ID=$task_id");
    }

    $proj->do_redirect("?cmd=banners_categs_diff");
}

sub repeat_diff_task : CMD {
    my ($proj, $vars) = @_;

    return unless $proj->user->rights->{right_spec_admin_actions};

    my $task_id = $proj->form->{task_id};
    $task_id =~ s/\D//g;
    my $task = $proj->banners_categories_diff_meta->Get($task_id);

    if($task_id and $task->{State} eq "Failed") {
        $proj->Do_SQL("update BannersCategoriesDiffMeta set State = 'New' where ID = $task_id");
        $proj->Do_SQL("delete from BannersCategoriesDiffResults where ID=$task_id");
    }

    $proj->do_redirect("?cmd=banners_categs_diff");
}

sub _banners_categs_diff_add {
    my ($proj, $data) = @_;

    my $input = $data->{Input}; delete $data->{Input};
    my $result = $data->{Result}; delete $data->{Result};

    $proj->banners_categories_diff_meta->Add($data);
    my $row_id = $proj->banners_categories_diff_meta->get_last_id();

    $proj->banners_categories_diff_input->Add({ ID => $row_id, Input => $input }) if defined $input;
    $proj->banners_categories_diff_results->Add({ ID => $row_id, Result => $result }) if defined $result;
}

sub banners_categs_diff : CMD {
    my ($proj, $vars) = @_;
    my $lang = $vars->{viewoptions}{lang};

    unless($vars->{rights}->{right_banners_categs_diff}){
        $vars->{text} = 'Error';
        return;
    }

    if($proj->form->{get_result_old}) {
        my $task = $proj->banners_categories_diff_meta->Get($proj->form->{ID});
        my $task_result = $proj->banners_categories_diff_results->Get($proj->form->{ID});

        $vars->{template} = 'text.tmpl';
        $vars->{is_pre} = 1;
        $vars->{text} = Encode::decode_utf8($task_result->{Result});

        return 1;
    } elsif($proj->form->{get_input}) {
        my $task = $proj->banners_categories_diff_meta->Get($proj->form->{ID});
        my $task_input = $proj->banners_categories_diff_input->Get($proj->form->{ID});

        $vars->{template} = 'text.tmpl';
        $vars->{is_pre} = 1;
        $vars->{text} = Encode::decode_utf8($task_input->{Input});

        return 1;
    }

    if($proj->form->{add}) {
        my $task_input = $proj->banners_categories_diff_input->Get($proj->form->{ID});
        my $url =  join("&",
            "ind.pl?cmd=edit_user_phrases",
            "viewoptionsstr=" . $proj->form->{viewoptionsstr},
            "redirect_path=" . URI::Escape::uri_escape_utf8("ind.pl?cmd=banners_categs_diff&viewoptionsstr=" . $proj->form->{viewoptionsstr}),
        );

        my $phl_count = 0;
        for my $line (split "\n", Encode::decode_utf8($task_input->{Input})) {
            my ($categ, $input) = split "\t", $line;
            my $phl = $proj->phrase_list($input);
            $categ = $proj->get_category_by_name($categ);

            next if !$categ;

            my $phl_id = $phl->cache_id;
            my $cat_id = $categ->{CatID};
            $proj->save_phrase_list($phl);
            $phl_count++;
            my $new_url = "$url&phlid$phl_count=$phl_id&catid$phl_count=$cat_id&action$phl_count=Add";
            if(length($new_url) < 2000) {
                $url = $new_url;
            } else {
                last; # слишком длинный урл для редиректа
            }
        }

        $proj->do_redirect($url);
    }

    if($proj->form->{get_result}) {
        my $task = $proj->banners_categories_diff_meta->Get($proj->form->{ID});
        my $dbt = $proj->dbtable('Categs_Diff_Data');
        unless($dbt->Count({ task => $task->{ID}  })){
            my $task_result = $proj->banners_categories_diff_results->Get($proj->form->{ID});
            my $txt = Encode::decode_utf8( $task_result->{Result} );

            my $flds = 'ID title body old_categs new_categs phrase decoded old_categs_input new_categs_input';
            my @arr =
                map {
                    my %h = (); @h{split ' ', $flds} = split /\t/, $_;
                    \%h;
                }
                shuffle( grep { ! /\]\tERROR:/ } split("\n", $txt) );
            for my $el ( @arr ){
                $el->{task} = $task->{ID};
                $el->{bid} = $el->{ID};
                delete($el->{ID});
            }
            $dbt->Add(\@arr);
        }
        $proj->do_redirect("?cmd=banners_categs_diff_data&task=".$task->{ID});

    }

    my $from_moderation = $proj->form->{from_moderation};

    $vars->{tinf} = {
        title => 'Banners Categories Diff Realm',
        table => 'BannersCategoriesDiffMeta',
        readonly => 1,
        idfield => 'ID',
        showid => 1,
        inline_add => 1,
        Moderate => { Add => sub {
            my ($self, $id, $h) = @_;
            _banners_categs_diff_add($self->proj, $h);
        } },
        action_before_add => sub {
            my ($proj, $data) = @_;

            unless ($data->{NumBanners}) {
                die "Number of items must not be void!";
            }

            if ($data->{DiffType} eq 'beta_t') {
                # Allow diff without uploaded file
                $data->{InputFileName} //= '';
            }

            return if $data->{DiffType} ne 'actions'; # return if not a moderation diff task

            my @list = map {
                my ($cat_id, $lang, $initial_phrase_id, $action) = split /~/, $_->{ID};
                {
                    CatID           => $cat_id,
                    Language        => $lang,
                    InitialPhraseID => $initial_phrase_id,
                    Action          => $action,
                }
            } @{$proj->dbtable('ModerationPhrasesBucket')->List({Login => $proj->login})};
            $_->{InitialPhrase} = $proj->get_initial_phrase_by_id($_->{CatID}, $_->{InitialPhraseID}, $_->{Language}, $_->{Action}) for @list;
            $data->{Input} = JSON::to_json(\@list);
            $data->{InputFileName} = 'dummy';
            $proj->dbtable('ModerationPhrasesBucket')->DelList({Login => $proj->login});
        },
        save_button => { title => 'Generate', right => 1, },
        default_field_params => { shlist => 1, edlist => 0, },
        add_filter => { Language => $lang, },
        #header_macro => 'bannermaker_header', #Макрос отрисовки шапки страницы с формой
        fields => [
               { extsql => ['select task ID, count(*) cc, sum(IF(Checked="bad", 1, 0)) checked, sum( IF(UserState="bad", 1, 0)) bads from Categs_Diff_Data where task in (?) group by task', 'ID'],
               },
               {  name => 'Login', title => "Автор", autoedit => sub { $proj->{login} }, },
               {  name => 'UpdateTime', title => "Время создания" },
               {  name => 'DataType', title => "Тип данных", edlist => 1, ftype => "hidden", default => 'banner', },
               {  name => 'DiffType', title => "Тип диффа", edlist => 1, ftype => "select",
                      selectlist => $from_moderation ?
                        [{name => 'Модерация', value => 'actions'}] :
                        [
                              { name => 'Добавление фраз',   value => 'phrases', },
                              { name => 'Преобразование фраз', value => 'transform', rights => 'right_banners_categs_diff_scripts', },
                              { name => 'Изменение в префильтре', value => 'prefilter', rights => 'right_banners_categs_diff_scripts', },
                              { name => 'Замена словаря', value => 'chng_dict', rights => 'right_banners_categs_diff_scripts', },
                              { name => 'Модерация', value => 'actions', },
                              { name => 'actions_custom', value => 'actions_custom', rights => 'right_banners_categs_diff_scripts', },
                              { name => 'chng_dict_light', value => 'chng_dict_light', rights => 'right_banners_categs_diff_scripts', }, # то же, что chng_dict, но создается 'light' Project (без load_minicategs)
                              { name => 'chng_dict_cmptr', value => 'chng_dict_cmptr', rights => 'right_banners_categs_diff_scripts', }, # то же, что chng_dict, но создается 'light' Project (без load_minicategs)
                              { name => 'custom', value => 'custom', },
                              { name => 'custom_light', value => 'custom_light', },
                              { name => 'beta_t', value => 'beta_t' },
                        ],
               },
               { name => "Language", title => 'Language', edlist => 1, shlist => 0, ftype => 'hidden', default => $lang, },
               {  name => 'Comment', title => 'Комментарий', edlist => 1, ftype => 'textarea' },
               {  name => 'Input',
                      edlist => 1,
                      title => 'Upload',
                      ftype => 'file',
                      filenamefield => 'InputFileName',
                      shlist => 0,
               },
               {  name => 'NumBanners', title => "Количество баннеров", edlist => 1 },
               {  name => 'State', title => "Статус" },
               {  extname => 'bads', title => 'B', },
               {  extname => 'checked', title => 'Chd', },
               {  showmacro => "banners_categs_diff_errors ID State Result" },
               {  showmacro => "banners_categs_diff_input ID" },
               {  showmacro => "banners_categs_diff_result ID State" },
               {  showmacro => "banners_categs_diff_add ID State DiffType" },
               {
                    shlist => 1,
                    showmacroel => 'show_btn_field',
                    icon        => 'remove',
                    # TODO fix desc!!!    desc        => "Отменить задание",
                    samewindow => 1,
                    geturl      => sub {
                        my ($el, $f) = @_;
                        if ($el->{State} eq 'Failed' and $proj->user->rights->{right_spec_admin_actions}) {
                            return "?cmd=repeat_diff_task&task_id=" . $el->{ID};
                        }
                        if ($el->{Login} ne $proj->login || $el->{State} ne 'New') {
                            return "";
                        }
                        return "?cmd=cancel_diff_task&task_id=" . $el->{ID};
                    },
               },
           ],
        search => { fields => [ 'Comment', ], name => 'text', title => 'Search by comment' },
        filters => [
               {  name => 'Логин', field => 'Login', grp => 1 },
        ],
        order_by => '-ID',
        pager => { name => 'p', cc => 40, },
        fixed_header => 1,
    };

    $proj->dblist({ vars => $vars })->make;
}

sub banners_categs_diff_data : CMDH {
    my ($proj, $vars) = @_;
    my $form = $proj->{form};

    my $title_add = '';
    my $data_type = $form->{data_type} // ''; # Костыль: $form->{data_type} - для cmd=banners_categs_diff_data_external
    if ($form->{task}) {
        my $id = $form->{task};
        my $tbl = $proj->dbtable('BannersCategoriesDiffMeta', 'ID');
        my %task = %{ $tbl->Get($id) // {} };
        $data_type ||= $task{DataType};
        $title_add .= "<br> $task{Comment}"   if ($task{Comment} // '');
        $title_add .= "<br> (ID=$id $task{DiffType}; $data_type($task{NumBanners}); ";
        $title_add .= " <a href='?cmd=banners_categs_diff&get_input=1&ID=$id'>Input</a>; ";
        $title_add .= " $task{UpdateTime} )";
    };

    return {
        idfield => 'ID',
        title => 'Categories Diff Data' . $title_add,
        showid => 0,
        table => 'Categs_Diff_Data',
        #idfield => 'ID',
        NN => 1,
        fix_sql_problem => 0,
        rights => 'right_banners_categs_diff_data',
        default_field_params => {
            shlist => 1,
            inlinefilter => { group => 0, },
            wwwshowmacro => 'space2nbsp',
        },
        action_onlist => sub {
            my ($proj, $el)=@_;
            my @phr = map { [ split " => ", $_ ] } split ' // ', $el->{'phrase_info'};
            my @phr2 = ();
            for my $el (@phr){
                my @ctgs = split '/', $el->[1];
                @ctgs = ('') unless @ctgs;
                push( @phr2, map { { phrase => $el->[0], categ => $_ } } @ctgs );
            }

            $el->{'phrasesctgs'} = \@phr2;

            if ($data_type eq 'banner') {
                my $bnr = $proj->bf->lbanner($el);
                $el->{title_body_links} = $bnr->title_body_phr->text;
            } else {
                $el->{title_body_links} = $el->{title};
                $el->{title_body_links} =~ s/[^a-zA-Zа-яёА-ЯЁ0-9_\-]/ /g;
            }
        },
        fields => [
            { name => 'task', shlist => 0, addform => 1, },
            { name => 'bid', shlist => 1,
                ( $data_type eq 'banner'  ?
                    (
                        title => 'bid',
                        showmacro => 'bid2link',
                    )  : (
                        title => 'id',
                    )
                ),
            },
            { title => ($data_type eq 'query'  ?  'Text' : 'Баннер'),
                grp => [
                    { name => 'title', },
                    ($data_type eq 'banner'  ?  { name => 'body', }  :  () ),
                    {
                        name => 'title_body_links',
                        showmacro => 'phrase2searchlinks',
                    },
                ],
                inlinefilter => '',
                shlist => 1,
            },
            { name => 'old_categs', showmacro => 'catlist2links', inlinefilter => { group => 1, }, },
            { name => 'old_categs_input', show_banner_saved_categs => 'old_categs_input', },
            { name => 'new_categs', showmacro => 'catlist2links', inlinefilter => { group => 1, }, },
            { name => 'new_categs_input', show_banner_saved_categs => 'new_categs_input', },
            { name => 'phrase',
                showmacroel => 'show_url_field',
                geturl => sub {
                    my ($el, $f) = @_;
                    return ((($el->{phrase} // '') eq ''  or  $el->{phrase} =~ m![\[\]\<\>\/\{\}]!)
                        ?  ""  :  "?cmd=get_phrase_info&phrase_text=".$el->{phrase}
                    )
                },
                inlinefilter => { group => 1, },
            },
            { name => 'decoded',
                inlinefilter => { group => 1, },
            },
            #{ name => 'phrase_info', shlist => 1, },
            { title => 'Фразы', shlist => 0, grp => [
                    { title => 'Фразы', extname => 'phrasesctgs', multigrp => [
                               { name => 'phrase', shlist => 1, },
                               { name => 'categ', shlist => 1, },
                         ], },
                    { name => 'PhraseActions', title => '',  shlist => 1, ftype => 'select', width=>70,
                          selectlist => [
                                  { name => 'Ничего не делаем',   value => '',   default => 1,  color => '#FFFFFF', },
                                  { name => 'Добавить фразы в категорию', value => 'addcateg', default => 1,  color => '#00FF00', },
                                  { name => 'Никогда не добавлять эти фразы', value => 'badcateg', color => '#cd261b', textcolor => '#FFFFFF', },
                                  { name => 'Широкая фраза', value => 'widephr', color => '#cd261b', textcolor => '#FFFFFF', },
                                  { name => 'Широкий брэнд', value => 'widebrand', color => '#cd261b', textcolor => '#FFFFFF', },
                                  { name => 'Неверная категория', value => 'widebrand', color => '#cd261b', textcolor => '#FFFFFF', },
                              ],
                          inline => 1,
                          edlist => 1,
                       },
                ]},
            { title => 'Оценка',
              inlinefilter => 0,
              grp => [
                { name => 'UserState', title => '',  shlist => 1, ftype => 'select', width=>100,
                      selectlist => [
                              { name => 'good',      value => '',   default => 1,  color => '#00FF00', },
                              { name => 'bad',       value => 'bad', color => '#cd261b', textcolor => '#FFFFFF', },
                              { name => 'same',      value => 'same', color => '#9999FF', textcolor => '#FFFFFF', },
                          ],
                      inline => 1,
                      edlist => 1,
                },
                { name => 'Checked', title => '',  shlist => 1, ftype => 'select', width=>100,
                      selectlist => [
                              { name => 'not checked',      value => '',   default => 1,  color => '#808080', },
                              { name => 'checked',       value => 'bad', color => '#0000ff', textcolor => '#FFFFFF', },
                          ],
                      inline => 1,
                      edlist => 1,
                },
            ]},
        ],
        data_import_export => {
            list => [ qw(bid title), (($data_type eq 'banner') ? qw(body) : qw()), qw(old_categs old_categs_input new_categs new_categs_input phrase decoded Checked)],
            no_import => 1,
            export => {
                use_grp => 1,
                filtered => 1,
                headers => 1,
                readable_checked_state => 1,
                readable_diff_categs => 1,
                readable_CategPhrases => [qw(old_categs_input new_categs_input)],
            },
        },

        fldselection => 1,
        filters => [
            {  name => 'Статус', field => 'UserState', grp => 1, use_other_filters => 1, },
            ( map { { field => $_, title => $_, grp => 1, use_other_filters => 1, order_by_count => 1, } } qw( decoded ) ),     # Фильтр по phrase тормозит?
        ],
        readonly => 1,
        pager => { name => 'p', cc => 100, },
        fixed_header => 1,
        order_by => 'bid', # TODO: сейчас баннеры отсортированы в порядке bid, а это плохо, потому что баннеры одной кампании идут друг за другом, переделать на сортировку по рандомному id
    };
}

# Для просмотра в веб-интерфейсе catmedia данных по диффам, загруженных извне
sub banners_categs_diff_data_external : CMDH {
    my ($proj, $vars) = @_;
    my $form = $proj->{form};

    my $task_id = '';
    if ($form->{task}) {
        $task_id = $form->{task};
    }

    return {
        base => 'banners_categs_diff_data',
        title => "Categories Diff Data loaded from external source <br> task_id: $task_id",
        table => 'Categs_Diff_Data_External',
    };
}

sub history_phrase_list : CMD {
    my ($proj, $vars) = @_;
    my $form = $proj->form;
    my $act = $form->{act} || '';

    #Получаем выбранные списки
    sub _get_sel_IDs { my ($proj, $name) = @_;  return split /\D+/, $proj->form->{$name}; }

    sub _get_sel_phlids {
        my ($proj, $name) = @_;
        return map { $_->{phlid} } @{ $proj->phrase_list_action_log->List( { ID => [ _get_sel_IDs($proj, $name) ] } ) };
    }

    if($proj->phrase_list->can($act)){ #Выполнение парных методов
        my @arra = _get_sel_phlids($proj,'slctd');
        my $phla = $proj->phrase_list;
        $phla += $proj->get_phrase_list($_) for @arra;
        my @arrb = _get_sel_phlids($proj,'slctdb');
        my $phlb = $proj->phrase_list;
        $phlb += $proj->get_phrase_list($_) for @arrb;
        my $resphl = $phla->$act($phlb);
        $proj->save_phrase_list($resphl);
        $proj->do_redirect(join("&",
            "ind.pl?cmd=edit_phrase_list",
            "act=showphl",
            "phlid=".$resphl->cache_id,
            "viewoptionsstr=".$form->{viewoptionsstr},
        ));

    }
    $vars->{list} = $proj->phrase_list_action_log->List( { login => $proj->login } );
    $_->{edit} = 1 for @{$vars->{list}};
    $vars->{sharing_list} = $proj->phrase_list_action_log->List( [[" sharing != ''"]] );
    my $h = {};
    push( @{$h->{$_->{dir}||'unsorted'}||=[]}, $_) for sort { $b->{date} cmp $a->{date} } @{$vars->{list}};

    #прописывает путь aa/ss/dd вниз по иерархии
    sub _path2h {
        my ($h, $path, $els) = @_;
        my @pth = split('/', $path);
        my $lvl = { sblist => $h };
        my $curpath = '';
        for my $p (@pth){
            $curpath .= ($curpath ? '/' : '').$p;
            my $dr = [ grep { $_->{name} eq $p} @{$lvl->{sblist}} ]->[0];
            unless( $dr ){
               $dr = {name => $p, path => $curpath, sblist => []};
               push(@{$lvl->{sblist}}, $dr);
            }
            $lvl = $dr;
        }
        push(@{$lvl->{sblist}}, @$els);
    }

    #$vars->{dirs} = [ map { { name => $_, sblist => $h->{$_} } } keys %$h ];
    $vars->{dirs} = [];
    _path2h( $vars->{dirs}, $_, $h->{$_} ) for keys %$h;
    my $shh = {};
    push( @{$shh->{$_->{sharing}}||=[]}, $_) for @{$vars->{sharing_list}};
    push(@{$vars->{dirs}}, { name => '_sharing', sblist => [  map { { name => $_, sblist => $shh->{$_} } } keys %$shh ]} );

    #Группируем несортированные по месяцам
    my $uns = [ grep { $_->{name} eq 'unsorted'} @{$vars->{dirs}} ]->[0];
    if($uns){
        $uns->{name} = '_unsorted';
        my $unsrt = $uns->{sblist};
        my $hh = {};
        push( @{$hh->{$proj->dates->trdate('db', 'week', $_->{date})||'nodate'}||=[]}, $_) for @$unsrt;
        $uns->{sblist} = [ map { { name => $_, sblist => $hh->{$_} } } sort { $b cmp $a } keys %$hh ];
    }

    $vars->{dirs} = [ sort { $a->{name} cmp $b->{name} } @{$vars->{dirs}} ];

    $vars->{template} = 'PraseInfo/history_phrase_list.tmpl';
}

sub ajax_history_phl_chdir : CMD {
    my ($proj, $vars) = @_;
    my $form = $proj->form;
    my %h = (
        dir => $form->{dir},
    );
    $proj->phrase_list_action_log->Edit($form->{ID}, \%h);
    $vars->{text} = Dumper(\%h);
}

sub change_phrase_list_inf : CMD {
    my ($proj, $vars) = @_;
    my $form = $proj->form;
    my $act = $form->{act} || '';
    if($act eq '_saveinf'){
        my %h = ( map { $_ => $form->{$_} }  qw{ name dir sharing comment } );
        $proj->phrase_list_action_log->Edit($form->{ID}, \%h);
        $proj->do_redirect(join("&",
            "ind.pl?cmd=history_phrase_list",
            "act=show",
            "viewoptionsstr=".$form->{viewoptionsstr},
        ));
    }
    $vars->{el} = $proj->phrase_list_action_log->Get($form->{ID});
    $vars->{template} = $form->{pagestyle} ? 'PraseInfo/change_phrase_list_inf.tmpl' : 'PraseInfo/change_phrase_list_inf_form.tmpl';
}

sub edit_user_phrases : CMD {
    my ($proj, $vars) = @_;

    my $lang = $vars->{viewoptions}{lang} || 'ru';
    $proj->current_lang($lang);

    my $cat_id_for_redirect;
    my $result = [];
    my $input = [ ["lang", $lang] ];
    my $saved = 0;
    my $do_save = $proj->form->{do_save};
    my @bad_phrases;
    my @all_categs;
    my $redirect_path = $proj->form->{redirect_path};

    # список всех упоминаемых категорий
    if(!$do_save) {
        for my $key_name (sort keys %{$proj->form}) {
            my ($i) = $key_name =~ /^phlid(\d+)$/;
            next if !$i || !$proj->form->{"catid$i"} || !$proj->form->{"action$i"};

            my $phlid = $proj->form->{"phlid$i"};
            my $phl = $proj->get_phrase_list($phlid);

            my %h;
            for my $phr ($phl->phrases) {
                $h{$_}++ for $phr->get_minicategs;
            }

            push @all_categs, $_ for
                sort{$a->{CategoryName} cmp $b->{CategoryName}}
                map{$proj->get_category_by_name($_, $lang)}
                keys %h;
        }
    }

    my %done_ids;
    for my $key_name (sort keys %{$proj->form}) {
        my ($i) = $key_name =~ /^phlid(\d+)$/;
        next if !$i || !$proj->form->{"catid$i"} || !$proj->form->{"action$i"};

        my $phlid = $proj->form->{"phlid$i"};
        my $phl = $proj->get_phrase_list($phlid);
        my $categ = $proj->get_category($proj->form->{"catid$i"}, $lang);
        my $action = $proj->form->{"action$i"};
        $cat_id_for_redirect ||= $proj->form->{"catid$i"};

        # проверяем, что категория задана корректно
        if(!$categ) {
            push @bad_phrases, $_->text for $phl->phrases;
            next;
        }

        push @$input, [$_, $proj->form->{$_}] for ("phlid$i", "catid$i", "action$i");

        for my $phr ($phl->phrases) {
            # пропускаем пустые фразы
            next if !$phr->text;

            my $id = "phr" . $phlid . "_" . $proj->get_user_phrase_id($phr->text);
            # пропускаем дублирующиеся фразы
            next if $done_ids{$id};
            $done_ids{$id}++;

            # сохранение фразы
            if($do_save) {
                next if !$proj->form->{$id};

                $categ = $proj->get_category($proj->form->{"categ_$id"}, $lang);
                next if !$categ;

                my $user_comment = $proj->form->{"comment_$id"};
                $proj->add_user_phrase($categ->{CatID}, $phr->text, $lang, $action, undef, $user_comment);
                $saved++;
            }

            push @$result, {
                ID              => $id,
                CatID           => $proj->form->{"catid$i"},
                Language        => $lang,
                InitialPhrase   => $phr->text,
                snorm_phr       => (($phr->text !~ /\[/) ? $phr->snorm_phr : "<i>атом</i>"),
                Category        => $categ,
                Categories      => [$categ, grep{$categ->{CatID} ne $_->{CatID}} @all_categs],
                CurrCategs      => join("/", map{$phr->category_from_ru($_)} $phr->get_minicategs),
                Action          => $action,
            };
        }
    }
    $proj->edit_user_phrases_warnings($result);

    if ($do_save) {
        if ($redirect_path =~ /^ind\.pl\?cmd=edit_phrase_list/) {
            my $h = $proj->phrase_list->web_get_category_full($cat_id_for_redirect, $lang);
            my $phl = $proj->phrase_list($h->{CategoryPhrases});
            $proj->save_phrase_list($phl);
            $proj->do_redirect($redirect_path . '&phlid=' . $phl->cache_id . '&act=showphl');
        } elsif ($redirect_path =~ /^ind\.pl\?cmd=show_phrases/) {
            $proj->do_redirect($redirect_path);
        }
    }

    if(@bad_phrases) {
        $vars->{bad_phrases} = join ",", @bad_phrases;
    }

    $vars->{redirect_path} = $redirect_path;
    $vars->{saved} = $saved;
    $vars->{input} = $input;
    $vars->{phrases} = $result;
    $vars->{template} = "edit_user_phrases.tmpl";
}

sub edit_user_phrases_bin {
    my ($proj, $vars) = @_;
    edit_user_phrases($proj, $vars);
    $vars->{template} = "edit_user_phrases_bin.tmpl";
}

sub edit_phrase_list : CMD {
    my ($proj, $vars) = @_;

    my $prlang = $vars->{viewoptions}{lang} || 'ru';
    $proj->current_lang($prlang);

    my $CurCatID = $vars->{viewoptions}{'curcatid'} || '';
    $vars->{'curcatinf'} = $proj->phrase_list->web_get_category($CurCatID, $prlang) if $CurCatID;

    # регионы
    my $regions = $proj->current_region;
    if(@$regions){
        require geobaselite;
        $vars->{regions} = join(",", map{$geobaselite::Region{$_}->{name}} @$regions);
    }else{
        $vars->{regions} = "все";
    }

    $vars->{'title'} = 'Редактирование группы фраз';
    my $form = $proj->form;
    my $act = $form->{act} || '';
    if($form->{phrases_text} || ($act eq 'showphl') || ($act eq '_load_file') || ($act =~ /^_urlinf/) || $form->{'loadphlid'} || ($act eq 'random_banners') ){
        #my $nltype = $form->{nltype} || ','; #тип разделителя между фразами
        my $nltype = ''; #тип разделителя между фразами
        $nltype = $vars->{viewoptions}{delim} if $vars->{viewoptions}{delim};
        $nltype ||= 'n';
        $vars->{viewoptions}{delim} = $nltype;
        if($act eq 'showphl'){ #Отображение фраз
            my @phrs = @{$proj->get_phrase_list($form->{phlid})->perl_array};
            $vars->{phrases_count} = grep {! /^#/} grep {$_} map {split /\n/, $_} @phrs;

            #Автоматическое ограничение показываемых фраз
            if(    ($vars->{phrases_count} > 20000)          # Много фраз
                && (! defined $vars->{viewoptions}{tshw})    # Нет явно установленных настроек
                && (! $CurCatID )                            # Это не отображение фраз категории
            ){
                $vars->{viewoptions}{tshw} ||= 20000;
            }

            @phrs = splice(@phrs, 0, $vars->{viewoptions}{tshw}) if $vars->{viewoptions}{tshw};
            my $rphl = $proj->phrase_list(\@phrs);
            #my $dphl = $proj->get_phrase_list($form->{'delphlid'});
            my $hndadd = $proj->get_phrase_list($form->{'hndadd'});
            my $hnddel = $proj->get_phrase_list($form->{'hnddel'});
            if( $vars->{viewoptions}{showctgs} ){
                $vars->{cattree} = $rphl->get_minicategs_tree;
                $vars->{cattree_text} = $rphl->get_minicategs_tree_text($vars->{cattree});
                $nltype = 'n';
                my ($atomsphl, $otherphl) = $rphl->divide(sub { "$_" =~ /\[/ });
                $vars->{categslist} = [ $otherphl->get_minicategs ];
                $rphl = $vars->{viewoptions}{showcounts} ? $otherphl->test_categs_count_info : $otherphl->test_categs_info;
                #print "Content-type: text/html\n\n";
                #print "<pre>";
                ##print Dumper($vars->{viewoptions}{showcounts});
                #print $otherphl->test_categs_info;
                #exit;
                if( $vars->{viewoptions}{logctgs} ){
                    $rphl = $rphl->test_categslog_info;
                    $rphl = $rphl->tmap( sub { s/ \/\/ /\n/g; $_ } );
                }
                $rphl += $atomsphl;
            }elsif($vars->{viewoptions}{showcounts}){
                $nltype = 'n';
                my ($atomsphl, $otherphl) = $rphl->divide(sub { "$_" =~ /\[/ });
                $rphl = $otherphl->test_count_info;
                $rphl += $atomsphl;
            } elsif($vars->{viewoptions}{showmediagroups}) {
                $vars->{groups} = $rphl->get_mediagroups_hash;
                $vars->{groups_type} = "mediagroups";
                $vars->{groups_title} = "Медиагруппы:";
                $vars->{groups_nogroup} = "без медиагрупп";
                $rphl = $rphl->add_mediagroups;
            } elsif($vars->{viewoptions}{showcatalogiaflags}) {
                $vars->{groups} = $rphl->get_catalogia_flags_hash;
                $vars->{groups_type} = "catalogia_flags";
                $vars->{groups_title} = "Флаги:";
                $vars->{groups_nogroup} = "без флагов";
                $rphl = $rphl->add_catalogia_flags;
            }
            my $dlmtr = $nltype eq 'n' ? "\n" : ', ';
            $vars->{phrases_text} = join($dlmtr, @$rphl);
            $vars->{phrases_text} =~ s/^\s+|\s+$//g;
            #$vars->{deleted_phrases} = "$dphl";
            $vars->{hndadd} = $hndadd;
            $vars->{hnddel} = $hnddel;
        }elsif($act){ #Обработка фраз
            my $text = $form->{phrases_text};
            $text =~ s/=(\&gt;|\>)[^\n]*(\n|$)/\n/g;
            $text =~ s/(^|\n)#[^\n]*(\n|$)/\n/g;

#            $text =~ s/<[^\<\>]+>/ /g;
#            $text =~ s/\r?\n/, /g if $nltype eq 'n';
            $text =~ s/(?:(?:\r?\n|,)\s*)+/,/g; #Перевод строки - тоже разделитель
            unless($act =~ /^web_/){
                #Удаляем тэги
                $text =~ s/<br>/\n/g; #Перевод строки - тоже разделитель
                $text =~ s/<\/p>/\n/g; #Конец абзаца - тоже разделитель
            }
            if(0){ #Отключили удаление тегов, так как это ломает логику фраз категорий
                #Удаляем тэги
                $text =~ s/<a [^<>]*>.*?<\/a>/ /g;
                $text =~ s/<[A-Za-z][^\<\>]+>/ /g; #Удаляем теги
            }
            $text =~ s/\s*,\s*/,/g; #Перевод строки - тоже разделитель
            my $phl = $proj->phrase_list($text)->lgrep(sub { ! /^#/ });
            if( $vars->{viewoptions}{tshw} || $form->{'nosavingdata'} ){
                $phl = $form->{'prevphlid'} ? $proj->get_phrase_list($form->{'prevphlid'}) : $proj->phrase_list;
                $phl = $phl->lgrep(sub { ! /^#/ })->tmap(sub { s/=(\&gt;|\>)[^\n]*(\n|$)//g; $_ }); #Очищаем фразы от мусора перед обработкой
            }
            if( $form->{'loadphlid'} ){
                $phl = $proj->get_phrase_list($form->{'loadphlid'});
            }
            my $prev_phl = $form->{'prevphlid'} ? $proj->get_phrase_list($form->{'prevphlid'}) : $proj->phrase_list;
            my $prev_dphl = $form->{'prevdelphlid'} ? $proj->get_phrase_list($form->{'prevdelphlid'}) : $proj->phrase_list;
            #$phl = $phl->tmap(sub { s/=\&gt;.*//g; $_ });
            my $rphl = undef;
            if($act =~ /^_/){
                my $rphlredir = 0;
                if( $form->{'textre'} && ($act eq '_text_delphrases')){
                    my $re = $form->{'textre'};
                    $rphl = $phl->tgrep(sub { ! /$re/i });
                }elsif( $act eq '_text_snormfilter' ){
                    my $fphl = $proj->phrase_list( $form->{'textre'} );
                    $rphl = $form->{'textre'} ? $phl->search_sub_phrases($fphl) : $phl;
                }elsif( $act eq '_text_delphrases2' ){
                    my $re = $form->{'textre'};
                    $rphl = $re ? $phl->tgrep(sub { /$re/i }) : $phl;
                }elsif( $act eq '_text_delsubphrase' ){
                    my $re = $form->{'textre'};
                    $rphl = $re ? $phl->tmap(sub { s/$re//ig; $_ }) : $phl;
                }elsif( $act eq '_text_analyse_phrase_list' ){
                    $proj->save_phrase_list($phl);
                    $proj->do_redirect( join("&",
                        #"ind.pl?cmd=insert_phrase_list",
                        "ind.pl?cmd=lite_insert_phrase_list",
                        "phlid=".$phl->cache_id,
                        "viewoptionsstr=".$form->{viewoptionsstr},
                    ));
                }elsif( $form->{'textre'} && ($act eq '_text_list2atoms')){
                    my @arr = $proj->phrase($form->{'textre'})->uniqsnormwords;
                    my $atom = '['.join('/', @arr).']';
                    my $flt = { map {$_=>1} @arr };
                    my @res = ();
                    for my $ph (@$phl){
                        my @wrds = $ph->uniqsnormwords;
                        if( grep {$flt->{$_}} @wrds ){
                            push(@res, $atom.' '.join(' ', grep {!$flt->{$_}} @wrds));
                        }else{
                            push(@res, $ph);
                        }
                    }
                    $rphl = $proj->phrase_list(\@res)->text_pack_list;
                }elsif( $act eq '_get_domain_site_tree' ){
                    my @domains = map {"$_"} $phl->phrases;
                    for my $d (@domains){
                        my $site = $proj->site($d);
                        my $res = $site->get_tree_nw;
                        my $htxt = $res->get_hierarchy_text;
                        $rphl = $proj->phrase_list($htxt);
                    }
                }elsif( $act eq '_get_domain_site_menu' ){
                    my @domains = map {"$_"} $phl->phrases;
                    for my $d (@domains){
                        my $site = $proj->site($d);
                        #my $res = $site->menu_pages->context_badre_filter->delete_bad_tmpls;
                        my $res = $site->menu_pages->filter_bad_urls->delete_bad_tmpls;
                        my $htxt = $res->pages2categstext;
                        $htxt =~ s/, ?/ /g;
                        $rphl = $proj->phrase_list($htxt);
                    }
                }elsif( $act eq '_get_listurls_phrases' ){
                    $rphl = $proj->phrase_list;
                    for my $inf ( $phl->phrases ){
                        my ($mrk, $url) = split(' =-> ', $inf);
                        my $urlphl = $proj->page($url)->get_hierarchy_subpages_phl->tmap(sub {s/=>.*$//; "$_ =*> $mrk"});
                        $rphl += $urlphl;
                    }
                }elsif( $act eq '_get_domain_phrases' ){
                    my @domains = map {"$_"} $phl->phrases;
                    $| = 1;
                    #print "Content-type: text/html\n\n";
                    #print "<pre>";
                    #print Dumper(\@domains);
                    my $dinf = {};
                    my $phltext = $proj->phrase_list->get_direct_domains_report(\@domains);
                    my $cacheid = md5int(time.'sdf'.rand(10000).join(',',@domains[0 .. 10]));
                    $proj->phrase_list_cache->Add(
                        {
                            PhraseListID    => $cacheid,
                            Phrases         => $phltext,
                        },
                        { replace => 1}
                    );
                    my $actid = $proj->add_phrase_list_action({
                        phlid => $cacheid,
                        act => $act,
                        prevphlid => $phl->cache_id,
                        name => 'PhraseList '.$cacheid,
                        dir => 'unsorted',
                    });
                    $proj->do_redirect(join("&",
                        "ind.pl?cmd=edit_phrase_list",
                        "act=showphl",
                        "phlid=".$cacheid,
                        "viewoptionsstr=".$form->{viewoptionsstr}."|delim_$nltype",
                    ));
                }elsif( $act eq '_get_fltrd_domain_phrases_detailed' ){
                    my @domains = map {"$_"} $phl->phrases;
                    $| = 1;
                    #print "Content-type: text/html\n\n";
                    #print "<pre>";
                    #print Dumper(\@domains);
                    my $phltext = $proj->phrase_list->get_direct_domains_top(\@domains);
                    my $cacheid = md5int(time.'sdf'.rand(10000).join(',',@domains[0 .. 10]));
                    $proj->phrase_list_cache->Add(
                        {
                            PhraseListID    => $cacheid,
                            Phrases         => $phltext,
                        },
                        { replace => 1}
                    );
                    my $actid = $proj->add_phrase_list_action({
                        phlid => $cacheid,
                        act => $act,
                        prevphlid => $phl->cache_id,
                        name => 'PhraseList '.$cacheid,
                        dir => 'unsorted',
                    });
                    $proj->do_redirect(join("&",
                        "ind.pl?cmd=edit_phrase_list",
                        "act=showphl",
                        "phlid=".$cacheid,
                        "viewoptionsstr=".$form->{viewoptionsstr}."|delim_$nltype",
                    ));
                }elsif( $act eq '_text_addphrases' ){
                    my $textre = $form->{'textre'} || '';
                    $textre =~ s/,/ /g if $textre =~ /=.?>/;
                    my $addphl = $proj->phrase_list($textre);
                    $rphl = $phl * $addphl;
                }elsif( $act eq '_text_addphrases_in_beginning' ){
                    my $textre = $form->{'textre'} || '';
                    $textre =~ s/,/ /g if $textre =~ /=.?>/;
                    my $addphl = $proj->phrase_list($textre);
                    $rphl = $addphl * $phl;
                }elsif( $act eq '_text_addphrases2' ){
                    my $textre = $form->{'textre'} || '';
                    $textre =~ s/,/ /g if $textre =~ /=.?>/;
                    my $addphl = $proj->phrase_list($textre);
                    $rphl = $phl * $addphl + $phl;
                #}elsif( $act eq '_text_check_phrase_context' ){
                #    $rphl = $phl->check_phrase_context($form->{'textre'} || '');
                }elsif( $act =~ /^_text_([-_A-Za-z0-9]+)$/ ){
                    my $mthd = $1;
                    if( $rphl = $phl->can($mthd) ){ #Если есть метод phrase_list'а
                        $rphl = $phl->$mthd($form->{'textre'} || '');
                    }
                }elsif( $act eq '_load_file' ){
                    $rphl = $proj->load_phl_from_form_file('loadedfile');
                    $rphlredir = 1;
                }elsif( $act eq '_urlinf_get_list_from_url' ){
                    $rphl = $proj->page($form->{'exturl'})->get_hierarchy_subpages_phl;
                    #print "Content-type: text/html\n\n";
                    #print '<pre>'.Dumper($form->{'exturl'});
                    #print "$rphl";
                    #my $p = $proj->page('http://euroncap.com/supermini.aspx');
                    #print $p->get_internal_subpages_pgl;
                    #exit;
                    $rphlredir = 1;
                }elsif( $act eq '_urlinf_get_list_from_url_textonly' ){
                    my $s = $proj->site($form->{'exturl'});
                    #my $nd = $s->node($form->{'exturl'});
                    #my $goods = $nd->goods;
                    my $goods = $proj->phrase_list([ $form->{'exturl'}])->pages2goods;
                    $rphl = $proj->phrase_list([ map { "$_"} @$goods ]);
                    $rphlredir = 1;
                }elsif( $act eq '_urlinf_get_list_from_url_log' ){
                    my $logtext = $proj->page($form->{'exturl'})->get_hierarchy_subpages_log; #->tmap(sub {s/=-?>.*$//; $_});
                    $rphl = $proj->phrase_list($logtext);
                    $rphlredir = 1;
                }elsif( $act eq '_urlinf_get_list' ){
                    my $pgl = $proj->page($form->{'exturl'})->get_subpages_pgl->pack_urls;
                    $rphl = $proj->phrase_list([ map {"$_"} @$pgl ]);
                    $rphl = $rphl->tmap(sub {s/,/ /g; $_});

                    #print "Content-type: text/html\n\n";
                    #print '<pre>'.Dumper($form->{'exturl'});
                    #print "$rphl";
                    #my $p = $proj->page('http://euroncap.com/supermini.aspx');
                    #print $p->get_internal_subpages_pgl;
                    #exit;
                    $rphlredir = 1;
                }elsif( $act eq '_delete_file' ){
                    my $filephl = $proj->load_phl_from_form_file('deletefile');
                    $filephl = $filephl->get_norm_phr_forms;
                    $phl = $phl->get_norm_phr_forms;
                    $rphl = $phl - $filephl;
                    $rphlredir = 1;
                }elsif( $act eq '_add_file' ){
                    my $filephl = $proj->load_phl_from_form_file('addfile');
                    $rphl = $phl + $filephl;
                    $rphlredir = 1;
                }elsif( $act eq '_intersect_files' ){
                    my $filephl = $proj->load_phl_from_form_file('intersectfile');
                    $_ = $_->get_norm_phr_forms for $filephl, $phl;
                    $rphl = $phl ** $filephl;
                    $rphlredir = 1;
                }elsif( $act eq '_get_new_phrases_from_file' ){
                    my $filephl = $proj->load_phl_from_form_file('newphrasefile');
                    $rphl = $proj->phrase_list($proj->phrase("$filephl")->lgt_phrases_filtering("$phl"));
                    #$_ = $_->get_norm_phr_forms for $filephl, $phl;
                    $rphlredir = 1;
                }elsif( $act eq '_categs_filter' ){
                    my $categs = $form->{'categtext'};
                    if($form->{'categtext_incl'}){
                        my %ctf = map {$_=>1} split ',', $categs;
                        $rphl = $phl->lgrep(sub { ! ((grep { $ctf{$_} } $_->get_minicategs) && (!grep {! $ctf{$_} } $_->get_minicategs)) });
                    }else{
                        my %ctf = map {$_=>1} split ',', $categs;
                        $rphl = $phl->lgrep(sub { ! grep { $ctf{$_} } $_->get_minicategs });
                    }
                }elsif( $act eq '_categs_svrl' ){
                    my $categs = $form->{'categtext'};
                    $rphl = $phl->lgrep(sub { my @ctgs = $_->get_minicategs; @ctgs > 1 });
                }elsif( $act eq '_categs_unctg' ){
                    my $categs = $form->{'categtext'};
                    $rphl = $phl->lgrep(sub { my @ctgs = $_->get_minicategs; @ctgs == 0 });
                }elsif( $act eq '_categs_catid_filter' ){
                    my $categs = $form->{'categtext'};
                    my %ctf = map { $phl->get_minicateg_by_id( $_ ) => 1 } split ',', $categs;
                    $rphl = $phl->lgrep(sub { grep { $ctf{$_} } $_->get_minicategs });
#                    print "content-type: text/html\n\n";
#                    print '<pre>'.Dumper(\%ctf,  $categs);
#                    print "$rphl";
#                    exit;
                }elsif( $act eq '_categs_delfreqcateg' ){
                    my %ctgs = map { $_->get_minicategs, } @$phl;
                    if(keys %ctgs){
                        my @top = h2sa(\%ctgs);
                        my $tpctg = shift @top;
                        $rphl = $phl->lgrep(sub { ! grep {$tpctg->[0] eq $_ } $_->get_minicategs });
                    }else{
                        $rphl = $phl;
                    }
                }elsif( $act eq '_categs_addphrases' ){
                    $proj->cmdslist->{'add_categs_phrases'}->($proj, $vars);
                }elsif($act eq "_categs_delphrases") {
                    my $h = parse_categs_phrases($proj, $vars, 1);
                    if(%$h) {
                        my $i = 1;
                        my @a;
                        for my $catid (keys %$h) {
                            push @a, "catid$i=$catid";
                            push @a, "phlid$i=" . $h->{$catid};
                            $i++;
                        }
                        $proj->do_redirect(join("&",
                            "ind.pl?cmd=delete_phrases",
                            "viewoptionsstr=".$form->{viewoptionsstr},
                            @a
                        ));
                    }
                }elsif($act eq "_categs_delete_from_categories") {
                    my @lines = split "\n", $proj->form->{phrases_text};
                    my $h = {};

                    for my $line (@lines) {
                        chomp $line;
                        $line =~ s/\r//g;
                        my @data = split "=>", $line;
                        next if @data != 3;
                        $data[2] =~ s/^\s+|\s+$|\r|\n//g;
                        next if !$data[1] || !$data[2];
                        ($h->{$data[1]} ||= {})->{$data[2]}++;
                    }

                    my @phlid;
                    my @fromcatid;
                    for my $categ_name (keys %$h) {
                        my $catid = $proj->phrase_list->get_minicateg_id_lang($categ_name, $vars->{viewoptions}{lang});
                        next if !$catid;

                        my $phl_to_del = $proj->phrase_list([ keys %{$h->{$categ_name}} ]);
                        $proj->save_phrase_list($phl_to_del);
                        push @phlid, $phl_to_del->cache_id;
                        push @fromcatid, $catid;
                    }

                    my $n = @phlid;
                    $proj->do_redirect(join("&",
                        "ind.pl?cmd=delete_phrases",
                        "viewoptionsstr=".$form->{viewoptionsstr},
                        (map{"phlid$_=".$phlid[$_ - 1]} 1..$n),
                        (map{"fromcatid$_=".$fromcatid[$_ - 1]} 1..$n),
                        (map{"catid$_="} 1..$n),
                    ));
                }elsif( $act eq '_saveinf' ){
                    $rphl = $phl;
                    $rphlredir = 1;
                }elsif( $act eq '_save_as' ){
                    $rphl = $phl;
                    $proj->save_phrase_list($_) for $phl;
                    my $actid = $proj->add_phrase_list_action({
                        phlid => $rphl->cache_id,
                        act => $act,
                        prevphlid => $phl->cache_id,
                        name => 'PhraseList '.$rphl->cache_id,
                        dir => 'unsorted',
                    });
                    $proj->do_redirect(join("&",
                        "ind.pl?cmd=change_phrase_list_inf",
                        "act=show",
                        "pagestyle=1",
                        "ID=".$actid,
                        "viewoptionsstr=".$form->{viewoptionsstr},
                    ));

                }elsif($act eq '_web_change_category_phrases'){
                    my @lines = map{$_ =~ s/ =>.*//; $_} split /(?:\r?\n)+/, $proj->form->{phrases_text};
                    my $phl = $proj->phrase_list(\@lines);
                    my ($phrases_to_add, $phrases_to_del) = $phl->web_change_category_diff($CurCatID);
                    $proj->save_phrase_list($phrases_to_add);
                    $proj->save_phrase_list($phrases_to_del);
                    $proj->do_redirect(join("&",
                        "ind.pl?cmd=edit_user_phrases",
                        "catid2=". $CurCatID,
                        "phlid2=". $phrases_to_add->cache_id,
                        "action2=Add",
                        "catid1=" . $CurCatID,
                        "phlid1=" . $phrases_to_del->cache_id,
                        "action1=Delete",
                        "viewoptionsstr=" . $proj->form->{viewoptionsstr},
                        "redirect_path=" . URI::Escape::uri_escape_utf8("ind.pl?cmd=edit_phrase_list&viewoptionsstr=" . $proj->form->{viewoptionsstr})
                    ));
                }else{
                    die("Unknown act='$act'");
                }
                if($rphlredir){
                    #$proj->save_phrase_list($rphl);
                    $proj->save_phrase_list($_) for $rphl, $phl;
                    if(1){ #Сохраняем лог действий
                        $proj->add_phrase_list_action({
                            phlid => $rphl->cache_id,
                            act => $act,
                            prevphlid => $phl->cache_id,
                            name => 'PhraseList '.$rphl->cache_id,
                            dir => 'unsorted',
                        });
                    }
                    $proj->do_redirect(join("&",
                        "ind.pl?cmd=edit_phrase_list",
                        "act=showphl",
                        "phlid=".$rphl->cache_id,
                        "viewoptionsstr=".$form->{viewoptionsstr}."|delim_$nltype",
                    ));
                }
            } elsif ($act eq 'get_txt_cp1251') {
                $vars->{_return_textfile} = 'phrases.txt';
                $vars->{text} = join("\n", @$phl);
                $vars->{text} = Encode::encode('windows-1251', $vars->{text});
                return;
            }elsif($act eq 'get_txt'){
                $vars->{'_return_textfile'} = 'phrases.txt';
                $vars->{text} = join("\n", @$phl);
                return;
            }elsif($act eq 'get_txt_tabdelim'){
                $vars->{'_return_textfile'} = 'phrases.txt';
                $vars->{text} = join("\n", @$phl);
                $vars->{text} =~ s/\s*=.?>\s*/\t/g;
                return;
            }elsif($act eq 'get_xls'){
                $vars->{'_return_cp1251_xls'} = 'phrases.xls';
                $vars->{text} = $form->{'phlid'}.join("\n", @$phl);
                return;
            }elsif($act eq 'get_real_xls'){
                $vars->{'_return_real_xls'} = 'phrases.xls';
#                $vars->{text} = $proj->phl2xls($rphl);
                $vars->{text} = $phl->get_xls_multicol;
                return;
            }elsif($act eq 'get_real_xls_info'){
                $vars->{'_return_real_xls'} = 'phrases.xls';
                $phl = $phl->tmap(sub { s/=>.*//g; $_ });
                $vars->{text} = $phl->get_xls_info;
                return;
            }elsif($act eq 'get_real_xls_info_by_mails'){
                #$vars->{'_return_real_xls'} = 'phrases.xls';
                #$phl = $phl->tmap(sub { s/=>.*//g; $_ });
                my $cc = 0;
                my @lst = $phl->split_by_count(100_000);
                for my $cphl ( @lst ){
                    $cc++;
                    my $tbeg = $proj->curtime;
                    my $data = $cphl->get_xls_info;
                    my $tend = $proj->curtime;
                    my $dt = {
                        filename => 'phrases_'.$proj->curdate.'_'.$cc.'.xls',
                        data => $data,
                        type => 'application/vnd.ms-excel',
                    };
                    $proj->SendMail({
                        subject => 'Выгрузка фраз '.$phl->cache_id.' '.$cc.' из '.@lst,
                        body    => "Выгрузка фраз с категориями и частотами\nначало $tbeg\nокончание $tend",
                        to => $proj->user->user_inf->{EMail},
                        attachments => [$dt],
                    });
                }
                #$vars->{text} = 'Данные были посланы письмом';
                #return;
            }elsif($act eq 'get_real_xls_testinfo'){
                my $rphl = $proj->phrase_list([split(/\r?\n|,/, $form->{phrases_text} )]);
                $vars->{'_return_real_xls'} = 'phrases.xls';
                $vars->{text} = $rphl->get_xls;
                return;
            }else{
                if($act =~ /atoms2list/ || $act =~ /^web_/){
                    $rphl = $phl->$act;
                }else{
                    #Убираем атомы
                    my ($atomsphl, $otherphl) = $phl->divide(sub { "$_" =~ /\[/ });
                    $otherphl->{'projservlogin'} = $proj->login;
                    $rphl = $form->{'spparam1'} ? $otherphl->$act($form->{'spparam1'}) : $otherphl->$act;
                    $rphl += $atomsphl; #Доклеиваем атомы обратно
                }
            }
            $rphl = $phl unless defined $rphl;
            $proj->save_phrase_list($_) for $rphl, $phl;
            if(1){ #Сохраняем лог действий
                $proj->add_phrase_list_action({
                    phlid => $rphl->cache_id,
                    act => $act,
                    prevphlid => $phl->cache_id,
                    name => 'PhraseList '.$rphl->cache_id,
                    dir => 'unsorted',
                });
            }
            my @additionalinf = ();
            if($vars->{viewoptions}{additionalinf}){
                my $dphl = $phl - $rphl;
                my $hndadd = $rphl - $prev_phl;
                my $hnddel = $prev_phl - $rphl;
                $proj->save_phrase_list($_) for $dphl, $hndadd, $hnddel;
                @additionalinf = (
                    "delphlid=".$dphl->cache_id,
                    "hndadd=".$hndadd->cache_id,
                    "hnddel=".$hnddel->cache_id,
                );
            }

            if($vars->{viewoptions}{markchanges}){
                $rphl = $prev_phl->delete_all_test_info->get_difference($rphl->delete_all_test_info);
                $proj->save_phrase_list($rphl);
            }
            $proj->do_redirect(join("&",
                "ind.pl?cmd=edit_phrase_list",
                "act=showphl",
                "phlid=".$rphl->cache_id,
                "viewoptionsstr=".$form->{viewoptionsstr}."|delim_$nltype",
                @additionalinf,
            ));
        }
        $vars->{nltype} = $nltype;
    }else{
        $vars->{viewoptions}{delim} = 'n';
    }
    my $use_new_interface = $ENV{HTTP_COOKIE} =~ /edit_phrase_list_beta=([^;]*)/;
    $vars->{template} = 'PraseInfo/' . ($use_new_interface ? 'edit_phrase_list_bin.tmpl' : 'edit_phrase_list.tmpl');
}

sub parse_categs_phrases {
    my ($proj, $vars, $allow_empty_categs) = @_;
    my $form = $proj->form;
    my $text = $form->{phrases_text};
    $text =~ s/^\s+//g;
    $text =~ s/(^|\n)#[^\n]*(\n|$)/\n/g;
    sub del_nommes { my $t = shift; $t =~ s/,/ /g; return $t; }
    $text =~ s/=>([^\n]*)(\n|$)/'=>'.del_nommes($1).$2/ge;
    $text =~ s/(?:(?:\r?\n|,)\s*)+/\n/g; #Перевод строки - тоже разделитель
    $text =~ s/\s*,\s*/\n/g; #Перевод строки - тоже разделитель
    my $phl = $proj->phrase_list;  #->get_minicateg_by_id
    my @data = map { [ $_->[0], $_->[1] ? $phl->get_minicateg_id_lang($_->[1], $vars->{viewoptions}{lang}) : "" ] }
        map { [ split /\s*=>\s*/, $_ ] }
        grep {$allow_empty_categs || /=>/}
        split "\n", $text;
    my $h = {};
    $h->{$_->[1]} .= $_->[0]."," for @data;
    $h->{$_} = $proj->phrase_list($h->{$_}) for keys %$h;
    $proj->save_phrase_list($_) for values %$h;
    $h->{$_} = $h->{$_}->cache_id for keys %$h;
    return $h;
}


sub user_list : CMDH {
    my ($proj, $vars) = @_;

    return {
        title => 'Список пользователей',
        table => 'Users',
        #getlist => sub {
        #    my ($self, %prm) = @_;
        #    #my $proj = $self->proj;
        #    my $list = $proj->List_SQL("
        #        select
        #            Login, Email, LastVisit, NumVisits, role
        #        from
        #            Users
        #        limit
        #            2
        #    ");
        #    return $list;
        #},
        readonly => 1,
        default_field_params => { shlist => 1, },
        fields => [
            { name => 'Login', title => 'Логин', inlinefilter => {}, },
            { name => 'EMail', title => 'EMail', },
            { name => 'LastVisit', title => 'Последнее посещение', },
            { name => 'NumVisits', title => 'Количество посещений', },
            { name => 'role', title => 'Роль', },
            {
                extname => 'NewCount',
                title   => 'На модерации',
                extsql  => ["
                    select
                        Login, count(*) NewCount
                    from
                        CatalogiaPhrases
                    where
                        Status = 'New' and Login in (?)
                    group by
                        Login",
                    ["Login"]
                ],
            },
            {
                extname => 'AcceptedCount',
                title   => 'Принято',
                extsql  => ["
                    select
                        Login, count(*) AcceptedCount
                    from
                        CatalogiaPhrases
                    where
                        Status in ('Accepted', 'Done') and Login in (?)
                    group by
                        Login",
                    ["Login"]
                ],
            },
            {
                extname => 'DeclinedCount',
                title   => 'Отклонено',
                extsql  => ["
                    select
                        Login, count(*) DeclinedCount
                    from
                        CatalogiaPhrases
                    where
                        Status = 'Declined' and Login in (?)
                    group by
                        Login",
                    ["Login"]
                ],
            },
            {
                extname => 'AcceptedPercent',
                title   => 'Процент принятых фраз',
                extsql  => ["
                    select
                        Login, AcceptedCount / (AcceptedCount + DeclinedCount) AcceptedPercent
                    from
                        (select
                            Login,
                            sum(IF(Status in ('Accepted', 'Done'), 1, 0)) AcceptedCount,
                            sum(IF(Status = 'Declined', 1, 0)) DeclinedCount
                        from
                            CatalogiaPhrases
                        where
                            Status != 'New' and Login in (?)
                        group by
                            Login)
                        SummaryInfo
                ", ['Login']],
                inlinefilter    => {},
            },
        ],
        order_by => 'Login',
        search => {
            fields => [
                'Login',
            ],
            name => 'text',
            title => 'Login',
        },
        pager => { name => 'p', cc => 100, },
    };
}


sub prefilter_list : CMDH {
    my ($proj, $vars) = @_;

    my $table = {
        title                   => 'Список префильтров',
        table                   => 'PrefilterList',
        idfield                 => 'ID',
        readonly                => (!$proj->user->rights->{right_edit_prefilters}),
        default_field_params    => { shlist => 1, },
        fields                  => [
            {
                name => 'Pattern',
                title => 'Шаблон',
                inlinefilter => {},
                showmacro => 'categphrase',
            },
            {   name        => 'ReplacingPattern',
                title       => 'Замещающий шаблон',
                inlinefilter    => {},
                showmacro    => 'categphrase',
                showsubel   => sub {
                    my ($el, $f) = @_;
                    return '"' . $el->{ReplacingPattern} . '"';
                },
            },
        ],
        Moderate => {
            Get     => sub {
                my ($self, $id) = @_;
                my $proj = $self->proj;
                if ($id =~ /prefilter_/) {
                    $id =~ s/prefilter_//;
                    my $get = $proj->List_SQL("
                        select
                            InitialPhrase
                        from
                            CatalogiaPhrases
                        where
                            InitialPhraseID = ?
                    ", [$id])->[0];
                    my ($pattern, $replacing_pattern) = split /\t/, $get->{InitialPhrase};
                    return {
                        ID                  => 'prefilter_' . $id,
                        Pattern             => $pattern,
                        ReplacingPattern    => $replacing_pattern,
                        Language            => $vars->{viewoptions}{lang},
                    };
                } else {
                    return $proj->dbtable('PrefilterList', 'ID')->Get($id);
                }
            },
            Add     => sub {
                my ($self, $id, $h) = @_;
                my $proj = $self->proj;
                $proj->add_user_phrase(
                    '',
                    $h->{Pattern} . "\t" . $h->{ReplacingPattern},
                    $vars->{viewoptions}{lang},
                    'AddPrefilter'
                );
            },
            List    => sub {
                my ($self, $list, $prms) = @_;
                my $proj = $self->proj;
                my $moderation_list = $proj->List_SQL("
                    select
                        InitialPhrase, Action, InitialPhraseID, CatID, Language
                    from
                        " . $proj->user_phrases->db_table . "
                    where
                        Action in ('AddPrefilter', 'DeletePrefilter') and Status in ('New', 'Accepted') and Language = ?
                ", [$vars->{viewoptions}{lang}]);
                for (grep{$_->{Action} eq 'AddPrefilter'} @$moderation_list) {
                    my ($pattern, $replacing_pattern) = split /\t/, $_->{InitialPhrase};
                    my $elem = {
                        ID                  => 'prefilter_' . $_->{InitialPhraseID},
                        Pattern             => $pattern,
                        ReplacingPattern    => $replacing_pattern,
                        Language            => $vars->{viewoptions}{lang},
                    };
                    push @$list, $elem;
                }
                my %prefilters = map{$_->{Language} . '_' . $_->{Pattern} . "\t" . $_->{ReplacingPattern} => 1} @{$list};
                $prefilters{$_->{Language} . '_' . $_->{InitialPhrase}} = 0 for grep{$_->{Action} eq 'DeletePrefilter'}@{$moderation_list};
                $list = [
                    grep {
                        $prefilters{$_->{Language} . '_' . $_->{Pattern} . "\t" . $_->{ReplacingPattern}}
                    } @$list
                ];
                return $list;
            },
        },
        default_filter          => { Language => $vars->{viewoptions}{lang}, },
        search                  => {
            fields  => [
                'Pattern',
            ],
            name    => 'text',
            title   => 'Pattern',
        },
        order_by    => 'ID',
        pager                   => { name => 'p', cc => 50, },
    };
    $table->{Moderate}{Edit}    = sub {
        my ($self, $id, $h) = @_;
        my $proj = $self->proj;
        my $old = $proj->dbtable('PrefilterList', 'ID')->Get($id);
        return if $h->{Pattern} eq $old->{Pattern} && $h->{ReplacingPattern} eq $old->{ReplacingPattern};
        $table->{Moderate}{Add}($self, $id, $h);
        $table->{Moderate}{Del}($self, $id, $old);
    };
    $table->{Moderate}{Del}     = sub {
            my ($self, $id) = @_;
            my $proj = $self->proj;
            my $h = $table->{Moderate}{Get}($self, $id);
            $proj->add_user_phrase(
                '',
                $h->{Pattern} . "\t" . $h->{ReplacingPattern},
                $vars->{viewoptions}{lang},
                'DeletePrefilter'
            );
    };
    return $table;
}

sub edit_profile : CMD {
    my ($proj, $vars) = @_;

    if ($proj->form->{edit_profile_do_save}) {
        $proj->user->dbt->Edit($proj->login, {"EMail" => $proj->form->{edit_profile_mail}});
        $proj->do_redirect("ind.pl?cmd=edit_profile");
    }

    $vars->{user_inf} = $proj->user->user_inf;
    $vars->{template} = "edit_profile.tmpl";
}



sub ajax_get_syn_cells_list : CMD {
    my ($proj, $vars) = @_;
    my $lang = $vars->{viewoptions}{lang};
    my $list = $proj->search_syn_cells($proj->form->{text}, $lang);

    for my $cell (@$list) {
        $cell->{Syns} =~ s/,/, /g;
        $cell->{Pairs} = [map{
            {
                ID => $cell->{MainSynID} . "_" . md5int($_),
                Text => $_,
                IsDeleted => $proj->is_syn_pair_deleted($_, $lang)
            }
        } split ",", $cell->{Pairs}];
    }

    $vars->{syn_cells} = $list;
    $vars->{template} = "ajax_get_syn_cells_list.tmpl";
}

sub get_syn_cells_list : CMD {
    my ($proj, $vars) = @_;
    my $lang = $vars->{viewoptions}{lang};

    # удаление пар синонимов
    if($proj->form->{delete_pairs}) {
        my $deleted_pairs = [];

        for (grep{/^pair_/} keys %{$proj->form}) {
            my $pair = $proj->form->{$_};
            $proj->add_user_phrase("", $pair, $lang, "DeleteSynPair");
            push @$deleted_pairs, $pair;
        }

        $vars->{deleted_pairs} = $deleted_pairs;
    }

    # добавление новой пары
    if($proj->form->{add_pair}) {
        my @words = map{$proj->form->{"syn$_"}} 1..2;
        s/\s//g for @words;
        if(!grep{!$_} @words) {
            my $pair = "$words[0]:$words[1]";
            $proj->add_user_phrase("", $pair, $lang, "AddSynPair");
            $proj->do_redirect("ind.pl?cmd=get_syn_cells_list&viewoptionsstr=".$proj->form->{viewoptionsstr});
        }
    }

    $vars->{user_pairs} = $proj->get_user_syn_pairs($lang);
    $vars->{template} = "get_syn_cells_list.tmpl";
}

# Возвращает CMDH для добавления слов в заданный словарь
# На входе
#   $proj
#   $vars
#   title => заголовок
#   act => Action (для таблицы CatalogiaPhrases)
#   fields_add => список полей, добавляемых к стандартным
#   fields     => дополнительные поля для отрисовки
#   func_phr => функция, используемая при добавлении в таблицу CatalogiaPhrases.
#       На входе:  ($proj, $h)      $h - элемент таблицы
#       На выходе: фраза, добавляемая в таблицу CatalogiaPhrases
sub get_tinf_add_to_dicts {
    my ($proj, $vars, %prm) = @_;

    my $title = $prm{title} // '';
    my $act = $prm{act};
    my $fields_add = $prm{fields_add};
    my $func_phr = ref($prm{func_phr}) eq 'CODE'  ?  $prm{func_phr}  :  sub { return ''; };
    my $bottom_views = $prm{bottom_views} // [];
    my $rights = $prm{rights};
    my $fields = $prm{fields} || [];

    my $res = {
        title => $title,
        table => 'CatalogiaPhrases',
        (defined $rights  ? (rights => $rights) : ()),
        readonly => 1,
        #idfield => 'ID',
        fix_sql_problem => 0,
        disable_del => 1,
        disable_add => 1,
        logchanges => 1,
        fields => [
            ( map { { %{$_},  edlist => 1,  shlist => 0, } }  @$fields_add),
            ( map { { name => $_,  edlist => 0,  inlinefilter => { group => 1, },  shlist => 1,  } } qw( InitialPhrase Language UpdateTime Login LastLogin Status Comment Action ) ),
            { edlist => 1,  name => 'UserComment',  title => 'Комментарий',  addform => 1,  ftype => 'textarea',  inlinefilter => { group => 1, },  shlist => 1,  after => 'InitialPhrase',  },
            @$fields
        ],
        default_field_params => {
        },
        order_by => 'UpdateTime DESC',
        filters => [
            ( map { { field => $_, title => $_, grp => 1, use_other_filters => 1, } } qw( Login Status ) ),
        ],
        add_filter => {
            Action => $act,
            Language => $vars->{viewoptions}{lang},
        },
        action_onlist => sub {
            my ($proj, $el)=@_;
        },
        compact_inline_add => 1,
        Moderate => {
            Add => sub {
                my ($self, $id, $h) = @_;
                my $proj = $self->proj;

                my $phr = &$func_phr($proj, $h);
                $proj->add_user_phrase(
                        '',
                        $phr,
                        $vars->{viewoptions}{lang},
                        $act,
                        '',
                        $h->{UserComment},
                );
            },
        },
        checkform => {
            action => sub {
                my ($self, $h) = @_;
                my $phr = &$func_phr($proj, $h);
                my $comment = $h->{UserComment};
                if (length($phr) >= 1000) {
                    return "Error: The length of the phrase must be less then 1000 symbols";
                };
                if (length($comment) >= 1000) {
                    return "Error: The length of the comment must be less then 1000 symbols";
                };
                return '';
            },
        },
        bottom_views => $bottom_views // [],
    };

    return $res;
}

sub search_csyns : CMDH {
    my ($proj, $vars) = @_;
    my $csyns_text = $proj->form->{csyns};
    $csyns_text =~ s/(\s*\@.+)//;
    my @csyns = split /\s*,\s*/, $csyns_text;
    my $data = [];
    my $max_lines_per_csyn = 5;
    my $links = BM::ContextSyns::ContextLinks->new({proj => $proj, init_phl => $proj->phrase_list( [$proj->form->{csyns}] ) });
    my %used_bids;
    my $start = [gettimeofday];
    my $max_time = 10;
    my %uniq_csyns = map{$proj->phrase($_)->snorm_phr => 1} @csyns;

    # парсим контекст
    my ($add_categ) = $proj->form->{csyns} =~ /\@\s*categ:\s*(.+)/;
    $add_categ = $proj->categs_tree->get_minicateg_id($add_categ);
    $add_categ = $add_categ ? "categ_$add_categ" : "";

    for my $csyn (keys %uniq_csyns) {
        last if tv_interval($start, [gettimeofday]) >= $max_time;

        my @ids = $proj->banners_bender->find_ids_atoms($proj->phrase("$csyn $add_categ active_flag"), 10000);
        my $bnl = $proj->bf->banner_list(\@ids);
        my $phl = $proj->phrase_list([ $csyn ]);
        my $csyn_data_count = 0;

        for my $bnr (@$bnl) {
            last if tv_interval($start, [gettimeofday]) >= $max_time;
            next if $used_bids{$bnr->id};
            $used_bids{$bnr->id}++;
            #$proj->dd($bnr->id, [map{$_->text} @{$bnr->phl}], $bnr->exminicategshash);
            my $new_phl = $links->extend_phl($bnr->phl, $bnr->exminicategshash);
            #$proj->dd($bnr->id, [map{$_->text} @{$new_phl}]);
            next if !@$new_phl;

            push @$data, {
                Title => $bnr->title,
                Body => $bnr->body,
                Phrases => join(",", map{$_->plus_text} $bnr->phrases),
                NewPhrases => join(",", map{$_->plus_text} @$new_phl),
            };

            $csyn_data_count++;
            last if $csyn_data_count >= $max_lines_per_csyn;
        }
    }

    return {
        getlistflt => sub { return $data; },
        disable_del => 1,
        readonly => 1,
        default_field_params => { shlist => 1 },
        fields => [
            { name => "Title" },
            { name => "Body" },
            { name => "Phrases" },
            { name => "NewPhrases" },
        ],
        toptext => "<h4>Уникальные слова: "  . join(", ", keys %uniq_csyns) . "</h4>"
    };
}

sub csyns_add : CMDH {
    my ($proj, $vars) = @_;

    my $res = get_tinf_add_to_dicts($proj, $vars,
        title => 'Контекстные синонимы',
        act => 'AddContextSyn',
        fields_add => [
            { extname => 'phr',  title => 'Добавляемые синонимы (на&nbsp;отдельных&nbsp;строках или через&nbsp;запятую)',  ftype => 'textarea', editformsearchurl => '?cmd=search_csyns&csyns=', },
        ],
        fields => [
            {
                icon => "search",
                showmacroel => 'show_btn_field',
                geturl => sub { my ($el, $f) = @_; return "?cmd=search_csyns&csyns=" . $el->{InitialPhrase}; },
                shlist => 1,
                after => "InitialPhrase"
            },
        ],
        func_phr => sub {
            my ($proj, $h) = @_;
            my $phr = $h->{phr};
            $phr =~ s/\n+/,/g;
            $phr =~ s/\s+/ /g;
            $phr =~ s/\s*,+\s*/,/g;
            $phr =~ s/,+/,/g;
            $phr =~ s/^,+//g;
            $phr =~ s/,+$//g;
            $phr =~ s/,/, /g;
            return $phr;
        },
    );

    return $res;
}

sub wide_spam_phrases : CMDH {
    my ($proj, $vars) = @_;

    my $lang = $vars->{viewoptions}{lang};

    my $res = get_tinf_add_to_dicts($proj, $vars,
        title => 'Wide spam-phrases',
        rights => 'right_add_spamphrases',
        act => 'AddWideSpamPhrase',
        fields_add => [
            { extname => 'phr1',  title => 'Phrase', },
        ],
        func_phr => sub {
            my ($proj, $h) = @_;
            my $phr = $proj->phrase($h->{phr1})->norm_phr;
            return $phr;
        },
        bottom_views => [
            { url => "?cmd=wide_spam_phrases_view&viewoptionsstr=lang_$lang&act=AjaxSearch", },
        ],
    );

    return $res;
}

sub wide_spam_phrases_view : CMDH {
    my ($proj, $vars) = @_;

    my $lang = $vars->{viewoptions}{lang};
    return get_tinf_view_dicts($proj, $vars,
        files => [
            $Utils::Common::options->{SpamPhrases_params}{"spam_phrases_wide_$lang"},
        ],
        fields => [qw( Phrase )],
        order_by => "Phrase",
        func_read => sub {
            my ($proj, $str) = @_;
            return {Phrase => $str};
        },
    );
}


sub lemmer_fix : CMDH {
    my ($proj, $vars) = @_;

    my $lang = $vars->{viewoptions}{lang};

    my $res = get_tinf_add_to_dicts($proj, $vars,
        title => 'Lemmer-fix',
        rights => 'right_add_lemmerfix',
        act => 'AddLemmerFix',
        fields_add => [
            { extname => 'phr1',  title => 'Word', },
            { extname => 'phr2',  title => 'Wrong lemma', },
        ],
        func_phr => sub {
            my ($proj, $h) = @_;
            my $phr_word = $h->{phr1};
            my $phr_lemma = $h->{phr2};
            for ($phr_word, $phr_lemma) {
                s/\s+/ /g;
                s/^\s+//g;
                s/\s+$//g;
                # TODO  несколько слов, знаки препинания, ...
            }
            my $phr = "$phr_word/$phr_lemma";
            return $phr;
        },
        bottom_views => [
            { url => "?cmd=lemmer_fix_view&viewoptionsstr=lang_$lang&act=AjaxSearch", },
        ],
    );

    return $res;
}


# Возвращает CMDH для отображения в интерфейсе содержимого словарей
# На входе:
#   $proj
#   $vars
#   title => заголовок
#   fields => список полей
#   func_read => функция, используемая Для чтения из файлов словарей
#       На входе:  ($proj, $str)      $str - строка файла
#       На выходе: хэш для отображения в строке таблицы
sub get_tinf_view_dicts {
    my ($proj, $vars, %prm) = @_;

    my $lang = $vars->{viewoptions}{lang};
    my $func_read = ref($prm{func_read}) eq 'CODE'  ?  $prm{func_read}  :  sub { return; };
    my @files = @{$prm{files} // []};
    my @fields = @{$prm{fields} // []};
    my $order_by = $prm{order_by} // '';
    my $title = $prm{title} // '';

    return {
        readonly => 1,
        title => $title,
        getlistflt => sub {
            my ($self, %prm) = @_;
            my @out;
            for my $file (@files) {
                my ($f) = ($file =~ /\/([^ \/]+)[ \| ]*$/);
                open FILE, " < " . $file or next; #die($!);
                while(<FILE>) {
                    chomp;
                    next if /^#/;
                    next if m/^\s*$/;
                    my $el = &$func_read($proj, $_) // next;
                    push @out, { %$el, File => $f, };
                }
                close FILE;
            }
            return \@out;
        },
        fields => [
            (map {{ name => $_, }} @fields),
        ],
        default_field_params => {
            shlist => 1,
            inlinefilter => { group => 0, },
        },
        order_by => $order_by,
        filters => [
            (map {{
                field => $_,
                grp => 1,
                use_other_filters => 1,
            }} qw()),
        ],
    };
}


# Отображает в интерфейсе содержимое словарей леммер-фикса (для языка, выбранного в интерфейсе)
sub lemmer_fix_view : CMDH {
    my ($proj, $vars) = @_;

    my $lang = $vars->{viewoptions}{lang};
    return get_tinf_view_dicts($proj, $vars,
        files => [
            $Utils::Common::options->{DictNorm}{"lemmer_fix_$lang"},
            $Utils::Common::options->{DictNorm}{"lemmer_fix_web_$lang"},
        ],
        fields => [qw( Word WrongLemma )],
        order_by => "Word",
        func_read => sub {
            my ($proj, $str) = @_;
            if ($str =~ /(.*)\/(.*)/) {
                return { Word => $1,  WrongLemma => $2, };
            } elsif ($str =~ /(.*)=(.*)/  and  $1 eq $2) {
                return { Word => $1,  WrongLemma => '', };
            } elsif (defined $str) {
                $proj->log("ERROR: bad line in the lemmer-fix file: '$_'");
            }
            return;
        },
    );
}


# Отображает в интерфейсе содержимое словарей контекстных синонимов (для языка, выбранного в интерфейсе)
sub csyns_view : CMDH {
    my ($proj, $vars) = @_;

    my $lang = $vars->{viewoptions}{lang};
    my @files;
    for my $type (qw( precise precise_tr diacr orfovars_tr )) {
        my $csyns = $Utils::Common::options->{ContextSyns}{$type}  //  do {
            $proj->log("ERROR: Could not get csyns ($type)");
            next;
        };
        #next  if defined $csyns->{lang}  and  $csyns->{lang} ne $lang;
        next  if ($csyns->{lang} // 'ru') ne $lang;
        my $file = $csyns->{file};
        my $dicts = $csyns->{dicts};
        my @files_part = grep {$_} (
            ($file  ?  (ref($file) eq 'ARRAY'  ?  @$file : $file)  :  ()),
            (map { $Utils::Common::options->{ContextSyns_params}{dir_dicts} . "/$_" }  ($dicts  ?  (ref($dicts) eq 'ARRAY'  ?  @$dicts : $dicts)  :  ())),
        );
        $proj->log("ERROR: Could not get csyns files ($type)")  unless @files_part;
        push @files, @files_part;
    }

    return get_tinf_view_dicts($proj, $vars,
        title => 'Контекстные синонимы',
        files => [@files],
        fields => [qw( Words )],
        order_by => "Words",
        func_read => sub {
            my ($proj, $str) = @_;
            return { Words => $str, };
            #my @w = split /,\s*/, $str;   # TODO context:    @categ, ...
            #@w = (uniq map {$proj->phrase($_)->norm_phr} @w);
            #return { Words => join(', ', @w), };
        },
    );
}


sub syns_path : CMD {
    my ($proj, $vars) = @_;

    if($proj->form->{syn1} && $proj->form->{syn2}) {
        $proj->current_lang($vars->{viewoptions}{lang});

        my $phr = $proj->phrase(join " ", map{$proj->form->{"syn$_"}} (1..2));

        $vars->{"syn$_"} = $proj->form->{"syn$_"} for 1..2;
        $vars->{path} = [map{ { ID => md5int($_), Text => $_, Words => [split /:/] } } @{$phr->analyze_synonyms_path}];

        $vars->{equal_norm}  =  (uniq (map { $proj->phrase( $proj->form->{"syn$_"} )->norm_phr } (1..2) )) < 2  ?  $proj->phrase( $proj->form->{"syn1"} )->norm_phr  :  "";
        $vars->{equal_snorm}  =  (uniq (map { $proj->phrase( $proj->form->{"syn$_"} )->snorm_phr } (1..2) )) < 2  ?  $proj->phrase( $proj->form->{"syn1"} )->snorm_phr  :  "";
        $vars->{words} = [ $proj->form->{syn1}, $proj->form->{syn2} ];
    }

    $vars->{template} = "syns_path.tmpl";
}

sub _grep_banners_list {
    my ($banners_list, $max_banners_count, @minus_words) = @_;

    my $proj = $banners_list->proj;
    my %minus_words = map {$_ => 1} @minus_words;
    my @grep_ids = ();
    for my $banner (@$banners_list) {
        last if @grep_ids >= $max_banners_count;
        my $banner_text = $banner->title . ' ' . $banner->body;
        my $language = $proj->get_language($banner->lang);
        my @norm_words = $language->phrase($banner_text)->uniqnormwords;
        next if grep {$minus_words{$_}} @norm_words;
        push @grep_ids, $banner->id;
    }

    return $proj->bf->banner_list(\@grep_ids);
}

sub search_banners : CMD {
    my ($proj, $vars) = @_;

    my $max_banners = $proj->form->{max_banners};
    my $log_categs = $proj->form->{log_categs};
    my $lang = $vars->{viewoptions}{lang};
    my $language = $proj->get_language($lang);

    $proj->current_lang($vars->{viewoptions}{lang});

    $max_banners = 100 if !$max_banners || $max_banners !~ /^\d+$/ || $max_banners > 10_000;

    if($proj->form->{do_search}) {
        # иногда баннеры фильтруются, потому что их нет в banners_extended, поэтому ищем с запасом
        my $count_search_banners = max(int($max_banners * 1.2), $max_banners + 5);
        my $t0 = [gettimeofday];
        my $query = URI::Escape::uri_unescape($proj->form->{phrase_text});
        my @minus_words = $language->phrase($query)->minuswords;
        my @norm_minus_words = $language->phrase(join(' ', @minus_words))->uniqnormwords;
        $query .= " lang_" . $vars->{viewoptions}{lang} if $vars->{viewoptions}{lang} ne 'ru';
        $query .= " active_flag" if $proj->form->{only_active};
        my @ids = $proj->banners_bender->find_ids_atoms($proj->phrase($query), (@minus_words ? 20_000 : $count_search_banners));
        my $t1 = [gettimeofday];
        my $bnrs = $proj->bf->banner_list(\@ids);
        $bnrs = _grep_banners_list($bnrs, $max_banners, @norm_minus_words);

        # это "категории catmedia"
        # todo: научиться считать "категории продакшна" на catmedia
        # https://st.yandex-team.ru/CATALOGIA-911
        for my $bnr (@$bnrs) {
            $bnr->{Categs} = [$bnr->get_minicategs];
        }

        my (%bid2cid, %cid2cnt);
        if ($proj->form->{with_cids} and @$bnrs) {
            %bid2cid = map { $_->{bid} => $_->{cid} } @{ $proj->dt_proxy_client->do_select(
                table => $proj->bf->get_banners_dyntable(),
                fields => ['bid', 'cid'],
                condition => "bid in (" . join(",", map { $_->{id} } @$bnrs) . ")",
            ) };

            # чтобы не удариться в лимит в 1 млн баннеров и не упасть в YQL-фоллбек, кампании выбираем по одной
            for my $cid (uniq values %bid2cid) {
                $cid2cnt{$cid} = scalar @{ $proj->dt_proxy_client->do_select(
                    table => $proj->bf->get_banners_dyntable(key_field => "cid"),
                    fields => ['bid'],
                    condition => "cid = $cid",
                ) };
            }
        }

        for my $bnr (@$bnrs) {
            $bnr->{Categs} = [ grep{$_} map{$proj->get_category_by_name($_, $bnr->lang)} @{$bnr->{Categs}} ];
            $bnr->{cid} = $bid2cid{$bnr->{id}};
            $bnr->{cid_cnt} = $cid2cnt{$bnr->{cid}};

            if($log_categs) {
                $bnr->{categs_log} = $bnr->preprocess_title_body->get_minicategs_test_info("<br/>");
            }
        }

        $vars->{banners} = \@$bnrs;
        $vars->{bnrs_count} = @$bnrs;
        my $t2 = [gettimeofday];
        $vars->{time_bender} = tv_interval($t0, $t1);
        $vars->{time_db} = tv_interval($t1, $t2);
    }

    $vars->{only_active} = $proj->form->{only_active};
    $vars->{with_cids} = $proj->form->{with_cids};
    $vars->{with_categs} = $proj->form->{with_categs};
    $vars->{max_banners} = $max_banners;
    $vars->{log_categs} = $log_categs;
    $vars->{phrase_text} = $proj->form->{phrase_text};
    $vars->{template} = "search_banners.tmpl";
}

sub debug_sleep : CMD {
    my ($proj, $vars) = @_;
    my $tm = $proj->form->{tm} || 10;
    $vars->{text} = "sleep $tm sec";
    sleep($tm);
}

sub get_page_info : CMD {
    my ($proj, $vars) = @_;

    if($proj->form->{url}) {
        my $lang = $vars->{viewoptions}{lang};
        $proj->current_lang($lang);

        my $page = $proj->page($proj->form->{url});
        my @categs = map{$proj->get_category_by_name($_, $lang)} $page->get_minicategs;
        my $h = $page->get_categs_subphrases;

        my @subphrases;
        for my $phr (keys %$h) {
            push @subphrases, {
                Phrase   => $phr,
                OrigTexts => [ keys %{$h->{$phr}{orig_texts}} ],
                Tags     => [ keys %{$h->{$phr}{tags}} ],
                Categs   => [ map{$proj->get_category_by_name($_, $lang)} keys %{$h->{$phr}{categs}} ]
            };
        }

        $vars->{info} = {
            categs  => \@categs,
            subphrases  =>  \@subphrases,
            debug_inf => $page->get_minicategs_debug_inf,
        };
    }

    $vars->{url} = $proj->form->{url};
    $vars->{template} = "get_page_info.tmpl";
}

sub inline_search_categs : CMD {
    my ($proj, $vars) = @_;
    my $text = $proj->form->{text};
    my $list = $proj->search_categs($text, $proj->viewoptions->{lang});
    $vars->{_return_text} = 1;
    $vars->{text} = join( "", map { $_->{'CategoryName'}."\n" } @$list );
    #return ['aaaaaa', 'bbbbbb']
};



sub urls_images_report : CMDH {
    my ($proj, $vars) = @_;

    $vars->{tinf} = {
        title => 'Отчёт по картинкам',
        readonly => 1,
        default_field_params => { shlist => 1, },
        getlistflt => sub {
            my ($self, %prm) = @_;
            my $phlid = $proj->form->{'phlid'};
            my $phl = $proj->get_phrase_list($phlid);
            my $list = [ map { { url => $_->[0], imgurl => $_->[1], phlid => $phlid } } map { [ split / =.> /, $_->text ] } @$phl ];
            return $list;
            #return [  { url => 'eee', imgurl => 'sdfsdf'} ];
        },
        fields => [
               {  name => 'url',    title => 'Ссылка', showmacro => 'exturl', },
               {  name => 'phlid',  shlist => 0, addform => 1, },
               {  name => 'imgurl', title => 'Картинка', showmacro => 'showurlimage' },
        ],
        pager => { name => 'p', cc => 100, },
        #search => { fields => [ 'Categories' ], name => 'text', },
    };
}

sub bnrs_images_report : CMDH {
    my ($proj, $vars) = @_;

    $vars->{tinf} = {
        title => 'Отчёт по картинкам',
        readonly => 1,
        default_field_params => { shlist => 1, },
        getlist => sub {
            my ($self, %prm) = @_;
            my $phl = $proj->get_phrase_list($proj->form->{'phlid'});
            my $list = [ map { { url => $_->[0], imgurl => $_->[1] } } map { [ split / =.> /, $_->text ] } @$phl ];
            return $list;
            #return [  { url => 'eee', imgurl => 'sdfsdf'} ];
        },
        fields => [
               {  name => 'url', title => 'Ссылка', showmacro => 'exturl', },
               {  name => 'imgurl', title => 'Картинка', showmacro => 'showurlimage' },
        ],
        #pager => { name => 'p', cc => 100, },
        #search => { fields => [ 'Categories' ], name => 'text', },
    };

}

sub changeuserlogin :CMD {
    my ($proj, $vars) = @_;
    if($proj->form->{'fakelogin'}){
        $vars->{viewoptions}{changeuserlogin} = $proj->form->{'fakelogin'};
        $proj->do_redirect(join("&",
            "ind.pl?cmd=changeuserlogin",
            "viewoptionsstr=".$proj->_join_viewoptions($vars->{viewoptions}),
        ));
    }
    #$proj->dd($vars->{viewoptions});
    $vars->{'fakelogin'} = $vars->{viewoptions}{changeuserlogin};
    $vars->{template} = "fake_login.tmpl";
}

sub select_regions : CMD {
    my ($proj, $vars) = @_;
    my $root = 10000;
    my $create_region;
    my %selected = map{$_ => 1} @{$proj->current_region || []};
    my %expanded;

    require geobaselite;

    # внесение изменений
    my $change_id = $proj->form->{uncheck} || $proj->form->{check};
    if($change_id) {
        my $ids = $proj->current_region || [];

        if($proj->form->{check}) {
            $selected{$change_id}++;
        } elsif($selected{$change_id}) {
            delete $selected{$change_id};
        } else {
            my $dont_check = $change_id;
            for my $parent (reverse @{$geobaselite::Region{$change_id}{path}}) {
                $selected{$_}++ for grep{$_ != $dont_check} @{$geobaselite::Region{$parent}{chld}};

                if($selected{$parent}) {
                    delete $selected{$parent};
                    last;
                }

                $dont_check = $parent;
            }
        }

        # удаление пометок с потомков изменяемого элемента
        for my $id (@$ids) {
            if(grep{$_ == $change_id} @{$geobaselite::Region{$id}->{parents}}) {
                delete $selected{$id};
            }
        }

        my $h = $vars->{viewoptions};
        $h->{region} = join("_", sort keys %selected);

        my $str = $proj->_join_viewoptions($h);
        my $retpath = $proj->form->{retpath};
        $retpath =~ s/\&viewoptionsstr=[^&]+//g;
        $proj->do_redirect("?cmd=select_regions&retpath=".uri_escape_utf8($retpath)."&viewoptionsstr=$str");
    }

    my @top_regions = qw(225 1 10174 213 166 187 11316 11162 2 11318);
    for my $id (@top_regions, keys %selected) {
        $expanded{$_}++ for @{$geobaselite::Region{$id}->{parents}};
    }

    $create_region = sub {
        my ($id, $checked) = @_;
        my %h = %{$geobaselite::Region{$id}};

        $h{id} = $id;
        $h{expanded} = $expanded{$id};
        $h{checked} = $checked || $selected{$id};
        $h{regions} = [ map{$create_region->($_, $h{checked})} @{$h{chld}}  ];

        return { %h };
    };

    $vars->{regions} = [$create_region->($root)];
    $vars->{template} = "select_regions.tmpl";
}

sub catalogia_manual_flags : CMDH {
    return {
        table       => "CatalogiaManualFlags",
        readonly    => 1,
        fields      => [
            { name => "bid", title => "BannerID", shlist => 1 },
            { name => "title", title => "Title", shlist => 1 },
            { name => "body", title => "Body", shlist => 1 },
            { name => "choosed", title => "Добавленные флаги", shlist => 1 },
            { name => "unchecked", title => "Удалённые флаги", shlist => 1 },
            { name => "flags", title => "Итого", shlist => 1 },
        ],
        pager       => { name => "p", cc => 50 }
    };
}

sub add_category_comment : CMDH {
    my ($proj, $vars) = @_;

    $proj->category_comments->Add({
        CatID       => $proj->form->{id},
        Comment     => $proj->form->{comment},
        Login       => $proj->{login}
    });

    $proj->do_redirect(join("&",
        "ind.pl?cmd=show_phrases",
        "id=".$proj->form->{id},
        "viewoptionsstr=".$proj->form->{viewoptionsstr}
    ));
}

sub delete_category_comment : CMDH {
    my ($proj, $vars) = @_;

    my $categ_id = $proj->category_comments->Get($proj->form->{id})->{CatID};

    $proj->Do_SQL("delete from CategoryComments where ID=".$proj->dbh->quote($proj->form->{id}));

    $proj->do_redirect(join("&",
        "ind.pl?cmd=show_phrases",
        "id=$categ_id",
        "viewoptionsstr=".$proj->form->{viewoptionsstr}
    ));
}

sub fcgi_process : CMDH {
    my ($proj, $vars) = @_;

    my %host2psfcgi = (); # host => [@ps_fcgi]

    my $table = {
        title                   => "FcgiProcess    [" . $proj->dates->cur_date('db_time') . "]",
        table                   => 'FcgiProcess',
        dbhname                 => 'catalogia_media_dbh',
        readonly                => 1,
        idfield                 => 'Pid',
        default_field_params    => {
            shlist => 1,
            inlinefilter    => { group => 1 },
        },
        fields      => [
            (map {{ name => $_ }}  qw[ Host Pid Login Cmd Act StartTime ]),
            {
                name            => 'Ps',
                inlinefilter    => {},
                showsubel       => sub {
                    my ($el, $f) = @_;
                    my $pid = $el->{Pid};
                    my $host = $el->{Host};
                    my $ps_fcgi = ($host2psfcgi{$host} //= [ do {
                            my @ps_fcgi = split /\n/, do_safely(sub { $proj->read_sys_cmd("timeout --kill-after 3 7 ssh $host 'ps ux | grep fcg[i]'"); }, no_die => 1) // '';
                    }]);
                    return (grep { m/^bmclient\s+$pid\s/ }  @$ps_fcgi)[0] // '';
                },
            },
        ],
        filters => [
            ( map {{ field => $_,  name => $_,  use_other_filters => 1,   grp => 1 }} qw( Host ) ),
            ( map {{ field => $_,  name => $_,  use_other_filters => 1,   }} qw( Pid ) ),
            ( map {{ field => $_,  name => $_,  use_other_filters => 1,   grp => 1 }} qw( Login Cmd Act ) ),
            #( map {{ field => $_,  name => $_,  use_other_filters => 1,   }} qw( StartTime ) ),
        ],
        order_by => '-StartTime',
        pager   => {
            name    => 'p',
            cc      => '50',
        },
        NN => 1,
    };
    return $table;
}

sub get_category_banners :CMD {
    my ($proj, $vars) = @_;
    my $categ_id = $proj->form->{cat_id};
    my $query = "categ_$categ_id";
    $query = "$query active_flag" if $proj->form->{active_flag};
    my @ids = $proj->banners_bender->find_ids_atoms($proj->phrase($query), 100_000_000);

    $vars->{'Content-type'} = "text/plain";
    $vars->{text} = join("\n", @ids);
}

sub mass_category_actions :CMD {
    my ($proj, $vars) = @_;
    my $categs_text = $proj->form->{categs} || "";
    my $phrases_text = $proj->form->{phrases} || "";
    my $action = $proj->form->{action} || "";
    my @log_lines;

    if($proj->form->{apply_action}) {
        my @categs;

        for my $categ_name(split "\n", $categs_text) {
            $categ_name =~ s/^\s+|\s+$//g;
            next if !$categ_name;

            my $categ = $proj->get_category_by_name($categ_name) || $proj->get_category($categ_name);

            if(!$categ) {
                push @log_lines, "ERROR: unknown category '$categ_name'";
            } else {
                push @categs, $categ;
            }
        }

        if(!@categs) {
            push @log_lines, "ERROR: no categories";
        }

        my @phrases = grep{$_} map{s/^\s+|\s+$//g; $_} split "\n", $phrases_text;
        if($action eq "AddFlag" || $action eq "DeleteFlag") {
            my @flags;

            for my $phr (@phrases) {
                if(!$proj->categs_tree->get_flag_description($phr)) {
                    push @log_lines, "ERROR: unknown flag '$phr'";
                } else {
                    push @flags, $phr;
                }
            }

            if(!@flags) {
                push @log_lines, "ERROR: no flags";
            }

            for my $categ (@categs) {
                for my $flag (@flags) {
                    if($action eq "AddFlag") {
                        $proj->category_interface->add_flag($categ->{CatID}, $flag);
                    } else {
                        $proj->category_interface->delete_flag($categ->{CatID}, $flag);
                    }

                    push @log_lines, "done: '" . $categ->{CategoryName} . "' / '$flag'";
                }
            }
        } else {
            push @log_lines, "ERROR: unknown action '$action'";
        }
    }

    $vars->{action} = $action;
    $vars->{categs} = $categs_text;
    $vars->{phrases} = $phrases_text;
    $vars->{log_text} = join("\n", @log_lines);
    $vars->{action_options} = [
        { name => "AddFlag", title => "Добавить флаги",  },
        { name => "DeleteFlag", title => "Удалить флаги",  },
    ];
    $vars->{title} = "Действия над категориями";
    $vars->{template} = "mass_category_actions.tmpl";
}

sub get_dse_samples :CMDH {
    my ($proj, $vars) = @_;

    my $get_result = sub {
        my $domain = $proj->form->{flt_domain};
        my $table = $proj->options->{'DSE_Banners_params'}{'dse_samples'};
        my $yt_client = $proj->yt_client->set_params(
            pool                => 'catalogia',
            tries               => 1,
            sleep_between_tries => 1,
        );
        return [ sort {$b->{'rank'} <=> $a->{'rank'}} @{$yt_client->read_table("'${table}[\"$domain\"]'", "'<encode_utf8=false>json'")}];
    };

    if ( $proj->form->{download} && $proj->form->{flt_domain}) {
        my $domain = $proj->form->{flt_domain};
        $domain =~ s/\W/_/g;
        my $data = [ ['Url', 'Title', 'TitlePerf', 'Phrase'], map {[$_->{url}, $_->{title}, $_->{title_perf}, $_->{phrase}]} @{$get_result->()}];
        if ( $proj->form->{download} eq 'xls') {
            $vars->{'_return_real_xls'} = "dse_samples_$domain.xls";
            $vars->{text} = array2xls($data);
        }
        elsif ( $proj->form->{download} eq 'csv') {
            $vars->{'_return_real_xls'} = "dse_samples_$domain.csv";
            $vars->{text} = join ("\r\n", map {join("\t", @$_)} @$data );
        }
        return;
    }

    return {
        title => 'Сэмплы DSE-баннеров',
        readonly => 1,
        topmenu => [
            $proj->form->{flt_domain} ? (
                {
                    title => 'Скачать таблицу',
                    sublist => [
                        { title => 'XLS', name => 'download', addformparams => { flt_domain => 'flt_domain', },
                          url => '?cmd=get_dse_samples&download=xls',
                        },
                        { title => 'CSV', name => 'download', addformparams => { flt_domain => 'flt_domain', },
                          url => '?cmd=get_dse_samples&download=csv',
                        },
                    ]
                }
            ) : (),
        ],

        default_field_params => { shlist => 1, },
        fields => [
             { name => 'url', title => 'Url' },
             { name => 'title', title => 'Title' },
             { name => 'title_perf', title => 'PerfTitle' },
             { name => 'phrase', title => 'Phrase' },
        ],
        filters => [
            { field => 'domain', title => 'Домен'},
        ],
        getlistflt => $get_result,
    }
}

sub test_timeout :CMD {
    my ($proj, $vars) = @_;
}

1;
