package Cmds::Moderation;

use strict;

use utf8;

use base qw(Cmds::Base);
use Encode;
use List::Util qw(min);
use Text::Iconv;
use Data::Dumper;
use URI::Escape;
use Utils::Sys qw(format_number h2sa);
use Time::HiRes qw(gettimeofday tv_interval);
use Utils::Sys qw(md5int);
use Utils::Array qw(array_intersection);
use JSON qw(to_json from_json);
use List::Util qw(shuffle sum);
use HTML::Entities;
use Cmds::BinaryRelation;
use URI::Escape qw(uri_escape_utf8);
use BaseForm;

my $DEBUG_TABLE_PREFIX = "";

sub moderation_report : CMDH {
    my ($proj, $vars) = @_;

    return {
        title => 'Отчёт по модерации',
        readonly => 1,
        table => 'CatalogiaPhrases',
        idfield => 'InitialPhraseId',
        rights => 'right_moderation_report',
        default_field_params => { shlist => 1, },
        fields => [
               {  name => 'CreateDate', title => 'Дата', inlinefilter => { group => 1 }, },
               {  name => 'Login',      title => 'Автор', inlinefilter => { group => 1 }, },
               {  name => 'NewCount',   title => 'На модерации', inlinefilter => {}, },
               {  name => 'AcceptedCount',   title => 'Принято', inlinefilter => {}, },
               {  name => 'DeclinedCount',   title => 'Отклонено', inlinefilter => {}, },
               {  name => 'DoneCount',   title => 'В продакшене', inlinefilter => {}, },
        ],
        order_by => '-CreateTime',
        group_by => 'CONCAT(Login, Left(CreateTime, 10))',
        group_by_add_fields    => [
            'Left(CreateTime, 10) CreateDate',
            'sum(IF(Status="New", 1, 0))  NewCount',
            'sum(IF(Status="Accepted", 1, 0))  AcceptedCount',
            'sum(IF(Status="Declined", 1, 0))  DeclinedCount',
            'sum(IF(Status="Done", 1, 0))  DoneCount',
        ],
        complicated_add_fields => ['Left(CreateTime, 10) CreateDate'],
        pager => { name => 'p', cc => 100, },
        add_filter => { 'Login LIKE' => '%' },
#        tree => { grpfields => [
#                      #{ field => 'UpdateTime', type => 'datemnthgrp', },
#                      { field => 'Login', },
#                  ],
#                  pager => { name => 'p', cc => 10, },
#              },
        filters => [
               {  name => 'Логин', field => 'Login', grp => 1 },
               {  name => 'Дата', field => 'CreateDate' },
        ],
    };
}

sub remoderate_phrase : CMD {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    my $cat_id = $form->{cat_id};
    my $initial_phrase_id = $form->{initial_phrase_id};
    my $action = $form->{action};
    my $lang = $form->{language};
    return if !$initial_phrase_id || !$cat_id;
    $proj->set_user_phrase_status_by_id(
        $cat_id,
        $initial_phrase_id,
        $action,
        $lang,
        'New',
        'Возвращена на модерацию пользователем ' . $proj->login
    );
}

sub users_history : CMDH {
    my ($proj, $vars) = @_;

    return {
        title => 'История модерации',
        readonly => 1,
        table => 'CatalogiaPhrases',
        idfield => 'InitialPhraseId',
        default_field_params => { shlist => 1, },
        fields => [
            {  name => 'UpdateTime   ', title => 'Дата', inlinefilter   => {}, },
            {  name => 'Login        ', title => 'Автор', inlinefilter   => {}, },
            {  name => 'LastLogin    ', title => 'Посл.изм.', },
            {  name => 'Action       ', title => 'Act', },
            {  name => 'InitialPhrase', title => 'Фраза', showmacro => 'categphrase', inlinefilter   => {}, },
            {  name => 'CatID        ', title => 'Категория', showmacro => 'catid2link CatID Language', inlinefilter => {}, },
            {  name => 'Language     ', title => 'Язык', },
            {  name => 'Status       ', title => 'Статус', },
            {  name => 'Comment      ', title => 'Комментарий', },
            {  name => 'UserComment  ', title => 'Комментарий пользователя', },
            {
                title           => 'Вернуть&nbsp;на&nbspмодерацию',
                showmacroel     => 'show_inline_btn_field',
                hide_on_click   => 1,
                geturl      => sub {
                    my ($el, $f) = @_;
                    return '?cmd=remoderate_phrase&cat_id=' . uri_escape_utf8($el->{CatID}) .
                           '&initial_phrase_id=' . uri_escape_utf8($el->{InitialPhraseID}) .
                           '&action=' . uri_escape_utf8($el->{Action}) .
                           '&language=' . uri_escape_utf8($el->{Language});
                },
            },
        ],
        order_by => '-UpdateTime',
        pager => { name => 'p', cc => 50, },
        search => { fields => [ 'InitialPhrase', ], name => 'text', },
        tree => { grpfields => [
                      #{ field => 'UpdateTime', type => 'dategrp', },
                      { field => 'UpdateTime', type => 'datemnthgrp', },
                      { field => 'Login', },
                      { field => 'CatID', type => 'catid', order_by => 'cc desc', },
                  ],
                  pager => { name => 'p', cc => 10, },
              },
        filters => [
               {  name => 'Логин', field => 'Login', like => 1, use_other_filters => 1, },
               {  name => 'Action', field => 'Action', grp => 1, use_other_filters => 1, multi => 1, },
               {  name => 'CaterogyID', field => 'CatID',
                      textmacro   => sub {
                          my ($proj, $val) = @_;
                          #return ($proj->get_category($val, $proj->{current_lang}) || {})->{'CategoryName'} || $val;
                          return $proj->get_category_name($val) || $val;
                      },
                      use_other_filters => 1,
                      order_by_count    => 1,
                      #grp    => 1,
                      like     => 1,
                      hide_count => 1,
               },
               {  title => 'Дата', field => 'UpdateTime', type => 'date2', },
               {  name => 'Статус', field => 'Status', grp => 1, use_other_filters => 1, },
        ],
    };
}

sub get_category_url {
    my ($proj, $cat_id) = @_;
    my $category = $proj->get_category_full($cat_id);
    return $cat_id if !$category;
    my $cat_name = $category->{CategoryName};

    return "<a href=\"http://catmedia.yandex.ru/ind.pl?cmd=show_phrases&id=$cat_id&viewoptionsstr=lang_ru\">$cat_name</a>" .
            "(" .
            "<a href=\"http://catmedia.yandex.ru/ind.pl?cmd=show_phrases&id=$cat_id&viewoptionsstr=lang_ru\">$cat_id</a>" .
            ")";
}

sub make_html_header_row {
    my $row = "<tr>\n";
    @_ = map{"<th>" . ($_ || "&nbsp;") . "</th>\n"} @_;
    $row .= (join '', @_);
    $row .= "</tr>\n";
    return $row;
}

sub make_html_row {
    my $row = "<tr>\n";
    @_ = map{"<td>" . ($_ || "&nbsp;") . "</td>\n"} @_;
    $row .= (join '', @_);
    $row .= "</tr>\n";
    return $row;
}

sub get_html_table_for_add {
    my ($proj, $phrases) = @_;

    my $table = "<b>Добавление фразы в категорию:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Добавляемая фраза', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $initial_phrase, $comment) = ($phr->{CatID}, $phr->{InitialPhrase}, $phr->{Comment});
        my $category_url = get_category_url($proj, $cat_id);
        $table .= make_html_row($category_url, $initial_phrase, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_delete {
    my ($proj, $phrases) = @_;

    my $table = "<b>Удаление фразы из категории:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Удаляемая фраза', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $initial_phrase, $comment) = ($phr->{CatID}, $phr->{InitialPhrase}, $phr->{Comment});
        my $category_url = get_category_url($proj, $cat_id);
        $table .= make_html_row($category_url, $initial_phrase, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_add_category {
    my ($proj, $phrases) = @_;

    my $table = "<b>Добавление категории:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('ID добавляемой категории', 'Название добавляемой категории', 'Родительская категория', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $parent_cat_id, $initial_phrase, $comment) = ((split /:/, $phr->{CatID}), $phr->{InitialPhrase}, $phr->{Comment});
        my $parent_category_url = get_category_url($proj, $parent_cat_id);
        $table .= make_html_row($cat_id, $initial_phrase, $parent_category_url, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_delete_category {
    my ($proj, $phrases) = @_;

    my $table = "<b>Удаление категории:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Удаляемая категория', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $comment) = ($phr->{CatID}, $phr->{Comment});
        my $category_url = get_category_url($proj, $cat_id);
        $table .= make_html_row($category_url, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_rename_category {
    my ($proj, $phrases) = @_;

    my $table = "<b>Переименование категории:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Новое название категории', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $initial_phrase, $comment) = ($phr->{CatID}, $phr->{InitialPhrase}, $phr->{Comment});
        my $category_url = get_category_url($proj, $cat_id);
        $table .= make_html_row($category_url, $initial_phrase, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_add_prefilter {
    my ($proj, $phrases) = @_;

    my $table = "<b>Добавление префильтра:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Шаблон', 'Замещающий шаблон', 'Комментарий');
    for my $phr (@$phrases) {
        my ($pattern, $replacing_pattern, $comment) = ((split /\t/, $phr->{InitialPhrase}), $phr->{Comment});
        $table .= make_html_row('"' . $pattern . '"', '"' . $replacing_pattern . '"', $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_delete_prefilter {
    my ($proj, $phrases) = @_;

    my $table = "<b>Удаление префильтра:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Шаблон', 'Замещающий шаблон', 'Комментарий');
    for my $phr (@$phrases) {
        my ($pattern, $replacing_pattern, $comment) = ((split /\t/, $phr->{InitialPhrase}), $phr->{Comment});
        $table .= make_html_row('"' . $pattern . '"', '"' . $replacing_pattern . '"', $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_add_flag {
    my ($proj, $phrases) = @_;

    my $table = "<b>Добавление флага на категорию:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Добавляемый флаг', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $initial_phrase, $comment) = ($phr->{CatID}, $phr->{InitialPhrase}, $phr->{Comment});
        my $category_url = get_category_url($proj, $cat_id);
        $table .= make_html_row($category_url, $initial_phrase, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_delete_flag {
    my ($proj, $phrases) = @_;

    my $table = "<b>Удаление флага из категории:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Удаляемый флаг', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $initial_phrase, $comment) = ($phr->{CatID}, $phr->{InitialPhrase}, $phr->{Comment});
        my $category_url = get_category_url($proj, $cat_id);
        $table .= make_html_row($category_url, $initial_phrase, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_change_parent {
    my ($proj, $phrases) = @_;

    my $table = "<b>Изменение родительской категории:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Новая родительская категория', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $parent_cat_id, $comment) = ($phr->{CatID}, $phr->{InitialPhrase}, $phr->{Comment});
        my $category_url = get_category_url($proj, $cat_id);
        my $parent_category_url = get_category_url($proj, $parent_cat_id);
        $table .= make_html_row($category_url, $parent_category_url, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_add_antiword {
    my ($proj, $phrases) = @_;

    my $table = "<b>Добавление антислова на категорию:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Добавляемое антислово', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $antiword, $comment) = ($phr->{CatID}, $phr->{InitialPhrase}, $phr->{Comment});
        my $category_url = get_category_url($proj, $cat_id);
        $table .= make_html_row($category_url, $antiword, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_add_syn_pair {
    my ($proj, $phrases) = @_;

    my $table = "<b>Добавление пары синонимов:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Добавляемая пара синонимов', 'Комментарий');
    for my $phr (@$phrases) {
        my ($syn0, $syn1, $comment) = ((split /:/, $phr->{InitialPhrase}), $phr->{Comment});
        $table .= make_html_row('"' . $syn0 . '" "'. $syn1 . '"', $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_add_csyn_pair {
    my ($proj, $phrases) = @_;

    my $table = "<b>Добавление контекстных синонимов:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Добавляемые синонимы', 'Комментарий');
    for my $phr (@$phrases) {
        my ($syn, $comment) = ($phr->{InitialPhrase}, $phr->{Comment});
        $table .= make_html_row('"' . $syn . '"', $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_add_lemmerfix {
    my ($proj, $phrases) = @_;

    my $table = "<b>Добавление слов в Леммер-фикс:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Добавляемые слова', 'Комментарий');
    for my $phr (@$phrases) {
        my ($syn, $comment) = ($phr->{InitialPhrase}, $phr->{Comment});
        $table .= make_html_row('"' . $syn . '"', $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_delete_syn_pair {
    my ($proj, $phrases) = @_;

    my $table = "<b>Удаление пары синонимов:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Удаляемая пара синонимов', 'Комментарий');
    for my $phr (@$phrases) {
        my ($syn0, $syn1, $comment) = ((split /:/, $phr->{InitialPhrase}), $phr->{Comment});
        $table .= make_html_row('"' . $syn0 . '" "'. $syn1 . '"', $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_vadd {
    my ($proj, $phrases) = @_;

    my $table = "<b>Добавление фразы в виртуальную категорию:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Добавляемая фраза', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $initial_phrase, $comment) = ($phr->{CatID}, $phr->{InitialPhrase}, $phr->{Comment});
        my $category_url = get_category_url($proj, $cat_id);
        $table .= make_html_row($category_url, $initial_phrase, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_vdelete {
    my ($proj, $phrases) = @_;

    my $table = "<b>Удаление фразы из виртуальной категории:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Удаляемая фраза', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $initial_phrase, $comment) = ($phr->{CatID}, $phr->{InitialPhrase}, $phr->{Comment});
        my $category_url = get_category_url($proj, $cat_id);
        $table .= make_html_row($category_url, $initial_phrase, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_add_nephew {
    my ($proj, $phrases) = @_;

    my $table = "<b>Добавление отношения Nephew:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Добавляемая категория-племянник', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $initial_phrase, $comment) = @{$phr}{qw(CatID InitialPhrase Comment)};
        my $domain_category_url = get_category_url($proj, $cat_id);
        my $image_category_url = get_category_url($proj, $initial_phrase);
        $table .= make_html_row($domain_category_url, $image_category_url, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table_for_delete_nephew {
    my ($proj, $phrases) = @_;

    my $table = "<b>Удаление отношения Nephew:</b><br><br>\n";
    $table .= "<table border='1'>\n";
    $table .= make_html_header_row('Категория', 'Удаляемая категория-племянник', 'Комментарий');
    for my $phr (@$phrases) {
        my ($cat_id, $initial_phrase, $comment) = @{$phr}{qw(CatID InitialPhrase Comment)};
        my $domain_category_url = get_category_url($proj, $cat_id);
        my $image_category_url = get_category_url($proj, $initial_phrase);
        $table .= make_html_row($domain_category_url, $image_category_url, $comment);
    }
    $table .= "</table><br><br><br>\n";

    return $table;
}

sub get_html_table {
    my ($proj, $phrases) = @_;

    my $action_function = {
        Add             => \&get_html_table_for_add,
        Delete          => \&get_html_table_for_delete,
        AddCategory     => \&get_html_table_for_add_category,
        DeleteCategory  => \&get_html_table_for_delete_category,
        RenameCategory  => \&get_html_table_for_rename_category,
        AddPrefilter    => \&get_html_table_for_add_prefilter,
        DeletePrefilter => \&get_html_table_for_delete_prefilter,
        AddFlag         => \&get_html_table_for_add_flag,
        DeleteFlag      => \&get_html_table_for_delete_flag,
        ChangeParent    => \&get_html_table_for_change_parent,
        AddAntiword     => \&get_html_table_for_add_antiword,
        AddSynPair      => \&get_html_table_for_add_syn_pair,
        AddContextSyn   => \&get_html_table_for_add_csyn_pair,
        AddLemmerFix    => \&get_html_table_for_add_lemmerfix,
        DeleteSynPair   => \&get_html_table_for_delete_syn_pair,
        VAdd            => \&get_html_table_for_vadd,
        VDelete         => \&get_html_table_for_vdelete,
        AddNephew       => \&get_html_table_for_add_nephew,
        DeleteNephew    => \&get_html_table_for_delete_nephew,
    };

    my $action_priority = {
        AddLemmerFix    => 16,
        AddContextSyn   => 15,
        Add             => 14,
        Delete          => 13,
        AddCategory     => 12,
        DeleteCategory  => 11,
        RenameCategory  => 10,
        AddPrefilter    => 9,
        DeletePrefilter => 8,
        AddFlag         => 7,
        DeleteFlag      => 6,
        ChangeParent    => 5,
        AddAntiword     => 4,
        AddSynPair      => 3,
        DeleteSynPair   => 2,
        VAdd            => 1,
        VDelete         => 0,
        AddNephew       => -1,
        DeleteNephew    => -2,
    };

    my $action_phrases = {};
    push @{$action_phrases->{$_->{Action}}}, $_ for @$phrases;
    my $full_html_table;

    for my $action (sort {$action_priority->{$b} <=> $action_priority->{$a}} (keys %$action_phrases)) {
        # TODO  $action_function->{$action} может быть undef
        if($action_function->{$action}) {
            $full_html_table .= $action_function->{$action}($proj, $action_phrases->{$action});
        } else {
            $full_html_table .= "action_function for '$action' is not implemented";
        }
    }

    return $full_html_table;
}

sub save_moderation {
    my ($proj, $list, $moderator_login) = @_;
    
    for (@$list) {
        $_->{InitialPhraseDecoded} = $_->{InitialPhrase};
        $_->{InitialPhrase} = encode_entities($_->{InitialPhrase});
    }

    my ($status_field, $pre_status_field, $last_login_field, $comment_field);
    if ($proj->user->rights->{right_moderate}) {
        ($status_field, $pre_status_field, $last_login_field, $comment_field) = qw/Status PreStatus LastLogin Comment/;
    } elsif ($proj->user->rights->{right_moderate_media}) {
        ($status_field, $pre_status_field, $last_login_field, $comment_field) = qw/MediaStatus MediaPreStatus LastMedia MediaComment/;

        # kostyl for sending MediaComment
        $_->{Comment} = $_->{MediaComment} for @$list;
    } else {
        # TODO no rights
    }

    # allows to apply changes made by logins with prefix AutoModerator
    my $purified_moderator_login = $moderator_login =~ /^AutoModerator/ ?
                                   'AutoModerator' :
                                   $moderator_login;
    my $moderator_object = $proj->List_SQL("
        select
            *
        from
            Users
        where
            Login = ?
    ", [$purified_moderator_login])->[0];
    my $moderation_table = $proj->user_phrases->db_table;


    # рассылаем уведомления об отклонённых изменениях
    
    my $declined_phrases_query = [
        grep {
            $_->{$pre_status_field} eq 'Declined' and
            $_->{$status_field} eq 'New' and
            $_->{$last_login_field} eq $moderator_login
        } @$list
    ];

    my %declined_phrases_per_login = ();
    push @{$declined_phrases_per_login{$_->{Login}} ||= []}, $_ for @$declined_phrases_query;

    my $superior_email = 'shelkovnikova@yandex-team.ru, stitov@yandex-team.ru';
    for my $login (keys %declined_phrases_per_login) {
        my $login_object = $proj->List_SQL("
            select
                *
            from
                Users
            where
                Login = ?
        ", [$login])->[0];
        
        my $moderator_email = $moderator_object->{EMail};
        
        my @notify_lists = ("moderation_notifications");
        push @notify_lists, "moderation_superiors" unless $login_object->{role} eq 'admin'; 
        
        my $message = {
            from_name => $moderator_email,
            subject => "CatMedia Moderation",
            to => $login_object->{EMail},
            mail_lists => \@notify_lists,
            type => 'text/html',
            body => "Some of " . $login . "'s changes were rejected by " . $moderator_login. ".<br><br>" . get_html_table($proj, $declined_phrases_per_login{$login}),
        };
        
        $proj->SendMail($message);
    }


    # рассылаем уведомления о добавлении/удалении категорий
    
    my %important_changes = (
        AddCategory => 'Добавлена категория "%s"',
        DeleteCategory => 'Удалена категория "%s"',
    );
    
    my @important_changes_list = (
        grep {
            defined($important_changes{$_->{Action}}) and
            $_->{$pre_status_field} eq 'Accepted' and
            $_->{$status_field} eq 'New' and
            $_->{$last_login_field} eq $moderator_login
        } @$list
    );

    for my $change (@important_changes_list) {
        my $subject = sprintf($important_changes{$change->{Action}}, $change->{InitialPhraseDecoded});
        my $message = {
            from => $moderator_object->{EMail},
            subject => $subject,
            to => 'catalogia@yandex-team.ru',
            cc => $moderator_object->{EMail},
            body => join("\n",
                $subject,
                "",
                "Действие совершил пользователь $change->{Login} в $change->{CreateTime} с комментарием '$change->{UserComment}'",
                "Действие подтвердил модератор $moderator_object->{Login} только что",
            ),
        };

        $proj->SendMail($message);        
    }


    # сохраняем пользовательские комментарии к добавленным категориям как описания категорий
    
    my @new_categories = (
        grep {
            $_->{Action} eq 'AddCategory' and
            $_->{$pre_status_field} eq 'Accepted' and
            $_->{$status_field} eq 'New' and
            $_->{$last_login_field} eq $moderator_login
        } @$list
    );

    for my $change (@new_categories) {
        my ($real_cat_id, $parent_id) = split(m/:/, $change->{CatID});
        $proj->add_category_description(
            $real_cat_id,
            $change->{Language},
            $change->{$last_login_field},
            $change->{UserComment},
        );
    }
    
    
    # сохраняем действия модератора
    
    $proj->set_user_phrase_status(
        $_->{CatID},
        $_->{InitialPhraseDecoded},
        $_->{Action},
        $_->{Language},
        $_->{$pre_status_field},
        $_->{$comment_field}
    ) for grep {
        $_->{$pre_status_field} ne 'New' and
        $_->{$last_login_field} eq $moderator_login
    } @$list;
}


sub show_phrase_moderation : CMDH {
    my ($proj, $vars) = @_;
    # фраза, для которой будут искаться эквивалентные
    my $representative_phrase = $proj->form->{phrase};
    # приводим фразы в вид, подходящий для сравнения
    # удаляем минус-слова
    $representative_phrase =~ s/ -\w+//g;
    my $table = {
        base    => 'moderation',
        filters => [ { field => 'GroupAction', delete_base => 1, } ],
    };
    $table->{getlistflt} = sub {
        my ($self, %prm) = @_;
        my $list = $self->proj->get_moderation_phrases;

        $list = [
            grep {
                $_->{ClearedInitialPhrase} = $_->{InitialPhrase};
                # удаляем минус-слова
                $_->{ClearedInitialPhrase} =~ s/ -\w+//g;
                $_->{ClearedInitialPhrase} eq $representative_phrase
            } @$list
        ];

        $_->{ID} = join("-", @{$_}{qw{ CatID Language InitialPhraseID Action }}) for @$list;

        my %phrase_count = ();
        for my $phr (@$list) {
            ++$phrase_count{$phr->{ClearedInitialPhrase}};
        }

        $_->{PhraseActionCount} = $phrase_count{$_->{ClearedInitialPhrase}} for @$list;

        return $list;
    };
    return $table;
}

sub moderation_phrases_bucket : CMDH {
    my ($proj, $vars) = @_;

    my $h = {
        base            => 'moderation',
        title           => 'Список выбранных фраз',
        getlistflt      => sub {
            my ($self, %prm) = @_;
            my $list = $vars->{rights}{right_moderate} ?
                       $self->proj->get_moderation_phrases :
                       $self->proj->get_moderation_media_phrases;
            $_->{ID} = join("~", @{$_}{qw{ CatID Language InitialPhraseID Action }}) for @$list;
            my %h = map {$_->{ID} => 1} @{$proj->dbtable('ModerationPhrasesBucket')->List({Login => $proj->login})};
            $_->{Bucket} = $h{$_->{ID}} ? 'On' : 'Off' for @$list;
            $list = [grep {$_->{Bucket} eq 'On'} @$list];
            return $list;
        },
        bottom_actions  => [
            {
                name            => 'diff_task_bucket',
                title           => 'Create diff task for selected phrases',
                ellist          => 1,
                redir_action    => sub {
                    my ($self, $lst1, $lst2, $prmsh, $fltrs, $ellist) = @_;

                    $ellist = [grep {$_->{Bucket} eq 'On'} @$ellist];
                    $proj->dbtable('ModerationPhrasesBucket')->DelList({Login => $proj->login});
                    $proj->dbtable('ModerationPhrasesBucket')->Add({
                        ID      => join('~', @{$_}{qw(CatID Language InitialPhraseID Action)}),
                        Login   => $proj->login,
                    }) for @$ellist;

                    return 'ind.pl?cmd=banners_categs_diff&from_moderation=1';
                },
            },
        ],
    };

    return $h;
}

# fixme: turn into a macro or something?
sub get_banner_html {
    my ($proj, $banner) = @_;

    my $title = BaseForm::html_encode($banner->{BannerTitle});
    my $title_extension = $banner->{BannerTitleExtension} ? BaseForm::html_encode("/// " . $banner->{BannerTitleExtension}) : "";
    my $body = BaseForm::html_encode($banner->{BannerBody});
    my $bid = BaseForm::html_encode($banner->{BannerID});
    my $url = defined($banner->{BannerUrl}) ? BaseForm::html_encode("http://h.yandex-team.ru/?" . uri_escape_utf8($banner->{BannerUrl})) : "about:blank";

    my $domain;
    if (defined $banner->{BannerDomain}) {
        $domain = $banner->{BannerDomain};
    } elsif (defined $banner->{BannerUrl}) {
        my $derived_domain = lc $proj->page($banner->{BannerUrl})->domain;
        $domain = "$derived_domain [derived from url]";
    } else {
        $domain = "[unknown]";
    }
    $domain = BaseForm::html_encode($domain);
    
    return qq{
        <div>
            <div> <a class="banner__link" href="$url">$title</a> $title_extension</div>
            <div class="banner__text">$body</div>
            <div class="banner__domain">Domain: $domain;&nbsp;&nbsp;BannerID: $bid</div>
        </div>
    };
}

sub score_group_recalculate : CMD {
    my ($proj, $vars) = @_;

    my $form = $proj->form;
    my $viewoptionsstr = $proj->form->{viewoptionsstr} // "";
    my $score_group_id = $form->{score_group_id};

    die "provide non-empty score_group_id" unless $score_group_id;

    $proj->Do_SQL("
        delete from " . $DEBUG_TABLE_PREFIX . "AutoModerationScores
        where ScoreGroupID = '$score_group_id'
    ");

    $proj->Do_SQL("
        delete from " . $DEBUG_TABLE_PREFIX . "AutoModerationScoreGroupExamples
        where ScoreGroupID = '$score_group_id'
    ");
    
    $vars->{text} = "Скоры и примеры баннеров для группы '$score_group_id' удалены и будут пересчитаны в течение суток";
}

sub get_moderation_exteditfld_updater {
    my %opts = @_;

    my $what_field = $opts{what_field} // die "provide what_field";
    my $who_field = $opts{who_field} // die "provide who_field";
    my $moderation_table = $opts{moderation_table} // die "provide moderation_table";
    
    return sub {
        my ($self, $id, $key, $value) = @_;
        my ($cat_id, $lang, $phrase_id, $action) = split /~/, $id;
        my $proj = $self->proj;
        $proj->Do_SQL("
            update
                " . $moderation_table . "
            set
                $what_field = ?, $who_field = ?
            where
                CatID = ? and Language = ? and InitialPhraseID = ? and Action = ?
        ", [$value, $proj->user->user_inf->{Login}, $cat_id, $lang, $phrase_id, $action]);
    };
}

sub moderation : CMDH {
    my ($proj, $vars) = @_;

    my $moderation_table = $proj->user_phrases->db_table;

    my $decorator = sub {
        my ($el, $f) = @_;
        my $h = {}; $h->{background} = 'pink' if $el->{Action} =~ /Delete/;
        return $h;
    };

    return {
        idfield             => 'ID',
        title               => 'Модерация',
        readonly            => 1,
        default_field_params => {
            shlist  => 1,
        },
        getlistflt => sub {
            my ($self, %prm) = @_;

            my $list = $vars->{rights}{right_moderate} ?
                       $self->proj->get_moderation_phrases :
                       $self->proj->get_moderation_media_phrases;
            if (!$vars->{rights}{right_moderate} && !$vars->{rights}{right_moderate_media}) {
                $list = [ grep {$_->{Login} eq $proj->login} @$list ];
            }

            $_->{ID} = join("~", @{$_}{qw{ CatID Language InitialPhraseID Action }}) for @$list;
            my %h = map {$_->{ID} => 1} @{$proj->dbtable('ModerationPhrasesBucket')->List({Login => $proj->login})};
            $_->{Bucket} = $h{$_->{ID}} ? 'On' : 'Off' for @$list;

            return $list;
        },
        fields => [
            {
                name            => 'CreateTime',
                title           => 'Дата',
                inlinefilter   => {},
            },
            {
                name    => 'Login',
                title   => 'Пользователь',
                inlinefilter   => {},
            },
            {
                name    => 'Action',
                title   => 'Дейст-<br>вие',
                inlinefilter   => {},
                decorator => $decorator,
            },
            {
                width => 50,
                showsubel   => sub {
                    my ($el, $f) = @_;
                    return '' unless grep {$_ eq $el->{Action}} qw(Add Delete AddAntiword AddLemmerFix);
                    my $lang = $el->{Language};
                    my $uritext = uri_escape_utf8($el->{InitialPhrase});
                    return join('',
                               map { ref($_) eq 'ARRAY' ? "<a target=\"_blank\" href=\"".$_->[1]."\" style='margin:0px 5px 0px 0px;'><img style=\"max-width: none;\" src='/".$_->[0]."' width=16 border=0></a>" : $_ }
                               "<div style='width:32'><nobr>",
                               [ 'banners2.gif', ($el->{Action} eq 'AddAntiword' ? $proj->macros->get_macros->{get_url_search_banners}->($el->{InitialPhrase}, $el->{CatID}) : $proj->macros->get_macros->{get_url_search_banners}->($el->{InitialPhrase})) ],
                               [ 'ph_gr.gif',    "?cmd=get_phrase_info&viewoptionsstr=$vars->{viewoptionsstr}&phrase_text=$uritext" ],
                               "</nobr></div><div style='width:32px'><nobr>",
                               [ 'yndx_gr.gif',  "http://yandex.".( $lang eq 'tr' ? 'com.tr' : 'ru' )."/yandsearch?text=$uritext" ],
                               [ 'ggl_gr.gif',   $proj->macros->get_macros->{hidereferer}->("https://www.google.".( $lang eq 'tr' ? 'com.tr' : 'ru' )."/#q=$uritext") ],
                               "</nobr></div>",
                           );
                },
            },
            {
                name            => 'InitialPhrase',
                title           => 'Фраза',
                showmacro       => 'categphrase',
                inlinefilter    => {},
                addform         => 1,
                get_showmacro => sub {
                    my ($el, $field) = @_;
                    return (grep {$el->{Action} eq $_} qw(ChangeParent AddNephew DeleteNephew)) ? 'catid2caturl' : 'categphrase';
                },
                decorator => $decorator,
            },
            {
                #rights      => 'right_moderate',
                showmacroel => 'show_btn_field',
                icon        => 'eye-open',
                geturl      => sub {
                    my ($el, $f) = @_;
                    return "" if $el->{Action} ne 'Delete' || $el->{TrackingUpdatesPhrasesCount} <= 1;
                    return "?cmd=tracking_updates&cat_id=" . $el->{CatID} . "&action=" . $el->{Action} . "&language=" . $el->{Language} . "&initial_phrase_id=" . $el->{InitialPhraseID};
                },
            },
            # new eye
            {
                #rights      => 'right_moderate',
                showmacroel => 'show_btn_field',
                icon        => 'eye-close',
                geturl      => sub {
                    my ($el, $f) = @_;
                    return "" if $el->{TrackingUpdatesPhrasesCount} <= 1;
                    return "?cmd=tracking_updates&cat_id=" . $el->{CatID} . "&action=" . $el->{Action} . "&language=" . $el->{Language} . "&initial_phrase_id=" . $el->{InitialPhraseID} . "&algorithm=new";
                },
            },
            {
                name        => 'CatID',
                title       => 'Категория',
                showmacro   => 'catid2link',
            },
            {
                name    => 'Language',
                title   => 'Язык',
            },
            {
                name            => 'Importance',
                title           => 'Охват<br>банн.',
                inlinefilter    => {},
            },
            {
                name            => 'Score',
                title           => 'AM<br>Score',
                inlinefilter    => {},
                rights          => 'right_moderate,right_moderate_media',
            },
            {
                # от автомодератора
                rights  => 'right_moderate',
                name    => 'Details',
                title   => 'Примечания',
            },
            {
                name        => 'Comment',
                title       => 'Комментарий',
                rights      => 'right_moderate',
                showmacroel => 'show_list_inline_edit_field',
                width => 200,
                inline      => 1,
                exteditfld  => get_moderation_exteditfld_updater(
                    what_field => 'Comment',
                    who_field => 'LastLogin',
                    moderation_table => $moderation_table,
                ),
                repeat_previous_btn => 1,
            },
            {
                name        => 'MediaComment',
                title       => $proj->user->rights->{'right_moderate'} ? 'Комментарий<br>Медиа' : 'Комментарий',
                rights      => 'right_moderate_media,right_moderate',
                showmacroel => 'show_list_inline_edit_field',
                width => 200,
                inline      => 1,
                exteditfld  => get_moderation_exteditfld_updater(
                    what_field => 'MediaComment',
                    who_field => 'LastMedia',
                    moderation_table => $moderation_table,
                ),
                repeat_previous_btn => 1,
            },
            {
                name    => 'PreStatus',
                title   => 'Модерация',
                rights  => 'right_moderate',
                inline  => 1,
                'showmacroel' => 'show_list_inline_select_field',
                'selectlist' => [
                      {
                        'value'     => 'New',
                        'name'      => 'New',
                        'default'   => 1
                      },
                      {
                        'color' => '#00FF00',
                        'value' => 'Accepted',
                        'name'  => 'Accepted',
                      },
                      {
                        'color' => '#FF0000',
                        'value' => 'Declined',
                        'name'  => 'Declined',
                        'textcolor' => '#FFFFFF'
                      },
                ],
                width => 85,
                mass_change => 1,
                ftype   => 'select',
                exteditfld  => get_moderation_exteditfld_updater(
                    what_field => 'PreStatus',
                    who_field => 'LastLogin',
                    moderation_table => $moderation_table,
                ),
            },
            {
                name    => 'MediaPreStatus',
                title   => $proj->user->rights->{'right_moderate'} ? 'Модерация<br>Медиа' : 'Модерация',
                rights  => 'right_moderate_media,right_moderate',
                inline  => 1,
                'showmacroel' => 'show_list_inline_select_field',
                'selectlist' => [
                      {
                        'value'     => 'New',
                        'name'      => 'New',
                        'default'   => 1
                      },
                      {
                        'color' => '#00FF00',
                        'value' => 'Accepted',
                        'name'  => 'Accepted',
                      },
                      {
                        'color' => '#FF0000',
                        'value' => 'Declined',
                        'name'  => 'Declined',
                        'textcolor' => '#FFFFFF'
                      },
                ],
                width => 85,
                mass_change => 1,
                ftype   => 'select',
                exteditfld  => get_moderation_exteditfld_updater(
                    what_field => 'MediaPreStatus',
                    who_field => 'LastMedia',
                    moderation_table => $moderation_table,
                ),
            },
            {
                name    => 'UserComment',
                title   => 'User<br>comment',
                showmacroel => 'show_list_inline_edit_field',
                exteditfld  => get_moderation_exteditfld_updater(
                    what_field => 'UserComment',
                    who_field => 'LastLogin',
                    moderation_table => $moderation_table,
                ),
            },
            {
                name    => 'LastLogin',
                title   => 'Модератор',
                rights  => 'right_moderate',
            },
            {
                name    => 'LastMedia',
                title   => 'Модератор',
                rights  => 'right_moderate_media,right_moderate',
            },
            {
                name    => 'Bucket',
                title   => 'Корзина',
                inline  => 1,
                'showmacroel' => 'show_list_inline_select_field',
                'selectlist' => [
                      {
                        'color' => '#00FF00',
                        'value' => 'On',
                        'name'  => 'On',
                      },
                      {
                        'color' => '#FF0000',
                        'value' => 'Off',
                        'name'  => 'Off',
                        'textcolor' => '#FFFFFF'
                      },
                ],
                width => 70,
                mass_change => 1,
                ftype   => 'select',
                exteditfld  => sub {
                    my ($self, $id, $key, $value) = @_;
                    if ($value eq 'On') {
                        $proj->dbtable('ModerationPhrasesBucket')->Add(
                            {ID => $id, Login => $proj->login},
                            {ignore => 1}
                        );
                    } else {
                        $proj->dbtable('ModerationPhrasesBucket')->DelList(
                            {ID => $id, Login => $proj->login}
                        );
                    }
                },
            },
        ],
        filters => [
            {
                title => 'InitialPhrase',
                field => 'InitialPhrase',
                like => 1,
                use_other_filters => 1,
            },
            {
                field       => 'GroupAction',
                grp         => 1,
                type        => 'tabs',
                top_tabs    => 1,
            },
            {
                title   => 'Login',
                field   => 'Login',
                grp     => 1,
                use_other_filters => 1,
            },
            {
                title   => 'Action',
                field   => 'Action',
                grp     => 1,
                multi => 1,
                use_other_filters => 1,
            },
            {
                title               => 'Category',
                field               => 'CatID',
                textmacro           => sub {
                    my ($proj, $val) = @_;
                    return $proj->get_category_name($val, $proj->{current_lang}) || $val;
                },
                use_other_filters   => 1,
                order_by_count      => 1,
                multi               => 1,
                grp                 => 1,
                multiselect_include_select_all => 1,
            },
            {
                title       => 'Lang',
                field       => 'Language',
                grp         => 1,
                use_other_filters => 1,
            },
            {
                rights      => 'right_moderate',
                title       => 'Moderator',
                field       => 'LastLogin',
                grp         => 1,
                use_other_filters => 1,
            },
            {
                rights      => 'right_moderate,right_moderate_media',
                field       => 'LastMedia',
                title       => 'ModeratorMedia',
                grp         => 1,
                use_other_filters => 1,
            },
            {
                rights      => 'right_moderate',
                title       => 'Модерация',
                field       => 'PreStatus',
                grp         => 1,
                use_other_filters => 1,
            },
            {  title => 'Дата', field => 'UpdateTime', type => 'date2', },
        ],
        pager => {
            name    => 'p',
            cc      => '300',
        },
        bottom_actions => [
            {
                # Положить в "корзину" все фразы с этой страницы
                # "корзина" - одна для всех открытых вкладок браузера
                name    => 'filtered_phrases_selection',
                title   => 'Select filtered phrases',
                ellist  => 1,
                action  => sub {
                    my ($self, $lst1, $lst2, $prmsh, $fltrs, $ellist) = @_;
                    my @elements = map {
                        {
                            ID      => join('~', @{$_}{qw(CatID Language InitialPhraseID Action)}),
                            Login   => $proj->login,
                        }
                    } @$ellist;
                    $proj->dbtable('ModerationPhrasesBucket')->Add(\@elements, {ignore => 1});
                },
            },
            {
                name    => 'filtered_phrases_deselection',
                title   => 'Deselect filtered phrases',
                ellist  => 1,
                action  => sub {
                    my ($self, $lst1, $lst2, $prmsh, $fltrs, $ellist) = @_;
                    my @ids = map {join('~', @{$_}{qw(CatID Language InitialPhraseID Action)})} @$ellist;
                    $proj->dbtable('ModerationPhrasesBucket')->DelList({ID => \@ids, Login => $proj->login});
                },
            },
            {
                name    => 'selected_phrases_list',
                title   => 'View selected phrases',
                ellist  => 1,
                redir_action => sub {
                    return "ind.pl?cmd=moderation_phrases_bucket";
                },
            },
            {
                name    => 'bottom_action',
                title   => 'Approve my moderation',
                ellist  => 1,
                rights  => 'right_moderate',
                addformparams => {
                    phrase => 'phrase',
                    (map {$_ => $proj->form->{$_}} qw/cat_id language initial_phrase_id action/),
                },
                action  => sub {
                    my ($self, $lst1, $lst2, $prmsh, $fltrs, $ellist) = @_;
                    save_moderation($proj, $ellist, $proj->user->user_inf->{Login});
                },
            },
            {
                name    => 'set_media_prestatuses',
                title   => 'Set media statuses',
                ellist  => 1,
                rights  => 'right_moderate',
                action  => sub {
                    my ($self, $lst1, $lst2, $prmsh, $fltrs, $ellist) = @_;
                    $proj->Do_SQL("
                        update
                            $moderation_table
                        set
                            PreStatus = MediaPreStatus,
                            Comment = MediaComment,
                            LastLogin = ?
                        where
                            CatID = ? and Language = ? and InitialPhraseID = ? and Action = ?
                    ", [ $proj->user->user_inf->{'Login'}, @{$_}{qw/CatID Language InitialPhraseID Action/} ]) for @$ellist;
                },
            },
            {
                name    => 'auto_moderator_approval',
                title   => 'Approve AutoModerator',
                ellist  => 1,
                rights  => 'right_moderate',
                addformparams => {
                    (map { $_ => $proj->form->{$_} } grep { defined $proj->form->{$_} } qw/cat_id language initial_phrase_id action algorithm/),
                },
                action  => sub {
                    my ($self, $lst1, $lst2, $prmsh, $fltrs, $ellist) = @_;

                    my %logins_hash = map {$_->{LastLogin} => 1} @$ellist;
                    my @logins = grep {/^AutoModerator/} keys %logins_hash;
                    
                    save_moderation($proj, $ellist, $_) for @logins;
                }
            },
            {
                name    => 'comment_all',
                title   => 'Дописать комментарий к списку фраз',
                rights  => 'right_moderate',
                ellist  => 1,
                fields => [
                    { name => 'Comment', title => 'Comment', edlist => 1, },
                ],
                action => sub {
                    my ($self, $lst1, $lst2, $prmsh, $fltrs, $ellist) = @_;

                    for (@$ellist) {
                        my ($cat_id, $lang, $phrase_id, $action) = split /-/, $_->{ID};
                        $self->proj->Do_SQL("
                            update
                                " . $moderation_table . "
                            set
                                Comment = ?
                            where
                                CatID = ? and Language = ? and InitialPhraseID = ? and Action = ?
                        ", [$prmsh->{Comment}, $cat_id, $lang, $phrase_id, $action]);
                    }
                }
            },
        ],
    };
}

sub tracking_updates : CMDH {
    my ($proj, $vars) = @_;
    my $form = $proj->form;

    my ($cat_id, $lang, $initial_phrase_id, $action, $algorithm) = @{$form}{qw(cat_id language initial_phrase_id action algorithm)};

    my @candidates = @{$proj->get_tracking_updates_phrases($cat_id, $lang, $initial_phrase_id, $action, $algorithm)};

    my $cmdh_table = {
        base    => 'moderation',
        fields  => [
            {
                name        => 'ColoredInitialPhrase',
                title       => 'Редактирование',
                after       => 'InitialPhrase',
            },
            {
                name        => 'PhraseActionCount',
                delete_base => 1,
            },
        ],
        filters => [
            {
                field       => 'GroupAction',
                delete_base => 1,
            },
        ],
    };
    $cmdh_table->{getlistflt} = sub {
        my ($self, %prm) = @_;

        my $proj = $self->proj;
        my $list = [];
        my $phrase = $candidates[0]->{InitialPhrase};
        for my $candidate (@candidates) {
            my $levenshtein_transformation = $proj->levenshtein_transformation($phrase, $candidate->{InitialPhrase});
            my $colored_candidate_phrase = '';
            for (@$levenshtein_transformation) {
                # kostyl
                $_->[0] =~ s/</&lt;/g;
                $_->[0] =~ s/>/&gt;/g;
                # /kostyl
                if ($_->[1] eq 'm') {
                    $colored_candidate_phrase .= '<span>' . $_->[0] . '</span>';
                } elsif ($_->[1] eq 'r') {
                    $colored_candidate_phrase .= '<span style="background-color:#E3F878">' . $_->[0] . '</span>';
                } elsif ($_->[1] eq 'i') {
                    $colored_candidate_phrase .= '<span style="background-color:#2FD57D">' . $_->[0] . '</span>';
                } elsif ($_->[1] eq 'd') {
                    $colored_candidate_phrase .= '<span style="background-color:#F78D78">' . $_->[0] . '</span>';
                } else {
                }
            }
            $candidate->{ColoredInitialPhrase} = $colored_candidate_phrase;
            $candidate->{ID} = join '~', @{$candidate}{qw(CatID Language InitialPhraseID Action)};
            push @$list, $candidate
        }
        return $list;
    };
    return $cmdh_table;
}


sub remoderate : CMD {
    my ($proj, $vars) = @_;

    my $phrases = $proj->get_accepted_user_phrases;
    for my $phr (grep{$_->{Login} eq $proj->login} @$phrases) {
        $proj->set_user_phrase_status(
            $phr->{CatID},
            $phr->{InitialPhrase},
            $phr->{Action},
            $phr->{Language},
            "New",
            "перемодерация"
        );
    }

    $proj->do_redirect("ind.pl?cmd=moderation");
}

# Фразы из cmd=moderation - для поиска бага с кодировкой
sub test_encode : CMDH {
    my ($proj, $vars) = @_;

    return {
        idfield             => 'ID',
        title               => 'test_encode',
        readonly            => 1,
        default_field_params => {
            shlist  => 1,
        },
        fields => [
            (map {{ name => $_, title => $_, }}
                ('InitialPhrase' , 'UserComment' , 'MediaPreStatus' , 'Status' , 'MediaComment' , 'PreStatus' , 'Login' , 'Action' , 'UpdateTime' , 'InitialPhraseID' , 'Language' , 'CreateTime' , 'Comment' , 'MediaStatus' , 'Tags' , 'CatID' , 'LastMedia' , 'GroupAction' , 'LastLogin', )
            ),
        ],
        pager => {
            name    => 'p',
            cc      => '100',
        },
        getlistflt => sub {
            my ($self, %prm) = @_;
            $self->proj->log("begin get list");
            my $list = $vars->{rights}{right_moderate} ? $self->proj->get_moderation_phrases :
                                                         $self->proj->get_moderation_media_phrases;
            $self->proj->log("end get list");
            #$self->proj->dd($list->[0]);

            #return [ $list->[0] ];
            return $list;
        }
    }
}

1;
