# coding=utf-8
import argparse
import filelock
import logging

import yt.wrapper as yt

from prepare_cdict_input import PrepareCdictInput
from generate_full_cdicts import Chronicle, Datoteka


logging.basicConfig(format="%(asctime)s\t[%(process)d]\t%(message)s", level=logging.INFO)
logger = logging.getLogger(__name__)


def main(gen_type):
    pc = PrepareCdictInput()
    dt = Datoteka(pc.bmyt_cl)
    ch = Chronicle(pc.bmyt_cl)
    cdicts = [ch, dt]
    if gen_type == 'gen-yt':
        lock_table = pc.lock_table_path
        lock_client = pc.yt_client
        yt.create("table", lock_table, ignore_existing=True)
        with lock_client.Transaction():
            try:
                # лок на YT для генерации на YT
                lock_client.lock(lock_table, mode="exclusive")
                if pc.is_old():
                    pc.generate_cdict_input_full()
                    for cdict in cdicts:
                        cdict.gen_cdict_yt()
                    pc.finish_yt_regeneration()

            except yt.YtHttpResponseError as err:
                logger.info("Fail take yt lock: yt generation running; YT msg: %s", err.error)
    elif gen_type == 'gen-binary':
        # локальный, лок на генерацию бинарей
        try:
            with filelock.FileLock('cdict_generator').acquire(timeout=0):
                time_last_yt_generation = pc.get_last_generation()
                # перегенерация железных cdict
                for cdict in cdicts:
                    if cdict.binary_is_old(time_last_yt_generation):
                        cdict.upload_from_yt()
                        cdict.generate_binary()
                    else:
                        logger.info("cdict is up-to-time - %s", cdict.target_binary_path)
        except filelock.Timeout:
            logger.warning('Another instance of this application currently holds the lock.')


if __name__ == '__main__':
    argparser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    argparser.add_argument('--gen-type', choices=['gen-yt', 'gen-binary'], help="gen-yt - generate cdicts on yt;gen-binary - generate local binary file", required=True)
    args = argparser.parse_args()
    main(args.gen_type)
