# -*- coding: utf-8 -*-

import argparse
import copy
import yt.wrapper as yt

import collections
import datetime
import time

import bannerland.yql.tools
import irt.monitoring.solomon.sensors


# YT path to logs
LOG_PATH = "//logs/bs-chevent-log/1d"

# Object in YT for getting distributed lock
LOCK_YT_OBJECT = "//home/bannerland/locks/bs-chevent-log"

YABS_PAGE_YT_TABLE = "//home/yabs/dict/Page"


def get_log_sensors(template, count_dict):
    """
    Возвращает массив и 3 сенсоров на основе шаблона и кол-ва категоризованных/некатегоризованных единиц
    :param template: шаблон сенсора
    :param count_dict: словарь-счётчик с количеством категоризованных/некатегоризованных единиц
    :return: массив из 3-х искомых сенсоров
    """
    res_arr = []

    categ_count = count_dict.get("categorized", 0)
    uncateg_count = count_dict.get("uncategorized", 0)
    if (categ_count == 0) and (uncateg_count == 0):
        raise ValueError("Failed slice with zero count of elements!")

    for property, value in {
        "categorized": categ_count,
        "uncategorized": uncateg_count,
        "recall": 100.0 * categ_count / (categ_count + uncateg_count)
    }.items():

        sensor = copy.deepcopy(template)
        sensor["labels"]["property"] = property
        sensor["value"] = value
        res_arr.append(sensor)

    return res_arr


def main():
    solomon_client = irt.monitoring.solomon.sensors.SolomonAgentSensorsClient()
    success_finish = True

    # Get personal script lock
    yt_lock_client = yt.YtClient(proxy="locke", config=yt.default_config.get_config_from_env())

    with yt_lock_client.Transaction():
        try:
            yt_lock_client.lock(LOCK_YT_OBJECT)
        except Exception:
            exit()

        # Get YQL-client
        yql_cl = bannerland.yql.tools.get_client("hahn")

        # Parse command line arguments
        parser = argparse.ArgumentParser()
        parser.add_argument("-l", "--last_days_count", type=int, default=1)
        args = parser.parse_args()

        # Extract information about last success sending to Graphite
        datetime_now = datetime.datetime.now()

        # Get recalls from neccessary count of last days
        for delta_day in range(args.last_days_count, 0, -1):

            # Get log-table name (from yesterday)
            dt = datetime_now - datetime.timedelta(days=delta_day)
            table_name = "{year}-{month}-{day}".format(year=str(dt.year).zfill(4),
                                                       month=str(dt.month).zfill(2),
                                                       day=str(dt.day).zfill(2))

            dt = dt.replace(hour=0, minute=0, second=0, microsecond=0)
            dt += datetime.timedelta(1, time.timezone)
            ts_datetime = dt.isoformat()

            # YQL request: filter initial data, extract values of neccessary categories
            yql_query = """
                PRAGMA yt.InferSchema;

                $filtered_info = SELECT
                    logt.contexttype AS contexttype, logt.producttype AS producttype, logt.selecttype AS selecttype,
                    (CASE WHEN CAST(logt.bmcategoryid AS Int64) = 0 then 0 else 1 END) AS is_categorized,
                    (CASE WHEN page.TargetType in (0, 2) then "Search" WHEN page.TargetType = 3 then "Network" else "Unknown" END) AS targettype
                FROM
                    `{full_path}` AS logt
                    LEFT JOIN
                    `{yabs_page}` AS page
                ON
                    CAST(logt.pageid AS Int64) = page.PageID
                WHERE
                    (CAST(logt.fraudbits AS Int64) = 0) AND
                    (CAST(logt.placeid AS Int64) in (542, 1542));
            """.format(full_path=yt.ypath_join(LOG_PATH, table_name), yabs_page=YABS_PAGE_YT_TABLE)

            slices = ["contexttype", "producttype", "selecttype", "targettype"]
            yql_query += """
                SELECT
                    {slices}, `is_categorized`, COUNT(*) AS `cc`
                FROM
                    $filtered_info
                GROUP BY
                    {slices}, `is_categorized`;
            """.format(slices=", ".join("`{}`".format(slice) for slice in slices))

            yql_response = bannerland.yql.tools.list_yql_result(bannerland.yql.tools.do_yql(yql_cl, yql_query))

            # Обрабатываем YQL-результат
            slices_stats = {slice: dict() for slice in slices}
            for row in yql_response:
                slice_value = row["cc"]
                categ_property = "categorized" if row["is_categorized"] == 1 else "uncategorized"

                for slice in slices:
                    slice_name = row[slice]
                    slice_elem_counter = slices_stats[slice].get(slice_name, collections.Counter())
                    slice_elem_counter[categ_property] += slice_value
                    slices_stats[slice][slice_name] = slice_elem_counter

            # Формируем результат для отправки в Solomon
            overall_dict = collections.Counter()
            selected_slice = slices[0]
            sensors_arr = []

            for slice_type, slice_dict in slices_stats.items():
                for slice_name, categs_counter in slice_dict.items():
                    sensors_arr += get_log_sensors(
                        template={
                            "cluster": "yt_hahn",
                            "service": "catalogia_categorization",
                            "sensor": "slices_statistics",
                            "labels": {
                                "slice_type": slice_type,
                                "slice_name": str(slice_name),
                            },
                            "ts_datetime": ts_datetime,
                        },
                        count_dict=categs_counter,
                    )

                    if slice_type == selected_slice:
                        for categ_property in ["categorized", "uncategorized"]:
                            overall_dict[categ_property] += categs_counter.get(categ_property, 0)

            sensors_arr += get_log_sensors(
                template={
                    "cluster": "yt_hahn",
                    "service": "catalogia_categorization",
                    "sensor": "overall_statistics",
                    "labels": {},
                    "ts_datetime": ts_datetime,
                },
                count_dict=overall_dict,
            )

            if solomon_client.push_sensors(sensors_arr):
                success_finish = False

    if success_finish:
        solomon_client.set_success_script_finish("get_log_percantage_slices")


if __name__ == "__main__":
    main()
