# coding=utf8

import copy
import filelock
import hashlib
import logging
import os
import time

import yt.wrapper as yt
import yt.yson as yson

import bm.yt_tools
import irt.broadmatching.common_options


if "YT_POOL" not in os.environ:
    os.environ["YT_POOL"] = "catalogia"

if "YT_TOKEN_PATH" not in os.environ:
    os.environ["YT_TOKEN_PATH"] = os.path.expanduser("/secrets/tokens/yt_plato")

if "YT_PROXY" not in os.environ:
    os.environ["YT_PROXY"] = "hahn"


YT_TEMP_DIR = "//home/catalogia/dyntables/resources/tmp"
YT_TABLET_CELL_BUNDLE = "catalogia"

logging.basicConfig(format="%(asctime)s\t[%(process)d]\t%(message)s", level=logging.INFO)
logger = logging.getLogger(__name__)


def make_fields_dict(schema):
    return {col["name"]: col for col in schema}


def put_keys_forward(schema, key_fields):
    # переставить ключевые колонки в начало схемы, объявить их сортированными, а все прочие - несортированными
    # https://wiki.yandex-team.ru/yt/userdoc/tables/#sortirovannost

    schema = copy.deepcopy(schema)
    new_schema = yson.YsonList()
    new_schema.attributes = schema.attributes

    fields_dict = make_fields_dict(schema)

    for key_field in key_fields:
        schema_el = fields_dict.pop(key_field)
        schema_el["sort_order"] = "ascending"
        new_schema.append(schema_el)

    for column in schema:
        if column["name"] in fields_dict:
            column.pop("sort_order", None)
            new_schema.append(column)

    return new_schema


class StripMapper(object):
    def __init__(self, keep_fields=(), add_fields=None):
        self.add_fields = add_fields or {}
        self.keep_fields = keep_fields

    def __call__(self, input_row):
        output_row = {field: input_row[field] for field in self.keep_fields if field in input_row}
        output_row.update(self.add_fields)
        yield output_row


def create_dyntable(src_table, dst_table, schema, key_fields, mapper=None, reducer=bm.yt_tools.FirstReducer(), tablet_cell_bundle=YT_TABLET_CELL_BUNDLE):
    logger.info("Create dyntable '%s' from static table '%s' started", dst_table, src_table)

    mapped_table = get_tmp_table(src_table + "_mapped")
    reduced_table = get_tmp_table(src_table + "_reduced")

    schema = put_keys_forward(schema, key_fields)
    schema.attributes["unique_keys"] = True

    logger.info("Remove temp tables")
    yt.remove(mapped_table, force=True)
    yt.remove(reduced_table, force=True)

    logger.info("Create reduced table '%s'", reduced_table)
    yt.create_table(
        reduced_table,
        recursive=True,
        attributes={
            "schema": schema,
            "tablet_cell_bundle": tablet_cell_bundle,
        },
    )

    table_for_reduce = src_table
    if mapper is not None:
        logger.info("Map '%s' to '%s'", src_table, mapped_table)
        yt.run_map(mapper, src_table, mapped_table)
        logger.info("Sort '%s'", mapped_table)
        yt.run_sort(mapped_table, sort_by=key_fields)
        table_for_reduce = mapped_table

    logger.info("Reduce '%s' to '%s'", src_table, reduced_table)
    yt.run_reduce(
        reducer,
        table_for_reduce,
        reduced_table,
        reduce_by=key_fields,
        spec={
            "job_io": {"table_writer": {"block_size": 256 * 2**10, "desired_chunk_size": 100 * 2**20}}
            # https://wiki.yandex-team.ru/yt/userdoc/dynamictablesmapreduce/#prevrashheniestaticheskojjtablicyvdinamicheskuju
        }
    )

    logger.info("Alter '%s'", reduced_table)
    bm.yt_tools.set_attribute(reduced_table, "enable_tablet_balancer", True, yt)  # https://wiki.yandex-team.ru/yt/userdoc/dynamicsortedtables/#avtomaticheskoeshardirovanie
    yt.alter_table(reduced_table, dynamic=True)

    logger.info("Check if '%s' mounts/unmounts", reduced_table)
    mount_frozen(reduced_table)
    unmount(reduced_table)

    if yt.exists(dst_table):
        logger.info("Remove old '%s'", dst_table)
        if bm.yt_tools.get_attribute(dst_table, "tablet_state", yt) == "mounted":
            unmount(dst_table)
        yt.remove(dst_table, force=True)

    logger.info("Rename '%s' to '%s'", reduced_table, dst_table)
    yt.move(reduced_table, dst_table)

    logger.info("Set medium to ssd_blobs '%s'", dst_table)
    bm.yt_tools.set_attribute(dst_table, "primary_medium", "ssd_blobs", yt)

    logger.info("Mount '%s'", dst_table)
    mount_frozen(dst_table)

    logger.info("Remove temp tables")
    yt.remove(mapped_table, force=True)
    yt.remove(reduced_table, force=True)

    logger.info("Create dyntable '%s' from static table '%s done'", dst_table, src_table)


def mount_frozen(table):
    started = time.time()
    timeout = 1800
    while 1:
        try:
            yt.mount_table(table, sync=True)
        except Exception as e:
            logger.exception(e)
            if time.time() - started > timeout:
                break
            time.sleep(1)
        else:
            break
    while 1:
        try:
            yt.freeze_table(table, sync=True)
        except Exception as e:
            logger.exception(e)
            if time.time() - started > timeout:
                break
            time.sleep(1)
        else:
            break


def unmount(table):
    yt.unmount_table(table, sync=True)


def get_tmp_table(table):
    basename = table.split("/")[-1]  # fixme: yt.ypath_dirname есть, a yt.ypath_basename нет, wtf?
    return yt.ypath_join(YT_TEMP_DIR, basename + "_" + hashlib.md5(table).hexdigest())


def update_resource(src_table, dst_table, primary_key, secondary_keys=None, extra_key_table_fields=None, force_update=False):
    logger.info("Check '%s' from '%s' at cluster '%s'", dst_table, src_table, os.environ["YT_PROXY"])
    secondary_keys = secondary_keys or []
    extra_key_table_fields = extra_key_table_fields or []

    # todo: mtime - не самый лучший способ следить за тем, что содержимое таблицы не поменялось
    # для продакшна хорошо бы следить за каким-нибудь своим атрибутом, чтобы не перестраивать динамические таблицы после каждого ночного пережатия статических

    src_mtime = bm.yt_tools.get_mtime(src_table)
    try:
        dst_mtime = bm.yt_tools.get_mtime(dst_table)
    except yt.YtHttpResponseError:
        dst_mtime = 0

    if (src_mtime < dst_mtime) and not force_update:
        logger.info("Table '%s' is up-to-date, skip", dst_table)
        return

    logger.info("Table '%s' needs updating, proceed", dst_table)

    src_table_schema = bm.yt_tools.get_attribute(src_table, "schema", yt)
    create_dyntable(
        src_table=src_table,
        dst_table=dst_table,
        schema=src_table_schema,
        key_fields=[primary_key]
    )

    # между перегенерацией основной таблицы и индексов может пройти изрядно времени
    # то есть, есть довольно продолжительный период, когда читатели ходят в индексы от старой таблицы, а потом по ним идут в новую
    # а ещё исходная таблица может измениться во время генерации индексов, и тогда индексы будут от разных версий таблицы
    # мы пока считаем, что это всё не проблема, потому что данные с которыми мы тут работаем имеют почти-инкрементальную природу
    # breqwas@, 23.05.2018

    all_key_table_fields = [primary_key] + secondary_keys + extra_key_table_fields
    for secondary_key in secondary_keys:
        fields_dict = copy.deepcopy(make_fields_dict(src_table_schema))
        key_table = dst_table + "_by_" + secondary_key
        add_fields = None

        key_table_schema = yson.YsonList()
        key_table_schema.attributes = src_table_schema.attributes
        key_table_schema.append(fields_dict[secondary_key])
        key_table_schema.append(fields_dict[primary_key])

        if len(secondary_keys) > 1:
            # в ключевых таблицах сохраняем все поля, по которым есть ключи (наверняка пригодятся)
            for other_key_field in secondary_keys:
                if other_key_field != secondary_key:
                    key_table_schema.append(fields_dict[other_key_field])
        else:
            # динамическая таблица обязана иметь неключевые поля
            key_table_schema.append({"name": "_dummy", "type": "boolean"})
            add_fields = {"_dummy": True}

        for field in extra_key_table_fields:
            key_table_schema.append(fields_dict[field])

        create_dyntable(
            src_table=src_table,
            dst_table=key_table,
            schema=key_table_schema,
            key_fields=[secondary_key, primary_key],
            mapper=StripMapper(keep_fields=all_key_table_fields, add_fields=add_fields)
        )

    logger.info("Table '%s' done", dst_table)


def main():
    logger.info("Started for cluster '%s'", os.environ["YT_PROXY"])

    logger.info("Read settings...")
    dt_proxy_params = irt.broadmatching.common_options.get_options()["dt_proxy_params"]

    logger.info("Update resources...")
    for resource in dt_proxy_params["resources"]:
        update_resource(
            src_table=resource["src_table"],
            dst_table=resource["dst_table"],
            primary_key=resource["primary_key"],
            secondary_keys=resource["secondary_keys"],
            extra_key_table_fields=resource["extra_key_table_fields"],
        )

    logger.debug("Done updating resources at cluster '%s'", os.environ["YT_PROXY"])


if __name__ == "__main__":
    lock = filelock.FileLock("prepare-dyn-tables.pid")
    try:
        with lock.acquire(timeout=0):
            main()
    except filelock.Timeout:
        logger.warning("Another instance of this application currently holds the lock.")
