#!/usr/bin/python
# -*- coding: utf-8 -*-

import filelock
import logging

import yenv
import yt.wrapper as yt

import irt.bannerland.options
import irt.iron.options as iron_opts

from yt.transfer_manager.client import TransferManager
from bm.yt_tools import get_upload_time, get_mtime
from irt.monitoring.solomon.sensors import SolomonAgentSensorsClient


logging.basicConfig(format="%(asctime)s\t[%(process)d]\t%(message)s", level=logging.INFO)
logger = logging.getLogger(__name__)


# Вычисляем время обновления таблицы.
# Если есть наш атрибут "upload_time", опираемся на него, иначе смотрим на системный "modification_time"
def get_update_time(table, yt_client):
    update_time = get_upload_time(table, yt_client=yt_client)
    if update_time is None:
        update_time = get_mtime(table)
    return update_time


# Трансфер таблицы с одного кластера на другой
def transfer_table(source_cluster, source_table, target_cluster, target_table, sync=False):

    from_to_string = "from {}.`{}` to {}.`{}`".format(source_cluster, source_table, target_cluster, target_table)
    logger.error("Start transferring %s ...", from_to_string)

    # Transfer-клиент, а также Source и Target YT-клиенты
    token_path = iron_opts.get('yt_token_path')

    with open(token_path, 'r') as f:
        token = f.read().strip()
    transfer_client = TransferManager(token=token)

    yt_source_client = yt.YtClient(
        config={
            'proxy': {'url': source_cluster},
            'token_path': token_path,
        }
    )
    yt_target_client = yt.YtClient(
        config={
            'proxy': {'url': target_cluster},
            'token_path': token_path,
        }
    )

    # Сравниваем время обновления таблиц
    original_mtime = get_update_time(source_table, yt_source_client)
    copy_mtime = 0
    if yt_target_client.exists(target_table):
        copy_mtime = get_update_time(target_table, yt_target_client)

    if copy_mtime < original_mtime:
        if yenv.type != yenv.STABLE:
            logger.error("DEBUG mode. Let's assume, that the table was transferred.")
        else:
            transfer_client.add_task(
                source_cluster,
                source_table,
                target_cluster,
                target_table,
                params={"mr_user": "catalogia", "schema_inference_mode": "from_input"},
                sync=sync,
            )
            if sync:
                logger.error("[OK] transferring '%s'", from_to_string)
    else:
        logger.error("The table is up to date. No transferring.")


# Трансфер таблиц Вебмастера с Арнольда на Хан
def transfer_webmaster_tables():

    logger.error("Transferring Webmaster tables start ...")
    dyn_src_opts = irt.bannerland.options.get_option('DynSources')
    for sources in ('queries', 'links'):
        transfer_table(
            'arnold',
            dyn_src_opts[sources]['yt_raw_table_arnold'],
            'hahn',
            dyn_src_opts[sources]['yt_raw_table'],
            sync=True,
        )

    logger.error("Transferring Webmaster tables done.")


def main():
    tables_for_arnold = [
        "//home/catalogia/catalogia-banners-recategorized-fast",
        "//home/catalogia/categ2mediagroup",
        "//home/catalogia/categories_tree",
        "//home/catalogia/id2mediagroup",
        "//home/catalogia/mediagroup2videodirectgroup",
        "//home/catalogia/id2videodirectgroup",
        "//home/catalogia/videodirectgroups_second_level",
        "//home/catalogia/interests_mapping",
    ]

    dyn_src_opts = irt.bannerland.options.get_option('DynSources')
    tables_for_arnold.append(dyn_src_opts['dse']['yt_path_domain'])

    tables_for_hume = [
        "//home/catalogia/banners_extended"
    ]

    for table_name in tables_for_arnold:
        transfer_table("hahn", table_name, "arnold", table_name)
    for table_name in tables_for_hume:
        transfer_table("hahn", table_name, "hume", table_name)

    transfer_webmaster_tables()
    logger.error('transfer-essential-tables_OK')
    solomon_client = SolomonAgentSensorsClient()
    solomon_client.set_success_script_finish("transfer-essential-tables")


# Main
if __name__ == '__main__':
    try:
        with filelock.FileLock('transfer-essential-tables').acquire(timeout=0):
            main()
    except filelock.Timeout:
        logger.warning('Another instance of this application currently holds the lock.')
