# coding=utf-8
import datetime
import logging
from six import integer_types
import requests

from irt.common.base_json_adapter import BaseJSONApiAdapter


class ABCAdapter(BaseJSONApiAdapter):
    API_URL = "https://abc-back.yandex-team.ru/api/v4/{request}"

    def __init__(self, token):
        self.token = token

    def get_members(self, service_id_or_slug, role_id_or_slug=None):
        """
        Get members of service
        :param service_id_or_slug: Service id or slug name in ABC
        :param role_id_or_slug: Role id or slug name in service in ABC
        :return: List of members with roles
        """
        params = dict()

        if isinstance(service_id_or_slug, integer_types):
            params['service'] = service_id_or_slug
        else:
            params['service__slug'] = service_id_or_slug

        if role_id_or_slug is not None:
            if isinstance(role_id_or_slug, integer_types):
                params['role'] = role_id_or_slug
            else:
                params['role__scope'] = role_id_or_slug

        req = self._request(requests.get,
                            url=self.API_URL.format(request="services/members/"),
                            result_field='results',
                            headers={"Authorization": "OAuth %s" % self.token},
                            params=params)

        if req is not None:
            users = {}
            for u in req:
                username = u['person']['login']
                if username not in users:
                    users[username] = {
                        'username': username,
                        'roles': []
                    }
                users[username]['roles'].append({
                    'id': u['role']['id'],
                    'slug': u['role']['scope']['slug']
                })

            return list(users.values())
        else:
            logging.error('Bad request ABCAdapter.get_members()')
            return []

    def get_service_slug(self, service_id):
        """
        Get slug name of service
        :param service_id: Service id in ABC
        :return: Service slug name in ABC
        """
        req = self._request(requests.get,
                            url=self.API_URL.format(request="services/"),
                            result_field='results',
                            headers={"Authorization": "OAuth %s" % self.token},
                            params={"id": service_id})

        if req is not None:
            return req[0]["slug"] if req else None
        else:
            logging.error('Bad request ABCAdapter.get_service_slug()')

    def get_duty(self, service_id_or_slug, date_from=None, date_to=None):
        """
        Get duty of service
        :param service_id_or_slug: Service id or slug name in ABC
        :param date_from: Start date for searching duty. Today if None
        :param date_to: End date for searching duty. Equal to date_from if None
        :return: List of duty
        """
        date_from = date_from or datetime.date.today()
        date_to = date_to or date_from
        params = {"date_from": date_from.isoformat(), "date_to": date_to.isoformat()}

        if isinstance(service_id_or_slug, integer_types):
            params['service'] = service_id_or_slug
        else:
            params['service__slug'] = service_id_or_slug

        req = self._request(requests.get,
                            url=self.API_URL.format(request="duty/shifts/"),
                            result_field='results',
                            headers={"Authorization": "OAuth %s" % self.token},
                            params=params)

        if req is not None:
            return [{
                'username': d['person']['login'],
                'name': d['schedule']['name'],
                'start': d['start'],
                'end': d['end'],
            } for d in req]
        else:
            logging.error('Bad request ABCAdapter.get_duty()')
            return []
