# -*- coding: utf-8 -*-

import json
import irt.logging
import requests

log = irt.logging.getLogger(None, __name__)


class ArtmonAPI(object):
    DATE_FORMAT = "%Y-%m-%d"

    def __init__(self, oauth_token):
        """
        Class initialization.

        :param oauth_token: Artmon OAuth token
        """
        self._oauth_token = oauth_token

    def do_request(self, main_start, main_end, cmp_start, cmp_end, timegroup, add_filters=None):
        """
        Implement an Artmon request.

        :param main_start: start date of the current interval
        :param main_end: end date of the current interval
        :param cmp_start: start date of the compared interval
        :param cmp_end: end date of the compared interval
        :param timegroup: timeseries' step (per minute/hour etc.)
        :param add_filters: additional Artmon filters, if need
        :return: If success - response (as a Python dict), otherwise - None
        """
        artmon_filters = {
            "YandexFilter": "all", "OrderFilter": "all", "TypeFilter": "all", "ContextFilter": "0",
            "group_context_type": None, "UserFilter": "all", "detailed": 1, "Slice": "sim_distance", "relative": 0,
            "dont_show_total": 1, "device_type": "all", "group_device_type": None, "group_product_type": None,
            "product_type": "all", "distribution": 0, "placeid_paid_filter": "542,1542", "placeid_filter": "542,1542",
            "pageid_filter": "", "impid_filter": "", "group_select_type": None, "group_sim_distance": None,
            "group_turbo": None, "group_inapp": None, "group_ya_service": None, "group_ssp": None, "group_video": None,
            "group_video_kind": None, "group_video_format": None, "group_preferred_deals": None, "group_is_adfox": None,
            "exclude_pageid_filter": "", "cross_filter": "0", "undefined": "", "domain_ajax": "", "login_ajax": "",
            "failed_money": 0, "show_visit_info": 0, "show_ad_serp_hits": 0, "not_only_good": 0, "discount": 0,
            "hide_ext": 1, "logarithm_axis": 0, "fraction_series": 1, "fraction_series_shift": 1, "ignore_fast": 0,
            "fast_traffic": 0, "clickhouse": 0, "bsclickhouse": 1, "fraud_series": 0,
        }
        if add_filters is not None:
            artmon_filters.update(add_filters)

        request_data = {
            "module": "Traffic2",
            "period_start": main_start.strftime(self.DATE_FORMAT),
            "period_end": main_end.strftime(self.DATE_FORMAT),
            "compare_start": cmp_start.strftime(self.DATE_FORMAT),
            "compare_end": cmp_end.strftime(self.DATE_FORMAT),
            "period_start_hour": "00",
            "compare_start_hour": "00",
            "period_end_hour": "23",
            "compare_end_hour": "23",
            "compare_enabled": "1",
            "timegroup": timegroup,
            "query": json.dumps({"filters": artmon_filters}, sort_keys=True),
        }

        log.info("Sending Artmon request...")
        log.debug("Request data: '%s'", request_data)
        req_res = requests.post(
            "https://artmon.bsadm.yandex-team.ru/cgi-bin/data.cgi",
            data=request_data,
            headers={"Authorization": "OAuth {}".format(self._oauth_token)},
            verify=False,
        )
        log.info("Artmon response code: %s", req_res.status_code)
        log.debug("Artmon response length (in bytes): %s", len(req_res.content))

        if not req_res.ok:
            log.error("Artmon request was failed!")
            log.debug("Response raw result: %s\nHeaders: %s", req_res.raw, req_res.headers)

            return None

        return req_res.json()
