import os
import copy
import pydash as _
import yaml
from six import with_metaclass

import irt.utils

__version__ = irt.utils.get_version()


class ConfigMetaclass(type):
    _instance = None

    def __new__(cls, name, bases, class_dict):
        for parent_base in bases:
            if isinstance(parent_base, ConfigMetaclass):
                raise TypeError("type '{0}' is not an acceptable base type".format(parent_base.__name__))

        return type.__new__(cls, name, bases, class_dict)

    def __call__(cls, *args, **kwargs):
        if cls._instance is None:
            if len(args) == 0 and 'module_name' not in kwargs:
                raise RuntimeError('Config is not initialized')
            ConfigMetaclass._instance = type.__call__(cls, *args, **kwargs)
        else:
            if len(args) > 0 or len(kwargs) > 0:
                raise RuntimeError('Attemp to reinit config')

        return cls._instance

    def __getitem__(self, key):
        return self._instance[key]

    def __getattr__(self, key):
        return self._instance.section(key)

    def __setattr__(self, key, val):
        raise RuntimeError('Attemp to modify config')


class Config(with_metaclass(ConfigMetaclass)):
    _config = {}

    @classmethod
    def config_filename(cls, suffix='config'):
        return '{}.yml'.format(suffix)

    def __init__(self, module_name=None, initial_config=None, config_patch=None):
        if not self._config:
            if initial_config is not None:
                type.__setattr__(Config, '_config', initial_config)

            if module_name is not None:
                locations = [
                    irt.utils.get_system_config_file(module_name, self.config_filename()),
                    irt.utils.get_user_config_file(module_name, self.config_filename()),
                    os.path.join(os.curdir, self.config_filename(module_name)),
                    os.environ.get(module_name.upper() + '_CONF')
                ]
            else:
                locations = [
                    os.path.join(os.curdir, self.config_filename())
                ]

            for location in locations:
                if location and os.path.isfile(location):
                    with irt.utils.suppress(IOError):
                        with open(location) as source:
                            config_dict = yaml.load(source, Loader=yaml.FullLoader)
                            _.merge(Config._config, config_dict)

            if config_patch is not None:
                _.merge(Config._config, config_patch)

    def section(self, section_name):
        cls = type.__call__(type(self))
        object.__setattr__(cls, '_config', _.get(self._config, section_name))
        return cls

    def get(self, item, default=None):
        return copy.deepcopy(self._config.get(item, default))

    def __getattr__(self, key):
        if key in self._config:
            return self.section(key)
        return super(Config, self).__getattr__(key)

    def __getitem__(self, key):
        return self.get(key)

    def __setattr__(self, key, val):
        raise RuntimeError('Attempt to modify config')
