#include <Python.h>
#ifndef PyMODINIT_FUNC /* declarations for DLL import/export */
#define PyMODINIT_FUNC static void
#endif

static PyObject* INIT_SUBCLASS_METHOD_NAME = PyString_FromString((char*)"__init_subclass__");
static newfunc original_tp_new = NULL;

/**
 * Replacing of original type.__new__ with some magic
 *
 * Working stages:
 *
 * Updating __dict__:
 * Special-case __init_subclass__ and __class_getitem__:
 *      if they are plain functions, make them classmethods
 *
 * Call original type.__new__
 *
 * Call super.__init_subclass__
 *
 * @param type Address of PyObject object with metaclass for class.
 * @param args Arguments for calling type.__new__.
 * @param kwds Keyword arguments for calling type.__new__.
 */
static PyObject* tp_new_with_init_subclass(struct _typeobject *type, PyObject *args, PyObject *kwds);

/**
 * Calling super.__init_subclass__ for class object
 *
 * @param class_type Address of PyObject object with class object
 */
int call_init_subclass(PyObject *class_type);

/**
 * Replacing type.__new__ to another function. That calls original type.__new__ and do magic
 */
void apply_patch();

PyMODINIT_FUNC
init_init_subclass(void) {
    PyObject* module;
    module = Py_InitModule("_init_subclass", NULL);
    if (module == NULL)
        return;
    apply_patch();
}

void apply_patch() {
    original_tp_new = PyType_Type.tp_new;
    PyType_Type.tp_new = &tp_new_with_init_subclass;
}

int call_init_subclass(PyObject *class_type) {
    PyObject *super, *func, *result;

    super = PyObject_Call((PyObject *)&PySuper_Type, PyTuple_Pack(2, class_type, class_type), NULL);
    if (super == NULL) {
        // Normal: no super class
        return 0;
    }
    if (!PyObject_HasAttr(super, INIT_SUBCLASS_METHOD_NAME)) {
        // Normal: no __init_subclass__ method in parent
        Py_DECREF(super);
        return 0;
    }
    func = PyObject_GetAttr(super, INIT_SUBCLASS_METHOD_NAME);
    Py_DECREF(super);

    if(!PyCallable_Check(func)) {
        // Error: __init_subclass__ not callable
        Py_DECREF(func);
        return -1;
    }

    result = PyObject_CallFunctionObjArgs(func, NULL);
    Py_DECREF(func);
    if (result == NULL) {
        // Error: result of __init_subclass__ is NULL
        return -1;
    }
    Py_DECREF(result);

    return 0;
}

PyObject* tp_new_with_init_subclass(struct _typeobject *type, PyObject *args, PyObject *kwds) {
    static char *kwlist[] = {(char*)"name", (char*)"bases", (char*)"dict", 0};
    PyObject* cls = NULL;
    PyObject* tmp = NULL;
    PyObject *name, *bases, *dict;

    const Py_ssize_t nargs = PyTuple_GET_SIZE(args);
    const Py_ssize_t nkwds = kwds == NULL ? 0 : PyDict_Size(kwds);
    if (nargs + nkwds == 3) {
        PyArg_ParseTupleAndKeywords(args, kwds, "SO!O!:type", kwlist,
                                    &name,
                                    &PyTuple_Type, &bases,
                                    &PyDict_Type, &dict);

        tmp = PyDict_GetItem(dict, INIT_SUBCLASS_METHOD_NAME);
        if (tmp != NULL && PyFunction_Check(tmp)) {
            tmp = PyClassMethod_New(tmp);
            if (tmp == NULL) {
                PyErr_SetString(PyExc_TypeError, "Error while converting __init_subclass__ to classmethod");
                return NULL;
            }
            if (PyDict_SetItem(dict, INIT_SUBCLASS_METHOD_NAME, tmp) < 0) {
                PyErr_SetString(PyExc_TypeError, "Error while converting __init_subclass__ to classmethod");
                Py_DECREF(tmp);
                return NULL;
            }
            Py_DECREF(tmp);
        }
    }

    cls = original_tp_new(type, args, kwds);
    if (call_init_subclass(cls) != 0) {
        if (!PyErr_Occurred()) {
            PyErr_SetString(PyExc_TypeError, "Error while calling super.__init_subclass__");
        }
        Py_DECREF(cls);
        return NULL;
    }
    return cls;
}
