from logging import *  # noqa
import logging
import os
import io
import six  # noqa


IRT_PROJECT = 'IRT'

BANNERLAND_PROJECT = 'BANNERLAND'
MULTIK_PROJECT = 'MULTIK'

PROJECTS = (
    BANNERLAND_PROJECT,
    MULTIK_PROJECT
)


TIME_LOG_PART = '%(asctime)s'
LEVEL_NAME_LOG_PART = '%(levelname)s'
LOGGER_NAME_LOG_PART = '%(name)s'
MESSAGE_LOG_PART = '%(message)s'
LINE_NUMBER_LOG_PART = '%(lineno)d'
PATHNAME_LOG_PART = '%(pathname)s'
FILENAME_LOG_PART = '%(filename)s'
MODULE_LOG_PART = '%(module)s'

MULTIK_FORMAT = '{{"time":"{}","level":"{}","logger":"{}","file":"{}","line":{},"message":"{}"}}'.format(
    TIME_LOG_PART,
    LEVEL_NAME_LOG_PART,
    LOGGER_NAME_LOG_PART,
    PATHNAME_LOG_PART,
    LINE_NUMBER_LOG_PART,
    MESSAGE_LOG_PART
)

BM_FORMAT = '%(asctime)s\t[%(process)d]\t%(message)s'

FORMATS = {
    IRT_PROJECT: BM_FORMAT,
    BANNERLAND_PROJECT: BM_FORMAT,
    MULTIK_PROJECT: MULTIK_FORMAT
}


def set_default_format(irt_project):
    """
    Set format of IRT_PROJECT logger similar to format of irt_project logger.

    :param irt_project: Project from which copy format of logger
    :return:
    """
    if irt_project not in PROJECTS:
        raise ValueError('Wrong project name')

    FORMATS[IRT_PROJECT] = FORMATS[irt_project]


def getLogger(irt_project,             # type: str
              name,                    # type: six.string_types
              log_level=logging.INFO,  # type: str
              ):                       # type: (...) -> logging.Logger
    """
    Get logger for module.

    :param irt_project: Name of project in IRT. None for common libs
    :param name: Name for logger
    :param log_level: Default level of logging, if ENV '`PROJECT`_LOG_LEVEL' not presented
    :return: logger
    """
    if irt_project is None:
        irt_project = IRT_PROJECT
    elif irt_project not in PROJECTS:
        raise ValueError('Wrong project name')

    logger = logging.getLogger(name=name)

    if logger.handlers:
        return logger

    log_level = os.getenv('{}_LOG_LEVEL'.format(irt_project), log_level)
    logger.setLevel(log_level)

    ch = logging.StreamHandler()
    ch.setLevel(log_level)
    formatter = logging.Formatter(FORMATS[irt_project])
    ch.setFormatter(formatter)
    logger.addHandler(ch)

    return logger


# Usage: tqdm_out = TqdmToLogger(logger); tqdm.tqdm(..., file=tqdm_out)
# Thanks to https://github.com/tqdm/tqdm/issues/313#issuecomment-267959111
class TqdmToLogger(io.StringIO):
    """
        Output stream for TQDM which will output to logger module instead of
        the StdOut.
    """
    logger = None
    level = None
    buf = ''

    def __init__(self, logger, level=None):
        super(TqdmToLogger, self).__init__()
        self.logger = logger
        self.level = level or logging.INFO

    def write(self, buf):
        self.buf = buf.strip('\r\n\t ')

    def flush(self):
        self.logger.info(self.buf)
        self.buf = ''


__all__ = logging.__all__ + ['IRT_PROJECT', 'BANNERLAND_PROJECT', 'MULTIK_PROJECT', 'set_default_format']
