import datetime as dt
import dateutil.parser
import shutil
import os
import six
import subprocess
import requests
from tqdm import tqdm
import tarfile
import tempfile
import time
import logging
import socket

try:
    from backports.tempfile import TemporaryDirectory
except ImportError:
    from tempfile import TemporaryDirectory

import sandbox.projects.release_machine.core.const as rm_const

from sandbox.common import rest
from sandbox.common import auth as common_auth
from sandbox.common.types import task as task_types

# NOTE Wait for skycore team
from api.copier import Copier

from .upload import _upload


logger = logging.getLogger(__name__)

HTTP_BLOCK_SIZE = 4096
RSYNC_PORT = 837


class Sandbox(object):
    """
    Usage examples:
    Sandbox().download(1376286432, dir)
    Sandbox().download_irt_resource('banners_test', temp_dir)
    """

    IRT_DATA_TYPE = 'IRT_DATA'
    COPY_RESOURCE_TASK = 'REMOTE_COPY_RESOURCE'

    SUCCESS_STATUS = 'SUCCESS'
    FAILURE_STATUSES = {'FAILURE', 'EXCEPTION'}

    def __init__(self, token=None):
        auth = common_auth.OAuth(token) if token is not None else None
        self._client = rest.Client(auth=auth) << rest.Client.HEADERS({'X-Links': '1'})
        self._token = token

    def _resource(self, res_type=None, state='READY', **kwargs):
        if kwargs:
            kwargs = {'attrs': kwargs}

        if state is not None:
            kwargs['state'] = state
        if res_type is not None:
            kwargs['type'] = res_type

        items = self._client.resource.read(limit=1, order="-id", **kwargs).get('items', [None])
        if len(items):
            return items[0]
        return None

    def download_resource(self, destination, mode=None, res_type=None, state='READY', **kwargs):
        logger.debug('Downloading resource attributes=%s to %s', kwargs, destination)
        resource = self._resource(res_type=res_type, state=state, **kwargs)
        if resource is not None:
            return self.download(resource, destination, mode)

    def download_irt_resource(self, sub_type, destination, mode=None, state='READY', **kwargs):
        logger.debug('Downloading IRT_DATA resource with sub_type=%s and attributes=%s in %s state to %s', sub_type, kwargs, state, destination)
        kwargs['sub_type'] = sub_type
        return self.download_resource(destination, mode=mode, res_type=Sandbox.IRT_DATA_TYPE, state=state, **kwargs)

    def download(self, resource_or_sbr_id, destination, mode=None):
        logger.debug('Downloading resource %s to %s', resource_or_sbr_id, destination)
        mode = mode or 'skynet'
        resource = resource_or_sbr_id if isinstance(resource_or_sbr_id, dict) else self._client.resource[resource_or_sbr_id].read()

        with TemporaryDirectory() as temp_dir:
            os.chmod(temp_dir, 0o777)
            return self._download(resource, temp_dir, destination, mode)

    def attributes_irt_resource(self, sub_type, state='READY', **kwargs):
        logger.debug('Getting attributes for IRT_DATA resource with sub_type=%s and attributes=%s in %s state', sub_type, kwargs, state)
        return self.attributes_resource(res_type=Sandbox.IRT_DATA_TYPE, state=state, **kwargs)

    def attributes_resource(self, res_type=None, state='READY', **kwargs):
        logger.debug('Getting attributes for resource with type=%s and attributes=%s in %s state', res_type, kwargs, state)
        resource = self._resource(res_type=res_type, state=state, **kwargs)
        if resource is not None:
            return self.attributes(resource)

    def attributes(self, resource_or_sbr_id):
        resource = resource_or_sbr_id if isinstance(resource_or_sbr_id, dict) else self._client.resource[resource_or_sbr_id].read()
        logger.debug('Attributes for resource %s', resource)
        return resource['attributes']

    def attribute(self, resource_or_sbr_id, key, default=None):
        logger.debug('Getting attribute %s for %s with default %s', key, resource_or_sbr_id, default)
        return self.attributes(resource_or_sbr_id).get(key, default)

    def new_bmgen_resource_needed(self, sub_type, hours_ttl=48):
        last_resource = self._resource(res_type=Sandbox.IRT_DATA_TYPE, sub_type=sub_type)
        if last_resource is None:
            raise RuntimeError('Can\'t find bmgen resource with sub_type="{}"'.format(sub_type))
        if (last_resource['task']['status'] == 'RELEASED') and (last_resource['attributes']['released'] == rm_const.ReleaseStatus.stable):
            return True
        if (time.time() - time.mktime(dateutil.parser.parse(last_resource['time']['created']).timetuple())) / 3600.0 > hours_ttl:
            raise RuntimeError('Too long waiting for the realesing of bmgen resource with sub_type="{}"'.format(sub_type))
        return False

    def not_released_bmgen_resources(self, sub_types, strict=True):
        result_resources = dict()
        for sub_type in sub_types:
            resource = self._resource(res_type=Sandbox.IRT_DATA_TYPE, sub_type=sub_type)
            if resource is None:
                raise ValueError('Invalid bmgen resource with sub_type="{}"'.format(sub_type))
            if (resource['task']['status'] != 'RELEASED') or (resource['attributes']['released'] != rm_const.ReleaseStatus.stable):
                result_resources[sub_type] = resource
            elif strict:
                return
        return result_resources

    def set_finite_ttl_for_previous_released_bmgen_resources(self, sub_type, released_type):
        res = self._client.resource.read(
            limit=1000,
            order='-id',
            type=Sandbox.IRT_DATA_TYPE,
            attrs={
                'ttl': 'inf',
                'sub_type': sub_type,
                'released': released_type
            }
        )

        if res is None or res['items'] is None or len(res['items']) == 0:
            return

        for resource in res['items']:
            resource_id = resource['id']
            logging.info('Set ttl=30 for resource with id {}'.format(resource_id))
            self._client.resource[resource_id].attribute['ttl'] = {'value': 30}

    def _download(self, resource, temp_dir, destination, mode):
        skynet_id = resource.get('skynet_id')
        rsync_list = resource.get('rsync', {}).get('links', [])
        mds = resource.get('mds')
        mds = mds.get('url') if mds is not None else None

        if mode != 'skynet' or not skynet_id or not self._is_skynet_available() or not self.download_sky(skynet_id, temp_dir):
            if mode == 'skynet':
                mode = 'rsync'
            if mode != 'rsync' or not rsync_list or not self.download_rsync(rsync_list[0], temp_dir):
                if mode == 'rsync':
                    mode = 'mds'
                if mode != 'mds' or mds is None or not self.download_mds(mds, temp_dir):
                    http = resource.get('http', {}).get('links', [])
                    if not http or not self.download_http(http[0], temp_dir, resource.get('size', 0)):
                        logger.error('Error while downloading resource %s' % resource['id'])
                        raise RuntimeError('Error while downloading resource %s' % resource['id'])
        return self._copy(temp_dir, destination)

    @classmethod
    def _check_port_ip(cls, host, port, ipv6=True):
        try:
            sock = socket.socket(socket.AF_INET6 if ipv6 else socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(1)
            return sock.connect_ex((host, port)) == 0
        except socket.gaierror:
            return False

    @classmethod
    def _check_port(cls, host, port):
        return cls._check_port_ip(host, port) or cls._check_port_ip(host, port, False)

    @classmethod
    def download_rsync(cls, link, dst):
        """
        Download data via rsync
        :param link: Rsync link to source
        :param dst: Destination directory
        :return:
        """
        logger.info('Downloading via rsync %s', link)

        if not link.startswith('rsync://'):
            logger.error('Wrong rsync link')
            return False

        host = link[8:].split('/')[0]
        if not cls._check_port(host, RSYNC_PORT):
            logger.error('Port is closed')
            return False

        cmd = ['rsync', '-r', link, dst]
        try:
            cls._call(cmd)
            return True
        except RuntimeError as e:
            logger.exception('Error occupied %s', e)
            return False

    @classmethod
    def _download_http(cls, stream_req, file_obj, file_size):
        with tqdm(total=file_size, unit='iB', unit_scale=True) as pbar:
            for data in stream_req.iter_content(HTTP_BLOCK_SIZE):
                pbar.update(len(data))
                file_obj.write(data)
        file_obj.flush()

    @classmethod
    def download_mds(cls, link, dst, file_size=None):
        try:
            logger.info('Downloading via mds %s', link)
            req = requests.get(link, stream=True)

            with tempfile.NamedTemporaryFile(mode='wb+', delete=False) as tmp_file:
                cls._download_http(req, tmp_file, file_size)
                tmp_file.seek(0)

                if os.path.exists(dst) and os.path.isdir(dst):
                    dst = os.path.join(dst, req.url.rsplit('/', 1)[-1])
            shutil.move(tmp_file.name, dst)

            return True
        except Exception as e:
            logger.exception('Exception raised %s during Downloading via MDS', e)
            return False

    @classmethod
    def download_http(cls, link, dst, file_size=None):
        """
        Download via http from sandbox
        :param link:
        :param dst:
        :param file_size:
        :return:
        """
        logger.info('Downloading via http %s', link)

        req = requests.get(link, params={'stream': 'tgz'}, stream=True)
        with tempfile.TemporaryFile(mode='wb+') as tmp_file:
            cls._download_http(req, tmp_file, file_size)
            tmp_file.seek(0)

            tar = tarfile.open(fileobj=tmp_file)
            tar.extractall(path=dst)
            tar.close()
        return True

    @classmethod
    def _sky_path(cls):
        """
        Path to skynet binary
        :return: Path to skynet binary
        """
        return '/usr/local/bin/sky'

    @classmethod
    def _is_skynet_available(cls):
        """
        Check if skynet available
        :return: True if skynet available
        """
        if not os.path.exists(cls._sky_path()):
            logger.error('Skynet is not available')
            return False
        # TODO check cmd output or check via api.Copier
        # example https://a.yandex-team.ru/arc/trunk/arcadia/build/scripts/fetch_from_sandbox.py?rev=6342766#L45
        return True

    @classmethod
    def _call(cls, cmd, cwd=None):
        """
        Call commandline command
        :param cmd: Command with args
        :return:
        """
        # TODO move to utils https://a.yandex-team.ru/review/1157995/details#comment-1679414
        process = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, cwd=cwd)
        data = process.communicate()
        if process.returncode != 0:
            raise RuntimeError('\n'.join(str(line) for line in data))

        return data

    @classmethod
    def _download_sky_py3_workaround(cls, sky_id, dst):
        """
        Host part of api.copier doesn't support Python3
        :param sky_id: id like rbtorrent:some_id
        :param dst: Path to destination
        :return:
        """
        logger.error('Using Py3 workaround')
        cmd = [cls._sky_path(), 'get', '-d', dst, '-w', sky_id]
        cls._call(cmd)

    def download_sky(self, sky_id, dst):
        """
        Download via skynet to destination.
        For Py3 using workaround via direct calling "sky get"
        :param sky_id: id like rbtorrent:some_id
        :param dst: Path to destination
        :return:
        """
        logger.info('Downloading via skynet %s', sky_id)

        if six.PY3:
            # TODO delete after https://a.yandex-team.ru/review/1156444/details
            try:
                self._download_sky_py3_workaround(sky_id, dst)
                return True
            except RuntimeError as e:
                logger.exception('Error occupied %s', e)
                return False
        else:
            try:
                Copier().handle(sky_id).get(dest=dst).wait()
            except Exception as e:
                logger.exception('Error occupied %s', e)
                # TODO check for exceptions
                return False
        return True

    @classmethod
    def _copy(cls, src, dst):
        """
        Copy all from directory src to directory dst
        :param src: Source directory
        :param dst: Destination directory
        :return:
        """
        # TODO check for https://a.yandex-team.ru/review/1157995/details#comment-1679370
        files = list(os.listdir(src))
        for f in files:
            source = os.path.join(src, f)
            if os.path.isdir(source):
                shutil.copytree(source, os.path.join(dst, f))
            else:
                shutil.copy(source, os.path.join(dst, f))
        return files

    def upload_irt_resource(self, path, resource_name, sub_type, description, owner, **kwargs):
        kwargs['sub_type'] = sub_type
        return self.upload_resource(path, resource_name, Sandbox.IRT_DATA_TYPE, description, owner, **kwargs)

    def upload_resource(self, path, resource_name, resource_type, description, owner, **kwargs):
        if self._is_skynet_available():
            try:
                resource_id = self.upload_resource_sky(path, resource_name, resource_type, description, owner, **kwargs)
                if resource_id is not None:
                    return resource_id
            except Exception as e:
                logger.exception(e)
            logger.error('Upload via sky failed')
        try:
            resource_id = self.upload_resource_http(path, resource_name, resource_type, description, owner, **kwargs)
            if resource_id is not None:
                return resource_id
        except Exception as e:
            logger.exception(e)
        logger.error('Upload via http failed')
        raise RuntimeError('Upload failed')

    @classmethod
    def _sky_share(cls, path):
        logger.error('Using Py3 workaround')
        dir_name, base_name = os.path.split(path)
        cmd = [cls._sky_path(), 'share', base_name]

        binary_rb_id = b''.join(cls._call(cmd, cwd=dir_name)).strip()

        if six.PY2:
            return binary_rb_id
        else:
            return binary_rb_id.decode('utf8')

    def upload_resource_sky(self, path, resource_name, resource_type, description, owner, **kwargs):
        logger.info('Uploading %s via skynet', path)
        rb_id = self._sky_share(path)
        logger.info('Uploading %s', rb_id)
        task_params = {
            'context': {
                'created_resource_name': resource_name,
                'resource_type': resource_type,
                'resource_attrs': ','.join('{}={}'.format(k, kwargs[k]) for k in kwargs),
                'remote_file_protocol': 'skynet',
                'remote_file_name': rb_id
            },
            'type': 'REMOTE_COPY_RESOURCE',
            'description': description,
            'owner': owner
        }

        task_id = self._client.task(task_params)['id']
        start_status = self._client.batch.tasks.start.update([task_id])[0]['status']
        if start_status != Sandbox.SUCCESS_STATUS:
            raise RuntimeError('Failed to start task {}. Status: "{}"'.format(task_id, start_status))
        logger.info('Uploading task https://sandbox.yandex-team.ru/task/%s/view', task_id)
        while True:
            # TODO add timeouts
            task_status = self._client.task[task_id].read()['status']
            if task_status == Sandbox.SUCCESS_STATUS:
                break
            elif task_status in Sandbox.FAILURE_STATUSES:
                raise RuntimeError('Task {} failed. Status: "{}"'.format(task_id, task_status))
            time.sleep(1)
        for resource in self._client.task[task_id].resources.read()['items']:
            if resource['type'] == resource_type:
                return resource['id']
        error = 'Resource with type {} not found in task {}'.format(resource_type, task_id)
        logger.error(error)
        raise RuntimeError(error)

    def upload_resource_http(self, path, resource_name, resource_type, description, owner, **kwargs):
        if os.path.basename(path) != resource_name:
            with TemporaryDirectory() as temp_dir:
                dst = os.path.join(temp_dir, resource_name)
                if os.path.isdir(path):
                    shutil.copytree(path, dst)
                else:
                    shutil.copy(path, dst)
                return self.upload_resource_http(dst, resource_name, resource_type, description, owner, **kwargs)
        args = ','.join('{}={}'.format(k, kwargs[k]) for k in kwargs)
        task_id = _upload(
            path,
            owner,
            resource_type,
            description,
            self._token,
            args
        )
        for resource in self._client.task[task_id].resources.read()['items']:
            if resource['type'] == resource_type:
                return resource['id']

    def create_task(self, task_name, owner, description, task_params=None, **kwargs):
        logging.info('Start creating task "%s"', task_name)
        if len(description) == 0:
            raise ValueError('Description is empty')

        task_info = self._client.task({'type': task_name})
        if (not isinstance(task_info, dict)) or (task_info.get('id') is None):
            raise RuntimeError('Fail to create task "{task_name}"'.format(task_name=task_name))
        task_id = task_info['id']

        kwargs.update({
            'owner': owner,
            'description': description,
        })
        if task_params is not None:
            kwargs['custom_fields'] = [{'name': name, 'value': value} for name, value in task_params.items()]
        self._client.task[task_id] = kwargs

        creating_response = self._client.batch.tasks.start.update(task_id)
        if (not isinstance(creating_response, list)) or (len(creating_response) == 0):
            raise RuntimeError('Fail for getting info for task_id={task_id}'.format(task_id=task_id))
        if creating_response[0]['status'] != 'SUCCESS':
            raise RuntimeError('Creating of task "{task_name}" (id={task_id}) has been failed with status {status}'.format(
                task_name=task_name,
                task_id=task_id,
                status=creating_response[0]['status'],
            ))

        logging.info('The task "%s" has been successfully created. See: %s', task_name, self.get_task_url(task_id))
        return task_id

    def wait_task(self, task_id):
        finished_statuses = set(task_types.Status.Group.FINISH) | set(task_types.Status.Group.BREAK)
        while self._client.task[task_id].read()['status'] not in finished_statuses:
            time.sleep(1)
        return self._client.task[task_id].read()['status']

    def get_task_url(self, task_id):
        return 'https://sandbox.yandex-team.ru/task/{task_id}'.format(task_id=task_id)

    def release_task(self, task_id, release_type, message, subject=None, timeout=dt.timedelta(seconds=300)):
        logger.info('Start auto release for task_id=%s to release type %s', task_id, release_type)
        self._client.release.create({
            'task_id': task_id,
            'type': release_type,
            'subject': subject if subject is not None else 'Auto release for SB-task {task_id} to release type "{release_type}"'.format(task_id=task_id, release_type=release_type),
            'message': message,
        })

        start_time = dt.datetime.now()
        task_url = self.get_task_url(task_id)
        fail_attempts = 0
        while dt.datetime.now() - start_time <= timeout:
            task_status = None
            try:
                task_status = self._client.task[task_id].read()['status']
            except:
                fail_attempts += 1
                if fail_attempts == 3:
                    raise RuntimeError('Too many failed requsts for the getting of task %s info', task_id)
            if task_status == task_types.Status.RELEASED:
                break
            if task_status in (task_types.Status.NOT_RELEASED, task_types.Status.FAILURE, task_types.Status.DELETED):
                raise RuntimeError('Releasing of SB-task has been failed, see "{task_url}"'.format(task_url=task_url))
            time.sleep(1)
        else:
            raise RuntimeError('Too long waiting for the success release finishing of SB-task "{task_url}"'.format(task_url=task_url))

        logger.info('Release for task_id=%s has been successfully finished.', task_id)


# NOTE: Not used (actual on 15.10.2020). Ticket for the deleting: IRT-2085
class YTWithSBResourceBase(object):
    # скачиваем из sb ресурсы по id (resource_ids) + по типу (resource_types)
    # находим файлы с именами из списка required_filenames
    # запоминаем абсолютные пути к ним в self.local_files
    # при запуске в run_* нужно передать local_files=mapper.local_files
    def __init__(self, required_filenames, resource_ids=None, resource_types=None, stable=True):
        logger.warning('You are using class "YTWithSBResourceBase". Please, remove "NOTE" about its unusing in the code.')

        self.tmp_dir = TemporaryDirectory()
        sb = Sandbox()

        if resource_ids:
            for resource_id in resource_ids:
                if stable and sb.attribute(resource_id, 'released') != rm_const.ReleaseStatus.stable:
                    raise ValueError('fail: sb resource %s is not stable', resource_id)
                sb.download(resource_id, self.tmp_dir.name)

        if resource_types:
            kwargs = {'destination': self.tmp_dir.name}
            if stable:
                kwargs['released'] = rm_const.ReleaseStatus.stable

            for resource_type in resource_types:
                sb.download_resource(res_type=resource_type, **kwargs)

        if not resource_ids and not resource_types:
            raise ValueError('set resource_ids or resource_types')

        required_filenames_set = set(required_filenames)
        founded_filepaths_set = set()
        # рекурсивно обходим скаченные папки и ищем там файлы из required_filenames
        # если в ресурсах нельколько файлов с одинковым именем, и это имя есть в needed_filenames, то бросается исключение, т.к. не знаем какой брать
        # речь только про имена файлов, даже если они лежат в разных папках, всё равно будет исключение
        for root, dirs, files in os.walk(self.tmp_dir.name):
            founded_resoures = [os.path.join(root, fn) for fn in files if fn in required_filenames_set]
            if any(res in founded_filepaths_set for res in founded_resoures):
                raise ValueError('found duplicate needed_filename in sb resources')
            founded_filepaths_set.update(founded_resoures)

        if len(required_filenames_set) != len(founded_filepaths_set):
            raise ValueError('not all needed_filenames find in sb resource')

        self.local_files = list(founded_filepaths_set)

    def __del__(self):
        self.tmp_dir.cleanup()
