import os
import requests

DEFAULT_TVM_DAEMON_SECRETS_PATH = '/var/lib/tvmtool/local.auth'


class TVM(object):
    """
    Class for getting tvm tickets for service, or make directly requests
    """
    def __init__(self, secret=None, host='localhost', port=18080, protocol='http'):
        """

        :param secret: secret of tvm application (default value from /var/lib/tvmtool/local.auth)
        :param host: host with tvmtool
        :param port: port of tvm tool
        :param protocol: protocol to use
        """
        if secret is None:
            if not os.path.exists(DEFAULT_TVM_DAEMON_SECRETS_PATH):
                raise ValueError('tvm daemon secret not selected and file %s not exists, tvm daemon is running?', DEFAULT_TVM_DAEMON_SECRETS_PATH)

            with open(DEFAULT_TVM_DAEMON_SECRETS_PATH) as token_f:
                secret = token_f.read().rstrip()
            if not secret:
                raise ValueError('tvm daemon secret not selected and not found in %s, tvm daemon is running?', DEFAULT_TVM_DAEMON_SECRETS_PATH)

        self._secret = secret
        self._url = '{}://{}:{}/tvm/tickets'.format(protocol, host, port)

    def get_ticket(self, service):
        """
        Get TVM ticket for service

        :param service: Name of service
        :return: ticket
        """
        return requests.get(
            self._url,
            params={
                'dsts': service
            },
            headers={
                'Authorization': self._secret
            }
        ).json()[service]['ticket']

    def process_requests_method(self, service, method, *args, **kwargs):
        """
        Make request to service with TVM ticket

        :param service: Name of service
        :param method: requests method to use
        :param args: args for requests (example: [Url] for .get .post, [method, Url] for .Request)
        :param kwargs: kwargs for requests method
        :return:
        """
        ticket = self.get_ticket(service)
        method = getattr(requests, method)
        if 'headers' not in kwargs:
            kwargs['headers'] = {}

        kwargs['headers'].update({'X-Ya-Service-Ticket': ticket})

        return method(*args, **kwargs)

    def __getattr__(self, service):
        """

        :param service: Name of service
        :return: TVMService
        """
        return TVMService(self, service)


class TVMService(object):
    """
    Class for making TVM requests to service
    """
    def __init__(self, tvm, service):
        """

        :param tvm: TVM Class
        :param service: Name of service
        """
        if not isinstance(tvm, TVM):
            raise TypeError('Expecting {} object, got {}'.format(TVM, type(tvm)))

        self._tvm = tvm
        self._service = service

    def _process_requests_method(self, method, *args, **kwargs):
        """
        Make request to service with TVM ticket
        :param method: requests method to use
        :param args: args for requests method
        :param kwargs: kwargs for requests method
        :return:
        """
        return self._tvm.process_requests_method(self._service, method, *args, **kwargs)

    def __getattr__(self, method):
        def _process_requests_method(*args, **kwargs):
            return self._process_requests_method(method, *args, **kwargs)

        return _process_requests_method
