import io
import yaml
import unicodecsv
import json
import os
import platform
import errno

from six import PY3

try:
    from backports.tempfile import TemporaryDirectory
except ImportError:
    from tempfile import TemporaryDirectory

from library.python import resource


def _to_bytes_py2(str_or_unicode):
    if type(str_or_unicode) == str:
        return str_or_unicode
    else:
        return str_or_unicode.encode('utf-8')


def _to_bytes_py3(str_or_bytes):
    if type(str_or_bytes) == str:
        return str_or_bytes.encode('utf-8')
    else:
        return str_or_bytes


def _to_bytes(str_or_unicode_or_bytes):
    if PY3:
        return _to_bytes_py3(str_or_unicode_or_bytes)
    return _to_bytes_py2(str_or_unicode_or_bytes)


def open_resource_or_file(key_or_path, binary=False, encoding=None):
    resource_data = resource.find(key_or_path)
    if resource_data:
        resource_data = io.BytesIO(resource_data)
        if binary:
            return resource_data
        else:
            return io.TextIOWrapper(resource_data, encoding=encoding)
    else:
        return io.open(key_or_path, 'rb' if binary else 'r', encoding=encoding)


def to_json(path, objects):
    with io.open(path, 'wb') as f:
        json_str = json.dumps(objects, ensure_ascii=False, sort_keys=True)
        f.write(_to_bytes(json_str))


def to_yml(path, objects):
    with io.open(path, 'w', encoding='utf-8') as f:
        yaml.safe_dump(objects, f, default_flow_style=False, allow_unicode=True)


def _to_separated_values(path, fieldnames, objects, delimiter=',', **kwargs):
    with io.open(path, 'wb') as f:
        writer = unicodecsv.DictWriter(f, fieldnames=fieldnames, delimiter=delimiter, encoding='utf-8', **kwargs)
        writer.writeheader()
        for row in objects:
            writer.writerow(row)


def to_csv(path, fieldnames, objects, **kwargs):
    _to_separated_values(path, fieldnames, objects, delimiter=',', **kwargs)


def to_tsv(path, fieldnames, objects, **kwargs):
    _to_separated_values(path, fieldnames, objects, delimiter='\t', **kwargs)


def from_json(path):
    with open_resource_or_file(path, False, encoding='utf-8') as f:
        return json.loads(f.read())


def from_yml(path):
    with open_resource_or_file(path, False, encoding='utf-8') as f:
        return yaml.safe_load(f)


def _from_separated_values(path, delimiter=',', **kwargs):
    with open_resource_or_file(path, True) as f:
        return list(unicodecsv.DictReader(f, delimiter=delimiter, **kwargs))


def from_csv(path, **kwargs):
    return _from_separated_values(path, delimiter=',', **kwargs)


def from_tsv(path, **kwargs):
    return _from_separated_values(path, delimiter='\t', **kwargs)


def get_system_config_file(app_name, config_filename):
    if platform.system() in ['Linux', 'Darwin']:
        return os.path.join('/etc', app_name, config_filename)
    elif platform.system() == 'Windows':
        if 'PROGRAMDATA' in os.environ:
            return os.path.join(os.path.expandvars('%PROGRAMDATA%'), app_name, config_filename)
        elif 'ALLUSERSPROFILE' in os.environ:
            return os.path.join(os.path.expandvars('%ALLUSERSPROFILE%'), app_name, config_filename)
        else:
            raise RuntimeError('Running script on unknown Windows version')
    else:
        raise RuntimeError('Running script on unknown platform')


def get_user_app_path(app_name):
    if platform.system() in ['Linux', 'Darwin']:
        return os.path.join(os.path.expanduser("~"), '.' + app_name)
    elif platform.system() == 'Windows':
        return os.path.join(os.path.expandvars('%APPDATA%'), app_name)
    else:
        raise RuntimeError('Running script on unknown platform')


def get_user_config_file(app_name, config_filename):
    if platform.system() in ['Linux', 'Darwin']:
        return os.path.join(os.path.expanduser("~"), '.' + app_name, config_filename)
    elif platform.system() == 'Windows':
        return os.path.join(os.path.expandvars('%APPDATA%'), app_name, config_filename)
    else:
        raise RuntimeError('Running script on unknown platform')


def read_env_or_file(env_name=None, env_path_or_path=None, env_path=None, path=None, default_value=None):
    if env_name is not None and env_name in os.environ:
        return os.environ.get(env_name)

    if env_path_or_path is not None:
        if env_path is None:
            env_path = env_path_or_path
        if path is None:
            path = env_path_or_path

    if env_path is not None and env_path in os.environ:
        env_path = os.environ.get(env_path)
        if os.path.isfile(env_path):
            with open(env_path) as f:
                return f.read().strip()

    if path is not None and os.path.isfile(path):
        with open(path) as f:
            return f.read().strip()

    return default_value


def makedirs(folder, *args, **kwargs):
    try:
        return os.makedirs(folder, exist_ok=True, *args, **kwargs)
    except TypeError:
        # Unexpected arguments encountered
        pass

    try:
        # Should work is TypeError was caused by exist_ok, eg., Py2
        return os.makedirs(folder, *args, **kwargs)
    except OSError as e:
        if e.errno != errno.EEXIST:
            raise

        if os.path.isfile(folder):
            # folder is a file, raise OSError just like os.makedirs() in Py3
            raise


__all__ = ['to_json', 'to_yml', 'to_csv', 'to_tsv',
           'from_json', 'from_yml', 'from_csv', 'from_tsv',
           'get_system_config_file', 'get_user_config_file', 'get_user_app_path',
           'read_env_or_file', 'open_resource_or_file',
           'makedirs', 'TemporaryDirectory']
