import re
from tqdm import tqdm
import requests
from six.moves.urllib.parse import urlparse


URI_REGEXP = r'^sbr://\d+$'

RESOURCE_DOWNLOAD_URL = 'https://proxy.sandbox.yandex-team.ru/{resource_id}'
RESOURCE_LATEST_URL = 'http://proxy.sandbox.yandex-team.ru/last/{name}'


def validate_uri(uri):
    return re.match(URI_REGEXP, uri) is not None


def download_latest(resource_name, path=None):
    resource_latest_url = RESOURCE_LATEST_URL.format(name=resource_name)
    return download_file(resource_latest_url, path)


def download(uri, path=None, tgz=False):
    if not validate_uri(uri):
        raise ValueError('Wrong uri')

    url = RESOURCE_DOWNLOAD_URL.format(resource_id=uri[6:])

    if tgz:
        url += '?stream=tgz'

    return download_file(url, path)


def download_file(url, path=None):
    """
    @param: url to download file
    @param: path place to put the file (default: from header)
    """
    if path is None:
        url_parts = urlparse(url)
        path_parts = url_parts[2].rpartition('/')
        path = path_parts[2]

    req = requests.get(url, stream=True)
    total_size = int(req.headers.get('content-length', 0))
    block_size = 1024
    pbar = tqdm(total=total_size, unit='iB', unit_scale=True)
    with open(path, 'wb') as f:
        for data in req.iter_content(block_size):
            pbar.update(len(data))
            f.write(data)
    pbar.close()
    if total_size != 0 and pbar.n != total_size:
        return False
    return True


__all__ = ['download_latest', 'download', 'download_file']
