# coding=utf-8
import datetime
import itertools
import logging

import matplotlib.pyplot as plt

import tempfile
from threading import Thread

from telegram import ParseMode, InputMediaPhoto
from telegram.ext import Updater, CommandHandler, MessageHandler

import ylock
from startrek_client import Startrek

from yabs_filter_record import decode as decode_count_link

import irt.utils
import irt.common.abc_adapter as abc_adapter
import irt.common.staff_adapter as staff_adapter

from irt.artmon import ArtmonAPI
from irt.multik.liveness_server import LivenessServer

DYN_CONTEXT_TYPE = '7'
SMART_CONTEXT_TYPE = '8'

LABELS = {
    DYN_CONTEXT_TYPE: 'Динамики',
    SMART_CONTEXT_TYPE: 'Смарты',
}

HOUR = 'hour'
DAY = 'day'
WEEK = 'week'

LOCK_NAME = 'irt_tlgrm_bot_{}'


COUNT_LINK_FIELDS = ('BannerID',
                     'OrderID',
                     'ParentBannerID',
                     'ParentExportID',
                     'ContextType',
                     'BMCategoryID',
                     'CTR',
                     'Cost',
                     'CostCur')


class IrtBot(object):
    """
IrtBot for IRT group
Version: {}
Contour: {}
Commands:
/help - Show this help
/artmon - Show artmon stats for dyn and smart. Please Use `/artmon dyn` and `/artmon smart` for showing only for selected product
/ticket_QUEUENAME ID - Get ticket data from QUEUENAME. Supported queues: IRT, DYNSMART, SUPBL, CATSUP, MULTIK, COVID
/new_ticket_QUEUENAME TITLE - Create ticket in QUEUENAME with title. If it will be in answer to another message, text of message will be placed into ticket description
/count Decode count-link
"""

    def __init__(self, tlgrm_token, artmon_token, yt_token, yt_account, st_token, queues=tuple(), abc_id=None, contour=False, liveness_port=5000):
        self._contour = contour

        self._artmon_api = ArtmonAPI(artmon_token)

        self._abc_adapter = abc_adapter.ABCAdapter(artmon_token)  # Temporary hack
        self._abc = abc_id
        self._staff_adapter = staff_adapter.StaffAdapter(artmon_token)  # Temporary hack

        self._tlgrm_updater = Updater(tlgrm_token, use_context=False)
        self._st_client = Startrek(useragent='irt-bot', token=st_token)

        dp = self._tlgrm_updater.dispatcher

        dp.add_handler(self._handler('help', self.help))
        dp.add_handler(self._handler('count', self.count_link))
        dp.add_handler(self._handler('artmon', self.artmon_stat))

        for q in queues:
            dp.add_handler(self._handler(self._create_cmd(['new', 'ticket', q]), self.new_ticket(q)))
            dp.add_handler(self._handler(self._create_cmd(['ticket', q]), self.ticket(q)))

        dp.add_handler(MessageHandler(None, self._all_messages))

        self._ylock_manager = ylock.create_manager(backend='yt',
                                                   prefix='//home/{}'.format(yt_account),
                                                   token=yt_token)

        self._liveness = LivenessServer(self._active, port=liveness_port)
        Thread(target=self._liveness.run).start()
        self._ready = False
        self._last_tickets = {}

    def _all_messages(self, bot, update):
        ticket_id, ts = self._last_tickets.get((update.message.chat.id, update.message.from_user.id), (None, None))

        if ts is not None and update.message.date - ts < datetime.timedelta(seconds=1):
            if update.message.text is not None:
                logging.info('Will add "%s" to ticket', update.message.text, ticket_id)
                ticket = self._st_client.issues[ticket_id]
                ticket.update(description='{}\n\n{}'.format(ticket.description or '', update.message.text))
        else:
            self._last_tickets.pop((update.message.chat.id, update.message.from_user.id), None)

    def help(self, bot, update, args, username=None):
        version = '{}-{}'.format(irt.utils.svn_branch(), irt.utils.svn_tag())
        update.message.reply_text(type(self).__doc__.format(version, self._contour))

    def _handler(self, cmd, handler):
        return CommandHandler(cmd, self._handler_wrapper(handler), pass_args=True)

    @classmethod
    def _create_cmd(cls, words):
        for w in range(pow(2, len(words) - 1)):
            str_format = '0{{:0{}b}}'.format(len(words) - 1)
            separators = ['_' if x == '1' else '' for x in str_format.format(w)]
            yield ''.join(x + y for x, y in zip(separators, words))

    def _active(self):
        return self._ready

    def loop(self, lock):
        self._ready = True
        if not lock:
            self._tlgrm_updater.start_polling()
            self._tlgrm_updater.idle()
            self._liveness.shutdown()
        else:
            with self._ylock_manager.lock(LOCK_NAME.format(self._contour), block=True, block_timeout=float('inf')):
                self.loop(not lock)
        self._ready = False

    def _handler_wrapper(self, handler):
        def _add_check_staff(bot, update, *args, **kwargs):
            user = self._check_username(update.message.from_user.username)
            if user is not None and self._check_abc(user['login']):
                handler(bot, update, username=user['login'], *args, **kwargs)
            else:
                update.message.reply_text('Ошибка. Проверьте свой телеграмм на стаффе')

        return _add_check_staff

    def _check_username(self, tlgrm):
        return self._staff_adapter.get_user_by_telegram(tlgrm)

    def _check_abc(self, username):
        if self._abc is None:
            return True
        for member in self._abc_adapter.get_members(self._abc):
            if member['username'] == username:
                return True
        return False

    def _artmon_request(self, context_types, first_period, second_period, period_duration, group_by):
        logging.info('Requesting artmon context {} dates: {} - {}'.format(context_types, first_period, second_period))
        return self._artmon_api.do_request(
            main_start=first_period,
            main_end=first_period + period_duration,
            cmp_start=second_period,
            cmp_end=second_period + period_duration,
            timegroup=group_by,
            add_filters={'Slice': 'context', 'group_context_type': context_types},
        )

    @classmethod
    def _extract_artmon_values(cls, rows, key, context_types):
        by_context_type = {ct: {} for ct in context_types}
        for row in rows:
            if key in row:
                series_id = row.get('series_id', '').split(',')
                if series_id[0] == 'bsclickhouse':
                    by_context_type[series_id[1]][row.get('utc')] = row.get(key)

        for ct in by_context_type:
            by_context_type[ct] = list(by_context_type[ct][utc] for utc in sorted(by_context_type[ct]))

        return by_context_type

    @classmethod
    def _get_artmon_now_prev(cls, data, key, context_types):
        by_context_type_now = cls._extract_artmon_values(data['items']['rows'], key, context_types)
        by_context_type_prev = cls._extract_artmon_values(data['items']['compared'], key, context_types)

        statictics = {
            ct: {
                'prev': by_context_type_prev[ct],
                'now': by_context_type_now[ct],
                'sum_prev': itertools.accumulate(by_context_type_prev[ct]),
                'sum_now': itertools.accumulate(by_context_type_now[ct])
            } for ct in context_types
        }
        for ct in context_types:
            statictics[ct]['ratio'] = [100 * (x - y) / y for x, y in zip(statictics[ct]['sum_now'], statictics[ct]['sum_prev'])]

        return statictics

    def _get_artmon_stat_with_delta(self, context_types, first_period, second_period, period_duration, group_by, reply_text, send_media_group, msg):
        data = self._artmon_request(context_types, first_period, second_period, period_duration, group_by)
        cost = self._get_artmon_now_prev(data, 'cost', context_types)
        clicks = self._get_artmon_now_prev(data, 'clicks', context_types)
        shows = self._get_artmon_now_prev(data, 'shows', context_types)

        if group_by == HOUR:
            td_delta = datetime.timedelta(hours=1)
        elif group_by == DAY:
            td_delta = datetime.timedelta(days=1)
        elif group_by == WEEK:
            td_delta = datetime.timedelta(weeks=1)
        else:
            return

        for ct in context_types:
            if reply_text is not None:
                header = ['*{}*\n_{:%Y-%m-%d} {:%Y-%m-%d}_'.format(LABELS[ct], first_period, second_period),
                          '```',
                          '       Дата       |  Деньги  |  Клики   |  Показы',
                          '---------------- | -------- | -------- | --------']
                date = first_period
                message = []
                for ratio, ratio_clicks, ratio_shows in zip(cost[ct]['ratio'], clicks[ct]['ratio'], shows[ct]['ratio']):
                    date += td_delta
                    message.append('{:%Y-%m-%d %H:%M} | {:7.03f}% | {:7.03f}% | {:7.03f}%'.format(date, ratio, ratio_clicks, ratio_shows))
                message.append('```')
                if len(message) < 30:
                    msg = reply_text('\n'.join(header + message), parse_mode=ParseMode.MARKDOWN)

            with tempfile.NamedTemporaryFile(suffix='.png') as shows_file, \
                tempfile.NamedTemporaryFile(suffix='.png') as clicks_file, \
                    tempfile.NamedTemporaryFile(suffix='.png') as cost_file:
                fig = self._plot(
                    list(range(len(cost[ct]['prev']))),
                    [cost[ct]['now'] + [0] * (len(cost[ct]['prev']) - len(cost[ct]['now'])), cost[ct]['prev']],
                    'Деньги {} {:.03f}%'.format(LABELS[ct], cost[ct]['ratio'][-2] if len(cost[ct]['ratio']) > 1 else 0)
                )
                fig.savefig(cost_file.name)

                fig = self._plot(
                    list(range(len(clicks[ct]['prev']))),
                    [clicks[ct]['now'] + [0] * (len(clicks[ct]['prev']) - len(clicks[ct]['now'])), clicks[ct]['prev']],
                    'Клики {} {:.03f}%'.format(LABELS[ct], clicks[ct]['ratio'][-2] if len(clicks[ct]['ratio']) > 1 else 0)
                )
                fig.savefig(clicks_file.name)

                fig = self._plot(
                    list(range(len(shows[ct]['prev']))),
                    [shows[ct]['now'] + [0] * (len(shows[ct]['prev']) - len(shows[ct]['now'])), shows[ct]['prev']],
                    'Показы {} {:.03f}%'.format(LABELS[ct], shows[ct]['ratio'][-2] if len(shows[ct]['ratio']) > 1 else 0)
                )
                fig.savefig(shows_file.name)

                send_media_group(msg.chat_id, [
                    InputMediaPhoto(open(cost_file.name, 'rb')),
                    InputMediaPhoto(open(shows_file.name, 'rb')),
                    InputMediaPhoto(open(clicks_file.name, 'rb'))
                ], reply_to_message_id=msg.message_id)

    def count_link(self, bot, update, args, username=None):
        if len(args) != 1:
            update.message.reply_text('Нужно прислать count ссылку')

        count_link = args[0]
        decoded = decode_count_link(count_link.encode('utf-8')).decode('utf-8')
        data = dict()
        for key_val in decoded.rstrip('/').split(','):
            splitted = key_val.split('=')
            if len(splitted) == 2:
                data[splitted[0]] = splitted[1]

        update.message.reply_text('\n'.join('{}={}'.format(k, data[k]) for k in COUNT_LINK_FIELDS))

    def artmon_stat(self, bot, update, args, username=None):
        today = datetime.date.today()
        today = datetime.datetime.combine(today, datetime.datetime.min.time())

        logging.info('User {} request artmon stats'.format(update.message.from_user.username))

        if 'dyn' in args:
            context_types = [DYN_CONTEXT_TYPE]
        elif 'smart' in args:
            context_types = [SMART_CONTEXT_TYPE]
        else:
            context_types = [DYN_CONTEXT_TYPE, SMART_CONTEXT_TYPE]

        if 'вчера' in args:
            today -= datetime.timedelta(days=1)

        if 'year' in args:
            period = datetime.timedelta(days=363)
            group_by = WEEK
            first_period = today - datetime.timedelta(days=364)
        elif 'week' in args:
            period = datetime.timedelta(days=6)
            group_by = HOUR
            first_period = today - datetime.timedelta(days=7)
        elif 'month' in args:
            period = datetime.timedelta(days=27)
            group_by = DAY
            first_period = today - datetime.timedelta(days=28)
        else:
            period = datetime.timedelta(days=0)
            group_by = HOUR
            first_period = today

        if '-2' in args:
            prev_period = first_period - datetime.timedelta(days=14)
        elif '-month' in args:
            prev_period = first_period - datetime.timedelta(days=28)
        elif '-year' in args:
            prev_period = first_period - datetime.timedelta(days=364)
        else:
            prev_period = first_period - datetime.timedelta(days=7)

        self._get_artmon_stat_with_delta(
            context_types,
            first_period,
            prev_period,
            period,
            group_by,
            update.message.reply_text if 'text' in args else None,
            bot.send_media_group,
            update.message
        )

    def _plot(self, x, y, title):
        fig = plt.figure()
        ax = fig.add_subplot(111)
        ax.set_title(title)
        plots = []
        for part in y:
            plots.append(x)
            plots.append(part)
        ax.plot(*plots)
        return fig

    def new_ticket(self, q):
        def _new_ticket(bot, update, args, username=None):
            reply_to = update.message.reply_to_message
            description = reply_to.text if reply_to else None
            issue = self._st_client.issues.create(createdBy=username, queue=q, summary=' '.join(args), description=description)
            update.message.reply_text('https://st.yandex-team.ru/{}'.format(issue.key))
            self._last_tickets[(update.message.chat.id, update.message.from_user.id)] = issue.key, update.message.date

        return _new_ticket

    def ticket(self, q):
        def _ticket(bot, update, args, username=None):
            try:
                ticket = self._st_client.issues['{}-{}'.format(q, args[0])]
                assignee = ticket.assignee.login if ticket.assignee is not None else 'Не назначено'
                msg_text = '''
*{}*
_Статус: {}_
_Автор: {}_
_Исполнитель: {}_
```
{}
```
https://st.yandex-team.ru/{}
'''.format(ticket.summary, ticket.status.name, ticket.createdBy.login, assignee, ticket.description or '', ticket.key)
            except Exception as e:
                msg_text = str(e)
            update.message.reply_text(msg_text, parse_mode=ParseMode.MARKDOWN)

        return _ticket
