from __future__ import print_function


from irt.utils import from_yml, suppress


from itertools import chain
import time
from subprocess import Popen, PIPE


def call(args, env=None):
    p = Popen(args, env=env, stdin=PIPE, stdout=PIPE, stderr=PIPE)
    output, err = p.communicate()
    return output.strip()


seconds_per_unit = {"s": 1, "m": 60, "h": 3600, "d": 86400, "w": 604800}


def convert_to_seconds(s):
    return int(s[:-1]) * seconds_per_unit[s[-1]]


class Page:
    def __init__(self, url, modules=None, variables=None, scroll=None, refresh_every=None, reload_every=None, active_time='1m', refresh_actions=None, reload_actions=None):
        self._globals = {}
        self._locals = {}
        self._url = url
        self._variables = variables or {}
        self._scroll = scroll or []
        self._modules = modules or []
        self._refresh_every = convert_to_seconds(refresh_every) if refresh_every else None
        self._reload_every = convert_to_seconds(reload_every) if reload_every else None

        self._active_time = convert_to_seconds(active_time)
        self._last_refresh = 0
        self._last_reload = 0
        self.refresh_actions = refresh_actions or []
        self.reload_actions = reload_actions or []

        for module in self._modules:
            self._globals[module] = __import__(module, self._globals, self._locals, [], 0)

    @property
    def url(self):
        for var in self._variables:
            self._locals[var] = eval(self._variables[var], self._globals, self._locals)
        with suppress(SyntaxError):
            return eval(self._url, self._globals, self._locals)
        return self._url

    def loop(self):
        time.sleep(self._active_time)

    @property
    def need_refresh(self):
        return self._refresh_every and time.time() - self._last_refresh > self._refresh_every

    def refresh(self):
        self._last_refresh = time.time()

    @property
    def need_reload(self):
        return self._reload_every and time.time() - self._last_reload > self._reload_every

    def reload(self):
        self._last_reload = time.time()
        self.refresh()
        return self.url


class Pages:
    def __init__(self, iterable):
        self._pages = [Page(**page) for page in iterable]
        if len(self._pages) == 0:
            raise RuntimeError('Where is no urls to show')
        self.current = 0

    @property
    def urls(self):
        return iter(p.url for p in self._pages)

    def loop(self):
        while True:
            p = self._pages[self.current]
            p.loop()
            self.current += 1
            self.current = self.current % len(self._pages)
            yield self._pages[self.current]


class Firefox:
    def __init__(self, config_path):
        config = from_yml(config_path)
        self._display = config['display']
        self._pages = Pages(config['pages'])

    def open(self):
        args = ['firefox'] + list(chain.from_iterable(['-new-tab', '-url', url] for url in self._pages.urls))
        print(args)
        Popen(args, stdin=None, stdout=None, stderr=None, close_fds=True, env={'DISPLAY': self._display})

    def find_window(self, timeout=None):
        print('Finding window...')
        start_time = time.time()
        while True:
            window = call(['xdotool', 'search', '--sync', '--onlyvisible', '--class', 'Firefox'], env={'DISPLAY': self._display})
            if window:
                window = str(int(window))
                print('Window found with id', window)
                return window
            if timeout and time.time() - start_time > timeout:
                raise RuntimeError('Timeout on searching window')
            time.sleep(100)

    def type(self, text):
        call(['xdotool', 'type', text], env={'DISPLAY': self._display})

    def press_key(self, key, window_id=None):
        if window_id:
            call(['xdotool', 'key', '--window', window_id, key], env={'DISPLAY': self._display})
        else:
            call(['xdotool', 'key', key], env={'DISPLAY': self._display})

    def enter_fullscreen(self):
        self.press_key('F11', window_id=self.find_window())

    def next_tab(self):
        self.press_key('Control_L+Tab')

    def execute_actions(self, actions):
        for action in actions:
            if action['type'] == 'key':
                print('Pressing key:', action['value'])
                self.press_key(action['value'])
            elif action['type'] == 'sleep':
                print('Sleeping for {}s'.format(action['value']))
                time.sleep(action['value'])
            else:
                raise RuntimeError('Wrong refresh action')

    def loop(self):
        for p in self._pages.loop():
            print('Go to next tab')
            self.next_tab()
            time.sleep(0.1)

            if p.need_reload:
                url = p.reload()
                print('Need to reload tab, new url:', url)
                self.press_key('Control_L+l')
                time.sleep(0.1)
                self.type(url)
                time.sleep(0.1)
                self.press_key('Return')
                self.execute_actions(p.reload_actions)
            elif p.need_refresh:
                p.refresh()
                print('Need to refresh tab')
                self.press_key('F5')
                self.execute_actions(p.refresh_actions)


print('Reading config...')
ff = Firefox('config.yml')
print('Opening FF...')
ff.open()
print('Enter fullscreen mode...')
ff.enter_fullscreen()
print('Start loop...')
ff.loop()
