import filelock
import logging
import six
import timeout_decorator

import irt.common.logging
import irt.bannerland.hosts
import irt.monitoring.atoms.config
import irt.monitoring.solomon.sensors


# Таймаут каждого атома по-умолчанию в секундах
ATOM_TIMEOUT = 5

logging.basicConfig(format="%(asctime)s\t[%(process)d]\t%(message)s", level=logging.INFO)
logger = logging.getLogger(__name__)


# Класс-заглушка для ловли исключений при неуспешной отправке в Соломон
class SendToSolomonException(Exception):
    pass


def atom_runner(atom_func, atom_args, timeout):
    """
    Запускает заданный атом с учётом таймаута
    :param atom_func: функция атома
    :param atom_args: аргументы атома
    :param timeout: таймаут выполнения атома
    :return: результат работы атома, если он завершился
    """
    @timeout_decorator.timeout(timeout, timeout_exception=timeout_decorator.TimeoutError, use_signals=False)
    def _raw_run():
        return atom_func(**atom_args)
    return _raw_run()


def main():
    """
    Вычисляет все атомы, которые нужно вычислить на этой машинке и отправляет результаты вычислений в качестве сенсоров в Соломон.
    """
    irt.common.logging.init_logging()
    solomon_client = irt.monitoring.solomon.sensors.SolomonAgentSensorsClient()

    curr_host = irt.bannerland.hosts.get_curr_host()
    host_info = irt.bannerland.hosts.get_host_info(curr_host)
    if "idle" in host_info["role"]:
        return

    # Берем файловый лок во избежание параллельного запуска данного процесса
    host_roles = set([irt.bannerland.hosts.get_host_role()])
    if host_info.get("master") == 1:
        host_roles.add("{}:master".format(host_info["role"]))

    # Здесь в цикле начинаем считать атомы
    for atom_conf in irt.monitoring.atoms.config.get_atoms_configuration():

        # Сначала проверяем, что данный атом нужно считать на этом хосте
        include_my_host = ("include_host_roles" not in atom_conf) and ("include_host_fqdns" not in atom_conf)

        include_my_host = include_my_host or \
                          (len(set(atom_conf.get("include_host_roles", [])) & host_roles) > 0) or \
                          (curr_host in atom_conf.get("include_host_fqdns", []))

        include_my_host = include_my_host and \
                          (len(set(atom_conf.get("exclude_host_roles", [])) & host_roles) == 0) and \
                          (curr_host not in atom_conf.get("exclude_host_fqdns", []))

        if not include_my_host:
            continue

        # Обрабатываем конфигурацию конкретного атома
        try:
            atom_func = atom_conf["atom"]
            atom_args = atom_conf["atom_args"]

            # Считаем атом
            logger.error("Calculating atom '{}' with arguments '{}' ...".format(atom_func.__name__, atom_args))
            atom_res = atom_runner(atom_func, atom_args, atom_conf.get("timeout", ATOM_TIMEOUT))

            # Численные атомы сразу отправляем в Соломон
            solomon_sensor = atom_conf.get("solomon_sensor_params")
            if solomon_sensor is not None:
                if not isinstance(atom_res, six.integer_types + (float,)):
                    raise TypeError("Invalid type of the atom result")
                push_res = solomon_client.push_single_sensor(
                    cluster=solomon_sensor["cluster"],
                    service=solomon_sensor["service"],
                    sensor=solomon_sensor["sensor"],
                    value=atom_res,
                    labels=solomon_sensor.get("labels"),
                )
                if push_res != 0:
                    raise SendToSolomonException("Can't send atom result to Solomon")

            # Если результат работы атомы требует дополнительной постобработки, производим её.
            # После этого отправляем полученный массив сенсоров в Соломон
            elif "special_handler" in atom_conf:
                solomon_sensors = atom_conf["special_handler"](**{"atom_res": atom_res})
                push_res = solomon_client.push_sensors(solomon_sensors)
                if push_res != 0:
                    raise SendToSolomonException("Can't send atom result to Solomon")
            else:
                raise ValueError("Invalid atom configuration (can't handle the atom result)")

            logger.error("Success!")
        except (TypeError, ValueError, SendToSolomonException) as err:
            logger.error("ERROR: Calculating of the atom was failed: '%s'", err)
        except timeout_decorator.TimeoutError as err:
            logger.error("ERROR: Atom timeout exception! %s", err)
        except Exception as err:
            logger.exception("ERROR: Unknown error: '%s'", err)


if __name__ == "__main__":
    try:
        with filelock.FileLock('calculate_atoms.lock').acquire(timeout=0):
            main()
    except filelock.Timeout:
        logger.warning('Another instance of this application currently holds the lock.')
